package freenet.client.cli;
import freenet.client.*;
import freenet.client.events.*;
import freenet.client.listeners.EventLogger;
import freenet.config.*;
import freenet.support.*;
import freenet.support.io.ReadInputStream;
import java.io.*;
import java.net.BindException;
import java.util.Stack;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Date;
import java.lang.reflect.InvocationTargetException;
import freenet.crypt.*;
import freenet.support.Fields;
import freenet.support.Loader;
import freenet.support.FileBucket;
import freenet.support.ArrayBucket;
/**
 * This class contains methods for reading options in CLI clients.
 *
 * @author oskar (Mostly ripped out of old the Freenet clients though.)
 **/

public class CLI {

    /** Configuration defaults **/
    static public final Config options = new Config();

    static {

        options.addOption("logLevel",             1, "normal", 5001    );     
        options.addOption("logFormat",            1, "m", 5002  );
        options.addOption("logDate",              1, "",  5003     );
        options.addOption("version",              0, null, 10 );
        options.addOption("help",            'h', 0, null, 20 ); 
        options.addOption("htl",                  1, 10, 100);
        options.addOption("clientFactory",   'c', 1, 
                           "freenet.client.cli.CLIFCPClient", 5100);   
        options.addOption("cipher",               1, "Rijndael", 5200);
        options.addOption("metadata",        'm', 1, "", 5300);
        options.addOption("noredirect",        0, null, 500);
        options.addOption("manual",           0, null, 30);
        // options.addOption("contentType",     1, "", 4500); // see below 
        options.addOption("dontGuessType",        0, null, 4600);
 
        // logLevel
        options.argDesc   ("logLevel", "<word>");
        options.shortDesc ("logLevel", "error, normal, minor, or debug");
        options.longDesc  ("logLevel",
            "The error reporting threshold, one of:",
            "  Error:   Errors only",
            "  Normal:  Report significant events",
            "  Minor:   Report minor events",
            "  Debug:   Report events only of relevance when debugging"
        );
        
        // logFormat
        options.setExpert ("logFormat", true);
        options.argDesc   ("logFormat", "<tmpl.>");
        options.shortDesc ("logFormat", "template, like d:c:h:t:p:m");
        options.longDesc  ("logFormat",
            "A template string for log messages.  All non-alphabet characters are",
            "reproduced verbatim.  Alphabet characters are substituted as follows:",
            "d = date (timestamp), c = class name of the source object,",
            "h = hashcode of the object, t = thread name, p = priority,",
            "m = the actual log message"
        );

        // logDate
        options.setExpert ("logDate", true);
        options.argDesc   ("logDate", "<tmpl.>");
        options.shortDesc ("logDate", "java style date/time template");
        options.longDesc  ("logDate",
            "A template for formatting the timestamp in log messages.  Defaults to",
            "the locale specific fully specified date format.  The template string",
            "is an ordinary java date/time template - see:",
            "http://java.sun.com/products/jdk/1.1/docs/api/java.text.SimpleDateFormat.html"
        );

        options.shortDesc("version","Print version information.");

        options.shortDesc("help","Print usage information.");

        options.argDesc("htl","<integer>");
        options.shortDesc("htl","The \"hops to live\" value to use");
        options.longDesc("htl",
                          "The number of nodes that the request should route through.",
                          "Greater HTL values may help find data, but the effect is limited.");
                       
        options.argDesc("clientFactory","<class name>");
        options.shortDesc("clientFactory","The type of client to use.");
        options.longDesc("clientFactory",
                          "The client can use a number of different pluggable client factories to",
                          "interface with nodes different manners. This includes FCP (Freenet client",
                          "protocol), FNP (Freenet node protocol), various kinds of RPC, and even ",
                          "directly interfacing a node instance.");

        options.argDesc("cipher","<name>");
        options.shortDesc("cipher","The default cipher to use for data.");

        options.argDesc("metadata","<filename>");
        options.shortDesc("metadata","File for metadata");

        options.shortDesc("noredirect","If set, won't follow or make redirects.");

        options.shortDesc("manual", "Generate an HTML manual");
        options.longDesc("manual", 
                         "Automatically generates a manual for the client in HTML. Note that manuals",
                         "are clientFactory specific, see \"clientFactory\" below.");

        /*
          There is no need for this. The only time it was useful can be
          better solved by giving the metadata manually
        options.argDesc("contentType","<mime type>");
        options.shortDesc("contentType","MIME Content-type to use as default");
        options.longDesc("contentType",
                         "The MIME Content-type to give new data during a \"put\" command and to",
                         "use as default otherwise");
        */

        options.shortDesc("dontGuessType","don't get MIME type from file ext");
        options.longDesc("dontGuessType",
                         "Disables the client from guessing the mime type of data that is retrieved",
                         "or inserted based on the file extension.");
    }

    /** version string **/
    static protected final String clientVersion = "2.002";
    /** default port to listen on **/
    static protected final int defaultListenPort = 0;
    /** default address of target node **/
    static protected final String defaultServerAddress = "tcp/127.0.0.1:19114";
    /** default hops to live to give request **/
    static protected final int defaultHopsToLive = 10;
    /** Default logging threshold **/
    static protected final String defaultLogging = "NORMAL";
    /** Default logging verbosity **/
    static protected final String defaultLogFormat = "m";
    /** Default cipher to use for encryption **/
    static protected final String defaultCipherName = "Twofish";

     /** The exit status to of the client **/
    public static int exitState = 0;
    /** Options */
    protected Params params;
    protected Logger logger;
    /** The standard input stream (rather than System.in) */
    protected InputStream in;
    /** The printwriter to use for output (rather than System.out) **/
    protected PrintStream out;
    /** The printwriter to use for error output (rather than System.err) **/
    protected PrintStream err;
    /** The client to use */
    protected CLIClientFactory clientFactory;

    /* List of available commands */
    private Hashtable commands;


    /**
     * Creates a new CLI.
     * @param params  An object containing the settings.
     * @exception  BindException is thrown if the client cannot listen on
     *             the given port.
     * @exception  BadAddressException is thrown if the address string cannot
     *             be parsed.
     * @exception  CLIException on failure..
     */
    public CLI(Params params) throws CLIException {
	this(params, loadLogger(params), System.in, System.out, System.err);
    }

    /**
     * Creates a new CLI with more control over logging.  This is just a 
     * temporary hack to minimize disruption; really we should make logging
     * a bit more configurable.
     *
     * @param params  An object containing the settings.
     * @param log     A Logger object.  If this is null, a new one will
     *                be created.
     * @exception  BindException is thrown if the client cannot listen on
     *             the given port.
     * @exception  BadAddressException is thrown if the address string cannot
     *             be parsed.
     * @exception  CLIException is thrown on failure.
     */
    public CLI(Params params, Logger log) throws CLIException {
	this(params, log, System.in, System.out, System.err);
    }


    /**
     * Creates a new CLI with more control over logging.  This is just a 
     * temporary hack to minimize disruption; really we should make logging
     * a bit more configurable.
     *
     * @param params  An object containing the settings.
     * @param log     A Logger object.  If this is null, a new one will
     *                be created.
     * @exception  BindException is thrown if the client cannot listen on
     *             the given port.
     * @exception  BadAddressException is thrown if the address string cannot
     *             be parsed.
     * @exception  CLIException is thrown on failure.
     **/
    public CLI(Params params, Logger log,
	       InputStream in, PrintStream out, PrintStream err)
	throws CLIException {
        
        this(params, loadClientFactory(params, log), log, in, out, err);
    }

    public CLI(Params params, CLIClientFactory cf, Logger log, InputStream in, 
               PrintStream out, PrintStream err) throws CLIException{
        try {
            this.params = params;

            params.addOptions(options.getOptions());
            //params.addOptions(Metadata.options.getOptions());

            this.in = in;
            this.out = out;
            this.err = err;

            logger = log;
            /* Tavin killed the bunny!
               if (params.getString("jump","no").equalsIgnoreCase("yes")) 
               jump();
            */
            
            commands = new Hashtable();
            addCommand(new GetCommand());
            addCommand(new PutCommand());
            addCommand(new SVKPairCommand());
            addCommand(new ComputeCHKCommand());
            addCommand(new PutSiteCommand());

            // process options
            //            htl = params.getInt("htl");
            //safer = !params.getString("safer","no").equalsIgnoreCase("no");

            this.clientFactory = cf;
            
        } catch (Throwable e) {
            if (clientFactory != null)
                clientFactory.stop();
            if (e instanceof CLIException) 
                throw (CLIException) e;
            else if (e instanceof RuntimeException)
                throw (RuntimeException) e;
            else 
                throw (Error) e;
        }
    }

    public static Logger loadLogger(Params params) {
        params.addOptions(options.getOptions());
        String logthresh = params.getString("logLevel");
        int thresh = Logger.priorityOf(logthresh);
        String format = params.getString("logFormat");
        String dformat = params.getString("logDate");
        Logger log = new Logger(thresh);
        log.addHook(new FileLoggerHook(System.err, format, dformat,
                                       thresh));
        return log;
    }

    public static CLIClientFactory loadClientFactory(Params params,
                                                     Logger logger) 
        throws CLIException {

        params.addOptions(options.getOptions());
        String cfactory = params.getString("clientFactory");
        try {
            Object o = 
                Loader.getInstance(cfactory,
                                   new Class[] {
                                       Params.class, Logger.class 
                                   },
                                   new Object[] {
                                       params, logger
                                   });
            if (!(o instanceof CLIClientFactory)) {
                throw new CLIException("Unsupported client:"
                                       + cfactory);
            }
            return (CLIClientFactory) o;
        } catch (InvocationTargetException e) {
            //            e.getTargetException().printStackTrace();
            Throwable t = e.getTargetException();
            if (t instanceof CLIException) {
                throw (CLIException) t;
            } else if (t instanceof RuntimeException) {
                throw (RuntimeException) t;
            } else if (t instanceof Error) {
                throw (Error) t;
            }
            throw new CLIException("Client " + cfactory 
                                   + " threw error:" + t.getMessage());
            
        } catch (NoSuchMethodException e) {
            throw new CLIException("Client " + cfactory 
                                   + " not supported");
        } catch (InstantiationException e) {
            throw new CLIException("Could not instantiate " +
                                   cfactory + " :" + e);
        } catch (IllegalAccessException e) {
            throw new CLIException("Access to " + cfactory 
                                   + " illegal.");
        } catch (ClassNotFoundException e) {
            throw new CLIException("No such client: " + cfactory);
        }
    }

    /**
     * Adds a command to the list available ones.
     */
    public void addCommand(ClientCommand cc) {
        commands.put(cc.getName() , cc);
    }

    public ClientCommand getCommand(String name) {
        return (ClientCommand) commands.get(name);
    }

    /**
     * Stops the client.
     */
    public void stop() {
        clientFactory.stop();
    }


    public boolean execute() {
        if (params.getParam("version") != null) {
            version();
            return false;
        } else if (params.getParam("help") != null) {
            usage();
            return true;
        } else if (params.getParam("manual") != null) {
            manual();
            return true;
        } else if (params.getNumArgs() < 1) {
            usage();
            return false;
        } else {
            ClientCommand cc = getCommand(params.getArg(0));
            if (cc == null) {
                err.println("Command time: " + cc + " not supported.");
                usage();
                return true;
            }
            //Bucket metadata = null;
            // bad. NPEs all ovah da playz
            Bucket metadata = new NullBucket();
            String mdf = params.getString("metadata");
            if (!"".equals(mdf)) 
                metadata = new FileBucket(new File(mdf));

            Bucket data;
            boolean stream;
            try {
                if (params.getNumArgs() > 1 + cc.argCount()) {
                    stream = false;
                    data = 
                        new FileBucket(new File(params.getArg(cc.argCount()
                                                              + 1)));
                } else {
                    stream = true;
                    data = new FileBucket();
                    if (cc.takesData()) {
                        OutputStream bout = data.getOutputStream();
                        byte[] b = new byte[0xffff];
                        int i;
                        while ((i = in.read(b)) != -1) {
                            bout.write(b, 0, i);
                        }
                    }
                }
            } catch (IOException e) {
                err.println("Error with data: " + e);
                return false;
            }

            RequestProcess rp;
            try {
                rp = cc.getProcess(params, metadata, data);
            } catch (CLIException e) {
                err.println(e.getMessage());
                return false;
            }

            Request r;
            EventLogger logl = new EventLogger(logger);
            while ((r = rp.getNextRequest()) != null) {
                if (!clientFactory.supportsRequest(r.getClass())) {
                    err.println("Current client cannot make request of type " 
                                + r);
                    rp.abort();
                    return false;
                }
                r.addEventListener(logl);
                try {
                    Client c = clientFactory.getClient(r);
                    c.start();
                } catch (IOException e) {
                    err.println("IO error: " + e);
                    return false;
                }
            }
            if (!rp.failed() && rp.getMetadata() != null) {
                err.println("Document metadata:");
                err.println(rp.getMetadata());
                if (metadata != null) {
                    try {
                        OutputStream out = metadata.getOutputStream();
                        rp.getMetadata().writeTo(err);
                        out.close();
                    } catch (IOException e) {
                        err.println("IO error writing metadata: " +
                                    e.getMessage());
                    }
                }                    
            } else if (rp.failed())
                err.println("Request failed.");

            if (!rp.failed() && cc.givesData() && stream) {
                try {
                    InputStream in = data.getInputStream();
                    int i;
                    byte[] b = new byte[0xffff];
                    while ((i = in.read(b)) != -1) {
                        out.write(b, 0, i);
                    }
                } catch (IOException e) {
                    err.println("IO error writing data: " + e.getMessage());
                    return false;
                }
            }

            return !rp.failed();
        }
        
    }


    /**
     * Print usage. By default prints options to out, subclasses should 
     * overried.
     */
    public void usage() {
        version();
        out.println("usage: cli <command> [file name] [options]...");
        out.println();
        out.println("Available Commands");
        out.println("------------------");
        for (Enumeration e = commands.elements() ; e.hasMoreElements() ;) {
            ClientCommand cc = (ClientCommand) e.nextElement();
            out.println(cc.getUsage());
        }
        out.println();
        printOptions(out);
    }

    /**
     * Prints the options in the manner the constructor of this class expects
     * then to the PrintWriter, with 80 character lines.
     * @param p  Where to write the options.
     **/
    public void printOptions(PrintStream p) {
        String s = "Standard options";
        StringBuffer l = new StringBuffer("----------------");
        p.println(s);
        p.println(l);
        options.printUsage(p);
        p.println();
        s = "Client: " + clientFactory.getDescription() + " options";
        for (l = new StringBuffer("-------") ; l.length() < s.length() ; 
             l.append('-')); 
        p.println(s);
        p.println(l);
        clientFactory.getOptions().printUsage(p);
    }

    /**
     * Prints version information about the client stdout.
     */
    public void version() {
        out.println(versionString());
    }

    public String versionString() {
	StringBuffer sb = new StringBuffer("cli version ");
        sb.append(clientVersion);
        sb.append(" (factory: ");
        String c = clientFactory.getDescription();
        if (sb.length() + c.length() + 1 > 80) {
            sb.append(System.getProperty("line.separator"));
        }
        sb.append(c);
        sb.append(')');
        return sb.toString();
    }

    public void manual() {
        out.println("<html><body>");
        out.println("<br><br>");
        out.println("<h2>Freenet Standard CLI Client Documentation</h2>");
        out.println("<h3>" + Config.htmlEnc(versionString()) + "</h3>");
        out.println("<br>");
        java.text.DateFormat df = java.text.DateFormat.getDateTimeInstance();
        out.println("<i>(This manual was automatically generated by the " + 
                    "--manual switch (see below) on " + 
                    Config.htmlEnc(df.format(new Date()))
                    + ". If you have updated Freenet since then, you " +
                    "may wish regenerate it.)</i>");
        out.println("<br><br>");
        out.println("freenet.client.cli.Main is a command line front end to "
                    + "the reference client libraries that come with Freenet. "
                    + "It is a rather powerful tool, with automatic metadata "
                    + "handling, and the ability to use arbitrary protocol "
                    + "backends.");
        out.println("<br><br>");
        out.println("See the <a href=\"http://www.freenetproject.org/"
                    + "index.php?page=documentation\"> project documentation" +
                    " pages</a> for more information, or ask pointed & " +
                    " specific questions on the <a href=\"" +
                    "http://www.freenetproject.org/index.php?page=lists\">" +
                    "mailing lists</a>.");
        out.println("<br><br>");
        out.println("<b>Usage: </b>java freenet.client.cli.Main &lt;command&gt; [file name] " 
                    + "[options]...");
        out.println("<br>");
        out.println("<h3>Commands:</h3>");
        out.println("<hr>");
        for (Enumeration e = commands.elements() ; e.hasMoreElements() ;) {
            ClientCommand cc = (ClientCommand) e.nextElement();
            out.println("<b>" + Config.htmlEnc(cc.getUsage()) + 
                        "</b><br><br>");
            String[] desc = cc.getDescription();
            for (int i = 0 ; i < desc.length ; i++)
                out.println(desc[i]);
            out.println("<br><hr>");
        }
        out.println("<br>");
        out.println("<h3>Standard Options:</h3>");
        out.println("<hr>");
        options.printManual(out);
        out.println("<br>");
        out.println("<h3>Client factory Options:</h3>");
        out.println("<hr>");
        clientFactory.getOptions().printManual(out);
        out.println("</body></html>");
    }
}





