/*
  This code is distributed under the GNU Public Licence (GPL)
  version 2.  See http://www.gnu.org/ for further details of the GPL.
*/

package freenet.client;

import freenet.Core;
import freenet.support.Logger;

import java.lang.reflect.Constructor;
import java.util.Hashtable;

/**
 * Factory class to dynamically load 
 * FECEncoder and FECDecoder implementations.
 * <p>
 * @author giannij
 **/
public class FECFactory {
    /**
     * Register and encoder implementation class.
     **/
    public boolean registerEncoder(String name, String className) {
        return register(name + "_encoder", className, FECEncoder.class);
    }

    /**
     * Register an decoder implementation class.
     **/
    public boolean registerDecoder(String name, String className) {
        return register(name + "_decoder", className, FECDecoder.class);
    }
    
    /**
     * Returns true if the specified endcoder or decoder name is
     * registered.
     **/
    public final boolean isRegistered(String decoderName, boolean encoder) {
        String suffix = "_decoder";
        if (encoder) {
            suffix = "_encoder";
        }
        return codecs.get(decoderName + suffix) != null;
    }

    /**
     * Creates a new instance of the requested encoder.
     * Returns null if the requested encoder can't be
     * created.
     **/
    public FECEncoder getEncoder(String name) {
        return (FECEncoder)makeInstance(name, "_encoder");
    }

    /**
     * Creates a new instance of the requested decoder.
     * Returns null if the requested encoder can't be
     * created.
     **/
    public FECDecoder getDecoder(String name) {
        return (FECDecoder)makeInstance(name, "_decoder");
    }

    ////////////////////////////////////////////////////////////

    private boolean register(String name, String className, Class mustAssign) {
        Class cls;
        try {
            cls = Class.forName(className.trim());
        }
        catch (ClassNotFoundException e) {
            Core.logger.log(this, "Couldn't load class: " + className + 
                            ".", 
                            Logger.ERROR); 

            return false;
        }
        
        if (!mustAssign.isAssignableFrom(cls)) {
            Core.logger.log(this, "Couldn't load class: " + className + 
                            " doesn't implement " + mustAssign.getName(), 
                            Logger.ERROR); 

            return false;
        }

        // find a default constructor
        Constructor[] constructors = cls.getConstructors();
        boolean found = false;
        int i;
        for (i=0; i<constructors.length; ++i) {
            Class[] types = constructors[i].getParameterTypes();
            if (types.length == 0) {
                found = true;
                break;
            }
        }
        
        if (!found) {
            Core.logger.log(this, "Couldn't load FECDecoder: " + name + 
                            ". No default constructor.", 
                            Logger.MINOR); 
           return false;
        }
        
        codecs.put(name,constructors[i]);
        return true;
    }

    private final Object makeInstance(String name, String suffix) {
        Constructor cons = (Constructor)codecs.get(name + suffix);
        if (cons == null) {
            return null;
        }

        Object ret = null;
        try {
            ret = cons.newInstance(new Object[0]);
        }
        catch (Throwable t) {
            // IMPORTANT: 
            // I want to make sure we don't crash on linkage 
            // errors.  This is especially important, 
            // because some decoder implementations may use JNI.
            // 
            Core.logger.log(this, "Couldn't make FEC codec instance: " + name + suffix +
                            ", " + t.toString(), 
                            Logger.ERROR); 

        }
        return ret;
    }

    private Hashtable codecs = new Hashtable();
}










