package freenet.client;

import freenet.*;
import freenet.client.events.*;
import freenet.client.listeners.CollectingEventListener;
import freenet.crypt.*;
import freenet.presentation.*;
import freenet.session.*;
import freenet.support.*;
import freenet.support.io.ReadInputStream;
import freenet.support.io.WriteOutputStream;
import java.io.*;
import java.net.*;
import java.math.BigInteger;
import java.util.Enumeration;

/** 
 * ClientFactory implementation for FCP
 *
 * @author <a href="mailto:rrkapitz@stud.informatik.uni-erlangen.de">Ruediger Kapitza</a>
 * @author <a href="mailto:giannijohansson@mediaone.net">Gianni Johansson</a>
 * @version 
 */
public class FCPClient implements ClientFactory {

    public static final int BUFFER_SIZE = 4096;

    public static final Presentation protocol = new ClientProtocol();

    protected Address target;


    /** 
     * Create a new ClientFactory for spawning Clients to process fcp-requests.
     * @param target  The fcp-service to send messages to.
     */    
    public FCPClient(Address target) {
        setTarget(target);
    }

    /** Requires setting of target manually.
      */
    protected FCPClient() {}
    
    protected void setTarget(Address target) {
        this.target = target;
    }


    ////////////////////////////////////////////////////////////
    // ClientFactory interface implementation
    public Client getClient(Request req) throws UnsupportedRequestException,
                                                KeyException, IOException {
        if (req instanceof GetRequest)
            return new FCPRequest((GetRequest) req);
        else if (req instanceof PutRequest)
            return new FCPInsert((PutRequest) req);
        else if (req instanceof ComputeCHKRequest)
            return new FCPComputeCHK((ComputeCHKRequest) req);
        else if (req instanceof ComputeSVKPairRequest)
            return new FCPComputeSVKPair((ComputeSVKPairRequest) req);
        else if (req instanceof HandshakeRequest)
            return new FCPHandshake((HandshakeRequest) req);
        else
            throw new UnsupportedRequestException();
    }

    public boolean supportsRequest(Class req) {
        return GetRequest.class.isAssignableFrom(req) ||
               PutRequest.class.isAssignableFrom(req) ||
               ComputeCHKRequest.class.isAssignableFrom(req) ||
               ComputeSVKPairRequest.class.isAssignableFrom(req) ||
               HandshakeRequest.class.isAssignableFrom(req);
    }    

    ////////////////////////////////////////////////////////////
    // Client implementations for various functions
    //
    private abstract class FCPInstance implements Client, Runnable {

        // FIXME -- FNP session support
        private final LinkManager linkManager = new PlainLinkManager();
        
        final Request req;
        final CollectingEventListener cel = new CollectingEventListener();
        
        Connection conn;
        InputStream in;
        OutputStream out;

        Thread workingThread;
        boolean started = false;
    

        FCPInstance(Request req) {
            this.req = req;
            req.addEventListener(cel);
        }

        final int state() {
            return req.state();
        }
        
        synchronized final StateReachedEvent state(int state) {
            req.state(state);
            return new StateReachedEvent(state);
        }

        // DON'T call with a lock on (this)
        final void unlockedProduceEvent(ClientEvent evt) {
            if (evt == null) {
                return;
            }
            if (req == null) {
                return;
            }
            req.produceEvent(evt);
        }

        final boolean tryState(int state) {
            ClientEvent evt = null;
            boolean ret = false;
            synchronized (FCPInstance.this) {
                if (state() != Request.CANCELLED) {
                    evt = state(state);
                    ret =  true;
                }
            }
            unlockedProduceEvent(evt);
            return ret;
        }

        public final void start() {
            if (started)
                throw new IllegalStateException("You can only start a request once.");
            else
                started = true;
            // Start a thread to handle the request asynchronously.
            workingThread = new Thread(this, "FCP Client: "+getDescription());
            workingThread.start();
        }

        public final int blockingRun() {
            if (started)
                throw new IllegalStateException("You can only start a request once.");
            else
                started = true;
            workingThread = Thread.currentThread();
            // Run the request to completion on this thread.
            run();
            return state();
        }

        public final boolean cancel() {
            ClientEvent evt = null;
            boolean ret = false;
            synchronized (FCPInstance.this) {
                if (state() >= Request.INIT && state() < Request.DONE) {
                    evt = state(Request.CANCELLED);
                    workingThread.interrupt();
                    ret = true;
                }
            }
            unlockedProduceEvent(evt);
            return ret;
        }
        
        public final Enumeration getEvents() { 
            return cel.events();
        }

        /** @return  the name of the FCP command handled by the subclass,
          *          perhaps with a URI identifier, etc.
          */
        abstract String getDescription();

        /** Called after the connection is set up and the request
          * is in the PREPARED state.
          */
        abstract void doit() throws Exception;
        
        public final void run() {
            try {
                ClientEvent evt = null;
                synchronized (FCPInstance.this) {
                    if (state() == Request.CANCELLED)
                        return;
                    negotiateConnection();
                    evt = state(Request.PREPARED);
                }
                unlockedProduceEvent(evt);
                doit();
            }
            catch (TerminalMessageException e) {
                ClientEvent evtError = null;
                ClientEvent evtFailed = null;
                synchronized (FCPInstance.this) {
                    if (state() != Request.CANCELLED) {
                        evtError = new ErrorEvent(e.getErrorString());
                        evtFailed = state(Request.FAILED);
                    }
                }
                unlockedProduceEvent(evtError);
                unlockedProduceEvent(evtFailed);
            }
            catch (Exception e) {
                ClientEvent evtException = null;
                ClientEvent evtFailed = null;
                synchronized (FCPInstance.this) {
                    if (state() != Request.CANCELLED) {
                        //System.err.println("FCPClient.FCPInstance.run -- exception:");
                        //e.printStackTrace();
                        evtException = new ExceptionEvent(e);
                        evtFailed = state(Request.FAILED);
                    }
                }
                unlockedProduceEvent(evtException);
                unlockedProduceEvent(evtFailed);
            }
            finally {
                if (conn != null) conn.close();
            }
        }
        
        private void negotiateConnection() throws Exception {
            conn = target.connect();
            out  = conn.getOut();
            
            int lnum = linkManager.designatorNum();
            int pnum = protocol.designatorNum();
            
            out.write((lnum >> 8) & 0xff);
            out.write(lnum & 0xff);
            // FIXME -- negotiate session

            out.write((pnum >> 8) & 0xff);
            out.write(pnum & 0xff);

            out.flush();
                    
            in  = new BufferedInputStream(conn.getIn());
            out = new BufferedOutputStream(out);
        }
                    
        final void send(String command) throws IOException {
            send(command, null, 0, 0);
        }
        
        final void send(String command, FieldSet fs) throws IOException {
            send(command, fs, 0, 0);
        }

        final void send(String command, long dlen, long mlen) throws IOException {
            send(command, new FieldSet(), dlen, mlen);
        }

        final void send(String command, FieldSet fs, long dlen, long mlen) throws IOException {

            if (state() == Request.CANCELLED)
                throw new InterruptedIOException();
            
            WriteOutputStream writer = new WriteOutputStream(out);
            writer.writeUTF(command, '\n');

            if (fs != null) {
                String tf;
                if (dlen > 0) {
                    tf = "Data";
                    fs.put("DataLength", Long.toHexString(dlen));
                    if (mlen > 0)
                        fs.put("MetadataLength", Long.toHexString(mlen));
                    else
                        fs.remove("MetadataLength");
                }
                else {
                    tf = "EndMessage";
                    fs.remove("DataLength");
                    fs.remove("MetadataLength");
                }
                fs.writeFields(writer, tf, '\n', '=', '.');
            }
            else {
                writer.writeUTF("EndMessage", '\n');
            }

            writer.flush();
        }

        final void sendData(long length, InputStream data) throws IOException {
            byte[] buf = new byte[BUFFER_SIZE];
            while (length > 0) {
                if (state() == Request.CANCELLED)
                    throw new InterruptedIOException();
                int n = data.read(buf, 0, (int) Math.min(length, buf.length));
                if (n == -1) throw new EOFException();
                length -= n;
                out.write(buf, 0, n);
            }
            out.flush();
        }

        final void receiveData(long length, OutputStream data) throws IOException {
            byte[] buf = new byte[BUFFER_SIZE];
            while (length > 0) {
                if (state() == Request.CANCELLED)
                    throw new InterruptedIOException();
                int n = in.read(buf, 0, (int) Math.min(length, buf.length));
                if (n == -1) throw new EOFException();
                length -= n;
                data.write(buf, 0, n);
            }
        }
            
        final RawMessage getResponse() throws IOException, TerminalMessageException {

            if (state() == Request.CANCELLED)
                throw new InterruptedIOException();
            
            RawMessage m;
            try {
                m = protocol.readMessage(in);
            }
            catch (InvalidMessageException e) {
                throw new TerminalMessageException("Got invalid message: "+e.getMessage());
            }
            if (m.messageType.equals("Failed")) {
                String err = (m.fs.get("Reason") == null
                              ? "Failed (no reason given)"
                              : "Failed, reason: "+m.fs.get("Reason"));
                throw new TerminalMessageException(err);
            }
            else if (m.messageType.equals("FormatError")) {
                String err = (m.fs.get("Reason") == null
                              ? "FormatError (no reason given)"
                              : "FormatError, reason: "+m.fs.get("Reason"));
                throw new TerminalMessageException(err);
            }
            else return m;
        }
    }

    
    /** For node messages that kill the request, like Failed or FormatError,
      * or invalid messages.
      */
    private static class TerminalMessageException extends Exception {
        private final String err;
        private TerminalMessageException(String err) {
            this.err = err;
        }
        final String getErrorString() {
            return err;
        }
    }


    //=== ClientHello =========================================================

    private class FCPHandshake extends FCPInstance {

        private static final String COMMAND = "ClientHello";

        private final HandshakeRequest req;
        
        private FCPHandshake(HandshakeRequest req) {
            super(req);
            this.req = req;
        }

        final String getDescription() {
            return COMMAND;
        }

        void doit() throws Exception {
            if (tryState(Request.REQUESTING)) {
                send(COMMAND);
                RawMessage m = getResponse();
                if (m.messageType.equals("NodeHello")) {
                    req.prot = m.fs.get("Protocol");
                    req.node = m.fs.get("Node");        
                    tryState(Request.DONE);
                }
                else throw new TerminalMessageException(
                    "Unexpected response: " + m.messageType
                );
            }
        }
    }

    
    //=== GenerateSVKPair =====================================================

    private class FCPComputeSVKPair extends FCPInstance {

        private static final String COMMAND = "GenerateSVKPair";

        private final ComputeSVKPairRequest req;

        private FCPComputeSVKPair(ComputeSVKPairRequest req) {
            super(req);
            this.req = req;
        }
        
        final String getDescription() {
            return COMMAND;
        }
        
        void doit() throws Exception {
            if (tryState(Request.REQUESTING)) {
                send(COMMAND);
                RawMessage m = getResponse();
                if (m.messageType.equals("Success")) {
                    BigInteger priv = new BigInteger(1, Base64.decode(m.fs.get("PrivateKey")));
                    ClientSVK svk   = new ClientSVK(null, null, null, new DSAPrivateKey(priv),
                                                    ClientSVK.getDefaultDSAGroup());
                    
                    req.clientKey = svk;
                    req.produceEvent(new GeneratedKeyPairEvent(svk.getPrivateKey(),
                                                               svk.getPublicKeyFingerPrint()));
                    tryState(Request.DONE);
                }
                else throw new TerminalMessageException(
                    "Unexpected response: " + m.messageType
                );
            }
        }
    }

    
    //=== GenerateCHK =========================================================

    private class FCPComputeCHK extends FCPInstance {

        private static final String COMMAND = "GenerateCHK";

        private final ComputeCHKRequest req;

        private FCPComputeCHK(ComputeCHKRequest req) throws IOException {
            super(req);
            this.req = req;
        }

        final String getDescription() {
            return COMMAND;
        }
        
        void doit() throws Exception {
            if (!tryState(Request.REQUESTING))
                return;

            long length = req.meta.size() + req.data.size();
            
            send(COMMAND, length, req.meta.size());

            InputStream data = null;
            try {
                data = req.meta.getInputStream();
                data = new SequenceInputStream(data, req.data.getInputStream());
                data = new EventInputStream(data, req, length >> 4, length);
                req.produceEvent(new TransferStartedEvent(length));
                sendData(length, data);
            }
            finally {
                if (data != null) data.close();
            }
            
            RawMessage m = getResponse();
            if (m.messageType.equals("Success")) {
                FreenetURI uri = new FreenetURI(m.fs.get("URI"));
                req.clientKey  = (ClientCHK) ClientCHK.createFromRequestURI(uri);
                req.produceEvent(new GeneratedURIEvent("Generated CHK", uri));
                tryState(Request.DONE);
            }
            else throw new TerminalMessageException(
                "Unexpected response: " + m.messageType
            );
        }
    }
    
    
    //=== ClientPut ===========================================================
    
    private class FCPInsert extends FCPInstance {

        private static final String COMMAND = "ClientPut";

        private final PutRequest req;

        private FCPInsert(PutRequest req) {
            super(req);
            this.req = req;
        }
        
        final String getDescription() {
            return COMMAND + ": " + req.uri;
        }
        
        void doit() throws Exception {
            if (!tryState(Request.REQUESTING))
                return;

            long length = req.meta.size() + req.data.size();
            
            FieldSet fs = new FieldSet();
            fs.put("URI", req.uri.toString());
            fs.put("HopsToLive", Integer.toHexString(req.htl));
            send(COMMAND, fs, length, req.meta.size());
            
            InputStream data = null;
            try {
                data = req.meta.getInputStream();
                data = new SequenceInputStream(data, req.data.getInputStream());
                data = new EventInputStream(data, req, length >> 4, length);
                req.produceEvent(new TransferStartedEvent(length));
                sendData(length, data);
            }
            finally {
                if (data != null) data.close();
            }

            while (true) {
                RawMessage m = getResponse();
                if (m.messageType.equals("Pending")) {
                    FreenetURI uri = new FreenetURI(m.fs.get("URI"));
                    req.produceEvent(new GeneratedURIEvent("Insert URI", uri));
                    req.clientKey = AbstractClientKey.createFromRequestURI(uri);
                    long time = (m.fs.get("Timeout") == null
                                 ? -1 : Fields.hexToLong(m.fs.get("Timeout")));
                    req.produceEvent(new PendingEvent(time));
                }
                else if (m.messageType.equals("Success")) {
                    tryState(Request.DONE);
                    return;
                }
                else if (m.messageType.equals("Restarted")) {
                    long time = (m.fs.get("Timeout") == null
                                 ? -1 : Fields.hexToLong(m.fs.get("Timeout")));
                    req.produceEvent(new RestartedEvent(time));
                }
                else if (m.messageType.equals("RouteNotFound")) {
                    int unreachable = 0, restarted = 0, rejected = 0;
                    try {
                        unreachable = Integer.parseInt(m.fs.get("Unreachable"), 16);
                    }
                    catch (Exception e) {}
                    try {
                        restarted = Integer.parseInt(m.fs.get("Restarted"), 16);
                    }
                    catch (Exception e) {}
                    try {
                        rejected = Integer.parseInt(m.fs.get("Rejected"), 16);
                    }
                    catch (Exception e) {}
                    req.produceEvent(new RouteNotFoundEvent(m.fs.get("Reason"),
                                                            unreachable,
                                                            restarted,
                                                            rejected));
                    tryState(Request.FAILED);
                    return;
                }
                else if (m.messageType.equals("KeyCollision")) {
                    FreenetURI uri = new FreenetURI(m.fs.get("URI"));
                    req.clientKey  = AbstractClientKey.createFromRequestURI(uri);
                    req.produceEvent(new GeneratedURIEvent("Insert URI", uri));
                    req.produceEvent(new CollisionEvent(req.clientKey));
                    tryState(Request.FAILED);
                    return;
                }
                else if (m.messageType.equals("URIError")) {
                    throw new TerminalMessageException("Bad URI: "+req.uri);
                }
                else {
                    throw new TerminalMessageException(
                        "Unexpected response: " + m.messageType
                    );
                }
            }
        }
    }

    
    //=== ClientGet ===========================================================
    
    private class FCPRequest extends FCPInstance {
        
        private static final String COMMAND = "ClientGet";
        
        private final GetRequest req;
        
        private FCPRequest(GetRequest req){
            super(req);
            this.req = req;
        }

        final String getDescription() {
            return COMMAND + ": " + req.uri;
        }

        void doit() throws Exception {
            if (!tryState(Request.REQUESTING))
                return;

            // create & send ClientGet-message
            FieldSet fs = new FieldSet();
            fs.put("URI", req.uri.toString());
            fs.put("HopsToLive", Integer.toHexString(req.htl)); 
            send(COMMAND, fs);

            while (true) {
                RawMessage m = getResponse();
                if (m.messageType.equals("DataFound")) {
                    if (doDataFound(m.fs)) {
                        tryState(Request.DONE);
                        return;
                    }
                }
                else if (m.messageType.equals("Restarted")) {
                    long time = (m.fs.get("Timeout") == null
                                 ? -1 : Fields.stringToLong(m.fs.get("Timeout")));
                    req.produceEvent(new RestartedEvent(time));
                }
                else if (m.messageType.equals("DataNotFound")) {
                    req.produceEvent(new DataNotFoundEvent());

                    tryState(Request.FAILED);

                    return;
                }
                else if (m.messageType.equals("RouteNotFound")) {
                    int unreachable = 0, restarted = 0, rejected = 0;
                    try {
                        unreachable = Integer.parseInt(m.fs.get("Unreachable"), 16);
                    }
                    catch (Exception e) {}
                    try {
                        restarted = Integer.parseInt(m.fs.get("Restarted"), 16);
                    }
                    catch (Exception e) {}
                    try {
                        rejected = Integer.parseInt(m.fs.get("Rejected"), 16);
                    }
                    catch (Exception e) {}
                    req.produceEvent(new RouteNotFoundEvent(m.fs.get("Reason"),
                                                            unreachable,
                                                            restarted,
                                                            rejected));
                    tryState(Request.FAILED);
                    return;
                }
                else if (m.messageType.equals("URIError")) {
                    throw new TerminalMessageException("Bad URI: "+req.uri);
                }
                else {
                    throw new TerminalMessageException(
                        "Unexpected response: " + m.messageType
                    );
                }
            }
        }

        private boolean doDataFound(FieldSet fs) throws TerminalMessageException,
                                                        IOException {
            if (fs.get("DataLength") == null)
                throw new TerminalMessageException("no DataLength field in DataFound");
            
            long dlen, mlen;
            try {
                dlen = Fields.stringToLong(fs.get("DataLength"));
            }
            catch (Exception e) {
                throw new TerminalMessageException("bad DataLength field in DataFound");
            }
            try {
                if (fs.get("MetadataLength") != null)
                    mlen = Fields.stringToLong(fs.get("MetadataLength"));
                else
                    mlen = 0;
            }
            catch (Exception e) {
                throw new TerminalMessageException("bad MetadataLength field in DataFound");
            }

            req.meta.resetWrite();
            req.data.resetWrite();

            Bucket[] buckets = { req.meta, req.data };
            long[] lengths   = { mlen, dlen-mlen, dlen };

            OutputStream data = new SegmentOutputStream(req, dlen >> 4,
                                                        buckets, lengths);

            req.produceEvent(new TransferStartedEvent(lengths));
            try {
                return readChunks(dlen, data);
            }
            finally {
                data.close();
            }
        }

        private boolean readChunks(long length, OutputStream data)
                                throws IOException, TerminalMessageException {
            while (length > 0) {
                RawMessage m = getResponse();
                if (m.messageType.equals("Restarted")) {
                    long time = (m.fs.get("Timeout") == null
                                 ? -1 : Fields.stringToLong(m.fs.get("Timeout")));
                    req.produceEvent(new RestartedEvent(time));
                    return false;
                }
                else if (!m.messageType.equals("DataChunk")) {
                    throw new TerminalMessageException(
                        "Unexpected response: "+m.messageType);
                }
                else if (m.fs.get("Length") == null) {
                    throw new TerminalMessageException(
                        "Bad DataChunk; no Length field");
                }
                // got DataChunk
                long size;
                try {
                    size = Fields.stringToLong(m.fs.get("Length"));
                }
                catch (Exception e) {
                    throw new TerminalMessageException(
                        "Bad DataChunk Length field: "+m.fs.get("Length"));
                }
                receiveData(size, data);
                length -= size;
            }
            return true;
        }
    }
}




