package freenet;
import freenet.thread.ThreadFactory;
import freenet.session.*;
import freenet.support.Heap;
import freenet.support.Comparable;
import freenet.support.Logger;
import freenet.support.MultiValueTable;
import freenet.support.LRUQueue;
import java.util.Hashtable;
import java.util.Vector;
import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.util.Date;
import java.io.OutputStream;
import java.io.IOException;
import java.io.PrintWriter;
/**
 * I moved out of the LinkManager code because I think that the 
 * crypto links and open ConnectionHandlers are fairly orthogonal.
 * Yeah, it means Another Fucking Hashtable (AFH (tm)), but the other
 * way was another fucking headache, and I've had enough of those too.
 *
 * Anyways, the job of this class is pretty obvious - it caches open
 * and active ConnectionHandler objects.
 *
 * @author oskar
 */

public class OpenConnectionManager {

    private final ThreadFactory tf;
    private final MultiValueTable chs;
    private int openConns;
    
    private final LRUQueue lru = new LRUQueue();
    private int maxConnections = -1;

    public OpenConnectionManager(ThreadFactory tf, int maxConnections) {
        this.tf = tf;
        this.maxConnections = maxConnections;

        chs = new MultiValueTable(50, 3);
        openConns = 0;
    }

    /**
     * This method is package only, meant only for ConnectionHandler
     * objects to add themselves.
     */
    void put(ConnectionHandler ch) {
        synchronized (this) {
            //if (ch.peerIdentity() == null) return;
            chs.put(ch.peerIdentity(), ch);
            //        v.addElement(ch);
            lru.push(ch); // hmmm nested locks
            Core.diagnostics.occurrenceCounting("liveConnections",1);
            if (ch.outbound && (Core.outboundContacts != null)) {
                // We really mean live, not active.  Some of the 
                // "active" connections may be idle.
                Core.outboundContacts.incActive(ch.peerAddress().toString());
                Core.outboundContacts.incSuccesses(ch.peerAddress().toString());
            }
            
            openConns++;
        }

        // Dump LRU connection.
        //System.err.println("OCM.put -- lru queue size: " + lru.size());
        ConnectionHandler oldest = null;
        synchronized (lru) {
            if (lru.size() > maxConnections) {
                oldest = (ConnectionHandler)lru.pop();
            }
        }
        if (oldest != null) {
            //System.err.println("OCM.put -- closed connection handler.");
            oldest.terminate();
        }
    }

    // Removing a connection that isn't in the OCM is a 
    // legal NOP.
    synchronized ConnectionHandler remove(ConnectionHandler ch) {
        //if (ch.peerIdentity() == null) return null;
        if (chs.removeElement(ch.peerIdentity(), ch)) {
            Core.logger.log(this, "Removed ConnectionHandler " + 
                            ch, Core.logger.DEBUG);
            Core.diagnostics.occurrenceCounting("liveConnections",
                                               -1);

            if (ch.outbound && (Core.outboundContacts != null)) {
                Core.outboundContacts.decActive(ch.peerAddress().toString());
            }
            lru.remove(ch);
            openConns--;
            return ch;
        } else
            return null;
    }

    /**
     * This will return an open and not busy ConnectionHandler to the 
     * node identified.
     */
    private synchronized ConnectionHandler findFreeConnection(Identity id) {
        for (Enumeration e = chs.getAll(id) ; e.hasMoreElements() ; ) {
            ConnectionHandler res = (ConnectionHandler) e.nextElement();
            if (!res.isOpen()) {
                //  System.err.println("ConnectionHandlers ARE LEAKING! (0)");

                //                  Core.logger.log(this, "ConnectionHandlers are leaking:"
                //                                  + res,
                //                                  Core.logger.MINOR);
                
                // BUG? This doesn't look right.  Won't remove() be called
                // when the leaked connection handler is finalized?
                Core.diagnostics.occurrenceCounting("liveConnections",
                                                   -1);
                if (chs.removeElement(id, res)) {
                    openConns--;
                    lru.remove(res);
                }
            } else if (!res.sending()) {
                // found one
                lru.push(res);
                return res;
            }
        } 
        return null;
    }

    /**
     * This attempts to search for the open connection to id best suited for
     * sending a message. The formula for this is the connection with the 
     * smallest value of sendQueueSize() divided by the preference value of
     * the transport.
     */
    private synchronized ConnectionHandler findBestConnection(Identity id) {
        ConnectionHandler res, best = null;
        double resval, bestval = Double.POSITIVE_INFINITY;
        for (Enumeration e = chs.getAll(id) ; e.hasMoreElements() ; ) {
            res = (ConnectionHandler) e.nextElement();
            if (!res.isOpen()) {
                //                  System.err.println("ConnectionHandlers ARE LEAKING!(1)");
                //                  Core.logger.log(this, "ConnectionHandlers are leaking",
                //                                  Core.logger.MINOR);
                if (chs.removeElement(id, res))  {
                    openConns--;
                    lru.remove(res);
                }
            } else {
                resval = res.sendQueueSize() / 
                    res.transport().preference();
                
                if (resval < bestval) {
                    bestval = resval;
                    best = res;
                }                           
            }
        }
        return best;
    }

    /**
     * Creates a new Connection which is started and added.
     * @param c     The Core to connect from
     * @param p     The Peer to connect to
     * @param long  The  number of millisseconds before returning 
     *              ConnectFailedException - this does not necessarily
     *              mean that it won't eventually succeed, but we some
     *              threads may not wish to hang around to find out.
     * @returns   A running ConnectionHandler
     * @exception  ConnectFailedException  if the connection fails or 
     *                                     timeout runs out.
     */ 
    public ConnectionHandler createConnection(Core c, Peer p, long timeout) 
                                            throws CommunicationException {

        ConnectionHandler ret = null;
        ConnectionJob ct = new ConnectionJob(c, p);
        synchronized (ct) {
            tf.getThread(ct, true).start();
            //tm.forceRun(ct);
            // Restored pooled threads (oskar 20020204)
            //Thread job = new Thread(ct, 
            //                        "Non-pooled connection thread: " 
            //                        + p.getAddress().toString());
            //job.start();


          
            long endtime = System.currentTimeMillis() + timeout;

            while (!ct.done) {
                try {
                    if (timeout == 0) {
                        ct.wait();
                    }
                    else {
                        long wait = endtime - System.currentTimeMillis();
                        if (wait <= 0) break;
                        ct.wait(wait);
                    }
                }
                catch (InterruptedException e) {}
            }
            
            if (ct.ch != null) {
                Core.diagnostics.occurrenceBinomial("connectionRatio",1,1);
                ret = ct.ch;
            } else if (ct.e == null) {
                Core.diagnostics.occurrenceBinomial("connectionRatio",1,0);
                throw new ConnectFailedException(p.getAddress(), 
                                                 p.getIdentity(),
                                           "Timeout reached while waiting",
                                                 true);
            } else {
                Core.diagnostics.occurrenceBinomial("connectionRatio",1,0);
                throw ct.e;
            }
        }
        
        if (ret != null) {
            return ret;
        }
        
        throw new RuntimeException("Assertion Failure: ret != null");
    }

    /**
     * Returns a free connection, making a new one if none is available.
     */
    public ConnectionHandler getConnection(Core c, Peer p, long timeout) 
                                        throws CommunicationException {
        // Let the race begin!
        // If non-null it was open and free when findFreeConnection returned
        // but nothing guarantees that some other thread won't have sent a
        // message with a huge trailing field by the time you actually
        // try to send a message on it.
        ConnectionHandler ch = findFreeConnection(p.getIdentity()); 

        if (ch == null) {
            ch = createConnection(c, p, timeout);
        }
        return ch;
    }

    /**
     * Gives the number of registered open connections.
     */
    public final int countConnections() {
        return openConns;
    }

    /**
     * Writes an HTML table with information about the open connections.
     */
    public synchronized void writeHtmlContents(PrintWriter pw) {
        pw.println("<h2>Fred OpenConnectionManager Contents</h2> <b>At date:");
        pw.println(new Date());
        pw.println("</b><table border=1>");
        pw.print("<tr><th>Peer addr</th><th>Open</th><th>Send Count</th>");
        pw.println("<th>Receiving</th><th>Lifetime</th><th>Messages</th></tr>");
        for (Enumeration e = lru.elements() ; e.hasMoreElements() ;) {
            ConnectionHandler ch = (ConnectionHandler) e.nextElement();
            pw.print("<tr><td>");
            String sep = "</td><td>";
            pw.print(ch.peerAddress() + sep);
            pw.print((ch.isOpen() ? "yes" : "no") + sep);
            pw.print(ch.sendingCount() + sep);
            pw.print((ch.receiving() ? "yes" : "no") + sep);
            pw.print(ch.runTime() + sep);
            pw.print(ch.messages());
            pw.println("</td></tr>");
        }

        pw.println("</table>");

    }


    ////////////////////////////////////////////////////////////
    // Helper functions to enforce hard limits on the maximum 
    // number of concurrent blocked connections we allow to 
    // a single address.
    //
    private Hashtable blockedConnections = new Hashtable();
    private int blockedConnectionCount = 0;

    private final int MAXBLOCKEDCONNECTIONS = 1;

    private final void incHardConnectionLimit(Address addr)
        throws ConnectFailedException {
        int val = 0;
        synchronized(blockedConnections) {
            blockedConnectionCount++;
            Integer count = (Integer)blockedConnections.get(addr.toString());
            if (count != null) {
                val = count.intValue();
                if (val >= MAXBLOCKEDCONNECTIONS) {
                    
                    Core.logger.log(OpenConnectionManager.this,
                                    " Too many blocked connection, aborting: " 
                                    + addr.toString() +
                                    " " + val,
                                    Logger.DEBUGGING);
                
                    // So that the arithmetic works when
                    // dec is called in finally block.
                    blockedConnections.put(addr.toString(), new Integer(val + 1));
                    
                    // Terminal.
                    throw new ConnectFailedException(addr, "Exceeded blocked connection limit: " +
                                                     val + " for " + addr);
                }
            }
            blockedConnections.put(addr.toString(), new Integer(val + 1));
        
        }
        Core.logger.log(OpenConnectionManager.this,
                        " blocked: " + addr.toString() +
                        " " + (val),
                        Logger.DEBUGGING);
        
    }

    private final void decHardConnectionLimit(Address addr) {
        synchronized(blockedConnections) {
            blockedConnectionCount--;
            Integer count = (Integer)blockedConnections.get(addr.toString());
            if (count != null) {
                int val = count.intValue();
                if (val > 0) {
                    blockedConnections.put(addr.toString(), new Integer(val - 1));
                }
                else {
                    blockedConnections.remove(addr.toString());
                }
            }
        }
    }



    ////////////////////////////////////////////////////////////

    private class ConnectionJob implements Runnable {

        private boolean done = false;
        private ConnectionHandler ch = null;
        private CommunicationException e = null;

        private final Core core;
        private final Peer p;

        public ConnectionJob(Core core, Peer p) {
            this.core = core;
            this.p = p;
        }

        public void run() {

            long start = System.currentTimeMillis();
            Connection c = null;
            
            boolean connected = false;
            int loops = 5;
            do {
                try {
                    LinkManager linkManager = p.getLinkManager();
                    Presentation presentation = p.getPresentation();

                    try {
                        core.logger.log(OpenConnectionManager.this,
                                        " blocked connections " + blockedConnectionCount,
                                        core.logger.DEBUGGING);

                        // IMPORTANT:
                        // The connect() call below can block for
                        // a long time before failing
                        // (3 minutes on rh7.1, IBM JDK 1.3).
                        // 
                        // Fail immediately if there are too 
                        // many blocked connections for 
                        // the requested address.
                        incHardConnectionLimit(p.getAddress());
                        c = p.getAddress().connect();

                    }
                    finally {
                        decHardConnectionLimit(p.getAddress());
                        if (Core.outboundContacts != null) {
                            String countAddr = null;
                            if (c != null) {
                                // Address we connected to.
                                countAddr = c.getPeerAddress().toString();
                            }
                            else {
                                // Address we tried to connect to.
                                countAddr = p.getAddress().toString();
                            }
                            // Keep track of outbound connection attempts.
                            Core.outboundContacts.incTotal(countAddr);
                        }

                    }
                    
                    OutputStream raw = c.getOut();
                    int i = linkManager.designatorNum();
                    raw.write((i >> 8) & 0xff);
                    raw.write(i & 0xff);
                    raw.flush();
                    Link l = linkManager.createOutgoing(core.privateKey,
                                                        core.identity, 
                                                        p.getIdentity(), c);
                    core.logger.log(OpenConnectionManager.this,
                                    "Connection between: " 
                                    + l.getMyAddress() + 
                                    " and " + l.getPeerAddress(),
                                    core.logger.DEBUGGING);
                    
                    
                    OutputStream crypt = l.getOutputStream();
                    int j = presentation.designatorNum();
                    crypt.write((j >> 8) & 0xff);
                    crypt.write(j & 0xff);
                    crypt.flush();
                    
                    ch = new ConnectionHandler(OpenConnectionManager.this,
                                               presentation, l, 
                                               core.ticker(),
                                               3, core.maxPadding, true);
                    //runCh = true;
                    //ch.start();
                    if (!core.hasInterfaceFor(ch.transport())) {
                        // if we don't have an interface for this transport, we
                        // will ask this connection to persist.
                        Message m = ch.presentationType().getSustainMessage();
                        if (m != null)
                            ch.sendMessage(m);
                    }

                    core.diagnostics.occurrenceContinuous("connectingTime", 
                                                    System.currentTimeMillis()
                                                         - start);       
                    
                    connected = true;
                    
                } catch (IOException e) {
                    this.e = new ConnectFailedException(p.getAddress(),
                                                        p.getIdentity(),
                                                        e.getMessage(),
                                                        false);
                    core.logger.log(OpenConnectionManager.this, 
                                    "Transport level connect failed to: "
                                    + p.getAddress() + " -- " + e, Logger.DEBUG);
                                    //this.e, Logger.DEBUG);
                } catch (SendFailedException e) {
                    this.e = new ConnectFailedException(e);
                    core.logger.log(OpenConnectionManager.this, 
                                    "Transport level connect failed to: "
                                    + p.getAddress() + " -- " + e, Logger.DEBUG); 
                                    //this.e, Logger.DEBUG);
                } catch (ConnectFailedException e) {
                    this.e = e;
                    core.logger.log(OpenConnectionManager.this, 
                                    "Transport level connect failed to: "
                                    + p.getAddress() + " -- " + e, Logger.DEBUG);
                                    //e, Logger.DEBUG);
                    // I'll attempt to fall back on an open connection.
                    // I can't decide if this is a nice hack or an ugly hack..
                    //ch = findBestConnection(p.getIdentity());
                    //if (ch == null) {
                    //    this.e = e;
                    //} else {
                    //    try {
                    //        for (int j = 0 ; j < 5 && !ch.sending();
                    //             j++) {
                    //            Core.logger.log(this, 
                    //                            "Waiting for CH. Sending:" +
                    //                            ch.sending() + " Count: " +
                    //                            ch.sendingCount(), 
                    //                            Logger.DEBUG);
                    //                            
                    //            ch.sendMessage(null);
                    //        }
                    //    } catch (SendFailedException sfe) {
                    //        this.e = sfe;
                    //    }
                    //}
                } catch (NegotiationFailedException e) {
                    this.e = e;
                    core.logger.log(OpenConnectionManager.this,
                                    "Negotiation failed with: "
                                    + p.getAddress() + " -- " + e, Logger.MINOR);
                                    //e, Logger.MINOR);
                } catch (AuthenticationFailedException e) {
                    this.e = e;
                    core.logger.log(OpenConnectionManager.this,
                                    "Authentication failed with: "
                                    + p.getAddress() + " -- " + e, Logger.MINOR);
                                    //e, Logger.MINOR);
                //} catch (IOException e) {
                //    core.logger.log(OpenConnectionManager.this,
                //                    "I/O error during negotiation with: "
                //                    + p.getAddress(),
                //                    e, Logger.MINOR);
                //    this.e = e;
                } catch (Throwable e) {
                    this.e = new ConnectFailedException(p.getAddress(),
                                                        p.getIdentity(),
                                                        e.getMessage(),
                                                        true);
                    core.logger.log(OpenConnectionManager.this,
                                    "Unknown exception while connecting to: "
                                    + p.getAddress(),
                                    e, Logger.ERROR);
                }
            } while (!connected && !e.isTerminal() && --loops > 0);

            synchronized (this) {
                done = true;
                this.notifyAll();
            }
            
            if (connected)
                ch.run();
            else if (ch != null)
                ch.terminate();
            else if (c != null)
                c.close();
        }
    }
}
 



