//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: MeshOfHexahedra.cpp,v 1.8 2004/02/13 19:54:08 delpinux Exp $

#include <MeshOfHexahedra.hpp>
#include <ConnectivityBuilder.hpp>
#include <set>
#include <stack>

#include <ConformTransformation.hpp>

void MeshOfHexahedra::buildLocalizationTools()
{
  if (this->numberOfVertices()==0) {
    fferr(0) << "Oops: mesh contains no vertices\n";
    std::exit(1);
  }

  __connectivity = new Connectivity<Hexahedron>(this->numberOfCells());
  ConnectivityBuilder<MeshOfHexahedra> (*this, *__connectivity, __surfaceMesh);

  if (__surfaceMesh != 0) {
    (*__surfaceMesh).setBackgroundMesh(this);
  }

  // getting bounding box size
  __a = (this->vertex(0));
  __b = __a;
  for (size_t i=1; i<this->numberOfVertices(); ++i) {
    Vertex& x = this->vertex(i);
    for (size_t k=0; k<3; ++k) {
      __a[k] = (__a[k]<x[k])?__a[k]:x[k];
      __b[k] = (__b[k]>x[k])?__b[k]:x[k];
    }
  }
  ffout(3) << "- Bounding box is " << __a << ',' << __b << '\n';

  ffout(3) << "- Building the octree\n";
  __octree = new Octree<size_t>(__a,__b);

  const TinyVector<3,real_t> center(0.5,0.5,0.5);

  for (size_t i = 0; i < this->numberOfCells(); ++i) {

    ConformTransformationQ1Hexahedron T(this->cell(i));

    TinyVector<3,real_t> X;
    T.value(center, X);

    (*__octree).add(i,X);
  }
}


MeshOfHexahedra::const_iterator MeshOfHexahedra::find(const double& x,
						      const double& y,
						      const double& z) const
{
  TinyVector<3,real_t> X(x,y,z);
  Octree<size_t>::iterator i = (*__octree).fuzzySearch(X);

  size_t cellNumber = (*i).value();
  
  MeshOfHexahedra::const_iterator h0(*this, cellNumber);

  bool found=false;

  std::set<MeshOfHexahedra::const_iterator> visited;
  std::set<MeshOfHexahedra::const_iterator> toVisitSet;
  std::stack<MeshOfHexahedra::const_iterator> toVisitStack;

  toVisitSet.insert(h0);
  toVisitStack.push(h0);
  MeshOfHexahedra::const_iterator h(*this);
  size_t numberOfCell = 1;
  do {
    fferr(0) << "Visiting cell n" << numberOfCell++ << '\n';
    // treating next cell
    h = toVisitStack.top();

    toVisitSet.erase(h);
    toVisitStack.pop();
    visited.insert(h);

    const Hexahedron& H = *h;
    
    TinyVector<3> Hmin=H(0);
    TinyVector<3> Hmax=H(0);

    for (size_t k=1; k<H.numberOfVertices(); ++k) {
      for (size_t l=0; l<3; ++l) {
	Hmin[l] = std::min(Hmin[l], H(k)[l]);
	Hmax[l] = std::max(Hmax[l], H(k)[l]);
      }
    }

    fferr(0) << "Cell is:\n";
    fferr(0) << '\t' << Hmin << '-' << Hmax << '\n';

    ConformTransformationQ1Hexahedron T(H);

    TinyVector<3, real_t> Xhat;
    TinyVector<6,bool> neighbours = false;
    found = false;
    if (not(T.invertT(x,y,z, Xhat))) {
      const double epsilon=1E-5;
      bool inside = true;
      for (size_t k=0; k<3; ++k) {
	neighbours[2*k] = (Xhat[k]<-epsilon);
	neighbours[2*k+1] = (Xhat[k]>1+epsilon);

	inside &= not(neighbours[2*k] or neighbours[2*k+1]);
      }
      found = inside;
    } else {
      found = true;
    }

    if(!found) {
      cellNumber = this->cellNumber(*h);

      for (size_t k=0; k<6; ++k) {
	if (neighbours[k]) {
	  h = (*__connectivity)(cellNumber)[k];
	  if (not(h.end())) {
	    if (visited.find(h) == visited.end()) {
	      if (toVisitSet.find(h) == toVisitSet.end()) {
		toVisitSet.insert(h);
		toVisitStack.push(h);
	      }
	    }
	  }
	}
      }
    }
  } while (not(found) and not(toVisitStack.empty()));

  if (not(found)) {
    fferr(0) << "#### Really did not found ####\n";
    h = 0;
  }

  return  h;
}
