//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: DivMuGrad.hpp,v 1.2 2003/09/20 21:31:09 delpinux Exp $

// Implement the - div(grad()) operator, in the domain.
// using finit differences on a Structured3DMesh

#ifndef _DIVMUGRAD_
#define _DIVMUGRAD_

#include <PDEOperator.hpp>
/*!
  \class DivMuGrad
  This class describes second order partial differencial operators of the form:
 
  \f$ \nabla\cdot \mu \nabla \f$ where 
  \f$
  \begin{array}{ll}
  \mu: &\Omega \to R   \\
       &x \mapsto \mu(x)
  \end{array}
  \f$.
  
  \par example:
  if \f$ \mu = 1 \f$, the operator is the Laplacian.

  \author Stphane Del Pino
*/

class DivMuGrad
  : public PDEOperator
{
  //! The function \f$ \mu \f$
  ReferenceCounting<UserFunction> __mu;

public:

  const UserFunction& coefficient(const size_t i)
  {
    assert (i<1);
    return (*__mu);
  }

  //! The name of the type of operator.
  std::string TypeName() const
  {
    return std::string("DivMuGrad");
  }

  //! Access to the function \f$ \mu \f$.
  UserFunction& mu()
  {
    return *__mu;
  }

  //! Read-only access to the function \f$ \mu \f$.
  const UserFunction& mu() const
  {
    return *__mu;
  }

  ReferenceCounting<PDEOperator> operator * (ConstReferenceCounting<UserFunction> u) const
  {
    ReferenceCounting<UserFunction> newMu = new MulUserFunction(__mu,u);
    return (new DivMuGrad(newMu));
  }

  //! Returns a pointer on the opposed DivMuGrad operator.
  ReferenceCounting<PDEOperator> operator - () const
  {
    ReferenceCounting<UserFunction> newMu = new UnaryMinusUserFunction(__mu);
    return (new DivMuGrad(newMu));
  }

  //! Constructs a function \a initMu.
  DivMuGrad(ReferenceCounting<UserFunction> initMu)
    : PDEOperator(PDEOperator::divmugrad,
		  1),
      __mu(initMu)
  {
    ;
  }

  //! Copy constructor.
  DivMuGrad(const DivMuGrad& D)
    : PDEOperator(D),
      __mu(D.__mu)
  {
    ;
  }

  //! Destrutor.
  ~DivMuGrad()
  {
    ;
  }
};

#endif // _DIVMUGRAD_

