//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Discretization.hpp,v 1.1.1.1 2003/02/17 16:32:51 delpinux Exp $


#ifndef   _DISCRETIZATION_HPP_
#define   _DISCRETIZATION_HPP_

#include <ReferenceCounting.hpp>

#include <BaseVector.hpp>
#include <BaseMatrix.hpp>

#include <PDESystem.hpp>

/*!
  \class Discretization

  this class is used to define discretizations of PDEProblems.

  \author Stephane Del Pino
*/

class Discretization
{
public:
  //! The set of discretization types
  enum Type {
    FEM
  };

protected:
  //! The discretization type
  const Discretization::Type __type;

  //! The PDEProblem to discretize.
  const Problem& __problem;

  //! The matrix which will contain the discretization.
  BaseMatrix& __A;

  //! The second member
  BaseVector& __b;

public:
  //! Computes the matrix using the problem
  virtual void assembleMatrix() = 0;

  //! Computes the second memeber using the problem
  virtual void assembleSecondMember() = 0;

  //! Provides interface for to allow computation without matrix
  virtual void getDiagonal(BaseVector&) const = 0;

  //! Provides interface for to allow computation without matrix
  virtual void timesX(const BaseVector&, BaseVector&) const = 0;

  //! Gives the Problem
  const Problem& problem() const
  {
    return __problem;
  }

  //! Which is the discretization type
  const Discretization::Type& type() const
  {
    return __type;
  }

  //! Access to the matrix.
  BaseMatrix& A()
  {
    return __A;
  }

  //! Access to the second member.
  BaseVector& b()
  {
    return __b;
  }

  /*!
    The constructor.
    The matrix A and the second member b should be computed here.
   */
  Discretization(const Discretization::Type  t,
		 const Problem& p,
		 BaseMatrix& a,
		 BaseVector& bb)
    : __type(t),
      __problem(p),
      __A(a),
      __b(bb)
  {
    ;
  }

  Discretization(const Discretization& D)
    : __type(D.__type),
      __problem(D.__problem),
      __A(D.__A),
      __b(D.__b)
  {
    ;
  }

  //! The virtual destructor
  virtual ~Discretization()
  {
    ;
  }
};

#endif // _DISCRETIZATION_HPP_

