;;; -*- Mode: LISP; Syntax: COMMON-LISP; Package: FLEXI-STREAMS; Base: 10 -*-
;;; $Header: /usr/local/cvsrep/flexi-streams/stream.lisp,v 1.26 2006/09/15 07:21:41 edi Exp $

;;; Copyright (c) 2005-2006, Dr. Edmund Weitz.  All rights reserved.

;;; Redistribution and use in source and binary forms, with or without
;;; modification, are permitted provided that the following conditions
;;; are met:

;;;   * Redistributions of source code must retain the above copyright
;;;     notice, this list of conditions and the following disclaimer.

;;;   * Redistributions in binary form must reproduce the above
;;;     copyright notice, this list of conditions and the following
;;;     disclaimer in the documentation and/or other materials
;;;     provided with the distribution.

;;; THIS SOFTWARE IS PROVIDED BY THE AUTHOR 'AS IS' AND ANY EXPRESSED
;;; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
;;; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;;; ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
;;; DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
;;; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
;;; GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
;;; INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;; NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;; SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

(in-package :flexi-streams)

(defclass flexi-stream (trivial-gray-stream-mixin)
  ((stream :initarg :stream
           :reader flexi-stream-stream
           :documentation "The actual stream that's used for
input and/or output.  It must be capable of reading/writing
octets with READ-SEQUENCE and/or WRITE-SEQUENCE.")
   (external-format :initform (make-external-format :iso-8859-1)
                    :initarg :flexi-stream-external-format
                    :accessor flexi-stream-external-format
                    :documentation "The encoding currently used
by this stream.  Can be changed on the fly.")
   (element-type :initform #+:lispworks 'lw:simple-char #-:lispworks 'character
                 :initarg :element-type
                 :accessor flexi-stream-element-type
                 :documentation "The element type of this stream."))
  (:documentation "A FLEXI-STREAM object is a stream that's
`layered' atop an existing binary/bivalent stream in order to
allow for multi-octet external formats.  FLEXI-STREAM itself is a
mixin and should not be instantiated."))

(define-condition flexi-stream-error (stream-error)
  ()
  (:documentation "Superclass for all errors related to
flexi streams."))

(define-condition flexi-stream-simple-error (flexi-stream-error simple-condition)
  ()
  (:documentation "Like FLEXI-STREAM-ERROR but with formatting
capabilities."))

(define-condition flexi-stream-element-type-error (flexi-stream-error)
  ((element-type :initarg :element-type
                 :reader flexi-stream-element-type-error-element-type))
  (:report (lambda (condition stream)
             (format stream "Element type ~S not allowed."
                     (flexi-stream-element-type-error-element-type condition))))
  (:documentation "Errors of this type are signaled if the flexi
stream has a wrong element type."))

(define-condition flexi-stream-encoding-error (flexi-stream-simple-error)
  ()
  (:documentation "Errors of this type are signaled if there is an
encoding problem."))

(defun signal-encoding-error (flexi-stream format-control &rest format-args)
  "Convenience function similar to ERROR to signal conditions of type
FLEXI-STREAM-ENCODING-ERROR."
  (error 'flexi-stream-encoding-error
         :format-control format-control
         :format-arguments format-args
         :stream flexi-stream))

(defun maybe-convert-external-format (external-format)
  "Given an external format designator \(a keyword, a list, or an
EXTERNAL-FORMAT object) returns the corresponding EXTERNAL-FORMAT
object."
  (typecase external-format
    (symbol (make-external-format external-format))
    (list (apply #'make-external-format external-format))
    (otherwise external-format)))

(defmethod initialize-instance :after ((flexi-stream flexi-stream) &rest initargs)
  "Makes sure the EXTERNAL-FORMAT and ELEMENT-TYPE slots contain
reasonable values."
  (declare (ignore initargs))
  (with-slots (external-format element-type)
      flexi-stream
    (unless (or (subtypep element-type 'character)
                (subtypep element-type 'octet))
      (error 'flexi-stream-element-type-error
             :element-type element-type
             :stream flexi-stream))
    (setq external-format (maybe-convert-external-format external-format))))

(defmethod (setf flexi-stream-external-format) :around (new-value (flexi-stream flexi-stream))
  "Converts the new value to an EXTERNAL-FORMAT object if
necessary."
  (call-next-method (maybe-convert-external-format new-value) flexi-stream))

(defmethod (setf flexi-stream-element-type) :before (new-value (flexi-stream flexi-stream))
  "Checks whether the new value makes sense before it is set."
  (unless (or (subtypep new-value 'character)
              (subtypep new-value 'octet))
    (error 'flexi-stream-element-type-error
           :element-type new-value
           :stream flexi-stream)))

(defmethod stream-element-type ((stream flexi-stream))
  "Returns the element type that was provided by the creator of
the stream."
  (flexi-stream-element-type stream))

(defmethod close ((stream flexi-stream) &key abort)
  "Closes the flexi stream by closing the underlying `real'
stream."
  (with-slots (stream)
      stream
    (cond ((open-stream-p stream)
           (close stream :abort abort))
          (t nil))))

(defmethod open-stream-p ((stream flexi-stream))
  "A flexi stream is open if its underlying stream is open."
  (open-stream-p (flexi-stream-stream stream)))

(defmethod stream-file-position ((stream flexi-stream))
  "Dispatch to method for underlying stream."
  (stream-file-position (flexi-stream-stream stream)))

(defmethod (setf stream-file-position) (position-spec (stream flexi-stream))
  "Dispatch to method for underlying stream."
  (setf (stream-file-position (flexi-stream-stream stream))
        position-spec))

(defclass flexi-output-stream (flexi-stream fundamental-binary-output-stream
                                            fundamental-character-output-stream)
  ((column :initform 0
           :accessor flexi-stream-column
           :documentation "The current output column.  A
non-negative integer or NIL."))
  (:documentation "A FLEXI-OUTPUT-STREAM is a FLEXI-STREAM that
can actually be instatiated and used for output.  Don't use
MAKE-INSTANCE to create a new FLEXI-OUTPUT-STREAM but use
MAKE-FLEXI-STREAM instead."))

(defclass flexi-input-stream (flexi-stream fundamental-binary-input-stream
                                           fundamental-character-input-stream)
  ((last-char :initform nil
              :accessor flexi-stream-last-char
              :documentation "This slot either holds NIL or a
cons which denotes the last character \(code) read successfully
and the external format that was used to read it.  This is mainly
used for UNREAD-CHAR sanity checks.")
   (last-octet :initform nil
               :accessor flexi-stream-last-octet
               :documentation "This slot either holds NIL or the last
octet read successfully from the stream using a `binary' operation
such as READ-BYTE.  This is mainly used for UNREAD-BYTE sanity
checks.")
   (octet-stack :initform nil
                :accessor flexi-stream-octet-stack
                :documentation "A small buffer which holds octets
that were already read from the underlying stream but not yet
used to produce characters.  This is mainly used if we have to
look ahead for a CR/LF line ending.")
   (position :initform 0
             :initarg :position
             :accessor flexi-stream-position
             :documentation "The position within the stream where each
octet read counts as one."))
  (:documentation "A FLEXI-INPUT-STREAM is a FLEXI-STREAM that
can actually be instatiated and used for input.  Don't use
MAKE-INSTANCE to create a new FLEXI-INPUT-STREAM but use
MAKE-FLEXI-STREAM instead."))

(defclass flexi-io-stream (flexi-input-stream flexi-output-stream)
  ()
  (:documentation "A FLEXI-IO-STREAM is a FLEXI-STREAM that can
actually be instatiated and used for input and output.  Don't use
MAKE-INSTANCE to create a new FLEXI-IO-STREAM but use
MAKE-FLEXI-STREAM instead."))

(defun make-flexi-stream (stream &rest args
                                 &key (external-format (make-external-format :iso-8859-1))
                                      element-type column position)
  "Creates and returns a new flexi stream.  STREAM must be an open
binary or `bivalent' stream, i.e. it must be capable of
reading/writing octets with READ-SEQUENCE and/or WRITE-SEQUENCE.  The
resulting flexi stream is an input stream if and only if STREAM is an
input stream.  Likewise, it's an output stream if and only if STREAM
is an output stream.  The default for ELEMENT-TYPE is LW:SIMPLE-CHAR
on LispWorks and CHARACTER on other Lisps.  EXTERNAL-FORMAT must be an
EXTERNAL-FORMAT object or a symbol or a list denoting such an object.
COLUMN is the initial column of the stream which is either a
non-negative integer or NIL.  The COLUMN argument is only used for
output streams.  POSITION \(only used for input streams) should be an
integer and it denotes the position the stream is in - it will be
increased by one for each octet read."
  ;; these arguments are ignored - they are only there to provide a
  ;; meaningful parameter list for IDEs
  (declare (ignore element-type column position))
  (unless (and (streamp stream)
               (open-stream-p stream))
    (error "~S should have been an open stream." stream))
  (apply #'make-instance
         ;; actual type depends on STREAM
         (cond ((and (input-stream-p stream)
                     (output-stream-p stream))
                'flexi-io-stream)
               ((input-stream-p stream)
                'flexi-input-stream)
               ((output-stream-p stream)
                'flexi-output-stream))
         :stream stream
         :flexi-stream-external-format external-format
         (sans args :external-format)))
