/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * fische-3.1
 * Copyright (C) Marcel Ebmer 2009 <marcel@26elf.at>
 * 
 * fische-3.1 is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * fische-3.1 is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <getopt.h>
#include <unistd.h>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <fstream>
#include <cstdio>
#include "driver.h"
#include "configuration.h"

using namespace std;

Configuration::Configuration(int count, char** args)
{
	__alsaDevice = (const char*)malloc(64);
	strcpy((char*)__alsaDevice, "default");
	__vecX = 0;
	__vecY = 0;
	__virtX = 0;
	__virtY = 0;
	__fs = false;
	__fps = 30;
	__MT = true;
	__eom = false;
	__predictive = false;
	__writeConfig = true;
#ifdef HAVE_ALSA
	__audioDriver = ALSA;
#endif
#ifdef HAVE_PULSE
	__audioDriver = PULSE;
#endif

	read_configfile();

	static struct option long_options[] = {
		{"geometry", required_argument, 0, 'g'},
		{"virtual", required_argument, 0, 'v'},
		{"device", required_argument, 0, 'd'},
		{"driver", required_argument, 0, 'D'},
		{"fullscreen", no_argument, 0, 'f'},
		{"fps", required_argument, 0, 's'},
		{"single", no_argument, 0, '1'},
		{"predictive", no_argument, 0, 'p'},
		{"exit-on-mouseevent", no_argument, 0, 'x'},
		{"nowrite", no_argument, 0, 'n'},
		{"help", no_argument, 0, 'h'},
		{0, 0, 0, 0}
	};

	int option_index = 0;
	int c;
	while((c = getopt_long(count, args, "g:v:d:D:fs:1pnh", long_options, &option_index)) != -1)
	{
		switch(c)
		{
			case 'g':
				set_geometry(optarg);
				break;
			case 'v':
				set_virtual(optarg);
				break;
			case 'd':
				set_device(optarg);
				break;
			case 'D':
				set_driver(optarg);
				break;
			case 'f':
				set_fullscreen((char*)("true"));
				break;
			case 's':
				set_fps(optarg);
				break;
			case '1':
				set_single((char*)("true"));
				break;
			case 'p':
				set_predictive((char*)("true"));
				break;
			case 'x':
				set_eom((char*)("true"));
				break;
			case 'n':
				__writeConfig = false;
				break;
			case 'h':
			case '?':
				help();
			default:
				break;
		}
	}
	if(!__vecX) __vecX = 800;
	if(!__vecY) __vecY = 400;
	if(!__virtX) __virtX = __vecX;
	if(!__virtY) __virtY = __vecY;
}

Configuration::~Configuration()
{
	if(__writeConfig) write_configfile();
}

int Configuration::fpsTarget()
{
	return __fps;
}

const char* Configuration::device()
{
	return __alsaDevice;
}

int Configuration::X()
{
	return __vecX;
}

int Configuration::Y()
{
	return __vecY;
}

int Configuration::screenX()
{
	return __virtX;
}

int Configuration::screenY()
{
	return __virtY;
}

bool Configuration::fullscreen()
{
	return __fs;
}

bool Configuration::multiThreaded()
{
	return __MT;
}

bool Configuration::exitOnMouse()
{
	return __eom;
}

bool Configuration::predictiveBeat()
{
	return __predictive;
}

void Configuration::togglePredictiveBeat()
{
	if(__predictive) __predictive = false;
	else __predictive = true;
}

void Configuration::toggleFullscreen()
{
	if(__fs) __fs = false;
	else __fs = true;
}

void Configuration::help()
{
	cout << endl;
	cout << "Usage: fische [-D (driver)] [-d (device)] [-g (X)x(Y)] [-v (X)x(Y)] [-f]" << endl;
	cout << "       [-s (fps)] [-1] [-p] [-h]" << endl;
	cout << endl;
	cout << "Details:" << endl;
	cout << "--------------------------------------------------------------------------------" << endl;
	cout << "-D --driver (driver)        Use the specified audio input driver. Presently" << endl;
	cout << "                            \"alsa\" and \"pulse\" are supported. default: pulse" << endl;
	cout << "--------------------------------------------------------------------------------" << endl;
	cout << "-d --device (device)        (device) names the alsa pcm capture device to get" << endl;
	cout << "                            the sound data from. On most systems, the default" << endl;
	cout << "                            will do just fine." << endl;
	cout << "                            This option is ignored with PulseAudio input." << endl;
	cout << "--------------------------------------------------------------------------------" << endl;
	cout << "-g --geometry (X)x(Y)       (X) and (Y) specify the width and height of the" << endl;
	cout << "                            animation. default: 800x400" << endl;
	cout << "--------------------------------------------------------------------------------" << endl;
	cout << "-v --virtual (X)x(Y)        (X) and (Y) specify the width and height of the" << endl;
	cout << "                            application window. Use this to prevent your" << endl;
	cout << "                            computer from trying to switch to non-existent" << endl;
	cout << "                            fullscreen resolutions." << endl;
	cout << "--------------------------------------------------------------------------------" << endl;
	cout << "-f --fullscreen             Start up in fullscreen mode. At runtime, you can" << endl;
	cout << "                            toggle fullscreen mode by pressing 'F'." << endl;
	cout << "--------------------------------------------------------------------------------" << endl;
	cout << "-s --fps (fps)              (fps) specifies the target frames per second." << endl;
	cout << "                            The default, 30, is what fische is designed for" << endl;
	cout << "--------------------------------------------------------------------------------" << endl;
	cout << "-1 --single                 Use only one CPU, even if there are more available" << endl;
	cout << "--------------------------------------------------------------------------------" << endl;
	cout << "-p --predictive             Use predictive beat detection (experimental)" << endl;
	cout << "--------------------------------------------------------------------------------" << endl;
	cout << "   --exit-on-mouseevent     Exit when a mouse button is clicked" << endl;
	cout << "                            (useful mainly for touchscreens)" << endl;
	cout << "--------------------------------------------------------------------------------" << endl;
	cout << "-n --nowrite                Do not update configuration file HOME/.fischerc" << endl;
	cout << "--------------------------------------------------------------------------------" << endl;
	cout << "-h --help                   This message." << endl;
	cout << "--------------------------------------------------------------------------------" << endl;
	cout << "RUNTIME CONTROLS ARE DOCUMENTED IN THE README AND ON THE MAN PAGE." << endl;
	cout << "--------------------------------------------------------------------------------" << endl;
	exit(EXIT_FAILURE);
}

void Configuration::set_geometry(char* value)
{
	if(!strstr(value, "x"))
	{
		cerr << "ERROR: " << value << " is not a valid geometry string" << endl;
		help();		
	}
	sscanf(value, "%dx%d", &__vecX, &__vecY);
	if((__vecX < 16) || (__vecX > 4096) || (__vecY < 16) || (__vecY > 4096))
	{
		cerr <<	"ERROR: goemetry string invalid" << endl;
		help();
	}
	cout << "* using custom animation geometry " << __vecX << " by " << __vecY << endl;
}

void Configuration::set_virtual(char* value)
{

	if(!strstr(value, "x"))
	{
		cerr << "ERROR: " << value << " is no valid virtual geometry string" << endl;
		help();		
	}
	sscanf(value, "%dx%d", &__virtX, &__virtY);
	if((__virtX < 16) || (__virtX > 4096) || (__virtY < 16) || (__virtY > 4096))
	{
		cerr <<	"ERROR: virtual goemetry string invalid" << endl;
		help();
	}
	if((__virtX < __vecX) || (__virtY < __vecY))
	{
		cerr << "ERROR: virtual cannot be smaller than actual geometry" << endl;
		help();
	}
	cout << "* using custom virtual geometry " << __virtX << " by " << __virtY << endl;
}

void Configuration::set_device(char* value)
{
	strncpy((char*)__alsaDevice, value, 64);
	cout << "* using custom capture device: \"" << __alsaDevice << "\"" << endl;
}

void Configuration::set_driver(char* value)
{
	if(!strcmp(value, "alsa")) __audioDriver = ALSA;
	else if(!strcmp(value, "pulse")) __audioDriver = PULSE;
	else cout << "* unknown audio driver \"" << value << "\"." << endl;
	switch(__audioDriver)
	{
		case ALSA:
			cout << "* using ALSA input" << endl;
			break;
		case PULSE:
		default:
			cout << "* using PulseAudio input" << endl;
			break;
	}
}

void Configuration::set_fps(char* value)
{
	sscanf(value, "%d", &__fps);
	cout << "* using custom target fps = " << __fps << endl;
}

void Configuration::set_fullscreen(char* value)
{
	if(!strcmp(value, "true")) __fs = true;
	else __fs = false;
}

void Configuration::set_predictive(char* value)
{
	if(!strcmp(value, "true"))
	{
		__predictive = true;
		cout << "* using predictive beat detection (experimental)" << endl;
	}
	else __predictive = false;
}

void Configuration::set_single(char* value)
{
	if(!strcmp(value, "true"))
	{
		__MT = false;
		cout << "* using single-threaded algorithms on user request" << endl;
	}
	else __MT = true;
}

void Configuration::set_eom(char* value)
{
	if(!strcmp(value, "true")) __eom = true;
	else __eom = false;
}

void Configuration::read_configfile()
{	
	__eom_in_cfile = false;
	__pred_in_cfile = false;
	__single_in_cfile = false;

	const char *homedir = getenv("HOME");
	if(homedir)
	{
		string filename = string(homedir) + string("/.fischerc");
		fstream configfile(filename.c_str(), fstream::in);
		if(configfile.is_open())
		{
			cout << "* using saved configuration from " << filename << endl;
			char line[256];
			char option[256];
			char value[256];
			while(!configfile.getline(line, 256).eof())
			{
				if(strlen(line) < 3) continue;
				if(line[0] == '#') continue;
				sscanf(line, "%s = %s", option, value);
				if(!strcmp(option, "geometry")) set_geometry(value);
				else if(!strcmp(option, "virtual")) set_virtual(value);
				else if(!strcmp(option, "device")) set_device(value);
				else if(!strcmp(option, "driver")) set_driver(value);
				else if(!strcmp(option, "fullscreen")) set_fullscreen(value);
				else if(!strcmp(option, "fps")) set_fps(value);
				else if(!strcmp(option, "single"))
				{
					__single_in_cfile = true;
					set_single(value);
				}
				else if(!strcmp(option, "predictive"))
				{
					__pred_in_cfile = true;
					set_predictive(value);
				}
				else if(!strcmp(option, "exit-on-mouseevent"))
				{
					__eom_in_cfile = true;
					set_eom(value);
				}
			}
			configfile.close();
			cout << "* -----END OF CONFIG FILE-----" << endl;
		}
	}
}

void Configuration::write_configfile()
{
	const char *homedir = getenv("HOME");
	if(homedir)
	{
		string filename = string(homedir) + string("/.fischerc");
		fstream configfile(filename.c_str(), fstream::out);
		if(configfile.is_open())
		{
			cout << "* saving configuration to " << filename << endl;
			configfile << "#############################" << endl;
			configfile << "# fische configuration file #" << endl;
			configfile << "#############################" << endl << endl;
			configfile << "geometry           = " << dec << __vecX << "x" << dec << __vecY << endl;
			configfile << "virtual            = " << dec << __virtX << "x" << dec << __virtY << endl;
			configfile << "device             = " << __alsaDevice << endl;
			configfile << "fps                = " << dec << __fps << endl;
			if(__audioDriver == ALSA)
				configfile << "driver             = alsa" << endl;
			else if(__audioDriver == PULSE)
				configfile << "driver             = pulse" << endl;
			if(__fs)
				configfile << "fullscreen         = true" << endl;
			else
				configfile << "fullscreen         = false" << endl;

			configfile << "### uncomment to set single-CPU mode ###" << endl;
			if(__single_in_cfile)
			{
				if(!__MT)
					configfile << "single             = true" << endl;
				else
					configfile << "single             = false" << endl;
			}
			else configfile << "#single             = true" << endl;

			configfile << "### uncomment to enable exit on mouse click ###" << endl;
			if(__eom_in_cfile)
			{
				if(__eom)
					configfile << "exit-on-mouseevent = true" << endl;
				else
					configfile << "exit-on-mouseevent = false" << endl;
			}
			else configfile << "#exit-on-mouseevent = true" << endl;

			configfile << "### uncomment to enable predictive beat detection ###" << endl;
			if(__pred_in_cfile)
			{
				if(__predictive)
					configfile << "predictive         = true" << endl;
				else
					configfile << "predictive         = false" << endl;
			}
			else configfile << "#predictive         = true" << endl;
		}
	}
}