/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * fische-3.1
 * Copyright (C) Marcel Ebmer 2009 <marcel@26elf.at>
 * 
 * fische-3.1 is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * fische-3.1 is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdint.h>
#include <cmath>
#include <iostream>
#include "analyst.h"
#include "ringbuffer.h"
#include "configuration.h"

using namespace std;

Analyst::Analyst(RingBuffer* r, Configuration* c)
{
	__ringbuffer = r;
	__config = c;
	__ma30 = 0;
	__sd = 0;
	__bima = 0;
	__bisd = 0;
	__state = WAITING;
	__frame = 0;
	__bpm = 0;
}

Analyst::~Analyst()
{
}

int Analyst::analyse()
{
	double bi = 0;

	__frame ++;
	int16_t* samples = (int16_t*)__ringbuffer->get();
	__dB = level(samples);
	if(__ma30 == 0) __ma30 = __dB;
	else __ma30 = __ma30 * 0.9667 + __dB * 0.0333;
	__sd = __sd * 0.9667 + abs(__dB - __ma30) * 0.0333;

	if((__frame - __lastbeat) > 90)
	{
		__bpm = 0;
		__beathistory.clear();
	}
	
	switch(__state)
	{
		case WAITING:
			if(__dB < __ma30 + __sd) break;

			if(__sd == 0) bi = 1;
			else bi = (__dB - __ma30) / __sd;
			if(__bima == 0) __bima = bi;
			else __bima = __bima * 0.95 + bi * 0.05;
			__bisd = __bisd * 0.95 + abs(__bima - bi) * 0.05;
			
			__state = BEAT;
			__beathistory.push_back(__frame - __lastbeat);
			if(__beathistory.size() > 30) __beathistory.erase(__beathistory.begin());
			__lastbeat = __frame;
			__ma3 = __dB;

			if(__config->predictiveBeat())
			{
				__bpm = guessBpm();
			}

			if(bi > __bima + 3 * __bisd) return 3;
			if(bi > __bima + __bisd) return 2;
			if(bi > 0) return 1;

		case BEAT:
		case MAYBEWAITING:
			__ma3 = __ma3 * 0.6667 + __dB * 0.3333;
			if(__ma3 < __ma30 + __sd)
			{
				if(__state == MAYBEWAITING) __state = WAITING;
				else __state = MAYBEWAITING;
				return 0;
			}
	}
	return 0;
}

double Analyst::level(int16_t* samples)
{
	double E = 0;
	for(int ii = 0; ii < __ringbuffer->size() / 2; ii +=2)
	{
		E += abs((double)*(samples + ii));
		E += abs((double)*(samples + ii + 1));
	}
	if(E < 1) E += 1;
	E = E / __ringbuffer->size() * 2;
	return log10(E/32768) * 10;
}

/* YES, I know this does not REALLY return beats per minute */
/* returns frames per beat instead */
double Analyst::guessBpm()
{
	int n[51];
	double s[51];
	int imax = 0;
	int max = 0;
	for(int i = 1; i < 51; i ++)
	{
		n[i] = 0;
		s[i] = 0;
		for(unsigned int ii = 0; ii < __beathistory.size(); ii ++)
		{
			int c = __beathistory.at(ii);
			if(abs((double)(i - c)) <= 1)
			{
				n[i] ++;
				s[i] += c;
			}
		}
		s[i] /= n[i];
		if(n[i] > max)
		{
			imax = i;
			max = n[i];
		}
	}

	if(max < 10) return 0;

	double guess = s[imax];
	int expl = max;
	if(guess < 25) expl += n[(int)(guess*2 + 0.5)];
	if(guess > 4) expl += n[(int)(guess/2 + 0.5)];

	for(unsigned int i = 0; i < __beathistory.size() - 1; i ++)
	{
		if(abs(__beathistory.at(i) + __beathistory.at(i + 1) - guess) <=1)
		{
			expl ++;
			i ++;
		}
	}
	
	if(expl < 20) return 0;

	int c0 = 0;
	int c2 = 0;
	int c5 = 0;
	for(int i = 1; i < 4; i ++)
	{
		int t = __beathistory.at(__beathistory.size() - i);
		if(abs(t - guess) <= 1) c0 ++;
		if(abs(t - guess / 2) <= 1) c5 ++;
		if(abs(t - guess * 2) <= 1) c2 ++;
	}
	if(c0 + c2 + c5 == 0) return 0;
	if(c0) return guess;
	if(c2 > c5) return guess * 2;
	return guess / 2;
}
