/* draw.c  */
/* COPYRIGHT (C) 2000 THE VICTORIA UNIVERSITY OF MANCHESTER and John Levon
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details. 
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA. 
 */
/* to draw a specific area of the draw window  */ 
/*
 * $Log: draw.c,v $
 * Revision 1.3  2000/12/17 00:57:41  moz
 * examples, filled open splines, highlight_objects
 *
 * Revision 1.2  2000/12/06 20:56:01  moz
 * GPL stuff.
 *
 * Revision 1.1.1.1  2000/08/21 01:05:30  moz
 *
 *
 * Revision 1.1.1.1  2000/07/19 22:45:30  moz
 * CVS Import
 *
 * Revision 1.30  2000/03/15 00:15:03  moz
 * Removed point toggles.
 *
 * Revision 1.29  2000/03/14 01:09:26  moz
 * Clear up zoom xors.
 *
 * Revision 1.28  2000/03/13 17:35:30  moz
 * Use just_zoomed.
 *
 * Revision 1.27  2000/03/13 02:34:33  moz
 * Increased drawing net range to compensate for change to send_redraw_object().
 *
 * Revision 1.26  2000/03/04 19:34:34  moz
 * Improve drawing window.
 *
 * Revision 1.25  2000/02/17 22:21:49  moz
 * Compiler fixes.
 *
 * Revision 1.24  2000/01/30 17:19:21  moz
 * grid_offset not used.
 *
 * Revision 1.23  2000/01/29 21:20:04  moz
 * Specify Complex for arrowheads.
 *
 * Revision 1.22  2000/01/26 18:11:16  moz
 * Memleak fixes.
 *
 * Revision 1.21  2000/01/25 01:35:38  moz
 * Comment fixup.
 *
 * Revision 1.20  2000/01/21 11:47:19  moz
 * Removed desertgc.
 *
 * Revision 1.19  1999/11/23 16:44:18  moz
 * Comment change.
 *
 * Revision 1.18  1999/11/16 01:50:23  moz
 * clip_line must take signed longs.
 *
 * Revision 1.17  1999/11/15 23:04:26  moz
 * Prototype fix.
 *
 * Revision 1.16  1999/11/15 02:12:04  moz
 * Use rounded trig.
 *
 * Revision 1.15  1999/07/29 20:45:08  moz
 * Pass rx,ry to draw_text() from draw_object().
 *
 * Revision 1.14  1999/07/04 15:53:08  moz
 * If screen width/height of arrow is silly, don't bother.
 *
 * Revision 1.13  1999/05/23 20:11:16  moz
 * Fixed broken prototype fix.
 *
 * Revision 1.12  1999/05/23 00:27:29  moz
 * Pedantic changes.
 *
 * Revision 1.11  1999/05/19 17:06:31  moz
 * 1.0 Checkin.
 *
 * Revision 1.10  1999/05/04 15:22:50  moz
 * Moved draw_grid over.
 *
 * Revision 1.9  1999/04/29 00:19:19  moz
 * Use toggle_guidelines instead of direct drawing.
 * Fixes compound/guidelines-droppings bug.
 *
 * Revision 1.8  1999/04/28 20:57:18  moz
 * Need to reclip each object.
 *
 * Revision 1.7  1999/04/28 00:37:03  moz
 * Don't draw a desert any more.
 *
 * Revision 1.6  1999/04/27 19:50:48  moz
 * Fixed rubberbanding problems.
 *
 * Revision 1.5  1999/04/27 04:24:45  moz
 * draw_view changed to remove XEvent param.
 *
 * Revision 1.4  1999/04/25 00:19:01  moz
 * draw_text now takes angle parameter.
 *
 * Revision 1.3  1999/04/24 22:23:18  moz
 * gc_colour needs view parameter.
 *
 * Revision 1.2  1999/04/23 00:38:46  moz
 * Moved draw_view into function from event_loop.c
 *
 * Revision 1.1  1999/03/30 00:04:47  moz
 * Initial revision
 *
 */    

#include <math.h> 
#include "include/figurine.h"
#include "include/extern.h"

/* redraws a section of a view */  
/* this should really take a number of disjoint regions */  
void
draw_view(View *view,long x, long y, long width, long height)
{
 	set_clip(whitegc, x, y, width, height);
 	set_clip(blackxorgc, x, y, width, height);
 	set_clip(dashgc, x, y, width, height);
	 
	/* background  */   
 	XFillRectangle(display,view->draw_window->win, whitegc, x, y, (uint)width,(uint) height); 
	if (intersects(0,0,(signed int)view->draw_window->w,(signed int)view->draw_window->h,DOC_PIXEL_BOUNDS_V(view))) 
		{
		XDrawRectangle(display,view->draw_window->win,dashgc,
						I2P(0-view->x_inches_offset,view), 
						I2P(0-view->y_inches_offset,view), 
						(uint)I2P(view->doc->width_in_inches,view),
						(uint)I2P(view->doc->height_in_inches,view)
						);
		
		if (view->showgrid) 
			draw_grid(view,x,y, width, height);    
		}; 
						 
	/* actual objects drawing  */ 
	draw_section(view,x,y, width, height); 
	 
	/* toggle back on XOR stuff  */   
  	if (view->guide_lines_displayed)  
		{ 
		set_clip(blackxorgc,x,y,width, height); 
		toggle_guidelines(view); 
		} 
	
	if (state.busy_drawing) 
		toggle(view,mouse_x, mouse_y); 
	
	SET_CLIP_WINDOW(blackxorgc, view->draw_window);
}

void 
draw_section_v(View *view, VRegion win)
{
	draw_section0(view, &win, 0, 0, 0, 0);
}

void 
draw_section(View *view, int x, int y, int w, int h)
{
	draw_section0(view, NULL, x, y, w, h);
}

/* draw object part of a document window  */  
void 
draw_section0(View *view, VRegion *bb, int x, int y, int w, int h)
{
	List obs=NULL;
	List obs2; 
	VRegion win; /* doc co-ords of screen x,y,w,h */ 
 	int lw = I2D(2.0,view); 
		 
	/* lw is used to try and catch very sharp fat line's originating objects */
	/* it's wasteful but drawing is clipped */
	 
	if (bb==NULL)
		{
		win.x1 = XP2D(x,view);
		win.y1 = YP2D(y,view);
		win.x2 = XP2D(x+w,view);
		win.y2 = YP2D(y+h,view);
		}
	else
		{ 
		win = *bb;
		x = XD2P(win.x1,view);
		y = YD2P(win.y1,view);
		w = XD2P(win.x2-win.x1,view);
		h = YD2P(win.y2-win.y1,view);
		}; 
		 

	/* account for stuff outside of bbox  */   
	win.x1 -= lw;
	win.y1 -= lw;
	win.x2 += lw;
	win.y2 += lw;
	/* find all intersecting objects  */  
	/* they are returned in depth-order  */  
	obs = intersecting_objects(view->doc->o, win, obs);
	obs2=obs; 

	while (obs!=NULL)
		{
		set_clip(ugc, x, y, w, h);
		set_clip(handlegc, x, y, w, h); 
		set_clip(fillgc, x, y, w, h); 
		set_clip(dashgc, x, y, w, h); 
		set_clip(blackxorgc, x, y, w, h);
		set_clip(whitegc, x, y, w, h); 
	 
		if (OB(obs)->type!=COMPOUND)
			{
			/* set up gc parameters for drawing object */  
			gc_end(ugc, OB(obs));
			gc_join(ugc, OB(obs));
			gc_lw(view, ugc, OB(obs)); 
			gc_line(ugc, OB(obs));
			gc_fill(fillgc, OB(obs));
			gc_colour(view,ugc, OB(obs));
			gc_fillcolour(view,fillgc, OB(obs));
			};

		draw_object(OB(obs),view,ugc,OB(obs)->bbox.x1, OB(obs)->bbox.y1, 1.0, 1.0, 0.0,x,y,w,h);
		obs = obs->next;
		}
		 
/* debugging */
#if 0
	stk_thin_outline(view->draw_window, x, y, w, h);
#endif
	delete_list(obs2); 
}

/* draw an individual object  */  
void 
draw_object(Object *ob,View *view, GC gc, long x, long y, double rx, double ry, double angle, long cx, long cy, long cw, long ch)
{
	switch (ob->type)
		{
		case SPLINE:
		case ARC: 
			draw_spline(ob,view,gc,x,y,rx,ry,angle);
			break;

		case ROUNDBOX:
			draw_roundbox(ob,view,gc,x,y,rx,ry);
			break;

		case POLYLINE:
		case POLYGON:
			draw_polyline(ob,view,gc, x, y, rx, ry,angle);
			break;
		
		case TEXT:
			draw_text(ob,view,gc,x,y,rx,ry,angle);
			break; 

		case ELLIPSE:
			/* we can't angle ellipses yet  */  
			draw_ellipse(ob,view,gc, x, y, rx, ry);
			break;
		
		case ARCELLIPSE:
			draw_arcellipse(ob,view,gc,x,y,rx,ry);
			break;
		
		case COMPOUND:
			draw_compound(ob,view,gc,x,y,rx,ry,cx,cy,cw,ch);
			break;
		
		default:
			v_error(ob->type);
			fprintf(stderr,"figurine: object %p\n",ob); 
			break; 
		};
	draw_bbox_handles(view);
		 
}

/* line clipping algorithm */ 
/* we need this because X clips very very big lines incorrectly */  
/* see Cohen & Sutherland */ 
/* TBRL encoding  */  
/* cruft is for linewidths>0 where drawing to exact point looks silly */ 
/* pass 2 * pixel linewidth for it  */
/* gives compiler warnings about shadowing y1 - this is bizarrely coming from
   some system header file and we can ignore it */  
VLine *
clip_line(long x1,long y1,long x2, long y2,VLine *vl, Boolean *clipped, int cruft)
{
	char mask1=0,mask2=0;
	VLine *l; 

	 
	if (vl->x1<(x1-cruft))
		mask1 |= 1;
	if (vl->x1>(x2+cruft))
		mask1 |= 2;
	if (vl->y1<(y1-cruft))
		mask1 |= 8;
	if (vl->y1>(y2+cruft))
		mask1 |= 4; 
	if (vl->x2<(x1-cruft))
		mask2 |= 1;
	if (vl->x2>(x2+cruft))
		mask2 |= 2;
	if (vl->y2<(y1-cruft))
		mask2 |= 8;
	if (vl->y2>(y2+cruft))
		mask2 |= 4; 
		
	if (mask1==0 && mask2==0) /* both points internal  */  
		{
		*clipped = FALSE;
		return vl;
		}; 
	
	/* must clip somehow now  */  
	*clipped = TRUE;
	 
	if ((mask1 & mask2) != 0)
		{
		/* trivially reject  */ 
		return NULL;
		}
	else
		{
		/* must divide */ 

		/* crosses at least boundary. find one that crosses, recurse with
			line "below" the boundary */

		/* top */  
		if ((mask1 & 8) != (mask2 & 8)) 
			{
			/* cut off top and recurse  */ 
			
			if (mask1 & 8)
				{
				/* x1,y1 is above line, so we recalculate that */
				vl->x1 = line_x_at_y(vl->x1,vl->y1,vl->x2,vl->y2, y1-cruft);
				vl->y1 = y1-cruft;
				}
			else
				{ 
				/* x2,y2 is above line, so we recalculate that */
				vl->x2 = line_x_at_y(vl->x1,vl->y1,vl->x2,vl->y2, y1-cruft);
				vl->y2 = y1-cruft;
				};

			l = clip_line(x1,y1,x2,y2,vl, clipped, cruft); 
			*clipped = TRUE;
			return l;
			};

		/* bottom  */  
		if ((mask1 & 4) != (mask2 & 4)) 
			{
			/* cut off below bottom and recurse  */ 
			
			if (mask1 & 4)
				{
				/* x1,y1 is below line, so we recalculate that */
				vl->x1 = line_x_at_y(vl->x1,vl->y1,vl->x2,vl->y2, y2+cruft);
				vl->y1 = y2+cruft;
				}
			else
				{ 
				/* x2,y2 is below line, so we recalculate that */
				vl->x2 = line_x_at_y(vl->x1,vl->y1,vl->x2,vl->y2, y2+cruft);
				vl->y2 = y2+cruft;
				};
			
			l = clip_line(x1,y1,x2,y2,vl, clipped, cruft); 
			*clipped = TRUE;
			return l;
			};

		/* right  */  
		if ((mask1 & 2) != (mask2 & 2)) 
			{
			/* cut off right and recurse  */ 
			
			if (mask1 & 2)
				{
				/* x1,y1 is right of line, so we recalculate that */
				vl->y1 = line_y_at_x(vl->x1,vl->y1,vl->x2,vl->y2, x2+cruft);
				vl->x1 = x2+cruft;
				}
			else
				{ 
				/* x2,y2 is right of line, so we recalculate that */
				vl->y2 = line_y_at_x(vl->x1,vl->y1,vl->x2,vl->y2, x2+cruft);
				vl->x2 = x2+cruft;
				};

			l = clip_line(x1,y1,x2,y2,vl, clipped, cruft); 
			*clipped = TRUE;
			return l;
			};

 		/* left  */   
			/* cut off left and recurse  */ 
			
			if (mask1 & 1)
				{
				/* x1,y1 is left of line, so we recalculate that */
				vl->y1 = line_y_at_x(vl->x1,vl->y1,vl->x2,vl->y2, x1-cruft);
				vl->x1 = x1-cruft;
				}
			else
				{ 
				/* x2,y2 is left of line, so we recalculate that */
				vl->y2 = line_y_at_x(vl->x1,vl->y1,vl->x2,vl->y2, x1-cruft);
				vl->x2 = x1-cruft;
				};
 
			l = clip_line(x1,y1,x2,y2,vl, clipped, cruft); 
			*clipped = TRUE;
			return l; 
		}; 


}

/* draw arrow  */  
/* given two points, draws an arrow described by *arrow */  
/* direction is from sx,sy -> ex,ey  */  
/* uses basic geometry to figure out arrow angle  */  
void 
draw_arrow(View *view, GC gc, Arrow *arrow, int sx, int sy, int ex, int ey)
{
	long x3,y3,x4,y4,x5,y5;
	VLine line; 
	double a,b; 
	double theta;
	double sint;
	double cost;
	int num; 
	XPoint xp[5]; 

	if (gc==blackxorgc)
		return;

 	gc_lw_v(view,gc,arrow->lw); 
	 
	line.x2 = ex;
	line.y2 = ey;
		 
	line.x1 = sx - ex;
	line.y1 = sy - ey;
	
	a = D2P(I2D(((double)arrow->h)/80.0,view),view); 
	b = D2P(I2D(((double)arrow->w)/80.0,view),view); 
	if (a==0 || b==0)
		return;
	line.x1 *= sqrt((double)(a*a)+(b*b));
	line.y1 *= sqrt((double)(a*a)+(b*b));
	line.x1 /= sqrt((double)((line.x2-sx)*(line.x2-sx)) + ((line.y2-sy)*(line.y2-sy)));
	line.y1 /= sqrt((double)((line.x2-sx)*(line.x2-sx)) + ((line.y2-sy)*(line.y2-sy)));
	theta = atan2((double)arrow->w,(double)arrow->h); 
	sint = sinround(theta);
	cost = cosround(theta);

	x3 = (long)(cost*line.x1 - sint*line.y1);
	y3 = (long)(sint*line.x1 + cost*line.y1);
	
	x3 += line.x2;
	y3 += line.y2;
	line.x1 = x3;
	line.y1 = y3;
	
	x4 = line.x1;
	y4 = line.y1;

		 
	line.x2 = ex;
	line.y2 = ey;
		 
	line.x1 = sx - ex;
	line.y1 = sy - ey;
	
	a = D2P(I2D(((double)arrow->h)/80.0,view),view); 
	b = D2P(I2D(((double)arrow->w)/80.0,view),view); 
	line.x1 *= sqrt((double)(a*a)+(b*b));
	line.y1 *= sqrt((double)(a*a)+(b*b));
	line.x1 /= sqrt((double)((line.x2-sx) * (line.x2-sx) + (line.y2-sy) * (line.y2-sy)));
	line.y1 /= sqrt((double)((line.x2-sx) * (line.x2-sx) + (line.y2-sy) * (line.y2-sy)));
	theta =  - atan2((double)arrow->w,(double)arrow->h); 
	sint = sinround(theta);
	cost = cosround(theta);

	x3 = (long)(cost*line.x1 - sint*line.y1);
	y3 = (long)(sint*line.x1 + cost*line.y1);
	
	x3 += line.x2;
	y3 += line.y2;
	line.x1 = x3;
	line.y1 = y3;
	
	x5 = line.x1;
	y5 = line.y1;
	 
	xp[0].x = x4; xp[0].y = y4;
	xp[1].x = ex; xp[1].y = ey;
	xp[2].x = x5; xp[2].y = y5;
	num = 3; 
	if (arrow->type!=ARROWSTICK)
		{
		if (arrow->type==ARROWINDENTED)
			{
			bent_midpoint(x4,y4,x5,y5,-0.261799388,&xp[3].x,&xp[3].y); 
			num++; 
			}
		else if (arrow->type==ARROWPOINTED)
			{
			bent_midpoint(x4,y4,x5,y5,0.261799388,&xp[3].x,&xp[3].y); 
			num++; 
			};

		xp[num].x = x4; xp[num].y = y4; 
		num++; 
		}; 

	if (arrow->type != ARROWSTICK)
		{ 
		if (arrow->filled)
			XFillPolygon(display,view->draw_window->win,gc,xp,num,Complex,CoordModeOrigin); 
		else
			XFillPolygon(display,view->draw_window->win,whitegc,xp,num,Complex,CoordModeOrigin);
		};

	XDrawLines(display,view->draw_window->win,gc,xp,num,CoordModeOrigin); 
}

/* draw grid on a view  */  
/* MIN_GRID_PIXEL_SEPARATION ensures we don't get totally covered in grid */  
void 
draw_grid(View *view, long x, long y, long w, long h)
{
   long startx,starty,endx,endy,a;
	long mx=1,my=1; 
 	long x2, y2;  

 
  	x2 = x+w; 
 	y2 = y+h; 

	force_in_box_l(&x, &y, XD2P(0,view), YD2P(0,view), 
						XD2P(I2D(view->doc->width_in_inches,view),view), YD2P(I2D(view->doc->height_in_inches,view),view)); 
	force_in_box_l(&x2, &y2, XD2P(0,view), YD2P(0,view), 
						XD2P(I2D(view->doc->width_in_inches,view),view), YD2P(I2D(view->doc->height_in_inches,view),view)); 
	w = x2-x;
	h = y2-y;
	set_clip(backgc, x, y, w, h); 
	
	startx = ((XP2D(x,view)/view->grid_x)-1)*view->grid_x;
	starty = ((YP2D(y,view)/view->grid_y)-1)*view->grid_y;
	endx = ((XP2D(x+w,view)/view->grid_x)+1)*view->grid_x;  
	endy = ((YP2D(y+h,view)/view->grid_y)+1)*view->grid_y;  
	
 	if (!intersects(startx, starty, endx, endy, DOCDBOUNDS(view)))
 		return;  
	
	while (D2P(view->grid_x*mx,view) <= MIN_GRID_PIXEL_SEPARATION) 
		mx++;

	startx = (startx/(view->grid_x*mx))*(view->grid_x*mx);
	 
	while (D2P(view->grid_y*my,view) <= MIN_GRID_PIXEL_SEPARATION) 
		my++;
		 
	starty = (starty/(view->grid_y*my))*(view->grid_y*my);
	 
	a = startx;
	while	(a<=endx)
		{
		if (a % (view->grid_x*mx) == 0) 
			XDrawLine(display,view->draw_window->win, backgc, 
						 XD2P(a,view), YD2P(starty,view), XD2P(a,view), YD2P(endy,view));
		a += view->grid_x*mx;
		};

	a = starty;
	while (a<=endy)
		{
		if (a % (view->grid_y*my) == 0) 
			XDrawLine(display,view->draw_window->win, backgc, 
						 XD2P(startx,view), YD2P(a,view), XD2P(endx,view), YD2P(a,view));
		a += view->grid_y*my;
		};
}                                                       
