/*
 * Copyright (C) 2000-2001 Chris Ross and Evan Webb
 * Copyright (C) 1999-2000 Chris Ross
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies of the Software, its documentation and marketing & publicity
 * materials, and acknowledgment shall be given in the documentation, materials
 * and software packages that this Software was used.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
/* -*- mode: c; mode: fold; -*- */
#include "ferite.h"

/*!
 * \fn FeriteFunction *__ferite_create_internal_function( FeriteScript *script, char *name )
 * \brief Allocate a FeriteFunction structure and set it up for use as a script function
 * \param script The script it is being used within
 * \param name The name of the function to be created
 * \return A pointer to a FeriteFunction structure
 * 
 * This function tends to be used only be ferite and is used purely to dump opcode data into
 * to create a native function see __ferite_create_external_function()
 */
FeriteFunction *__ferite_create_internal_function( FeriteScript *script, char *name )
{
   FeriteFunction *ptr;
   int i = 0;

   FE_ENTER_FUNCTION;
   FUD(("Creating Function: %s\n", name));
   ptr = fmalloc( sizeof( FeriteFunction ) );
   ptr->name = fstrdup( name );
   ptr->type = FNC_IS_INTRL;
   ptr->localvars = __ferite_variable_hash_alloc( script, FE_FUNCTION_VARIABLE_SIZE );
   ptr->ccode = __ferite_create_opcode_list( FE_FUNCTION_OPCODE_INIT_SIZE );
   ptr->signature = fmalloc( sizeof( FeriteParameterRecord* ) * FE_FUNCTION_PARAMETER_MAX_SIZE );
   for( i = 0; i < FE_FUNCTION_PARAMETER_MAX_SIZE; i++ )
     ptr->signature[i] = NULL;
   ptr->arg_count = 0;
   ptr->returnt = NULL;
   ptr->next = NULL;
   FE_LEAVE_FUNCTION( ptr );
}

/*!
 * \fn FeriteFunction *__ferite_create_external_function( FeriteScript *script, char *name, void *(*funcPtr)(FeriteScript *, FeriteVariable **), char *description, int is_static )
 * \brief Allocate a FeriteFunction structure and set it up for use as a native function
 * \param script The script it is being used within
 * \param name The name of the function to be created
 * \param funcPtr A pointer to the native c function handling it
 * \param description A description of the functions signiture
 * \param is_static Whether or not the function is a static function (only affects classes). Non-Zero value means true.
 * \return A pointer to a FeriteFunction structure
 * 
 * The description is pretty straight forward - it is simply a sequence of characters eg "oosna" They mean the following:
 * <ul>
 * <li> <b>n</b> - number
 * <li> <b>s</b> - string
 * <li> <b>a</b> - array
 * <li> <b>o</b> - object
 * <li> <b>v</b> - void
 * <li> <b>.</b> - variable argument list
 * </ul>
 * 
 * This means that the function will only actually ever be called with the correct parameters.
 */
FeriteFunction *__ferite_create_external_function( FeriteScript *script, char *name, void *(*funcPtr)(FeriteScript *, FeriteVariable **), char *description, int is_static )
{
   FeriteFunction *ptr;
   int i, in_quote = 0, start = 0, count = 0, has_variable = 0, has_name = 0;
   char buf[4096], type, variablename[1024], initialiser[1024];
   FeriteVariable *new_variable;

   FE_ENTER_FUNCTION;
   FUD(("Creating Function: %s\n", name));
   ptr = fmalloc( sizeof( FeriteFunction ) );
   ptr->name = fstrdup( name );
   ptr->type = FNC_IS_EXTRL;
   ptr->fncPtr = funcPtr;
   ptr->signature = fmalloc( sizeof( FeriteParameterRecord* ) * FE_FUNCTION_PARAMETER_MAX_SIZE );
   for( i = 0; i < FE_FUNCTION_PARAMETER_MAX_SIZE; i++ )
     ptr->signature[i] = NULL;
   ptr->arg_count = 0;

   for( i = 0; i < strlen( description ); i++ )
   {
      if( description[i] == '\"' )
		in_quote = !in_quote;
      if( (description[i] == ',' && !in_quote) || ((i+1) == strlen(description)) )
      {
		 count++;
		 if( strlen(description) == (i+1) )
		   i++;
		 memset( buf, '\0', 4096 );
		 memset( variablename, '\0', 1024 );
		 memset( initialiser, '\0', 1024 );

		 has_name = 0;
		 has_variable = 0;
		 strncpy( buf, description + start, i - start );
		 if( strlen( buf ) > 0 )
		 {
			type = buf[0];
			has_name = __ferite_find_string( buf+1, ":" );
			if( has_name >= 0 )
			{
			   has_variable = __ferite_find_string( buf+2, "=" );
			   if( has_variable >= 0 )
			   {
				  strncpy( variablename, buf+2, __ferite_find_string( buf+2, "=" ) );
				  strcpy( initialiser, buf+2+strlen(variablename)+1 );
			   }
			   else
				 strcpy( variablename, buf+2 );
			   if( strlen( variablename ) <= 0 )
				 sprintf( variablename, "%s_arg%d", name, count );
			}
			else
			  sprintf( variablename, "%s_arg%d", name, count );
			FUD(( "%c %s = %s\n", type, variablename, initialiser ));
			start = i + 1;
			new_variable = NULL;
			switch( type )
			{
			 case 'a': new_variable = __ferite_create_uarray_variable( variablename, 0); break;
			 case 's': new_variable = __ferite_create_string_variable( variablename, initialiser ); break;
			 case 'n': new_variable = __ferite_create_number_long_variable( variablename, atoi(initialiser) ); break;
			 case 'o': new_variable = __ferite_create_object_variable( variablename ); break;
			 case 'v': new_variable = __ferite_create_void_variable( variablename ); break;
			 case '?': new_variable = __ferite_create_void_variable( "?" ); break;
			 case '.': new_variable = __ferite_create_void_variable( "." ); break;
			 default:
			   ferite_error( NULL, "Type '%c' not allowed for function signatures (%s)\n", name );
			   break;
			}
			if( new_variable != NULL )
			{
			   ptr->signature[ptr->arg_count] = fmalloc( sizeof( FeriteParameterRecord* ) );
			   ptr->signature[ptr->arg_count]->variable = new_variable;
			   ptr->signature[ptr->arg_count]->has_default_value = 0;
			   ptr->arg_count++;
			}
		 }
		 else
		   ferite_warning( NULL, "Empty parameter (#%d) in signature for function %s\n", count, name );
      }
   }
   ptr->is_static = is_static;
   ptr->returnt = NULL;
   ptr->next = NULL;
   FE_LEAVE_FUNCTION( ptr );
}

/*!
 * \fn FeriteFunction *__ferite_function_get( FeriteScript *script, char *name )
 * \brief Find a function in the top level namespace of a script
 * \param script The script to search
 * \param name The name of the function
 * \return A pointer to the function on success, NULL on fail
 */
FeriteFunction *__ferite_function_get( FeriteScript *script, char *name )
{
   FeriteNamespaceBucket *nsb = NULL;

   FE_ENTER_FUNCTION;
   FUD(("Searching for function %s\n", name ));
   nsb = __ferite_namespace_element_exists( script, script->mainns, name );
   if( nsb && nsb->type == FENS_FNC )
   {
      FE_LEAVE_FUNCTION( nsb->data );
   }
   FUD(("ERROR: Could not find function %s\n", name));
   FE_LEAVE_FUNCTION( NULL );
}

/*!
 * \fn FeriteOp **__ferite_function_get_compiled_code( FeriteScript *script, char *name )
 * \brief Find a function in the top level namespace of a script and return it's opcode data
 * \param script The script to search
 * \param name The name of the function
 * \return A pointer to the function's opcode on success, NULL on fail
 */
FeriteOp **__ferite_function_get_compiled_code( FeriteScript *script, char *name )
{
   FeriteFunction *ptr;

   FE_ENTER_FUNCTION;

   FUD(("Searching for function %s\n", name ));
   ptr = __ferite_function_get( script, name );
   if( ptr != NULL )
   {
      FUD(("Found %s\n", name ));
      FE_LEAVE_FUNCTION( ptr->ccode->list );
   }
   FUD(("ERROR: Could not find function %s\n", name));
   FE_LEAVE_FUNCTION( NULL );
}

/*!
 * \fn void __ferite_delete_function_list( FeriteScript *script, FeriteFunction *funcs )
 * \brief Delete a list of FeriteFunction's - on most occasions this is only deleteing on function
 * \param script The current script
 * \param funcs The top function in the list
 */
void __ferite_delete_function_list( FeriteScript *script, FeriteFunction *funcs )
{
   int i;

   FE_ENTER_FUNCTION;
   if( funcs )
   {
      if( funcs->next )
		__ferite_delete_function_list( script, funcs->next );

      FUD(("Deleting Function: %s\n", funcs->name ));
      if( funcs->type == FNC_IS_INTRL )
		__ferite_delete_variable_hash( script, funcs->localvars );

      if( funcs->type == FNC_IS_INTRL )
		__ferite_delete_opcode_list( script, funcs->ccode );

      for( i = 0; i < FE_FUNCTION_PARAMETER_MAX_SIZE; i++ )
      {
		 if( funcs->signature[i] != NULL )
		 {
			__ferite_variable_destroy( script, funcs->signature[i]->variable );
			ffree( funcs->signature[i] );
		 }
      }
      ffree( funcs->signature );
      ffree( funcs->name );
      ffree( funcs );
   }
   FE_LEAVE_FUNCTION( NOWT );
}

/*!
 * \fn void __ferite_delete_function_hash( FeriteScript *script, FeriteHash *hash )
 * \brief Delete a hash aswell as cleaning up all functions
 * \param script The current script
 * \param hash The hash to be deleted
 */
void __ferite_delete_function_hash( FeriteScript *script, FeriteHash *hash )
{
   FE_ENTER_FUNCTION;
   __ferite_delete_hash( script, hash, (void (*)(FeriteScript*,void *))__ferite_delete_function_list );
   FE_LEAVE_FUNCTION( NOWT );
}

/*!
 * \fn int __ferite_old_check_params( FeriteScript *script, FeriteVariable **params, FeriteParameterRecord **signature )
 * \brief Check the parameters against the function signiture and see if they match
 * \param script The current script
 * \param params The parameters to be checked
 * \param signature The funcions signature
 * \return 0 on fail, 1 on success
 */
int __ferite_old_check_params( FeriteScript *script, FeriteVariable **params, FeriteParameterRecord **signature )
{
   int correct_params = 1, i = 0;
   int sig_count = ferite_get_parameter_count( (FeriteVariable **)signature );
   int arg_count = ferite_get_parameter_count( params );

   FE_ENTER_FUNCTION;

   FUD(( "sig_count: %d arg_count: %d\n", sig_count, arg_count ));
   
   if( arg_count < sig_count - 1 && signature[arg_count]->variable->name[0] != '.' )
   {
      correct_params = 0;
	  /* ferite_error( "Incorrect Number of paramters\n" ); */
      FE_LEAVE_FUNCTION( correct_params );
   }
   else
   {
      FUD(("\t PASSED TEST 1...\n"));
      for( i = 0; i < sig_count; i++ )
      {
		 FUD(("sig_count: type [%d] [%d]\n", signature[i]->variable->type, PTR2VAR(params[i])->type ));
		 if( signature[i]->variable->name[0] == '.' )
		 {
			correct_params = 1;
			/* ferite_warning( "Found '.' this is a good enough escape caluse for me :)\n" ); */
			FE_LEAVE_FUNCTION( correct_params );
		 }
		 if( signature[i]->variable->type != PTR2VAR(params[i])->type &&
			 !(signature[i]->variable->type == F_VAR_NUM && PTR2VAR(params[i])->type == F_VAR_DOUBLE ) && 
			 signature[i]->variable->type != F_VAR_VOID /* this should allow the person to pass any value */ )
		 {
			correct_params = 0;
			ferite_warning( script, "Found type %s but expected %s for argument #%d\n", __ferite_variable_id_to_str( script, PTR2VAR(params[i])->type ),
							__ferite_variable_id_to_str( script, signature[i]->variable->type ),
							i+1 );
			break;
		 }
      }
	  /* if we get here and i < arg_count we have too many args */
	  if( i < arg_count ){
		 correct_params = 0;
	  }
   }
   FE_LEAVE_FUNCTION( correct_params );
}

/*!
 *  * \fn int __ferite_check_params( FeriteScript *script, FeriteVariable **params, FeriteParameterRecord **signature )
 *  * \brief Check the parameters against the function signiture and see if they match
 *  * \param script The current script
 *  * \param params The parameters to be checked
 *  * \param signature The funcions signature
 *  * \return 0 on fail, 1 on success
 *  */
int __ferite_check_params( FeriteScript *script, FeriteVariable **params, FeriteParameterRecord **signature) {
   int arg_count, sig_count;
   int i, has_period = 0;
   
   FE_ENTER_FUNCTION;
	 
   arg_count = ferite_get_parameter_count (params);
   sig_count = ferite_get_parameter_count ((FeriteVariable **)signature);
   
   /* If both the signature and the parameters are empty, we can safely assume that the params are correct */
   if (arg_count == 0 && sig_count == 0) {
	  FE_LEAVE_FUNCTION( 1 );
   }                              
   
   for (i = 0; signature[i]; i++) {
	  if (signature[i]->variable->name[0] == '.') {
		 has_period = 1;
		 break;
	  }
	  if (!params[i]) {
		 ferite_warning (script, "Insufficient parameters\n");
		 FE_LEAVE_FUNCTION( 0 );
	  }
	  
	  if (signature[i]->variable->type != PTR2VAR(params[i])->type &&
		  !(signature[i]->variable->type == F_VAR_NUM && PTR2VAR(params[i])->type == F_VAR_DOUBLE ) &&
		  signature[i]->variable->type != F_VAR_VOID) { 
		 /* Found an invalid type */
		 ferite_warning (script, "Found type %s but expected %s for argument #%d\n",
						 __ferite_variable_id_to_str( script, PTR2VAR(params[i])->type ),
						 i+1 );
		 FE_LEAVE_FUNCTION( 0 );
	  }
   }        
   
   if (has_period) {
	  int offset;
	  
	  /* for when we dont have a var for the '...' */
	  if( signature[i]->variable->name[0] == '.' && !params[i] ){
		 FE_LEAVE_FUNCTION( 1 );
	  }
	  
	  i++; /* Skip the period signature */
	  offset = arg_count - i;
	  
	  if (offset < 0) {
		 /* Invalid Parameter count, an offset of less than zero means ABS(offset) parameters are missing */
		 ferite_warning (script, "Insufficient parameters.\n");
		 FE_LEAVE_FUNCTION( 0 );
	  }
	  
	  for (; params[i + offset] && signature[i]; i++) {
		 /* We cannot find another period.  If we do, something is WRONG */
		 if (signature[i]->variable->name[0] == '.') {
			ferite_warning (script, "Found multiple period's in signature!\n");
			break;
		 }
		 
		 if (signature[i]->variable->type != PTR2VAR(params[i + offset])->type &&
			 !(signature[i]->variable->type == F_VAR_NUM && PTR2VAR(params[i + offset])->type == F_VAR_DOUBLE ) &&
			 signature[i]->variable->type != F_VAR_VOID) {
			/* Found an invalid type */
			ferite_warning (script, "Found type %s but expected %s for argument #%d\n",
							__ferite_variable_id_to_str( script, PTR2VAR(params[i])->type ),
							__ferite_variable_id_to_str( script, signature[i]->variable->type ),
							i+1 );
			FE_LEAVE_FUNCTION( 0 );
		 }
	  }
   } else {
	  if ((params[i] && !signature[i]) || (!params[i] && signature[i])) {
		 /* Signature and parameters didn't run out at the same time, invalid parameter count */
		 if (params[i] && !signature[i]) {
			ferite_warning (script, "Too many parameters\n");
		 } else {       
			ferite_warning (script, "Insufficient parameters.\n");
		 }
		 FE_LEAVE_FUNCTION( 0 );
	  }
   }
   FE_LEAVE_FUNCTION( 1 );
}
