/* -*- mode: c; mode: fold; -*- */
/*
 * Copyright (C) 2000-2001 Chris Ross and Evan Webb
 * Copyright (C) 1999-2000 Chris Ross
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *   
 * The above copyright notice and this permission notice shall be included in
 * all copies of the Software, its documentation and marketing & publicity 
 * materials, and acknowledgment shall be given in the documentation, materials
 * and software packages that this Software was used.
 *    
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER 
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "ferite.h"

/*!
 * \fn void __ferite_stop_execution( FeriteScript *script )
 * \brief Stop the script executing
 * \param script The script to stop
 */
void __ferite_stop_execution( FeriteScript *script )
{
   FE_ENTER_FUNCTION;
   script->keep_execution = 0;
   FE_LEAVE_FUNCTION( NOWT );
}

/*!
 * \fn int __ferite_is_executing( FeriteScript *script )
 * \brief Return whether a script is executing
 * \param script The script to query
 */
int __ferite_is_executing( FeriteScript *script )
{
   FE_ENTER_FUNCTION;
   FE_LEAVE_FUNCTION( script->is_executing );
}

/*!
 * \fn int ferite_script_execute( FeriteScript *script )
 * \brief Run a script
 * \param script The script to run
 */
int ferite_script_execute( FeriteScript *script )
{
   FeriteNamespaceBucket *nsb = NULL;
   FeriteFunction *func = NULL;
   FeriteVariable *err = NULL, *errstr = NULL, *erno = NULL;
   int rval = 0;

   FE_ENTER_FUNCTION;
   script->error_state = 0;
   script->is_executing = 1;
   nsb = __ferite_namespace_element_exists( script, script->mainns, "_start" );
   func = nsb->data;
   rval = __ferite_script_function_execute( script, func, NULL );

   /* clean up the system */
   __ferite_variable_destroy( script, func->returnt );
   nsb = __ferite_namespace_element_exists( script, script->mainns, "err" );
   if( script->error_state == ERROR_THROWN ) /* error propogated up the system */
   {
      err = nsb->data;
      errstr = __ferite_get_variable_from_hash( script, VAO(err)->variables, "errstr" );
      erno = __ferite_get_variable_from_hash( script, VAO(err)->variables, "errno" );
      if( script->error_cb != NULL )
	  {
		 (script->error_cb)( script, VAS(errstr), VAI(erno) );
      }
	  else
	  {
		 fprintf( stderr, "[ferite]: Fatal Error: Execution stopped:\n" );
		 fprintf( stderr, "  Reason: %s", VAS(errstr), VAI(erno) );
		 fprintf( stderr, "  On line %d, in file '%s'\n\n", script->current_op_line, script->current_op_file );
      }
   }
   /* this basically removes the issues lying around with the err object - the only one that still lies around after
    * execution is over */
   if( VAO(PTR2VAR(nsb->data)) != NULL )
   {
      VAO(PTR2VAR(nsb->data))->refcount--;
      __ferite_check_gc(); /* needed to remove shody stuff from t'system */
   }
   script->is_executing = 0;
   FE_LEAVE_FUNCTION( rval );
}

/*!
 * \fn int __ferite_script_function_execute( FeriteScript *script,  FeriteFunction *function, FeriteVariable **params )
 * \brief Execute a function within a script
 * \param script The current script
 * \param function The function to execute
 * \param params The calling arguments
 */
int __ferite_script_function_execute( FeriteScript *script,  FeriteFunction *function, FeriteVariable **params )
{
   FeriteExecuteRec    *exec = NULL;
   FeriteVariable      *targetvar = NULL, *fargs = NULL;
   int i = 0, rval = 0, stop_execute = 0;

   FE_ENTER_FUNCTION;

   /*{{{ Checks before we do anything */
   if( !script->is_executing )
   {
      script->is_executing = 1;
      stop_execute = 1;
   }
   if( function == NULL )
   {
      ferite_error( script, "Trying to execute a NULL function" );
      return 0;
   }
   if( script == NULL )
   {
      ferite_error( script, "Trying to execute a function without a script" );
      return 0;
   }
   script->keep_execution = 1;
   /*}}}*/

   /*{{{ obtain function, and create local variable hash */
   exec = fmalloc( sizeof( FeriteExecuteRec ) );
   exec->function = function;
   exec->variable_hash = __ferite_duplicate_variable_hash( script, function->localvars );
   exec->stack = __ferite_create_stack( FE_EXECUTOR_STACK_SIZE );
   /*}}}*/

   /*{{{ Copy paramceter values over to the local variable stack */
   if( params != NULL ) /* handle function parameters */
   {
      int arg_count = ferite_get_parameter_count( params ); 
	  int doMoreArgs = 1;
	  
	  fargs = __ferite_create_uarray_variable( "fncArgs", arg_count+1 ); 
	  for( i = 0; i < arg_count; i++ )
      {
		 if( function->signature[i] != NULL && doMoreArgs ){
			targetvar = __ferite_get_variable_from_hash( script, exec->variable_hash, function->signature[i]->variable->name );
			__ferite_variable_destroy( script, __ferite_op_assign( script, targetvar, params[i] ) );
			if( function->signature[i]->variable->name[0] == '.' ){ /* we stop assigning to the variable hash once we hit the
																	 * ... marker -> we need to mod the compiler 
																	 */
			   doMoreArgs = 0;
			}
		 }		 
		 /* we add the variable to the fargs array */
		 FUD(( "EXECUTER: function %s: param[%d] == '%s' %d\n", function->name, i, params[i]->name, arg_count )); 
		 
		 /* get self into the variable hash by skipping the fncArgs */
		 if( strcmp( params[i]->name, "super" ) == 0 && (i+2) == arg_count && strcmp( params[i+1]->name, "self" ) == 0 )
		   continue;
		 
		 if( strcmp( params[i]->name, "self" ) == 0 && (i+1) == arg_count )
		   break;
		 
		 targetvar = __ferite_create_void_variable( "fncArgs-value" );
		 __ferite_variable_destroy( script, __ferite_op_assign( script, targetvar, params[i] ) );
		 __ferite_uarray_add( script, VAUA(fargs), targetvar, NULL, i );
      }
	  __ferite_add_variable_to_hash( script, exec->variable_hash, fargs );
   }
   /*}}}*/

   /* push the exec rec onto the script execution stack */
   __ferite_stack_push( script->exec_stack, exec );

   FUD(("EXECUTOR: Executing function %s %p\n", function->name, function->ccode->list[1] ));
   rval = __ferite_script_real_function_execute( script, function, script->mainns, exec );

   /* clean up */
   __ferite_stack_pop( script->exec_stack );
   __ferite_clean_up_exec_rec( script, exec );

   if( stop_execute )
     script->is_executing = 0;

   __ferite_check_gc(); /* needed to remove shody stuff from t'system */
   FE_LEAVE_FUNCTION( rval );
}

/*!
 * \fn int __ferite_script_eval_execute( FeriteScript *script, FeriteFunction *function, FeriteNamespace *ns, FeriteExecuteRec *exec )
 * \brief Execute a eval operator compiled script
 * \param script The current script to run
 * \param function The function to execute
 * \param ns The eval operators namespace
 * \param exec The execute records for the current script
 */
int __ferite_script_eval_execute( FeriteScript *script, FeriteFunction *function, FeriteNamespace *ns, FeriteExecuteRec *exec )
{
   int rval;

   FE_ENTER_FUNCTION;
   __ferite_stack_push( script->exec_stack, exec );
   rval = __ferite_script_real_function_execute( script, function, script->mainns, exec );
   __ferite_stack_pop( script->exec_stack );
   __ferite_check_gc(); /* needed to remove shody stuff from t'system */
   FE_LEAVE_FUNCTION( rval );
}

/*!
 * \fn int __ferite_script_real_function_execute( FeriteScript *script, FeriteFunction *function, FeriteNamespace *ns, FeriteExecuteRec *exec )
 * \brief Execute a eval operator compiled script
 * \param script The current script to run
 * \param function The function to execute
 * \param ns The scripts main namespace
 * \param exec The execute records for the current script
 */
int __ferite_script_real_function_execute( FeriteScript *script, FeriteFunction *function, FeriteNamespace *mainns, FeriteExecuteRec *exec )
{
   /*{{{ Variables */
   FeriteNamespaceBucket *nsb = NULL;
   FeriteOp              *current_op = NULL;
   FeriteOp             **opcode_list = NULL;
   FeriteVariable        *varone = NULL, *vartwo = NULL, *result = NULL, *result2 = NULL;
   FeriteVariable        *(*unaryop)( FeriteScript *s, FeriteVariable *a );
   FeriteVariable        *(*binaryop)( FeriteScript *s, FeriteVariable *a, FeriteVariable *b );
   FeriteVariable       **param_list = NULL;
   FeriteFunction        *trgt_function_call = NULL;
   FeriteClass           *sclass;
   FeriteStack           *sorter;
   int                    current_op_loc = 0, keep_function_running = 1, i = 0, j = 0, error_op_location = 0, error_array[100];
   /*}}}*/

   FE_ENTER_FUNCTION;

   sorter = __ferite_create_stack(FE_FUNCTION_PARAMETER_MAX_SIZE);
   opcode_list = function->ccode->list;
   current_op = opcode_list[0];
   current_op_loc++;
   
   script->current_op_file = function->ccode->filename;

   FUD(("EXECUTION STARTING\n"));
   while( keep_function_running && script->keep_execution )
   {
      varone = NULL;
      vartwo = NULL;
      FUD(("[%p] ", current_op ));
	  
	  script->current_op_line = current_op->line;
      switch( current_op->OP_TYPE )
      {
       /*{{{ F_OP_PUSH      (push something onto the stack)   */
       case F_OP_PUSH:
		 FUD(("PUSH\n"));
		 __ferite_stack_push( exec->stack, current_op->opdata );
		 break;
       /*}}}*/
       /*{{{ F_OP_POP       (pop something off the stack)     */
       case F_OP_POP:
		 FUD(("POP\n"));
		 varone = __ferite_stack_pop( exec->stack );
		 if( varone && varone->flags.disposable )
		 {
			FUD(("Deleting Variable: %s\n", varone->name ));
			__ferite_variable_destroy( script, varone );
		 }
		 break;
       /*}}}*/
       /*{{{ F_OP_PUSHVAR   (get variable and put onto stack) */
       case F_OP_PUSHVAR:
		 FUD(("PUSHVAR\n"));
		 if( ((char *)(current_op->opdata))[0] != '.' )
		 {
			/* try and get it from the local scoped variable hash */
			varone = __ferite_get_variable_from_hash( script, exec->variable_hash, (char *)(current_op->opdata) );
			if( varone == NULL ) /* search main:: */
			{
			   nsb = __ferite_namespace_element_exists( script, mainns, (char *)(current_op->opdata) );
			   if( nsb != NULL )
			   {
				  FE_ASSERT( nsb->type == FENS_VAR || nsb->type == FENS_NS || nsb->type == FENS_CLS );
				  switch( nsb->type ){
				   case FENS_NS:
					 /* we need to do some trickery here. I just have to work out what that is :)
					  * ok plan of action, create a varaible, set it's type to F_VAR_NS, push it onto the stack,
					  * it should *never* get out of the executor. */
					 varone = __ferite_variable_alloc();
					 varone->name = fstrdup( (char *)(current_op->opdata) );
					 varone->type = F_VAR_NS;
					 varone->data.pval = nsb->data;
					 MARK_VARIABLE_AS_DISPOSABLE( varone );
					 break;
				   case FENS_VAR:
					 varone = nsb->data;
					 break;
				   case FENS_CLS:
					 /* we need to do some trickery here. I just have to work out what that is :)
					  * ok plan of action, create a varaible, set it's type to F_VAR_NS, push it onto the stack,
					  * it should *never* get out of the executor. */
					 varone = __ferite_variable_alloc();
					 varone->name = fstrdup( (char *)(current_op->opdata) );
					 varone->type = F_VAR_CLASS;
					 varone->data.pval = nsb->data;
					 FUD(( "pushing class %s\n", (char *)(current_op->opdata) ));
					 MARK_VARIABLE_AS_DISPOSABLE( varone );					 
					 break;
				  }
			   }
			   else
				 varone = NULL;
			}
			/* we should only ever get here on a eval() as the variables declared on the eval() will only be present within
			 * the function->localvars */
			if( varone == NULL )
			{
			   varone = __ferite_get_variable_from_hash( script, function->localvars, (char *)(current_op->opdata) );
			}
		 }
		 else
		 {
			vartwo = __ferite_stack_pop( exec->stack );
			switch( vartwo->type )
			{
			 case F_VAR_OBJ:
			   FUD(( "Executing: Searching for '%s' in '%s'\n", (char *)(current_op->opdata)+1, vartwo->name ));
			   if( VAO(vartwo) != NULL ){
				  varone = __ferite_get_variable_from_hash( script, VAO(vartwo)->variables,  (char *)(current_op->opdata)+1 );
				  if( varone != NULL ){
					 if( varone->flags.is_static ){
						ferite_error( script, "Trying to access static variable '%s' in object '%s'\n", (char *)(current_op->opdata)+1, vartwo->name );
					 }
				  } else {
					 ferite_error( script, "Trying to access non-existant variable '%s' in obejct '%s'\n", (char *)(current_op->opdata)+1, vartwo->name );
				  }
			   } else {
				  ferite_error( script, "Trying to access variable '%s' in obejct '%s' which is null\n", (char *)(current_op->opdata)+1, vartwo->name );
			   }
			   break;
			 case F_VAR_NS:
			   nsb = __ferite_namespace_element_exists( script, (FeriteNamespace *)(vartwo->data.pval), (char *)(current_op->opdata)+1 );
			   if( nsb == NULL )
			   {
				  ferite_error( script, "Can't find '%s' in namespace '%s'\n", (char *)(current_op->opdata)+1, vartwo->name );
				  break;
			   }
			   if( nsb->type != FENS_VAR && nsb->type != FENS_NS && nsb->type != FENS_CLS)
			   {
				  ferite_error( script, "Expecting variable, class or namespace for '%s' in '%s'\n", (char *)(current_op->opdata)+1, vartwo->name );
				  break;
			   }
			   switch( nsb->type ){
				case FENS_NS:
				  /* we need to do some trickery here. I just have to work out what that is :)
				   * ok plan of action, create a varaible, set it's type to F_VAR_NS, push it onto the stack,
				   * it should *never* get out of the executor. */
				  varone = __ferite_variable_alloc();
				  varone->name = fstrdup( (char *)(current_op->opdata)+1 );
				  varone->type = F_VAR_NS;
				  varone->data.pval = nsb->data;
				  MARK_VARIABLE_AS_DISPOSABLE( varone );
				  break;
				case FENS_VAR:
				  varone = nsb->data;
				  break;
				case FENS_CLS:
				  /* we need to do some trickery here. I just have to work out what that is :)
				   * ok plan of action, create a varaible, set it's type to F_VAR_NS, push it onto the stack,
				   * it should *never* get out of the executor. */
				  varone = __ferite_variable_alloc();
				  varone->name = fstrdup( (char *)(current_op->opdata)+1 );
				  varone->type = F_VAR_CLASS;
				  varone->data.pval = nsb->data;
				  FUD(( "EXEC: pushing class %s\n", (char *)(current_op->opdata) ));
				  MARK_VARIABLE_AS_DISPOSABLE( varone );					 
				  break;
			   }
			   break;
			 case F_VAR_CLASS:
			   varone = __ferite_get_variable_from_hash( script, ((FeriteClass*)(vartwo->data.pval))->variables, (char *)(current_op->opdata)+1 );
			   if( varone == NULL ){
				  ferite_error( script, "Unable to find variable %s in class %s\n", (char *)(current_op->opdata)+1, vartwo->name );
			   } else {
				  if( varone->flags.is_static != 1 ){
					 ferite_error( script, "Trying to access non-static variable %s in class %s\n", (char *)(current_op->opdata)+1, vartwo->name );					 
				  }
			   }
			   break;
			 default:
			   ferite_error( script, "Can not get variable '%s' from '%s', expecting object/namespace/class but found a %s.\n",
							(char *)(current_op->opdata),
							vartwo->name,
							__ferite_variable_id_to_str( script, vartwo->type ) );
			   break;
			}
		 }
		 if( vartwo && vartwo->flags.disposable )
		   __ferite_variable_destroy( script, vartwo );
		 if( script->error_state == ERROR_THROWN )
		   break;

		 if( varone != NULL )
		   __ferite_stack_push( exec->stack, varone );
		 else
		 {
			ferite_error( script, "Can't Find Variable '%s'\n",  (char *)(current_op->opdata) );
			break;
		 }
		 break;
	 /*}}}*/
       /*{{{ F_OP_UNARY     (do a unary operation)            */
       case F_OP_UNARY:
		 FUD(("UNARY\n"));
		 varone = __ferite_stack_pop( exec->stack );
		 unaryop = (FeriteVariable *(*)( FeriteScript *, FeriteVariable * ))current_op->op;
		 FUD(("Unary Op Parameters: %s\n", varone->name ));
		 result = unaryop( script, varone );
		 if( varone->flags.disposable )
		 {
			FUD(("Deleting Variable: %s\n", varone->name ));
			__ferite_variable_destroy( script, varone );
		 }
		 if( result == NULL )
		 {
			break;
		 }
		 else
		   __ferite_stack_push( exec->stack, result );
		 break;
       /*}}}*/
       /*{{{ F_OP_BINARY    (do a binary operation)           */
       case F_OP_BINARY:
		 FUD(("BINARY\n"));
		 vartwo = __ferite_stack_pop( exec->stack );
		 varone = __ferite_stack_pop( exec->stack );
		 binaryop = (FeriteVariable *(*)( FeriteScript *, FeriteVariable *, FeriteVariable * ))current_op->op;
		 FUD(("Binary Op Parameters: %s, %s\n", varone->name, vartwo->name ));
		 result =  binaryop( script, varone, vartwo );
		 if( vartwo->flags.disposable )
		 {
			FUD(("Deleting Variable: %s\n", vartwo->name ));
			__ferite_variable_destroy( script, vartwo );
		 }
		 if( varone->flags.disposable )
		 {
			FUD(("Deleting Variable: %s\n", varone->name ));
			__ferite_variable_destroy( script, varone );
		 }
		 if( result == NULL )
		 {
			break;
		 }
		 else
		   __ferite_stack_push( exec->stack, result );
		 break;
       /*}}}*/
       /*{{{ Function stuff (function, method and new)        */
       case F_OP_METHOD:
       case F_OP_FUNCTION:
       case F_OP_NEWOBJ:
		 /*{{{ Create Parameter List */
		 i = 0;
		 FUD(("CREATING PARAMETER LIST.\n"));
		 do
		 {
			__ferite_stack_push( sorter, (varone=__ferite_stack_pop( exec->stack )) );
			i++;
		 }
		 while( strcmp( varone->name, "paramstrt" ) != 0 );
		 __ferite_stack_pop( sorter );
		 param_list = __ferite_create_parameter_list( sorter->stack_ptr + 4 );
		 for( j = 0; j < i; j++ )
		   param_list[j] = __ferite_stack_pop( sorter );
		 FUD(("PARAMETER LIST CREATED.\n"));
		 /*}}}*/
		 switch( current_op->OP_TYPE )
		 {
			/*{{{ F_OP_METHOD (object and namespace) */
		  case F_OP_METHOD:
			vartwo = __ferite_stack_pop( exec->stack );
			switch( vartwo->type )
			{
			 case F_VAR_OBJ:
			   FUD(( "Trying to find '%s' in '%s'\n", (char *)(current_op->opdata), vartwo->name ));
			   /* what we do here is check to see if they are searching the super class, if so, we swap the objects           */
			   /* template with that of it's parents - if it has no parent we don't bother..., we then look for the function  */
			   /* that we are after..... i personally would consider this to be, as we english say, "dash cunning"            */
			   if( VAO(vartwo) == NULL )
			   {
				  ferite_error( script, "Trying to access method '%s' in a null object '%s'", (char *)(current_op->opdata), vartwo->name );
				  break;
			   }
			   sclass = VAO(vartwo)->tmpl;
			   if( strcmp( vartwo->name, "super" ) == 0 )
			   {
				  if( VAO(vartwo)->tmpl->parent != NULL )
				  {
					 VAO(vartwo)->tmpl = VAO(vartwo)->tmpl->parent;
				  }
			   }
			   trgt_function_call = __ferite_find_function_in_object( script, VAO(vartwo), (char *)(current_op->opdata) );
			   VAO(vartwo)->tmpl = sclass;
			   if( trgt_function_call == NULL )
			   {
				  /* ok this is where we try and be dash cunning, fail miserably and settle for an autoload function. */
				  trgt_function_call = __ferite_find_function_in_object( script, VAO(vartwo), ".autoload" );
				  if( trgt_function_call == NULL ){					 
					 ferite_error( script, "Unable to find method '%s' in object '%s'", (char *)(current_op->opdata)+1, vartwo->name );
					 break;
				  } else { /* we have an autoload function, soooooooo, what do we do? */
					 /* There are two ways in which we can handle the autoload call, we either shift the parameter stack 
					  * and place the name of the function first, this would make life easier, but then again when has
					  * life ever been easy? */
					 varone = fe_new_str( "function-name", (char *)(current_op->opdata)+1 );
					 param_list = __ferite_add_to_parameter_list( param_list, varone );
					 MARK_VARIABLE_AS_DISPOSABLE( varone );
				  }
			   }
			   if( trgt_function_call->is_static ){
				  ferite_error( script, "Trying to access static member method %s through object %s", (char *)(current_op->opdata)+1, vartwo->name );
				  break;
			   }
			   /* den swap it back :)                                                                                         */
			   /*                                                                                                             */
			   /* NB: we dont need to do this for varaibles because each class automatically get a duplicae of it's parents   */
			   /*     varaibles hash -> so all objects within the system upn definition.                                      */
			   break;
			 case F_VAR_NS:
			   nsb = __ferite_namespace_element_exists( script, (FeriteNamespace *)(vartwo->data.pval), (char *)(current_op->opdata)+1 );
			   if( nsb == NULL || nsb->type != FENS_FNC )
			   {
				  ferite_error( script, "Unable to find method '%s' in namespace '%s'\n", (char *)(current_op->opdata)+1, vartwo->name );
				  break;
			   }
			   trgt_function_call = nsb->data;
			   break;
			 case F_VAR_CLASS:
			   trgt_function_call = __ferite_hash_get( script, ((FeriteClass*)(vartwo->data.pval))->functions, (char *)(current_op->opdata)+1 );
			   if( trgt_function_call == NULL ){
				  ferite_error( script, "Unable to access method %s within class %s", (char *)(current_op->opdata)+1, vartwo->name );
				  break;
			   } else {
				  if( !(trgt_function_call->is_static) ){
					 ferite_error( script, "Trying to access non-static method %s within class %s", (char *)(current_op->opdata)+1, vartwo->name );
				  }
			   }
			   break;
			 default:
			   ferite_error( script, "Expecting container found %s when trying to call %s.%s()",
							 __ferite_variable_id_to_str( script, vartwo->type ),
							 vartwo->name,
							 (char *)(current_op->opdata)+1 );
			   if( vartwo && vartwo->flags.disposable ) /* the var was created */
				 __ferite_variable_destroy( script, vartwo );
			   break;
			}
			/* make a quick gettaway */
			if( script->error_state == ERROR_THROWN )
			{
			   if( vartwo && vartwo->flags.disposable ) /* the var was created */
				 __ferite_variable_destroy( script, vartwo );
			   break;
			}
			if( trgt_function_call != NULL )
			{
			   if( vartwo->type == F_VAR_OBJ )
			   {
				  FUD(( "Adding super and self to the objects method call\n" ));
				  /* another pointer - but with a special name ;) (see above for information.... */
				  result2 = __ferite_duplicate_variable( script, vartwo );
				  MARK_VARIABLE_AS_DISPOSABLE( result2 );
				  ffree( result2->name );
				  result2->name = fstrdup( "super" );
				  param_list = __ferite_add_to_parameter_list( param_list, result2 );

				  /* a pointer to the object it self in the object's function */
				  result = __ferite_duplicate_variable( script, vartwo );
				  MARK_VARIABLE_AS_DISPOSABLE( result );
				  ffree( result->name );
				  result->name = fstrdup( "self" );
				  param_list = __ferite_add_to_parameter_list( param_list, result );
			   }
			   /* ferite_warning( "Return from check params: %d\n", __ferite_check_params( param_list, trgt_function_call->signature ) ); */
			   if( __ferite_check_params( script, param_list, trgt_function_call->signature ) != 0 )
			   {
				  if( trgt_function_call->type == FNC_IS_EXTRL )
				  {
					 __ferite_stack_push( exec->stack, (trgt_function_call->fncPtr)( script, param_list ) );
				  }
				  else /* internal function call */
				  {
					 __ferite_script_function_execute( script, trgt_function_call, param_list );
					 __ferite_stack_push( exec->stack, trgt_function_call->returnt );
				  }
			   }
			   else
			   {
				  ferite_error( script, "Wrong parameter types for function %s.%s\n", ( vartwo->type == F_VAR_OBJ ? VAO(vartwo)->tmpl->name : vartwo->name ), trgt_function_call->name );
				  if( vartwo && vartwo->flags.disposable ) /* the var was created */
					__ferite_variable_destroy( script, vartwo );
				  break;
			   }
			}
			else
			{
			   ferite_error( script,"Error: Can't find function %s() in object %s %s\n",
							(char *)(current_op->opdata),
							VAO(vartwo)->name,
							(VAO(vartwo)->name == NULL ? "\n       (cause: object has not been created?)" : ""));
			   if( vartwo && vartwo->flags.disposable ) /* the var was created */
				 __ferite_variable_destroy( script, vartwo );
			   break;
			}

			if( vartwo && vartwo->flags.disposable ) /* the var was created */
			  __ferite_variable_destroy( script, vartwo );
			break;
	    /*}}}*/
			/*{{{ F_OP_FUNCTION (script) */
		  case F_OP_FUNCTION:
			trgt_function_call = NULL;
			nsb = __ferite_namespace_element_exists( script, mainns, (char *)(current_op->opdata) );
			if( nsb != NULL )
			  trgt_function_call = nsb->data;
			if( trgt_function_call == NULL && script ) /* we are in an eval check parent script */
			{
			   nsb = __ferite_namespace_element_exists( script, script->mainns, (char *)(current_op->opdata) );
			   if( nsb != NULL )
				 trgt_function_call = nsb->data;
			}
			if( trgt_function_call == NULL ) /* we really can't fiund it */
			{
			   __ferite_raise_script_error( script, 0, "Can't find function '%s'.", (char *)(current_op->opdata) );
			   break;
			}
			if( __ferite_check_params( script, param_list, trgt_function_call->signature ) )
			{
			   __ferite_script_function_execute( script, trgt_function_call, param_list );
			   __ferite_stack_push( exec->stack, trgt_function_call->returnt );
			}
			else
			{
			   ferite_error( script, "Wrong parameter types for function %s\n", (char *)(current_op->opdata) );
			   break;
			}
			break;
	  /*}}}*/
			/*{{{ F_OP_NEWOBJ */
		  case F_OP_NEWOBJ:
			FUD(("Creating new object\n"));
			vartwo = __ferite_stack_pop( exec->stack );
			if( vartwo->type != F_VAR_CLASS ){
			   ferite_error( script, "%s is not a class, bad luck, try again :)", vartwo->name );
			   break;
			}
			__ferite_stack_push( exec->stack, (void *)__ferite_new_object( script, (FeriteClass *)(vartwo->data.pval), param_list ) );
			if( vartwo && vartwo->flags.disposable ) /* the var was created */
			  __ferite_variable_destroy( script, vartwo );
			break;
	  /*}}}*/
		 }
		 /*{{{ delete parameter list */
		 __ferite_delete_parameter_list( script, param_list );
	 /*}}}*/
		 break;
       /*}}}*/
       /*{{{ F_OP_BNE       (branch if not equal)             */
       case F_OP_BNE:
		 FUD(("BNE\n"));
		 varone = __ferite_stack_pop( exec->stack );
		 if( __ferite_variable_is_false( script, varone) )
		 {
			FUD(("BNE: Branching\n" ));
            current_op_loc = current_op->addr;
			if( varone->flags.disposable ) /* the var was created */
			  __ferite_variable_destroy( script, varone );
			break;
		 }
		 FUD(("BNE: Not Branching\n" ));
		 if( varone->flags.disposable ) /* the var was created */
		   __ferite_variable_destroy( script, varone );
		 break;
       /*}}}*/
       /*{{{ F_OP_BIE       (branch if equal)                 */
       case F_OP_BIE:
		 FUD(("BIE\n"));
		 varone = __ferite_stack_pop( exec->stack );
		 if( !__ferite_variable_is_false( script, varone ) )
		 {
			FUD(("BIE: Branching\n" ));
			current_op_loc = current_op->addr;
			if( varone->flags.disposable ) /* the var was created */
			  __ferite_variable_destroy( script, varone );
			break;
		 }
		 FUD(("BIE: Not Branching\n" ));
		 if( varone->flags.disposable ) /* the var was created */
		   __ferite_variable_destroy( script, varone );
		 break;
       /*}}}*/
       /*{{{ F_OP_JMP       (jump to a address)               */
       case F_OP_JMP:
		 FUD(("JMP\n"));
		 current_op_loc = current_op->addr;
		 break;
       /*}}}*/
       /*{{{ F_OP_NOP       (do nothing)                      */
       case F_OP_NOP:
		 FUD(("NOP. Nothing Done :)\n"));
		 break;
       /*}}}*/
       /*{{{ F_OP_RGX       (regular expression instruction)  */
       case F_OP_RGX:
		 varone = __ferite_stack_pop( exec->stack );
		 FUD(( "REGEX: Applying Regex '%s' to %s\n", PARGX(current_op->opdata)->compile_buf, varone->name ));
		 vartwo = __ferite_execute_regex( PARGX(current_op->opdata), varone, script );
		 MARK_VARIABLE_AS_DISPOSABLE( vartwo );
		 __ferite_stack_push( exec->stack, vartwo );
		 if( varone && varone->flags.disposable )
		 {
			FUD(("Deleting Variable: %s\n", varone->name ));
			__ferite_variable_destroy( script, varone );
		 }
		 break;
	 /*}}}*/
       /*{{{ F_OP_EXIT      (exit and return from a function) */
       case F_OP_EXIT:
		 FUD(("Exiting\n"));
		 keep_function_running = 0; /* quit the function */
		 varone = __ferite_stack_pop( exec->stack );
		 function->returnt = __ferite_duplicate_variable( script, varone );
		 MARK_VARIABLE_AS_DISPOSABLE( function->returnt );
		 if( varone && varone->flags.disposable )
		 {
			FUD(("Deleting Variable: %s\n", varone->name ));
			__ferite_variable_destroy( script, varone );
		 }
		 /*__ferite_stack_push( exec->stack, varone );*/ /* so we dont lose memory */
		 break;
		 /*}}}*/
       /*{{{ F_OP_ERR       (set an error handler)            */
       case F_OP_ERR:
		 if( current_op->addr == -1 )
		 {  /* reset the error counter */
			script->error_state = NO_ERROR;
			error_array[--error_op_location] = 0;
		 }
		 else
		 {
			FUD(("ERROR HANDLER: Setting Error location to %ld",  current_op->addr ));
			error_array[error_op_location++] = current_op->addr;
		 }
		 break;
       /*}}}*/
       default:
		 ferite_error( script, "Unknown op type [%d]", current_op->OP_TYPE );
      }

      if( !keep_function_running || !script->keep_execution)
		break;

      current_op = opcode_list[current_op_loc];
      current_op_loc++;

      /*{{{ GARBAGE COLLECTOR */
      genv->total_ops_run++;
      if( genv->total_ops_run > FE_GC_RUN_AFTER_OPS )
      {
		 __ferite_check_gc();
		 genv->total_ops_run = 0;
      }
      /*}}}*/

      /*{{{ error checking */
      FUD(( "ERROR STATE: %d\n", script->error_state ));
      switch( script->error_state )
      {
       case ERROR_THROWN:
		 FUD(( "ERROR STATE: reported...\n" ));
		 if( error_op_location < 1 || error_array[error_op_location-1] == 0 )
		 { /* there is no error handler propogate upwards */
			FUD(( "ERROR STATE: No error handler found... bombing out.\n" ));
			FUD(( "EXEC: detected error - stoping execution\n" ));
			keep_function_running = 0;
		 }
		 else
		 {
			FUD(( "ERROR STATE: Going to error handler code\n" ));
			current_op_loc = error_array[error_op_location-1];
			current_op = opcode_list[current_op_loc];
			current_op_loc++;
		 }
		 break;
      }
      /*}}}*/
   }
   __ferite_delete_stack( sorter );
   FUD(("EXECUTION COMPLETE. Have a nice day :). (%s)\n", function->name));
   FE_LEAVE_FUNCTION( script->keep_execution );
}

/*!
 * \fn void __ferite_clean_up_exec_rec( FeriteScript *script, FeriteExecuteRec *exec )
 * \brief Clean up and Execution Record
 * \param script The current script
 * \param exec   Pointer to the execution record
 */
void __ferite_clean_up_exec_rec( FeriteScript *script, FeriteExecuteRec *exec )
{
   int counter, i;
   FeriteVariable *targetvar;

   FE_ENTER_FUNCTION;
   /*{{{ Clean up local scope variables */
   FUD(("DELETING LOCAL VARIABLES\n" ));
   __ferite_delete_variable_hash( script, exec->variable_hash );
   /*}}}*/

   /*{{{ Clean up execution stack */
   counter = 0;
   for( i = 1; i <= exec->stack->stack_ptr; i++ )
   {
      targetvar = exec->stack->stack[i];
      if( targetvar && targetvar->flags.disposable )
      {
		 FUD(("[%d/%d] DESTROYING STACK VARIABLE %s (%s)\n", i,
			exec->stack->stack_ptr,
			targetvar->name,
			__ferite_variable_id_to_str( script, targetvar->type) ));
		 __ferite_variable_destroy( script, targetvar );
		 counter++;
      }
   }
   FUD(( "%d variables lefton stack\n", counter ));
   FUD(("IN TOTAL %d VARIABLES WHERE NOT DEALLOCATED\n", counter));
   __ferite_delete_stack( exec->stack );
   /*}}}*/

   ffree( exec );
   FE_LEAVE_FUNCTION( NOWT );
}

/*!
 * \fn void __ferite_clean_executor( FeriteScript *script )
 * \brief Clean up the executor
 * \param script The current script
 */
void __ferite_clean_executor( FeriteScript *script )
{
   FeriteExecuteRec *rec;
   int i;

   FE_ENTER_FUNCTION;
   if( script )
   {
      for( i = 1; i <= script->exec_stack->stack_ptr; i++ )
      {
		 rec = script->exec_stack->stack[i];
		 if( rec != NULL )
		 {
			__ferite_clean_up_exec_rec( script, rec );
		 }
      }
   }
   FE_LEAVE_FUNCTION( NOWT );
}

/*!
 * \fn FeriteVariable **__ferite_create_parameter_list( int size )
 * \brief Create a parameter list, NULLify it and then return it
 * \param size The number of parameters to hold
 */
FeriteVariable **__ferite_create_parameter_list( int size )
{
   FeriteVariable **list = NULL;
   int i = 0;

   FE_ENTER_FUNCTION;
   list = fmalloc( sizeof( FeriteVariable* ) * size );
   for( i = 0; i < size; i++ )
     list[i] = NULL;
   FE_LEAVE_FUNCTION( list );
}

/*!
 * \fn FeriteVariable **__ferite_add_to_parameter_list( FeriteVariable **list, FeriteVariable *var )
 * \brief Place a parameter within the next availible place within the parameter list
 * \param list The list to place it in
 * \param var  The variable to place within the list
 */
FeriteVariable **__ferite_add_to_parameter_list( FeriteVariable **list, FeriteVariable *var )
{
   int i = 0;
   int size = ferite_get_parameter_count( list );

   FE_ENTER_FUNCTION;
   while( list[i] != NULL && i < size )
     i++;
   list[i] = var;
   FE_LEAVE_FUNCTION( list );
}

/*!
 * \fn void __ferite_delete_parameter_list( FeriteScript *script, FeriteVariable **list )
 * \brief Delete a parameter list, and destroy any disposable variables
 * \param script The current script
 * \param list   The list to be deleted
 */
void __ferite_delete_parameter_list( FeriteScript *script, FeriteVariable **list )
{
   int i = 0;
   int size = ferite_get_parameter_count( list );

   FE_ENTER_FUNCTION;
   while( list[i] != NULL && i < size )
   {
      if( list[i] && list[i]->flags.disposable )
		__ferite_variable_destroy( script, PTR2VAR(list[i]) );
      i++;
   }
   ffree( list );
   FE_LEAVE_FUNCTION( NOWT );
}

/*!
 * \fn FeriteVariable **__ferite_create_parameter_list_from_data( FeriteScript *script, char *format, ... )
 * \brief This function is used to make creating a parameter list very easy.
 * \param script The current script
 * \param format The signiture for the parameters
 *
 * This function makes the whole process of creating a parameter list very easy. It allows
 * you to create a list from the actual data and not have to worry about the finer details of
 * ferite variables. It returns a parameter list.<br>
 * <br>
 * e.g.:<br>
 * __ferite_create_parameter_list_from_data( script, "nns", 2.3, 2, "Jello" );<br>
 * <br>
 * The formats are as follows:<br>
 *   n = number<br>
 *   s = string<br>
 *   o = object<br>
 *   a = array<br>
 */
FeriteVariable **__ferite_create_parameter_list_from_data( FeriteScript *script, char *format, ... )
{
   FeriteVariable **retval = NULL;
   FeriteVariable *var = NULL;
   va_list          ap;
   int              i = 0;

   retval = fmalloc( sizeof( FeriteVariable * ) * (strlen(format) + 1) );
   memset( retval, '\0', sizeof( FeriteVariable * ) * (strlen(format) + 1) );

   va_start( ap, format );
   for( i = 0; i < strlen(format); i++ )
   {
      switch( format[i] )
      {
       case 'n':
		 var = __ferite_create_number_double_variable( "__ferite_create_parameter_list_from_data", va_arg( ap, double ) );
		 break;
       case 's':
		 var = __ferite_create_string_variable( "__ferite_create_parameter_list_from_data", va_arg( ap, char * ) );
		 break;
       case 'o':
		 var = __ferite_create_object_variable( "__ferite_create_parameter_list_from_data" );
		 VAO(var) = va_arg( ap, FeriteObject * );
		 VAO(var)->refcount++;
		 break;
       case 'a':
		 var = __ferite_create_uarray_variable( "__ferite_create_parameter_list_from_data", 0 );
		 __ferite_uarray_destroy( script, VAUA(var) );
		 VAUA(var) = __ferite_uarray_dup( script, va_arg( ap, FeriteUnifiedArray *), (void *(*)(FeriteScript*,FeriteVariable*))__ferite_duplicate_variable );
		 break;
      }
      MARK_VARIABLE_AS_DISPOSABLE( var );
      retval[i] = var;
   }
   va_end( ap );
   return retval;
}

/*!
 * \fn FeriteVariable *__ferite_call_function( FeriteScript *script, FeriteFunction *function, FeriteVariable **params )
 * \brief This will call any function from it's function pointer and a parameter list
 * \param script The current script
 * \param function The function to be called
 * \param params The parameter list to be passed to the function
 * 
 * This function will work on either an internal or native function.
 */
FeriteVariable *__ferite_call_function( FeriteScript *script, FeriteFunction *function, FeriteVariable **params )
{
   FeriteVariable *retval = NULL;

   FE_ENTER_FUNCTION;
   if( __ferite_check_params( script, params, function->signature ) == 1 )
   {
      if( function->type == FNC_IS_EXTRL )
      {
		 retval = (function->fncPtr)( script, params );
      }
      else
      {
		 __ferite_script_function_execute( script, function, params );
		 retval = function->returnt;
         if( script->error_state == ERROR_THROWN )
		   retval = __ferite_create_void_variable( "error..." );
      }
   }
   else
   {
      printf( "Incorrect parameters for function %s.\n", function->name );
   }
   FE_LEAVE_FUNCTION( retval );
}
