/* Copyright (C) 20011126  Pete Vassoff <pfv@grex.org>
 * This code is released to the ferite project as below.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *   
 * The above copyright notice and this permission notice shall be included in
 * all copies of the Software, its documentation and marketing & publicity 
 * materials, and acknowledgment shall be given in the documentation, materials
 * and software packages that this Software was used.
 *    
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER 
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef _FILE_UTILITY_H
# define _FILE_UTILITY_H

# include "config.h"
# include <dirent.h>
# include <errno.h>
# include <fcntl.h>
# include <limits.h>
# include <stdlib.h>
# include <string.h>

# ifndef USING_SOLARIS
#  include <sys/file.h>
# else
#  include "/usr/ucbinclude/sys/file.h"
# endif

# ifdef USING_DARWIN /* NASTY!! DARWIN has no poll.h, but it does have the poll call!! */
#  include "poll.h"
# else
#  include <sys/poll.h>
# endif

# include <sys/stat.h>
# include <sys/types.h>
# include <unistd.h>
# include <utime.h>

extern int errno; 

typedef struct file_data
{  
   char	*pathspec;	/* NULL: unused - "<filepathspec>" */
   char	*modes;		/* NULL: unused - "[[rwa][b]+]" */
   FILE	*fp;		/* NULL: unused - obvious. */
   int		fd;		/*   -1: unused - fileno() of above */
   int		err;		/* we'll save this for you - see strerror() */
   int		nonB:1,		/* Special flags - INTERNAL USE */
	 pipe:1,			/* popen() */
	 fifo:1,			/* mkFifo() */
	 sock:1,			/* mkSocket() & connect() */
	 serv:1;			/* mkSocket() */
} FileData;

# define OAF( o ) ((FileData*)(o->odata))
# define FileObj  ((FileData*)(self->odata))
# define FileObj2(obj) ((FileData*)(obj->odata))
# define FileException(s) __ferite_raise_script_error(script, \
														 FileObj->err, "ERROR: File::%s: %d (%s)\n", (s), \
														 FileObj->err, strerror(FileObj->err) )

typedef struct stat_data
{  
   char	*pathspec;
   struct stat statbuf;
   int		err;		/* we'll save this for you - see strerror() */
} StatData;

# define StatObj  ((StatData*)(self->odata))
# define StatException(s) __ferite_raise_script_error(script, \
														 StatObj->err, "ERROR: Stats::%s: %d (%s)\n", (s), \
														 StatObj->err, strerror(StatObj->err) )
# define AM_SUID(m)      (((m) & S_ISUID) == S_ISUID)
# define AM_SGID(m)      (((m) & S_ISGID) == S_ISGID)
# define AM_SVTX(m)      (((m) & S_ISVTX) == S_ISVTX)

typedef struct dir_data
{  
   char	*pathspec;
   DIR		*dp;
   int		err;		/* we'll save this for you - see strerror() */
} DirData;

# define DirObj  ((DirData*)(self->odata))
# define DirException(s) __ferite_raise_script_error(script, \
														DirObj->err, "ERROR: Directory::%s: %d (%s)\n", (s), \
														DirObj->err, strerror(DirObj->err) )

#ifndef USING_SOLARIS
typedef struct pollfd pollfd_t;
#endif

typedef struct poll_data
{ 
   pollfd_t *block;
   int active;
   int total;
   int err;
} PollData;

# define POLLBLK 3
# define PollObj		((PollData*)(self->odata))
# define PollException(s) __ferite_raise_script_error(script, \
														 PollObj->err, "ERROR: Poll::%s: %d (%s)\n", (s), \
														 PollObj->err, strerror(PollObj->err) )    

# define ExtendedException(ss,s) __ferite_raise_script_error(script, \
																FileObj->err, "ERROR: %s::%s: %d (%s)\n", (ss), (s), \
																FileObj->err, strerror(FileObj->err) )



# define masksense(n) ((~(n))&0777)

/* File class utility.: close open shit and reset the rest.. */
void FileRecycle( FileData *fo );

/* File class: always returns OLD mask - wants 'mask' and may assign 'real'. */
/* returns oldmask: if !mask, then the current umask is ZERO. */
/* You STILL need to manually "umask(the_return);" to restore the STATE. */
int FileUmasking(char *mask, int *real);


/* Sorting functions.. */
/* int alphasort(const struct dirent **a, const struct dirent **b); */
int nosort(const struct dirent **a, const struct dirent **b);

/* Selection functions.. */
int dironly(const struct dirent *a);
int fileonly(const struct dirent *a);
int symlonly(const struct dirent *a);
int fifoonly(const struct dirent *a);
int sockonly(const struct dirent *a);

/* The Report workhorse. */
struct dirent **realscan(const char *path,
	int (*select)(struct dirent *),
	int (*compar)(struct dirent **, struct dirent **),
	int *count );

/* The Report cleanup */
void scancleanup( struct dirent ***given, int N);


/* A Recursive mkdir() */
int mkpath(const char *complete, mode_t mode);

/* A utility to determine absolute pathname, and (optionally) filename. */
char *abspath( char *pathbuf, char *filebuf, const char *filepath);


/* Convert a string have 3 of "[r-][w-][xsStT-]" to an int. */
unsigned perm2int(char *owner, char *group, char *other);

/* Two tools for creating/connecting to unix-sockets: */
int breakSocket(char *name, int sockfd, int bound);
int makeSocket(char *name, int mask, int bound);


/* Check an PollData for 'what' matching some 'fd', return a valid index. */
int poll_match(PollData *p, int what);

/* Add a unique handle & events to the PollData. */
void* poll_add(PollData *p, int handle, char *events);

/* Called by Poll::poll() when you have events: re-sort the array to place */
/* the limited set of event-marked contents foremost. */
int poll_sort(PollData *p);

#endif /* _FILE_UTILITY_H */
