/*
   my_syslog.c : a BSD syslog wrapper 
   Part of fbgetty 
   Copyright (C) 1999 2000 2001 Yann Droneaud <ydroneaud@meuh.eu.org>. 

   fbgetty is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   fbgetty is distributed in the hope that it will be useful, but 
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

 */

#include <fbgetty/global.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <syslog.h>

#include <string.h>
#include <stdarg.h>

#ifndef MY_VSYSLOG_BASIC_SIZE
#define MY_VSYSLOG_BASIC_SIZE 80
#endif

#ifdef MY_VSYSLOG_DEBUG
#undef HAVE_VSYSLOG
#define MY_VSYSLOG_LOOP_MAX 4  /* 80*(2^4) > 1k  */
#define MY_VSYSLOG_LOOP_INIT \
  int _loop_count;  \
  _loop_count = 0;  
#define MY_VSYSLOG_LOOP_CHECK \
  _loop_count ++;  \
  my_assert(_loop_count >= MY_VSYSLOG_LOOP_MAX, "infinite loop suspected");
#define MY_VSYSLOG_LOOP_FINISH
#else
#define MY_VSYSLOG_LOOP_INIT
#define MY_VSYSLOG_LOOP_CHECK
#define MY_VSYSLOG_LOOP_FINISH
#endif

#ifdef HAVE_VSYSLOG
int
my_vsyslog(int priority, const char *format, va_list args)
{
  vsyslog(priority, format, args);

  return(0);
}
#else
/*
 * this function call snprintf(), resize the buffer to match the real size.
 * then it duplicate '%' to avoid being interpreted by a BSD syslog 
 * (it will probably print double %% if it's not a BSD syslog ...)
 */
int
my_vsyslog(int priority, const char *format, va_list args)
{
  char *template, *template_p;
  char *message, *message_p;

  int size = MY_VSYSLOG_BASIC_SIZE;

  MY_VSYSLOG_LOOP_INIT

  /* allocate basic size */
  template = (char *) malloc(size * sizeof(char));
  if (template == NULL)
    return(-1);

  /* get the formated string */
  while(1)
    {
      int resize;
      char *new_template;

      MY_VSYSLOG_LOOP_CHECK

      /* parse the string */
      resize = vsnprintf(template, size, format, args);
      
      if (resize == -1)
	{
	  free(template);
	  return(-1);
	}

      /* check bound */
      if (resize < size)
	goto out;

      /* double size */
      size <<= 1; /* size *= 2 */
      
      new_template = (char *) realloc(template, size);
      if (new_template == NULL)
	{
	  free(template);
	  return(-1);
	}
      
      template = new_template;
    }

 out:

  MY_VSYSLOG_LOOP_FINISH
  
  /* count number of char, count % twice */
  size = 0;
  template_p = template;
  while(*template_p != '\0')
    {
      size ++;
      
      if (*template == '%')
	size ++;
      
      template_p ++;
    }
  
  message = (char *) malloc((size + 1) * sizeof(char));
  if (message == NULL)
    {
      free(template);
      return(-1);
    }
  
  /* copy the string, 
     prevent '%' being interpreted by syslog(), if it's a BSD syslog() */
  template_p = template;
  message_p = message;
  while(*template_p != '\0')
    {
      *message_p++ = *template_p;

      if (*template_p == '%')
	*message_p++ = *template_p;
      
      *message_p = '\0';
      
      template_p ++;
    }
  
  free(template);
  
  /* then send it with a standart syslog() */
  syslog(priority, message);
  
  free(message);
  
  return(0);
}
#endif
