/*
 * $Id: chip_fau_parca.c,v 1.37 2010-05-26 09:02:01 vrsieh Exp $
 *
 * Copyright (C) 2010 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG_CONTROL_FLOW	0
#define DEBUG_CONTROL_REGS	0
#define DEBUG_TIMING		0

#define PE_COUNT_MAX	256
#define ROW_COUNT_MAX	256
#define RREG_COUNT	7
#define LREG_COUNT	8

/* DCACHESIZE >= (2+PE_COUNT_MAX)*(2+ROW_COUNT_MAX)*RREG_COUNT*sizeof(uint16_t) */
#define DCACHESIZE	(1024*1024)
#define ICACHESIZE	(4096*sizeof(uint32_t))

#include "config.h"
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "compiler.h"
#include "glue-main.h"

#include "chip_fau_parca.h"

#define COMP_(x) chip_fau_parca_ ## x

struct cpssp {
	struct process process;

#if DEBUG_TIMING
	unsigned int inst_cnt;
#endif

	unsigned int pe_count;
	unsigned int row_count;

	struct sig_boolean_or *port_intA;
	unsigned int state_power;
	unsigned int state_n_reset;
	struct sig_pci_bus_main *port_pci_bus;
	struct sig_pci_bus_idsel *port_idsel;

	unsigned int pcicmd;
	unsigned int pcists;
	unsigned int pcibase0;
	unsigned int pcibase1;
	unsigned int pcibase2;
	unsigned int pciintline;

	uint16_t dcache[2][(2+PE_COUNT_MAX)*(2+ROW_COUNT_MAX)*RREG_COUNT];

	uint32_t icache[ICACHESIZE / sizeof(uint32_t)];

	unsigned int io_n_reset;
	unsigned int io_mux;
	unsigned int io_intmask;

	unsigned int io_ready;

	unsigned int parca_n_reset;

	struct {
		unsigned int line;
		unsigned int pc;
		uint32_t inst;
	} parca_cu;

	struct {
		unsigned int pc;
		unsigned int opA;
		unsigned int opB;
		unsigned int res;
		uint16_t reg[LREG_COUNT];
	} parca_pe[PE_COUNT_MAX];
};

static void
COMP_(irq_update)(struct cpssp *cpssp)
{
	sig_boolean_or_set(cpssp->port_intA, cpssp,
			cpssp->io_ready & cpssp->io_intmask);
}

static uint16_t *
COMP_(parca_reg)(
	struct cpssp *cpssp,
	unsigned int line,
	unsigned int pe,
	unsigned int emer,
	unsigned int reg
)
{
	static uint16_t null = 0x0000;

	if (reg == 0) {
		/* Null Register */
		return &null;
	}

	reg--;

	if (reg < RREG_COUNT) {
		/* BRAM Register */
		if (emer & 4) {
			switch (emer & 3) {
			case 0: /* West */
				pe--;
				break;
			case 1: /* East */
				pe++;
				break;
			case 2: /* North */
				line--;
				break;
			case 3: /* South */
				line++;
				break;
			default: assert(0); /* Cannot happen. */
			}
		}
		pe++;
		line++;
		return &cpssp->dcache[cpssp->io_mux][((1 + cpssp->pe_count + 1) * line + pe) * RREG_COUNT + reg];
	}

	reg -= RREG_COUNT;

	if (reg < LREG_COUNT) {
		/* Local Register */
		return &cpssp->parca_pe[pe].reg[reg];
	}

	reg -= LREG_COUNT;

	assert(0);
}

static uint16_t
COMP_(parca_read)(
	struct cpssp *cpssp,
	unsigned int pe,
	unsigned int emer,
	unsigned int reg
)
{
	return *COMP_(parca_reg)(cpssp, cpssp->parca_cu.line, pe, emer, reg);
}

static void
COMP_(parca_write)(
	struct cpssp *cpssp,
	unsigned int pe,
	unsigned int reg,
	uint16_t val
)
{
	if (reg == 0) {
		/* Don't write Null register. */
		return;
	}
	*COMP_(parca_reg)(cpssp, cpssp->parca_cu.line, pe, 0, reg) = val;
}

static void
COMP_(parca_step)(struct cpssp *cpssp)
{
	unsigned int pe;
	int gnl;
	int gnp;
	unsigned int clks;

	/* Calculate global PC. */
	cpssp->parca_cu.pc = 0xffffffff;
	for (pe = 0; pe < cpssp->pe_count; pe++) {
		if (cpssp->parca_pe[pe].pc < cpssp->parca_cu.pc) {
			cpssp->parca_cu.pc = cpssp->parca_pe[pe].pc;
		}
	}

	/*
	 * IFDE State
	 */
	/* Get instruction. */
	cpssp->parca_cu.inst = cpssp->icache[cpssp->parca_cu.pc];

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "ParCA: Executing 0x%08x at 0x%03x\n",
				cpssp->parca_cu.inst, cpssp->parca_cu.pc);
	}
	if (DEBUG_CONTROL_REGS) {
		unsigned int reg;

		fprintf(stderr, "line %u\n", cpssp->parca_cu.line);
		for (reg = 1; reg < 16; reg++) {
			fprintf(stderr, "%c%u:", reg < 8 ? 'R' : 'A', reg & 7);
			for (pe = 0; pe < cpssp->pe_count; pe++) {
				fprintf(stderr, " %04x", COMP_(parca_read)(
					cpssp, pe, 0, reg));
			}
			fprintf(stderr, "\n");
		}
	}

	gnl = 0;
	gnp = 0;

	/*
	 * EX1/EX2 State
	 */
	/* Get operands. */
	for (pe = 0; pe < cpssp->pe_count; pe++) {
		if (cpssp->parca_pe[pe].pc != cpssp->parca_cu.pc) {
			continue;
		}

		switch (cpssp->parca_cu.inst >> 27) {
		case 0x00: /* NOP */
			break;

		case 0x01: /* ADD */
		case 0x02: /* SUB */

		case 0x04: /* JEQ */
		case 0x05: /* JGR */
		case 0x06: /* JGE */
		case 0x07: /* JLS */
		case 0x08: /* JLE */

		case 0x0d: /* AND */
		case 0x0e: /* ORS */
			cpssp->parca_pe[pe].opA
				= COMP_(parca_read)(cpssp, pe,
					0,
					(cpssp->parca_cu.inst >> 20) & 0xf);
			cpssp->parca_pe[pe].opB
				= COMP_(parca_read)(cpssp, pe,
					(cpssp->parca_cu.inst >> 24) & 0x7,
					(cpssp->parca_cu.inst >> 16) & 0xf);
			break;

		case 0x03: /* JMP */
			break;

		case 0x09: /* SLS */
		case 0x0a: /* SLR */
		case 0x0b: /* SRS */
		case 0x0c: /* SRR */

		case 0x0f: /* NOT */
			cpssp->parca_pe[pe].opA
				= COMP_(parca_read)(cpssp, pe,
					(cpssp->parca_cu.inst >> 24) & 0x7,
					(cpssp->parca_cu.inst >> 20) & 0xf);
			break;

		case 0x10: /* ADI */
		case 0x11: /* SBI */
		case 0x12: /* ANI */
		case 0x13: /* ORI */
			cpssp->parca_pe[pe].opA
				= COMP_(parca_read)(cpssp, pe,
					(cpssp->parca_cu.inst >> 24) & 0x7,
					(cpssp->parca_cu.inst >> 20) & 0xf);
			cpssp->parca_pe[pe].opB
				= cpssp->parca_cu.inst & 0xffff;
			break;

		case 0x14: /* SET */
			cpssp->parca_pe[pe].opB
				= cpssp->parca_cu.inst & 0xffff;
			break;

		case 0x15: /* BST */
			cpssp->parca_pe[pe].opA
				= COMP_(parca_read)(cpssp, pe,
					0,
					(cpssp->parca_cu.inst >> 20) & 0xf)
				& ~(1 << ((cpssp->parca_cu.inst >> 16) & 0xf));
			cpssp->parca_pe[pe].opB
				= ((cpssp->parca_cu.inst >> 15) & 1)
					<< ((cpssp->parca_cu.inst >> 16) & 0xf);
			break;

		case 0x16: /* JIS */
		case 0x17: /* JNS */
			cpssp->parca_pe[pe].opA
				= COMP_(parca_read)(cpssp, pe,
					(cpssp->parca_cu.inst >> 24) & 0x7,
					(cpssp->parca_cu.inst >> 20) & 0xf);
			cpssp->parca_pe[pe].opB
				= (cpssp->parca_cu.inst >> 16) & 0xf;
			break;

		case 0x1e: /* GNL */
		case 0x1f: /* GNP */
			break;

		default:
			assert(0); /* FIXME */
		}
	}

	/* Execute instruction. */
	for (pe = 0; pe < cpssp->pe_count; pe++) {
		if (cpssp->parca_pe[pe].pc != cpssp->parca_cu.pc) {
			continue;
		}

		switch (cpssp->parca_cu.inst >> 27) {
		case 0x00: /* NOP */
			break;

		case 0x01: /* ADD */
		case 0x10: /* ADI */
			cpssp->parca_pe[pe].res = 
				cpssp->parca_pe[pe].opA + cpssp->parca_pe[pe].opB;
			break;

		case 0x02: /* SUB */
		case 0x11: /* SBI */
			cpssp->parca_pe[pe].res = 
				cpssp->parca_pe[pe].opA - cpssp->parca_pe[pe].opB;
			break;

		case 0x03: /* JMP */
			cpssp->parca_pe[pe].res = 1;
			break;

		case 0x04: /* JEQ */
			cpssp->parca_pe[pe].res =
				cpssp->parca_pe[pe].opA == cpssp->parca_pe[pe].opB;
			break;

		case 0x05: /* JGR */
			cpssp->parca_pe[pe].res =
				cpssp->parca_pe[pe].opA > cpssp->parca_pe[pe].opB;
			break;

		case 0x06: /* JGE */
			cpssp->parca_pe[pe].res =
				cpssp->parca_pe[pe].opA >= cpssp->parca_pe[pe].opB;
			break;

		case 0x07: /* JLS */
			cpssp->parca_pe[pe].res =
				cpssp->parca_pe[pe].opA < cpssp->parca_pe[pe].opB;
			break;

		case 0x08: /* JLE */
			cpssp->parca_pe[pe].res =
				cpssp->parca_pe[pe].opA <= cpssp->parca_pe[pe].opB;
			break;

		case 0x09: /* SLS */
			cpssp->parca_pe[pe].res =
				cpssp->parca_pe[pe].opA << 1;
			break;

		case 0x0a: /* SLR */
			cpssp->parca_pe[pe].res =
				(cpssp->parca_pe[pe].opA << 1)
				| (cpssp->parca_pe[pe].opA >> 15);
			break;

		case 0x0b: /* SRS */
			cpssp->parca_pe[pe].res =
				cpssp->parca_pe[pe].opA >> 1;
			break;

		case 0x0c: /* SRR */
			cpssp->parca_pe[pe].res =
				(cpssp->parca_pe[pe].opA >> 1)
				| (cpssp->parca_pe[pe].opA << 15);
			break;

		case 0x0d: /* AND */
		case 0x12: /* ANI */
			cpssp->parca_pe[pe].res =
				cpssp->parca_pe[pe].opA & cpssp->parca_pe[pe].opB;
			break;

		case 0x0e: /* OR */
		case 0x13: /* ORI */
		case 0x15: /* BST */
			cpssp->parca_pe[pe].res =
				cpssp->parca_pe[pe].opA | cpssp->parca_pe[pe].opB;
			break;

		case 0x0f: /* NOT */
			cpssp->parca_pe[pe].res =
				~cpssp->parca_pe[pe].opA;
			break;

		case 0x14: /* SET */
			cpssp->parca_pe[pe].res =
				cpssp->parca_pe[pe].opB;
			break;

		case 0x16: /* JIS */
			cpssp->parca_pe[pe].res =
				(cpssp->parca_pe[pe].opA >> cpssp->parca_pe[pe].opB) & 1;
			break;

		case 0x17: /* JNS */
			cpssp->parca_pe[pe].res =
				! ((cpssp->parca_pe[pe].opA >> cpssp->parca_pe[pe].opB) & 1);
			break;

		case 0x1e: /* GNL */
			gnl = 1;
			break;

		case 0x1f: /* GNP */
			gnp = 1;
			break;

		default:
			assert(0); /* FIXME */
		}
	}

	/*
	 * WB State
	 */
	/* Write result. */
	for (pe = 0; pe < cpssp->pe_count; pe++) {
		if (cpssp->parca_pe[pe].pc != cpssp->parca_cu.pc) {
			continue;
		}

		switch (cpssp->parca_cu.inst >> 27) {
		case 0x00: /* NOP */
		case 0x03: /* JMP */
		case 0x04: /* JEQ */
		case 0x05: /* JGR */
		case 0x06: /* JGE */
		case 0x07: /* JLS */
		case 0x08: /* JLE */
		case 0x16: /* JIS */
		case 0x17: /* JNS */
		case 0x1e: /* GNL */
		case 0x1f: /* GNP */
			break;

		case 0x01: /* ADD */
		case 0x02: /* SUB */
		case 0x0d: /* AND */
		case 0x0e: /* ORS */
			COMP_(parca_write)(cpssp, pe,
				(cpssp->parca_cu.inst >> 12) & 0xf,
				cpssp->parca_pe[pe].res);
			break;

		case 0x09: /* SLS */
		case 0x0a: /* SLR */
		case 0x0b: /* SRS */
		case 0x0c: /* SRR */
		case 0x0f: /* NOT */
		case 0x10: /* ADI */
		case 0x11: /* SBI */
		case 0x12: /* ANI */
		case 0x13: /* ORI */
			COMP_(parca_write)(cpssp, pe,
				(cpssp->parca_cu.inst >> 16) & 0xf,
				cpssp->parca_pe[pe].res);
			break;

		case 0x14: /* SET */
		case 0x15: /* BST */
			COMP_(parca_write)(cpssp, pe,
				(cpssp->parca_cu.inst >> 20) & 0xf,
				cpssp->parca_pe[pe].res);
			break;

		default:
			assert(0); /* FIXME */
		}
	}

	/*
	 * PCUP State
	 */
	/* Calculate next PC. */
	for (pe = 0; pe < cpssp->pe_count; pe++) {
		if (cpssp->parca_pe[pe].pc != cpssp->parca_cu.pc) {
			continue;
		}

		switch (cpssp->parca_cu.inst >> 27) {
		case 0x00: /* NOP */
		case 0x01: /* ADD */
		case 0x02: /* SUB */
		case 0x09: /* SLS */
		case 0x0a: /* SLR */
		case 0x0b: /* SRS */
		case 0x0c: /* SRR */
		case 0x0d: /* AND */
		case 0x0e: /* ORS */
		case 0x0f: /* NOT */
		case 0x10: /* ADI */
		case 0x11: /* SBI */
		case 0x12: /* ANI */
		case 0x13: /* ORI */
		case 0x14: /* SET */
		case 0x15: /* BST */
			cpssp->parca_pe[pe].pc++;
			break;

		case 0x03: /* JMP */
		case 0x04: /* JEQ */
		case 0x05: /* JGR */
		case 0x06: /* JGE */
		case 0x07: /* JLS */
		case 0x08: /* JLE */
		case 0x16: /* JIS */
		case 0x17: /* JNS */
			if (cpssp->parca_pe[pe].res) {
				unsigned int off;
				
				off = cpssp->parca_cu.inst & 0x00000fff;

				if ((cpssp->parca_cu.inst >> 12) & 1) {
					cpssp->parca_pe[pe].pc -= off;
				} else {
					cpssp->parca_pe[pe].pc += off;
				}
			} else {
				cpssp->parca_pe[pe].pc++;
			}
			break;

		case 0x1e: /* GNL */
			cpssp->parca_pe[pe].pc = 0;
			break;
			
		case 0x1f: /* GNP */
			break;
		}

		cpssp->parca_pe[pe].pc &= 0x00000fff;
	}

	if (gnl) {
		/*
		 * GNL State
		 */
		cpssp->parca_cu.line++;
		if (cpssp->parca_cu.line == cpssp->row_count) {
			cpssp->parca_cu.line = 0;
		}

	} else if (gnp) {
		/*
		 * GNP State
		 */
		cpssp->io_ready = 1;
		COMP_(irq_update)(cpssp);
	}

	/*
	 * Calculate clock ticks.
	 */
	/* IFDE State */
	clks = 4;

	/* EX1 State */
	clks += 16;

	/* EX2 State */
	switch (cpssp->parca_cu.inst >> 27) {
	case 0x00: /* NOP */
	case 0x01: /* ADD */
	case 0x02: /* SUB */
	case 0x03: /* JMP */
	case 0x04: /* JEQ */
	case 0x05: /* JGR */
	case 0x06: /* JGE */
	case 0x07: /* JLS */
	case 0x08: /* JLE */
	case 0x0d: /* AND */
	case 0x0e: /* ORS */
	case 0x0f: /* NOT */
	case 0x10: /* ADI */
	case 0x11: /* SBI */
	case 0x12: /* ANI */
	case 0x13: /* ORI */
	case 0x14: /* SET */
	case 0x15: /* BST */
	case 0x16: /* JIS */
	case 0x17: /* JNS */
	case 0x1e: /* GNL */
	case 0x1f: /* GNP */
		break;

	case 0x09: /* SLS */
	case 0x0a: /* SLR */
	case 0x0b: /* SRS */
	case 0x0c: /* SRR */
		clks += 17;
		break;

	default:
		assert(0);
	}

	/* WB State */
	switch (cpssp->parca_cu.inst >> 27) {
	case 0x00: /* NOP */
	case 0x03: /* JMP */
	case 0x04: /* JEQ */
	case 0x05: /* JGR */
	case 0x06: /* JGE */
	case 0x07: /* JLS */
	case 0x08: /* JLE */
	case 0x16: /* JIS */
	case 0x17: /* JNS */
	case 0x1e: /* GNL */
	case 0x1f: /* GNP */
		break;

	case 0x01: /* ADD */
	case 0x02: /* SUB */
	case 0x0d: /* AND */
	case 0x0e: /* ORS */
		if (((cpssp->parca_cu.inst >> 12) & 0xf) < 1 + RREG_COUNT) {
			clks += 32;
		}
		break;

	case 0x09: /* SLS */
	case 0x0a: /* SLR */
	case 0x0b: /* SRS */
	case 0x0c: /* SRR */
	case 0x0f: /* NOT */
	case 0x10: /* ADI */
	case 0x11: /* SBI */
	case 0x12: /* ANI */
	case 0x13: /* ORI */
		if (((cpssp->parca_cu.inst >> 16) & 0xf) < 1 + RREG_COUNT) {
			clks += 32;
		}
		break;

	case 0x14: /* SET */
	case 0x15: /* BST */
		if (((cpssp->parca_cu.inst >> 20) & 0xf) < 1 + RREG_COUNT) {
			clks += 32;
		}
		break;

	default:
		assert(0);
	}

	/* PCUP State */
	switch (cpssp->parca_cu.inst >> 27) {
	case 0x00: /* NOP */
	case 0x01: /* ADD */
	case 0x02: /* SUB */
	case 0x03: /* JMP */
	case 0x04: /* JEQ */
	case 0x05: /* JGR */
	case 0x06: /* JGE */
	case 0x07: /* JLS */
	case 0x08: /* JLE */
	case 0x09: /* SLS */
	case 0x0a: /* SLR */
	case 0x0b: /* SRS */
	case 0x0c: /* SRR */
	case 0x0d: /* AND */
	case 0x0e: /* ORS */
	case 0x0f: /* NOT */
	case 0x10: /* ADI */
	case 0x11: /* SBI */
	case 0x12: /* ANI */
	case 0x13: /* ORI */
	case 0x14: /* SET */
	case 0x15: /* BST */
	case 0x16: /* JIS */
	case 0x17: /* JNS */
		clks += 14;
		break;
	case 0x1e: /* GNL */
	case 0x1f: /* GNP */
		break;
	default:
		assert(0);
	}

	/* GNL/GNP State */
	switch (cpssp->parca_cu.inst >> 27) {
	case 0x00: /* NOP */
	case 0x01: /* ADD */
	case 0x02: /* SUB */
	case 0x03: /* JMP */
	case 0x04: /* JEQ */
	case 0x05: /* JGR */
	case 0x06: /* JGE */
	case 0x07: /* JLS */
	case 0x08: /* JLE */
	case 0x09: /* SLS */
	case 0x0a: /* SLR */
	case 0x0b: /* SRS */
	case 0x0c: /* SRR */
	case 0x0d: /* AND */
	case 0x0e: /* ORS */
	case 0x0f: /* NOT */
	case 0x10: /* ADI */
	case 0x11: /* SBI */
	case 0x12: /* ANI */
	case 0x13: /* ORI */
	case 0x14: /* SET */
	case 0x15: /* BST */
	case 0x16: /* JIS */
	case 0x17: /* JNS */
		break;
	case 0x1e: /* GNL */
		clks += 34;
		break;
	case 0x1f: /* GNP */
		clks += 34; /* Minimum */
		break;
	default:
		assert(0);
	}

	cpssp->process.inst_cnt += clks;

#if DEBUG_TIMING
	cpssp->inst_cnt += clks;

	if (gnp) {
		fprintf(stderr, "inst_cnt %u\n", cpssp->inst_cnt);
	}
#endif
}

static void __attribute__((__noreturn__))
COMP_(process)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

again:	;
	while (cpssp->process.inst_cnt < cpssp->process.inst_limit) {
		if (unlikely(! cpssp->state_power)) {
			/*
			 * Power-off
			 */
			if (DEBUG_CONTROL_FLOW) {
				fprintf(stderr, "ParCA: power-off\n");
			}
			if (cpssp->process.inst_cnt < cpssp->process.inst_limit) {
				cpssp->process.inst_cnt = cpssp->process.inst_limit;
			}

		} else if (unlikely(! cpssp->parca_n_reset)) {
			/*
			 * Reset
			 */
			if (DEBUG_CONTROL_FLOW) {
				fprintf(stderr, "ParCA: reset\n");
			}
			if (cpssp->process.inst_cnt < cpssp->process.inst_limit) {
				cpssp->process.inst_cnt = cpssp->process.inst_limit;
			}

		} else if (unlikely(cpssp->io_ready)) {
			/*
			 * Halted
			 */
			if (DEBUG_CONTROL_FLOW) {
				fprintf(stderr, "ParCA: ready\n");
			}
			if (cpssp->process.inst_cnt < cpssp->process.inst_limit) {
				cpssp->process.inst_cnt = cpssp->process.inst_limit;
			}

		} else {
			/*
			 * Do steps...
			 */
			if (DEBUG_CONTROL_FLOW) {
				fprintf(stderr, "ParCA: step\n");
			}
			COMP_(parca_step)(cpssp);
		}
	}

	sched_to_scheduler();

	goto again;
}

static void
COMP_(parca_reset_update)(struct cpssp *cpssp)
{
	unsigned int n_reset;
	unsigned int pe;
	unsigned int lreg;

	n_reset = cpssp->state_n_reset && cpssp->io_n_reset;

	if (cpssp->parca_n_reset == n_reset) {
		return;
	}

	cpssp->parca_n_reset = n_reset;

	cpssp->parca_cu.line = 0;

	for (pe = 0; pe < cpssp->pe_count; pe++) {
		cpssp->parca_pe[pe].pc = 0x000;
		for (lreg = 0; lreg < LREG_COUNT; lreg++) {
			cpssp->parca_pe[pe].reg[lreg] = 0x0000;
		}
	}

	cpssp->io_ready = 0;
}

static void
COMP_(power_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_power = val;
}

static void
COMP_(n_reset_set)(void *_cpssp, unsigned int n_val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_n_reset = n_val;

	cpssp->pcicmd = 0x0000;
	cpssp->pcists = 0x0000;
	cpssp->pcibase0 = 0x00000000;
	cpssp->pcibase1 = 0x00000000;
	cpssp->pcibase2 = 0x00000000;
	cpssp->pciintline = 0x00;

	cpssp->io_n_reset = 0;
	cpssp->io_mux = 0;
	cpssp->io_intmask = 0;

	COMP_(parca_reset_update)(cpssp);
}

static int
COMP_(ior)(void *_cpssp, uint32_t port, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: port=0x%04x, bs=%x\n", __FUNCTION__, port, bs);
	}

	assert(! (port & 3));

	if (! ((cpssp->pcicmd >> 0) & 1)) {
		/* I/O disabled. */
		return 1;
	}
	if (port < cpssp->pcibase2 || cpssp->pcibase2 + 8 <= port) {
		/* Not our ports. */
		return 1;
	}

	port &= 7;

	*valp = 0x00000000;

	switch (port) {
	case 0x00:
		/* Command Register */
		if ((bs >> 0) & 1) {
			*valp |= (cpssp->io_n_reset << 0)
				| (cpssp->io_mux << 1)
				| (cpssp->io_intmask << 2);
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x04:
		/* Status Register */
		if ((bs >> 0) & 1) {
			*valp |= cpssp->io_ready << 0;
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	default:
		assert(0);
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: val=0x%08x\n", __FUNCTION__, *valp);
	}

	return 0;
}

static int
COMP_(iow)(void *_cpssp, uint32_t port, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = _cpssp;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: port=0x%04x, bs=%x, val=0x%08x\n",
				__FUNCTION__, port, bs, val);
	}

	assert(! (port & 3));

	if (! ((cpssp->pcicmd >> 0) & 1)) {
		/* I/O disabled. */
		return 1;
	}
	if (port < cpssp->pcibase2 || cpssp->pcibase2 + 8 <= port) {
		/* Not our ports. */
		return 1;
	}

	port &= 7;

	switch (port) {
	case 0x00:
		/* Command Register */
		if ((bs >> 0) & 1) {
			cpssp->io_n_reset = (val >> 0) & 1;
			COMP_(parca_reset_update)(cpssp);
			cpssp->io_mux = (val >> 1) & 1;
			cpssp->io_intmask = (val >> 2) & 1;
			COMP_(irq_update)(cpssp);
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x04:
		/* Status Register */
		if ((bs >> 0) & 1) {
			/* Read-only */
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	default:
		assert(0);
	}

	return 0;
}

static int
COMP_(mr)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=%x\n",
				__FUNCTION__, addr, bs);
	}

	if (! ((cpssp->pcicmd >> 1) & 1)) {
		/* Memory disabled. */
		return 1;
	}
	
	if (cpssp->pcibase0 <= addr
	 && addr < cpssp->pcibase0 + DCACHESIZE) {
		/* Accessing partition. */
		uint32_t *dcache;

		addr &= DCACHESIZE - 1;
		addr /= sizeof(uint16_t);
		dcache = (uint32_t *) &cpssp->dcache[! cpssp->io_mux][addr];

		*valp = *dcache;
		return 0;

	} else if (cpssp->pcibase1 <= addr
		&& addr < cpssp->pcibase1 + ICACHESIZE) {
		/* Accessing instruction cache. */
		uint32_t *icache;

		addr &= ICACHESIZE - 1;
		addr /= sizeof(uint32_t);
		icache = (uint32_t *) &cpssp->icache[addr];

		*valp = *icache;
		return 0;

	} else {
		return 1;
	}
}

static int
COMP_(mw)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = _cpssp;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=%x, val=0x%08x\n",
				__FUNCTION__, addr, bs, val);
	}

	assert(! (addr & 3));

	if (! ((cpssp->pcicmd >> 1) & 1)) {
		/* Memory disabled. */
		return 1;
	}
	
	if (cpssp->pcibase0 <= addr
	 && addr < cpssp->pcibase0 + DCACHESIZE) {
		/* Accessing partition. */
		uint32_t *dcache;

		addr &= DCACHESIZE - 1;
		addr /= sizeof(uint16_t);
		dcache = (uint32_t *) &cpssp->dcache[! cpssp->io_mux][addr];

		if ((bs >> 0) & 1) {
			*dcache &= ~(0xff << 0);
			*dcache |= ((val >> 0) & 0xff) << 0;
		}
		if ((bs >> 1) & 1) {
			*dcache &= ~(0xff << 8);
			*dcache |= ((val >> 8) & 0xff) << 8;
		}
		if ((bs >> 2) & 1) {
			*dcache &= ~(0xff << 16);
			*dcache |= ((val >> 16) & 0xff) << 16;
		}
		if ((bs >> 3) & 1) {
			*dcache &= ~(0xff << 24);
			*dcache |= ((val >> 24) & 0xff) << 24;
		}
		return 0;

	} else if (cpssp->pcibase1 <= addr
		&& addr < cpssp->pcibase1 + ICACHESIZE) {
		/* Accessing instruction cache. */
		uint32_t *icache;

		addr &= ICACHESIZE - 1;
		addr /= sizeof(uint32_t);
		icache = &cpssp->icache[addr];

		if ((bs >> 0) & 1) {
			*icache &= ~(0xff << 0);
			*icache |= ((val >> 0) & 0xff) << 0;
		}
		if ((bs >> 1) & 1) {
			*icache &= ~(0xff << 8);
			*icache |= ((val >> 8) & 0xff) << 8;
		}
		if ((bs >> 2) & 1) {
			*icache &= ~(0xff << 16);
			*icache |= ((val >> 16) & 0xff) << 16;
		}
		if ((bs >> 3) & 1) {
			*icache &= ~(0xff << 24);
			*icache |= ((val >> 24) & 0xff) << 24;
		}
		return 0;

	} else {
		return 1;
	}
}

static int
COMP_(c0r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=%x\n",
				__FUNCTION__, addr, bs);
	}

	assert(! (addr & 3));

	if ((addr >> 8) & 7) {
		/* Only function 0 supported. */
		return 1;
	}
	addr &= 0xff;

	*valp = 0x00000000;

	switch (addr) {
	case 0x00:
		/* Vendor Identitification Register */
		if ((bs >> 0) & 1) {
			*valp |= 0xfe << 0; /* FIXME */
		}
		if ((bs >> 1) & 1) {
			*valp |= 0xff << 8; /* FIXME */
		}

		/* Device Identitification Register */
		if ((bs >> 2) & 1) {
			*valp |= 0xfe << 16; /* FIXME */
		}
		if ((bs >> 3) & 1) {
			*valp |= 0xff << 24; /* FIXME */
		}
		break;

	case 0x04:
		/* PCI Command Register */
		if ((bs >> 0) & 1) {
			*valp |= (cpssp->pcicmd >> 0) << 0;
		}
		if ((bs >> 1) & 1) {
			*valp |= (cpssp->pcicmd >> 8) << 8;
		}

		/* PCI Device Status Register */
		if ((bs >> 2) & 1) {
			*valp |= (cpssp->pcists >> 0) << 16;
		}
		if ((bs >> 3) & 1) {
			*valp |= (cpssp->pcists >> 8) << 24;
		}
		break;

	case 0x08:
		/* Revision Identification Register */
		if ((bs >> 0) & 1) {
			*valp |= 1 << 0;
		}

		/* Class Code Register */
		if ((bs >> 1) & 1) {
			*valp |= 0xff << 8; /* FIXME */
		}
		if ((bs >> 2) & 1) {
			*valp |= 0xff << 16; /* FIXME */
		}
		if ((bs >> 3) & 1) {
			*valp |= 0x0b << 24; /* Processor */
		}
		break;

	case 0x0c:
		/* Cache Line Size */
		if ((bs >> 0) & 1) {
			/* Unused */
		}

		/* Latency Timer */
		if ((bs >> 1) & 1) {
			/* Unused */
		}

		/* Header Type Register */
		if ((bs >> 2) & 1) {
			*valp |= 0x00 << 16; /* No multifunction, version 0 */
		}

		/* BIST */
		if ((bs >> 3) & 1) {
			*valp |= 0x00 << 24; /* No BIST */
		}
		break;

	case 0x10:
		/* PCI Base Address Register 0 */
		if ((bs >> 0) & 1) {
			*valp |= (((cpssp->pcibase0 >> 0) & 0xf0) << 0)
				| (1 << 3) /* prefetchable */
				| (0 << 1) /* locate anywhere in 32bit */
				| (0 << 0); /* memory-mapped */
		}
		if ((bs >> 1) & 1) {
			*valp |= ((cpssp->pcibase0 >> 8) & 0xff) << 8;
		}
		if ((bs >> 2) & 1) {
			*valp |= ((cpssp->pcibase0 >> 16) & 0xff) << 16;
		}
		if ((bs >> 3) & 1) {
			*valp |= ((cpssp->pcibase0 >> 24) & 0xff) << 24;
		}
		break;

	case 0x14:
		/* PCI Base Address Register 1 */
		if ((bs >> 0) & 1) {
			*valp |= (((cpssp->pcibase1 >> 0) & 0xf0) << 0)
				| (1 << 3) /* prefetchable */
				| (0 << 1) /* locate anywhere in 32bit */
				| (0 << 0); /* memory-mapped */
		}
		if ((bs >> 1) & 1) {
			*valp |= ((cpssp->pcibase1 >> 8) & 0xff) << 8;
		}
		if ((bs >> 2) & 1) {
			*valp |= ((cpssp->pcibase1 >> 16) & 0xff) << 16;
		}
		if ((bs >> 3) & 1) {
			*valp |= ((cpssp->pcibase1 >> 24) & 0xff) << 24;
		}
		break;

	case 0x18:
		/* PCI Base Address Register 2 */
		if ((bs >> 0) & 1) {
			*valp |= (((cpssp->pcibase2 >> 0) & 0xf0) << 0)
				| (0 << 1) /* reserved */
				| (1 << 0); /* I/O-mapped */
		}
		if ((bs >> 1) & 1) {
			*valp |= ((cpssp->pcibase2 >> 8) & 0xff) << 8;
		}
		if ((bs >> 2) & 1) {
			*valp |= ((cpssp->pcibase2 >> 16) & 0xff) << 16;
		}
		if ((bs >> 3) & 1) {
			*valp |= ((cpssp->pcibase2 >> 24) & 0xff) << 24;
		}
		break;
		
	case 0x1c:
		/* PCI Base Address Register 3 */
		/* Unused */
		/* Unused */
		/* Unused */
		/* Unused */
		break;

	case 0x20:
		/* PCI Base Address Register 4 */
		/* Unused */
		/* Unused */
		/* Unused */
		/* Unused */
		break;

	case 0x24:
		/* PCI Base Address Register 5 */
		/* Unused */
		/* Unused */
		/* Unused */
		/* Unused */
		break;

	case 0x28:
		/* Cardbus CIS Pointer */
		/* Unused */
		/* Unused */
		/* Unused */
		/* Unused */
		break;

	case 0x2c:
		/* Subsystem Vendor ID Register */
		/* Unused */
		/* Unused */

		/* Subsystem ID Register */
		/* Unused */
		/* Unused */
		break;

	case 0x30:
		/* PCI ROM Address Register */
		/* Unused */
		/* Unused */
		/* Unused */
		/* Unused */
		break;

	case 0x34:
		/* PCI Capability List Register */
		/* Unused */
		/* Unused */
		/* Unused */
		/* Unused */
		break;

	case 0x38:
		/* Unknown FIXME */
		/* Unused */
		/* Unused */
		/* Unused */
		/* Unused */

	case 0x3c:
		/* PCI Interrupt Line Register */
		if ((bs >> 0) & 1) {
			*valp |= cpssp->pciintline << 0; /* IntA */
		}

		/* PCI Interrupt Pin Register */
		if ((bs >> 1) & 1) {
			*valp |= 0x01 << 8; /* IntA */
		}

		/* PCI Min Gnt Register */
		/* Unused */

		/* PCI Max Lat Register */
		/* Unused */
		break;

	default:
		fprintf(stderr, "WARNING: FAU_PARCA: Reading");
		if ((bs >> 0) & 1) {
			fprintf(stderr, " 0x%02x", addr + 0);
			*valp |= 0 << 0;
		}
		if ((bs >> 1) & 1) {
			fprintf(stderr, " 0x%02x", addr + 1);
			*valp |= 0 << 8;
		}
		if ((bs >> 2) & 1) {
			fprintf(stderr, " 0x%02x", addr + 2);
			*valp |= 0 << 16;
		}
		if ((bs >> 3) & 1) {
			fprintf(stderr, " 0x%02x", addr + 3);
			*valp |= 0 << 24;
		}
		fprintf(stderr, " (Returning 0)\n");
		break;
	}

	return 0;
}

static int
COMP_(c0w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = _cpssp;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=%x, val=0x%08x\n",
				__FUNCTION__, addr, bs, val);
	}

	assert(! (addr & 3));

	if ((addr >> 8) & 7) {
		/* Only function 0 supported. */
		return 1;
	}
	addr &= 0xff;

	switch (addr) {
	case 0x00:
		/* Vendor Identification Register */
		/* Read-only */
		/* Read-only */

		/* Device Identification Register */
		/* Read-only */
		/* Read-only */
		break;

	case 0x04:
		/* PCI Command Register */
		if ((bs >> 0) & 1) {
			/* Some bits are hardwired to '0'. */
			val &= ~(1 << 7);	/* Address and Data Stepping Enable */
			val &= ~(1 << 6);	/* Parity Error Detect Enable */
			val &= ~(1 << 5);	/* VGA Palette Snoop Enable */
			val &= ~(1 << 4);	/* Memory Write and Invalidate */
			cpssp->pcicmd &= ~(0xff << 0);
			cpssp->pcicmd |= ((val >> 0) & 0xff) << 0;
		}
		if ((bs >> 1) & 1) {
			/* Some bits are hardwired to '0'. */
			val &= ~(1 << 15);	/* Reserved */
			val &= ~(1 << 14);	/* Reserved */
			val &= ~(1 << 13);	/* Reserved */
			val &= ~(1 << 12);	/* Reserved */
			val &= ~(1 << 11);	/* Reserved */
			val &= ~(1 << 10);	/* Reserved */
			val &= ~(1 <<  9);	/* Fast Back-to-Back Enable */
			cpssp->pcicmd &= ~(0xff << 8);
			cpssp->pcicmd |= ((val >> 8) & 0xff) << 8;
		}
		break;

		/* PCI Device Status Register */
		if ((bs >> 2) & 1) {
			/* Bit 7 is read-only. */
			/* Bits 6-0 are reserved. */
		}
		if ((bs >> 3) & 1) {
			/* Bit 15 is read-only. */
			if ((val >> (14+16)) & 1) {
				cpssp->pcists &= ~(1 << 14);
			}
			if ((val >> (13+16)) & 1) {
				cpssp->pcists &= ~(1 << 13);
			}
			if ((val >> (12+16)) & 1) {
				cpssp->pcists &= ~(1 << 12);
			}
			if ((val >> (11+16)) & 1) {
				cpssp->pcists &= ~(1 << 11);
			}
			/* Bits 10-9 are read-only. */
			/* Bit 8 is read-only. */
		}
		break;

	case 0x08:
		/* Revision Identification Register */
		if ((bs >> 0) & 1) {
			/* Read-only */
		}

		/* Class Code Register */
		if ((bs >> 1) & 1) {
			/* Read-only */
		}
		if ((bs >> 2) & 1) {
			/* Read-only */
		}

		/* Header Type Register */
		if ((bs >> 3) & 1) {
			/* Read-only */
		}
		break;

	case 0x10:
		/* PCI Base Address Register 0 */
		val &= ~(DCACHESIZE - 1);
		val &= ~0xf;

		if ((bs >> 0) & 1) {
			cpssp->pcibase0 &= ~(0xff << 0);
			cpssp->pcibase0 |= ((val >> 0) & 0xff) << 0;
		}
		if ((bs >> 1) & 1) {
			cpssp->pcibase0 &= ~(0xff << 8);
			cpssp->pcibase0 |= ((val >> 8) & 0xff) << 8;
		}
		if ((bs >> 2) & 1) {
			cpssp->pcibase0 &= ~(0xff << 16);
			cpssp->pcibase0 |= ((val >> 16) & 0xff) << 16;
		}
		if ((bs >> 3) & 1) {
			cpssp->pcibase0 &= ~(0xff << 24);
			cpssp->pcibase0 |= ((val >> 24) & 0xff) << 24;
		}
		break;

	case 0x14:
		/* PCI Base Address Register 1 */
		val &= ~(ICACHESIZE - 1);
		val &= ~0xf;

		if ((bs >> 0) & 1) {
			cpssp->pcibase1 &= ~(0xff << 0);
			cpssp->pcibase1 |= ((val >> 0) & 0xff) << 0;
		}
		if ((bs >> 1) & 1) {
			cpssp->pcibase1 &= ~(0xff << 8);
			cpssp->pcibase1 |= ((val >> 8) & 0xff) << 8;
		}
		if ((bs >> 2) & 1) {
			cpssp->pcibase1 &= ~(0xff << 16);
			cpssp->pcibase1 |= ((val >> 16) & 0xff) << 16;
		}
		if ((bs >> 3) & 1) {
			cpssp->pcibase1 &= ~(0xff << 24);
			cpssp->pcibase1 |= ((val >> 24) & 0xff) << 24;
		}
		break;

	case 0x18:
		/* PCI Base Address Register 2 */
		val &= ~(8 - 1);
		val &= ~0x3;

		if ((bs >> 0) & 1) {
			cpssp->pcibase2 &= ~(0xff << 0);
			cpssp->pcibase2 |= ((val >> 0) & 0xff) << 0;
		}
		if ((bs >> 1) & 1) {
			cpssp->pcibase2 &= ~(0xff << 8);
			cpssp->pcibase2 |= ((val >> 8) & 0xff) << 8;
		}
		if ((bs >> 2) & 1) {
			cpssp->pcibase2 &= ~(0xff << 16);
			cpssp->pcibase2 |= ((val >> 16) & 0xff) << 16;
		}
		if ((bs >> 3) & 1) {
			cpssp->pcibase2 &= ~(0xff << 24);
			cpssp->pcibase2 |= ((val >> 24) & 0xff) << 24;
		}
		break;

	case 0x3c:
		/* PCI Interrupt Line Register */
		if ((bs >> 0) & 1) {
			cpssp->pciintline = val;
		}

		/* PCI Interrupt Pin Register */
		if ((bs >> 1) & 1) {
			/* Read-only */
		}

		/* PCI Min Gnt Register */
		if ((bs >> 2) & 1) {
			/* Read-only */
		}

		/* PCI Max Lat Register */
		if ((bs >> 3) & 1) {
			/* Read-only */
		}
		break;

	default:
		if (0x40 <= addr) {
			fprintf(stderr, "WARNING: FAU_PARCA: Writing 0x%08x to reserved register 0x%02x.\n", val, addr);
		}
		break;
	}
	
	return 0;
}

void *
COMP_(create)(
	const char *name,
	const char *pe_count,
	const char *row_count,
	struct sig_manage *manage,
	struct sig_boolean *port_power,
	struct sig_boolean *port_n_reset,
	struct sig_pci_bus_main *port_pci_bus,
	struct sig_pci_bus_idsel *port_idsel,
	struct sig_boolean_or *port_intA
)
{
	static const struct sig_boolean_funcs power_funcs = {
		.set = COMP_(power_set),
	};
	static const struct sig_boolean_funcs n_reset_funcs = {
		.set = COMP_(n_reset_set),
	};
	static const struct sig_pci_bus_main_funcs pci_bus_funcs = {
		.ior = COMP_(ior),
		.iow = COMP_(iow),
		.mr = COMP_(mr),
		.mw = COMP_(mw),
	};
	static const struct sig_pci_bus_idsel_funcs idsel_funcs = {
		.c0r = COMP_(c0r),
		.c0w = COMP_(c0w),
	};
	struct cpssp *cpssp;

	assert((2+PE_COUNT_MAX)*(2+ROW_COUNT_MAX)*RREG_COUNT*sizeof(uint16_t)
			<= DCACHESIZE);

	cpssp = malloc(sizeof(*cpssp));
	assert(cpssp);
	memset(cpssp, 0, sizeof(*cpssp));

	cpssp->process.inst_hz = 33*1000*1000;
	sched_process_init(&cpssp->process, COMP_(process), cpssp);

	cpssp->pe_count = atoi(pe_count);
	cpssp->row_count = atoi(row_count);

	/* Call */
	cpssp->port_pci_bus = port_pci_bus;
	sig_pci_bus_main_connect(port_pci_bus, cpssp, &pci_bus_funcs);

	cpssp->port_idsel = port_idsel;
	sig_pci_bus_idsel_connect(port_idsel, cpssp, &idsel_funcs);

	/* Out */
	cpssp->port_intA = port_intA;
	sig_boolean_or_connect_out(port_intA, cpssp, 0);

	/* In */
	cpssp->state_power = 0;
	sig_boolean_connect_in(port_power, cpssp, &power_funcs);

	sig_boolean_connect_in(port_n_reset, cpssp, &n_reset_funcs);

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	free(cpssp);
}
