/* $Id: bridge.c,v 1.250 2009-01-27 17:06:39 potyra Exp $ 
 *
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#ifdef OPENBSD
#include <sys/uio.h>
#endif
#include <arpa/inet.h>
#include <netinet/tcp.h>

#include <assert.h>
#include <errno.h>
#include <sys/fcntl.h>
#include <sys/ioctl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "compiler.h"

#include "bridge.h"
#include "glue-io.h"
#include "glue-log.h"
#include "glue-main.h"


struct conn {
	unsigned int node_myself;
	struct {
		int available;
		struct sockaddr_in addr;
		int fd;
	} node[MAX_PEERS];
};

static struct conn *conn;

static struct bridge *bridge_first = 0;
static struct bridge *bridge_last = 0;

int
cim_get_peer_count(struct bridge *bridge)
{
	return bridge->count;
}

void
cim_create(struct bridge *bridge)
{
	assert(bridge);

	bridge->prev = (struct bridge *) 0;
	bridge->next = (struct bridge *) 0;

	bridge->receive = (void (*)(void *, void *, unsigned int)) 0;
	bridge->s = (void *) 0;
}

void
cim_destroy(struct bridge *bridge)
{
	assert (bridge);
}

void
cim_send(struct bridge *bridge, const void *content, unsigned int conlen)
{
	int n;
	int ret;

	for (n = 0; n < MAX_PEERS; n++) {
		struct cim_msg header;

		if (! (bridge->hosts & (1 << n))
		 || n == conn->node_myself) {
			continue;
		}

		header.timestamp = time_virt();
		header.conn_id = bridge->conn_id;
		header.length = conlen;

		ret = io_write(conn->node[n].fd, &header, sizeof(header));
		assert(0 <= ret);

		ret = io_write(conn->node[n].fd, content, conlen);
		assert(0 <= ret);
	}
}

static void
cim_recv(int fd, void *_b)
{
	struct bridge *b;
	char buf[2048];
	int ret;

	for (;;) {
		struct cim_msg header;

		ret = io_read(fd, &header, sizeof(header));
		if (ret < 0 && errno == EAGAIN) {
			break;
		}
		if (ret == 0) {
			/* Might happen at end of simulation. */
			break;
		}
		assert(0 <= ret);
		assert(ret == sizeof(header));
		assert(header.length < sizeof(buf));

	again:	;
		ret = io_read(fd, buf, header.length);
		if (ret < 0 && errno == EAGAIN) {
			/* sched_yield(); */ /* FIXME */
			goto again;
		}
		if (ret == 0) {
			/* Might happen at end of simulation. */
			break;
		}
		assert(0 <= ret);
		assert(ret == header.length);

		for (b = bridge_first; ; b = b->next) {
			if (! b) {
				/* ID not found. */
				/* Packet dropped... */
				break;
			}
			if (b->conn_id == header.conn_id) {
				/* ID found. */
				(b->receive)(b->s, buf, header.length);
				break;
			}
		}
	}
}

void
cim_connect(
	struct bridge *b,
	void (*receive)(void *, void *, unsigned int),
	void *s
)
{
	assert(b);
	assert(receive);
	/* s might be NULL */

	b->receive = receive;
	b->s = s;

	b->prev = bridge_last;
	b->next = 0;
	if (b->prev) {
		b->prev->next = b;
	} else {
		bridge_first = b;
	}
	bridge_last = b;
}

int dummy = 0; /* FIXME */

void
cim_reconfig(struct bridge *bridge)
{
	assert(bridge);

	assert(dummy);
}

void
conn_init(void)
{
	unsigned int n;

	for (n = 0; n < MAX_PEERS; n++) {
		if (n == conn->node_myself) {
			continue;
		}
		if (conn->node[n].available) {
			assert(0 <= conn->node[n].fd);
			io_register(conn->node[n].fd, (void *) 1, cim_recv);
		}
	}
}

void
conn_exit(void)
{
	unsigned int n;

	for (n = 0; n < MAX_PEERS; n++) {
		if (n == conn->node_myself) {
			continue;
		}
		if (conn->node[n].available) {
			io_unregister(conn->node[n].fd);
		}
	}
}

void
conn_create(void)
{
#if 0 /* FIXME */
	static int on = 1;
	struct cfg *cfg;
	int listen_fd;
	struct sockaddr_in addr;
	socklen_t addrlen;
	int n;
	int ret;
#endif

	conn = malloc(sizeof(*conn));
	assert(conn);
	memset(conn, 0, sizeof(*conn));

#if 0 /* FIXME */
	/*
	 * Read connection info.
	 */
	cfg = cfg_open(basedir, "nodes", -1, "nodes");
	if (! cfg) {
		/* Leave everything set to '0'. */
		return;
	}
	assert(cfg);

	ret = cfg_get_int(cfg, "myself", &conn->node_myself);
	assert(0 <= ret);
	assert(conn->node_myself < sizeof(conn->node) / sizeof(conn->node[0]));

	for (n = 0; n < sizeof(conn->node) / sizeof(conn->node[0]); n++) {
		char key[20];

		sprintf(key, "node%d", n);
		ret = cfg_get_sockaddr_in(cfg, key, &conn->node[n].addr);
		if (ret < 0) {
			assert(n != conn->node_myself);
			conn->node[n].available = 0;
			continue;
		}
		conn->node[n].available = 1;
	}

	cfg_close(cfg);

	/*
	 * Create 'listen' socket.
	 */
	listen_fd = socket(PF_INET, SOCK_STREAM, 0);
	assert(0 <= listen_fd);

	ret = setsockopt(listen_fd, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));
	assert(0 <= ret);

	memset(&addr, 0, sizeof(addr));
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = htonl(INADDR_ANY);
	addr.sin_port = conn->node[conn->node_myself].addr.sin_port;
	addrlen = sizeof(addr);
	ret = bind(listen_fd, (struct sockaddr *) &addr, addrlen);
	assert(0 <= ret);

	ret = listen(listen_fd, 10);
	assert(0 <= ret);

	/*
	 * Let all nodes with lower ID connect.
	 */
	for (n = 0;
	    n < conn->node_myself;
	    n++) {
		if (! conn->node[n].available) {
			conn->node[n].fd = -1;
			continue;
		}

		addrlen = sizeof(addr);
		conn->node[n].fd = accept(listen_fd,
				(struct sockaddr *) &addr, &addrlen);
		assert(0 <= conn->node[n].fd);
	}

	/*
	 * No connection to myself...
	 */
	conn->node[conn->node_myself].fd = -1;

	/*
	 * Connect to all nodes with higher ID.
	 */
	for (n = sizeof(conn->node) / sizeof(conn->node[0]) - 1;
	    conn->node_myself < n;
	    n--) {
		if (! conn->node[n].available) {
			conn->node[n].fd = -1;
			continue;
		}

		conn->node[n].fd = socket(PF_INET, SOCK_STREAM, 0);
		assert(0 <= conn->node[n].fd);

		memset(&addr, 0, sizeof(addr));
		addr.sin_family = AF_INET;
		addr.sin_addr.s_addr = htonl(INADDR_ANY);
		addr.sin_port = htons(0);
		addrlen = sizeof(addr);
		ret = bind(conn->node[n].fd,
				(struct sockaddr *) &addr, addrlen);

	again:	;
		memset(&addr, 0, sizeof(addr));
		addr.sin_family = AF_INET;
		addr.sin_addr.s_addr = conn->node[n].addr.sin_addr.s_addr;
		addr.sin_port = conn->node[n].addr.sin_port;
		addrlen = sizeof(addr);
		ret = connect(conn->node[n].fd,
				(struct sockaddr *) &addr, addrlen);
		while (ret < 0 && errno == ECONNREFUSED) {
			sleep(1);
			goto again;
		}
		assert(0 <= ret);
	}

	/*
	 * Destroy 'listen' socket.
	 */
	ret = close(listen_fd);
	assert(0 <= ret);
#endif
}

void
conn_destroy(void)
{
	free(conn);
}
