/* $Id: Architecture.hpp 4323 2009-01-27 13:48:12Z potyra $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __ARCHITECTURE_HPP_INCLUDED
#define __ARCHITECTURE_HPP_INCLUDED

#include <list>
#include "frontend/ast/LibUnit.hpp"
#include "frontend/ast/ConcurrentStat.hpp"
#include "frontend/ast/Location.hpp"
#include "frontend/ast/Name.hpp"
#include "frontend/ast/Entity.hpp"
#include "frontend/ast/CompInstStat.hpp"

namespace ast {

/** One VHDL architecture */
class Architecture : public LibUnit {
public:
	//! c'tor
	/** @param declName name of the architecture.
          * @param entName name of the referring Entity.
          * @param decls local declarations.
          * @param concStats concurrent statements.
	  * @param libs library clauses
          * @param useCs use clauses
          * @param loc location of the Architecture declaration.
          */
	Architecture(
		std::string *declName,
		Name *entName,
		std::list<SymbolDeclaration*>* decls,
		std::list<ConcurrentStat*>* concStats,
		std::list<SimpleName*> *libs,
		std::list<Name*>* useCs,
		Location loc
		) :	LibUnit(declName, libs, useCs, decls, loc),
		     	entityName(entName),
			concurrentStats(concStats) {

		assert(entName != NULL);
		assert(entName->candidates.size() < 2);
		if (entName->candidates.size() == 1) {
			Symbol *s = entName->candidates.front();
			this->entity = dynamic_cast<Entity*>(&s->declaration);
			assert(this->entity != NULL);
		} 
		// otherwise a name error was reported by the parser already.
	}

	//! Accept a Visitor.
	//
 	/** All leaf AST nodes need to implement this method.
         *
         *  @param visitor the Visitor that can visit this node.
         */
	virtual void accept(Visitor& visitor) {
		visitor.visit(*this);
	}

	/** Put a textual representation of the AstNode on the stream.
	 *  @param stream stream to put the textual representation to.
	 */
	virtual void put(std::ostream &stream) const {
		assert(this->name != NULL);
		stream << "Architecture "
			<< *this->name
			<< " of " << this->entityName;
	}

	/** The referring entity name */
	Name *entityName;
	/** concurrent stats of Entity */
	std::list<ConcurrentStat*> *concurrentStats;
	/** the referring entity */
	Entity *entity;

protected:
	/** Destructor, decrease reference counts */
	virtual ~Architecture() {
		util::MiscUtil::terminate(this->entityName);
		util::MiscUtil::terminate(this->entity);
		util::MiscUtil::lterminate(this->concurrentStats);
	}

};

}; /* namespace ast */

#endif /* __ARCHITECTURE_HPP_INCLUDED */
