/*
 * dcc.cpp -- Part of the ezbounce IRC proxy
 *
 * (C) 1999-2001 Murat Deligonul
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 *
 * ---
 * This handles the dcc proxying and sending for ezbounce. 
 * The DCC Proxying is actually nothing more than a virtual
 * network pipe. The dcc command interception and redirecting is
 * done in conn.cpp.
 * ---
 * TODO/FIXME:
 *
 *      - Memory allocation errors and full buffer errors not quite handled
 *      - look at close_all() again.. should we close the file fd in there? Should we make
 *         one for all derived classes and have it call base to do root cleaning?      
 *
 */
                   
/*
 * Don't relay as soon as it gets the text. Wait till the other side
 * is writable. Off by default. May improve reliability of connections for
 * you if you're experiencing any problems such as stalling or corruption.
 */
#undef GENTLE


/* 
 * How big shall the buffer be expandable too?
 * If you keep needing to increase this it means I fu*ked up somewhere!
 */
 
#define BUFFSIZE 32720

/*
 * Packetsize for sending files 
 * Ought to be runtime configurable 
 */
 
#define PACKETSIZE 2048
 

#include "autoconf.h"

#include <sys/types.h>
#include <sys/socket.h>
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#ifdef HAVE_SYS_FILIO_H
    #include <sys/filio.h>
#endif
#ifdef HAVE_SYS_IOCTL_H
    #include <sys/ioctl.h>
#endif
#include <sys/time.h>
#include <netinet/in.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/stat.h>
#include "dcc.h"
#include "linkedlist.h"
#include "general.h"
#include "config.h" 


#ifdef __DEBUG__
#   define DCC_DEBUG
#endif

#ifdef DCC_DEBUG
#   include <stdio.h> 
#   define DEBUG(format, args...) fprintf (stderr, format, ## args)
#else
#   define DEBUG(format, args...)
#endif

 
int dcc::num_waiting = 0;

/* Status of a DCC connection: */
enum {
        ACCEPTED     = 0x8,              /* Connection accepted */   
        CONNECTING   = 0x10,             /* DCC Proxy trying to establish full link */
        CONNECTED    = 0x20,             /* Full proxy link established */

/*
 * More stupid flags so I can do an efficient way of checking for
 * timeouts on waiting for receiver to connect. The timeout right now
 * is 90 seconds.
 */
        TIMES_UP     = 0x800
};



/* Does the port binding for the DCC listen sockets,
 * finding an open port in the config-specified range if
 * required */
static unsigned short bind_port(int fd, struct sockaddr_in * psin)
{
    if (config.dcc_ports)
    {
        char tok[20];
        int x = 0;
        while (gettok(config.dcc_ports, tok, sizeof(tok), ',', ++x))
        {
            char _lbound[8], _ubound[8];
            unsigned short lbound, ubound = 0;
            gettok(tok, _lbound, sizeof _lbound, '-', 1);
            lbound = atoi(_lbound);
            if (strchr(tok, '-'))
            {
                gettok(tok, _ubound, sizeof _ubound, '-', 2);
                ubound = atoi(_ubound);
            }

            DEBUG("Testing listen ports lower: %d upper: %d\n", lbound, ubound);
            if (!ubound)
                ubound = lbound;
            for (int i = lbound; i <= ubound; i++)
            {
                DEBUG("Now calling bind() on port %d\n", i);
                psin->sin_port = htons(i);
                if (!bind(fd, (struct sockaddr *) psin, sizeof(struct sockaddr_in)))
                    return psin->sin_port;
            }
        }
    }

    /* No dice? Bind to somewhere random */
    DEBUG("Binding to random port\n");
    psin->sin_port = htons(0);
    return bind(fd, (struct sockaddr *) psin, sizeof(struct sockaddr_in));
}


/*
 * The base dcc constructor, sets up the listening socket
 *
 */
dcc::dcc(struct sockaddr_in * host_info, unsigned short * listen_port)
{
    struct sockaddr_in sin;
    int listen_fd; 
    
    sender = receiver = 0;

    memset(&sin, 0, sizeof(sin));
    if (host_info)
        memcpy(&sin, host_info, sizeof(sin));
    sin.sin_family = AF_INET;

    listen_fd = socket(AF_INET, SOCK_STREAM, 0);

    if (listen_fd < 0 
        || bind_port(listen_fd, &sin) == (u_short) -1
        || listen(listen_fd,2) < 0)
    {
        *listen_port = 0;
        close_all();
        return;
    }

    socklen_t dummy = sizeof(sin);
    getsockname(listen_fd, (struct sockaddr *) &sin, &dummy);
    *listen_port = ntohs(sin.sin_port);
    if (host_info)
        memcpy(host_info, &sin, sizeof(sin));
    
    lsock = new dccsock(this, listen_fd, POLLIN);
    num_waiting++;

    stat = 0;
    dcc_list_entry = 0;
    start_time = time(NULL);
}

dcc::~dcc()
{
    close_all();
    
    if (sender)
        delete sender->buffer; 
    if (receiver)
        delete receiver->buffer;
    delete sender;
    delete receiver;
}

/*
 * FIXME: Should this also delete the objects?
 * Or at least destroy their buffers? 
 */
void dcc::close_all(void)
{
    if (receiver)
    {
        delete receiver->sock;
        receiver->sock = 0;
    }
    if (sender)
    {
        delete sender->sock;
        sender->sock = 0;
    }
    if (lsock)
        num_waiting--;
    delete lsock;
    lsock = 0;

    stat = 0;
}

/*
 * Create a dcc pipe object. Fill 'sender' with info given.
 * By default binds to default interface on random port, and
 * waits for the receiver to connect.  
 * By supplying something non-null for host_info you can 
 * control what port or interface. Also when function completes
 * you will know what IP address the sock was bound.
 *
 * On success: listen_port is assigned non-0 value.
 */
dccpipe::dccpipe(unsigned long sender_address, unsigned short sender_port, 
         struct sockaddr_in * host_info, unsigned short * listen_port) : dcc::dcc(host_info, listen_port)
{
    /* 
     * At this point the constructor for the base class has been called 
     * and we should have a socket setup, waiting for connections.
     */
    if (!*listen_port)
    {
        if (lsock)
            num_waiting --;
        delete lsock;
        lsock = 0;
        return;
    }

    sender = new dccinfo_t;
    receiver = new dccinfo_t;

    sender->address = htonl(sender_address);
    sender->port = htons(sender_port);
    sender->sock = 0;
    sender->buffer = 0;
    receiver->sock = 0;
    receiver->port = 0;
    receiver->address = 0;
    receiver->buffer = 0;
}


/*
 * Check an indiviual dccpipe object's sockets for events. 
 * Handle connect completion, accepting connections and relaying data.
 *
 * Return any of the following:
 *
 * DCC_ACCEPTED    -- connection has been accepted - now trying to connect to the sender
 * DCC_ESTABLISHED -- async connect worked ok. transfer now in progress
 * DCC_CLOSED      -- dcc connection closed by eof from one side. (transfer complete? who knows) Kill me now
 * DCC_ERROR       -- fatal error somewhere. Kill me now.
 * 
 */
int dccpipe::poll(struct sockaddr_in * psin)
{
    int ret = 0;

    if (stat & TIMES_UP)
        return DCC_PIPE_TIMEDOUT;
    if (lsock && lsock->revents(POLLIN))
    {
        socklen_t size = sizeof (struct sockaddr_in);
        int f = accept(lsock->fd, (struct sockaddr *) psin, &size);
        receiver->sock    = new dccsock(this, f, POLLIN);
        receiver->buffer  = new dynbuff(256, BUFFSIZE);
        receiver->port    = psin->sin_port;
        receiver->address = psin->sin_addr.s_addr;
        stat |= ACCEPTED;
        stat |= CONNECTING;
        nonblocking(receiver->sock->fd);
        delete lsock;
        num_waiting--;
        lsock = 0;
        ret = DCC_ACCEPTED;
        if (!do_connect())
        {
            close_all();
            return DCC_ERROR;
        }
    } 
    /* 
     * Remember: the receiver connects to the socket
     * we setup. So by now any read/write errors
     * from it mean that he's gone. So reset if that happens
     */
    else if (receiver && receiver->sock->revents(POLLIN))
    {
        unsigned long size = receiver->buffer->get_size();
        switch (receiver->buffer->add(receiver->sock->fd, (size ? 2048 : 8192)))
        {
        case DYNBUFF_ERR_READ:
        case DYNBUFF_ERR_WRITE:
            close_all();
            return DCC_ERROR;

        default:
            /* If we're connected to the other guy then
             *  we can send it to him. If we're still waiting
             *  on that we can save any text from this guy until
             *  that happens. This might happen if the connection is a 
             *  dcc chat for example
             *  
             *  Make sure that the flush operation worked w/o any errors
             *  in write.
             */
#ifndef GENTLE
            if (sender->sock && (stat & CONNECTED)) 
            {
                if (receiver->buffer->flush(sender->sock->fd) == DYNBUFF_ERR_WRITE && errno != EAGAIN)
                {
                    close_all();
                    return DCC_ERROR;
                }
                DEBUG("RECEIVER: new size after read and flush: %ld\n", receiver->buffer->get_size());
            }
#endif
            ret = 1;
        }
    }

    if ((stat & CONNECTED) && sender->sock->revents(POLLIN))
    {
        /* 
         * Just check that we can get the data from the sender w/o
         * errors..
         */
        unsigned long size = sender->buffer->get_size();     
        if (sender->buffer->add(sender->sock->fd, (size ? 2048 : 8192)) < 1)
        {
            close_all();
            return DCC_ERROR;
        }
#ifndef GENTLE
        sender->buffer->flush(receiver->sock->fd);
        DEBUG("SENDER: new size after read and flush: %ld\n", sender->buffer->get_size());
#endif
        ret = 1;
    }
 
    if (receiver)
    {
        if (receiver->sock->revents(POLLOUT))
        {
             /* Stuff that got queued */
            DEBUG("Writability status on receiver: trying to write %ld bytes\n", sender->buffer->get_size());
            sender->buffer->flush(receiver->sock->fd);
            DEBUG("   New size: %ld\n", sender->buffer->get_size());
            ret = 1;
        }
        else if (receiver->sock->revents(POLLERR | POLLHUP))
        {
            /* receiver is gone?? */
            DEBUG("Lost connection to recevier: POLLERR and/or POLLHUP detected\n");
            close_all();
            return DCC_ERROR;
        }
    }
    

    if (sender->sock)
    {
        if (sender->sock->revents(POLLOUT))
        {
            if (stat & CONNECTING)
            {
                /* 
                 * To test if the nonblocking connect worked, we 
                 * need to another stupid recv_test and if that works
                 * then the connection is ok. 
                 */
                if (recv_test(sender->sock->fd) < 1)
                {
                    on_connect(errno);
                    return DCC_ERROR;
                }
                else 
                {
                    /* 
                     * Connection to the other guy worked. 
                     * Send any crap that the receiver has sent to us to 
                     * the sender that we have queued here.
                     */
                    ret = DCC_PIPE_ESTABLISHED;
                    on_connect(0);
                    receiver->buffer->flush(sender->sock->fd);
                }
            }
            else 
            {
                /* Stuff that got queued */
                DEBUG("Writability status on sender: trying to write %ld bytes\n", receiver->buffer->get_size());
                receiver->buffer->flush(sender->sock->fd);
                DEBUG("   New size: %ld\n", receiver->buffer->get_size());
                ret = 1;
            }            
        } else if (sender->sock->revents(POLLERR | POLLHUP))
        {
            DEBUG("DCC: POLLHUP and/or POLLERR
                   Shouldn't happen!\n");
            close_all();
            return DCC_ERROR;
        }
    }

    /* Check buffer sizes. We must restrict them somewhat or a fast sender 
     * sending to say, a 28.8 modem can easily explode the buffers */
    if (ret == 1 && sender && receiver)
    {
        u_long snd_size = (sender->buffer) ? sender->buffer->get_size() : 0;
        u_long rcv_size = 0; 
        rcv_size = receiver->buffer ? receiver->buffer->get_size() : 0;
        if ((stat & CONNECTED) && snd_size)
        {
            DEBUG("Checking for writability ONLY on receiver\n");
            receiver->sock->set_events(POLLOUT);
            /* Try to limit the queue size */
        } else if (rcv_size < 2048)     
            receiver->sock->set_events(POLLIN);
        if ((stat & CONNECTING) || rcv_size)
        {            
            sender->sock->set_events(POLLOUT);
            DEBUG("Checking for writability ONLY on sender\n");
        }  /* Try to limit the queue size */
        else if (stat & CONNECTED) 
        {
            if (snd_size < 2048)
                sender->sock->set_events(POLLIN);
            else 
                sender->sock->set_events(0);
        }
    }
    return ret;
}

bool dccpipe::on_connect(int err)
{
    stat &= ~CONNECTING;
    if (err)
        goto out;
    
    sender->buffer = new dynbuff(2048, BUFFSIZE);
    if (!sender->buffer)
        goto out;
    
    stat |= CONNECTED;

    return 1;

out:
    close_all();
    return 0;
}

/*
 * Do a nonblocking connect to the receiver
 */
bool dccpipe::do_connect(void)
{
    struct sockaddr_in sin;
    int f = socket(AF_INET, SOCK_STREAM, 0);
    if (f < 0) 
        return 0;

    sender->sock = new dccsock(this, f, POLLIN | POLLOUT);

    sin.sin_family = AF_INET;
    sin.sin_port = 0;
    sin.sin_addr.s_addr = sender->address;
    sin.sin_port = sender->port;

    
    if (!nonblocking(sender->sock->fd))
        return 0;

    switch (connect(sender->sock->fd, (struct sockaddr *) &sin, sizeof sin))          
    {
    case -1:
        if (errno != EINPROGRESS)
            return 0;
    default:     
        return 1;
    }
}

dccsend::dccsend(const char * filename, struct sockaddr_in * host_info, unsigned short * listen_port, unsigned long * filesize)
            : dcc::dcc(host_info, listen_port)
{
    /* 
     * At this point the constructor for the base class has been called 
     * and we should have a socket setup, waiting for connections.
     * *listen_port will be > 0 if this all ok.
     * Also check that we can actually read and send this file.
     */
    this->filename = 0;
    file = -1;
    sent = pos = packets = bytes2send = 0;

    if (!*listen_port || access(filename, R_OK))
    {
        *listen_port = 0;
        if (lsock)
            num_waiting--;
        delete lsock;
        lsock = 0;
        return;
    }

    struct stat st;
    ::stat(filename, &st);

    *filesize = st.st_size;

    this->filename = my_strdup(filename);
}

dccsend::~dccsend()
{
    close_all();
    delete[] filename;
    filename = 0;
}

/* 
 * Again like dcc::poll. Call once you've got a confirmation
 * that there is data waiting. Does the work needed.
 * Return:
 * DCC_SEND_ESTABLISHED:    Connection accepted. Send is happening.
 * DCC_CLOSED:              Socket was closed, but send is not complete. Kill object.
 * DCC_SEND_COMPLETE:       Ditto. Kill object.
 */
int dccsend::poll(struct sockaddr_in * psin)
{
    int ret = 0;

    if (stat & TIMES_UP)
        return DCC_SEND_TIMEDOUT;
    if (lsock && lsock->revents(POLLIN))
    {
        socklen_t size = sizeof (struct sockaddr_in);
        if ((file = open(filename, O_RDONLY)) < 0)
        {
            close_all();
            return DCC_ERROR;
        }

        struct stat st;
        fstat(file, &st);
        bytes2send = st.st_size;
        
        int f = accept(lsock->fd, (struct sockaddr *) psin, &size);
        receiver = new dccinfo_t;
        receiver->sock = new dccsock(this, f, POLLIN);
        receiver->buffer =  0;                           /* Don't need one */
        receiver->port = psin->sin_port;
        receiver->address = psin->sin_addr.s_addr;
        stat |= ACCEPTED;
                                     							/* Hi Sara!! */
        nonblocking(receiver->sock->fd);
        delete lsock;
        lsock = 0;
        num_waiting--;
        ret = DCC_SEND_ESTABLISHED;
        start_time = time(NULL);
        
        int q = send_next_packet();
        if (q <= 0)
        {
            /* This checks for sending zero byte files.  I don't know
             * if I'm supposed to deny the send in the first place, but
             * oh well here are the checks for it and this will also detect
             * if the first packet sent never got off the ground because of a failed
             * read() */
            close_all();
            if (q < 0)
                return DCC_ERROR;
            else 
                return DCC_SEND_COMPLETE;
        }
    }
    if (receiver && receiver->sock->revents(POLLIN))
    {
        /* Receiver is updating his position on the file */
        int numbytes = 0;
        unsigned long bytes;

        if ((ioctl(receiver->sock->fd, FIONREAD, &numbytes) == -1)    /* ioctl() call bombs */
            || (!numbytes)      /* or you get 0 bytes, even though it says we got data waiting */
            ||  ((unsigned)read(receiver->sock->fd, &bytes, sizeof(unsigned int)) < sizeof(unsigned int))) /* read some weird shit */
        {
            /* ... then the connection was shut down by him */
            close_all();
            return DCC_CLOSED;
        }
        else if (numbytes)
        {	
            /* update our internal counters .. 
             * if # acked (pos variable) = # sent,
             *    its time to send a new packet
             * if its not, then.. should we send a new packet?
             *   im not sure.. unless we want to be fast.
             * if acked = bytes2send, we're done man.
             *   - do some accounting regarding cps
             */
            
            bytes = (unsigned int) ntohl(bytes);
            DEBUG("Receiver acknowledges %lu/%lu bytes\n", bytes, sent);
            
            pos = bytes;

            if (pos == bytes2send)
            {
                close_all();
                end_time = time(NULL);
                return DCC_SEND_COMPLETE;
            }
            if (pos == sent)
            {
                if (send_next_packet() < 0)
                {
                    close_all();
                    end_time = time(NULL);
                    return DCC_CLOSED;
                }
            }
        }   
        ret = 1;
    }
    return ret;
}

/*
 * Send the next packet and update internal counters
 * Return:
 * -1: Fatal error. Object will need to be killed
 * 0: Nothing to send, or could not send anything because of non-fatal error or
 *      something else
 * >0: # of bytes sent in this packet
 *
 * Does NOT close any files on error! 
 */
int dccsend::send_next_packet(void)
{
    static char buffer[PACKETSIZE + 1];

    int bytesread = read(file, buffer, PACKETSIZE);

    if (bytesread == 0)
        /* Looks like we're at the end. Waiting for the final ack i guess */
        return 0;
    else if (bytesread < 0)
    {
        DEBUG("Error in read: %s\n", strerror(errno));
        return -1;
    }
    
    int sent = send(receiver->sock->fd, buffer, bytesread, 0);
    if (sent != bytesread)
    {
        if (sent == -1)
        {
            if (errno == EWOULDBLOCK || errno == ENOBUFS)
	            lseek(receiver->sock->fd, -bytesread, SEEK_CUR);
            else 
            {
                DEBUG("Non-recoverable failure in send(): %s\n", strerror(errno));
                return -1;
            }                              
         }
  		 else 
            lseek(receiver->sock->fd, -(bytesread - sent), SEEK_CUR);
    }
    this->sent += sent;
    this->packets++;
    
    return sent;
}


/* int dcc::dccsock::event_handler(struct pollfd * pfd)
 * written in conn.cpp */

