/* 
 * config.cpp
 *
 * (C) 1998-2001 Murat Deligonul
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 *
 * ----    
 * does:
 *    Parsing configuration file, loading options to various data structures
 *    rulesets, etc.
 *    A more flexible and human readable configuration file format was chosen 
 *    over something easy to parse.. So the code is kind of largish.
 *
 *  TODO/FIXME:
 *     * Possible wildcard matching capabilites for port_in_set()
 *        
 */

#include "autoconf.h"

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#ifdef HAVE_FCNTL_H
    #include <fcntl.h>
#endif
#include "config.h"
#include "ruleset.h"
#include "general.h"
#include "dynbuff.h"
#include "ezbounce.h"
#include "logfile.h"

#if defined(__DEBUG__) || defined(ENABLE_DEBUG_LOGGING)
#   define DEBUG(format, args...) fprintf (stderr, "DEBUG: " ## format, ## args)
#else
#   define DEBUG(format, args...)
#endif


struct cfg_command_set
{
    const char *command;
    int id;
    bool set;
};

static struct cfg_command_set * match_set_command(const char *);
static int do_set_command(struct config_struct *, const char *, const char *);
static int do_rule_set(struct config_struct *, obj_set<rule_set> *, int, const char *, char *, u_long);
static int do_misc_block(struct config_struct *, int type, const char * block, char *, u_long);
static bool check_config(struct config_struct *, const obj_set<rule_set> *);

enum {
    SET = 50000, ALLOW, DENY, VHOSTS, ADMIN, LISTEN,      /* basic commands */

    LOGFILE, PASSWORD,     /* set-able stuff */
    MAX_CLIENT_IQ, MAX_SERVER_IQ,
    MIN_CLIENT_IQ, MIN_SERVER_IQ,
    MAX_IDLE_TIME,
    MAX_REGISTER_TIME,
    MAX_DNS_WAIT_TIME, MAX_FAILED_PASSWORDS, MAX_FAILED_ADMINS, PIDFILE, 
    LOG_DIR, DEFAULT_LOG_OPTIONS, MAX_LOGFILE_SIZE, MOTD_FILE, DCC_LISTEN_PORT_RANGE, AUTOSERVER
};

/* FIXME: move this into a function. */
static struct cfg_command_set set_commands[] = {
    { "LOGFILE", LOGFILE, 0},
    { "PASSWORD",  PASSWORD, 0},
    { "KILL-ON-FULL-QUEUE",  KILL_WHEN_FULL,   0},
    { "MAX-IDLE-TIME",      MAX_IDLE_TIME,     0},
    { "MAX-REGISTRATION-TIME", MAX_REGISTER_TIME, 0},
    { "MAX-DNS-WAIT-TIME", MAX_DNS_WAIT_TIME, 0},
    { "PREVENT-SELF-CONNECTS", PREVENT_SELF_CONNECTS, 0},
    { "MAX-FAILED-ADMINS", MAX_FAILED_ADMINS, 0},
    { "MAX-FAILED-PASSWORDS", MAX_FAILED_PASSWORDS, 0},
    { "DROP-ON-DISCONNECT", DROP_ON_DISCONNECT, 0},
    { "ENABLE-DETACH-COMMAND", ENABLE_DETACH_COMMAND, 0},
    { "NO-REVERSE-LOOKUPS", NO_REVERSE_LOOKUPS, 0},
    { "ENABLE-VHOST-COMMAND", ENABLE_VHOST_COMMAND, 0},
    { "ENABLE-FAKE-IDENTS", ENABLE_FAKE_IDENTS, 0},
    { "AUTO-FAKE-IDENTS", AUTO_FAKE_IDENT, 0},
    { "ENABLE-AUTO-DETACH", ENABLE_AUTO_DETACH, 0},
    { "WRITE-PIDFILE", WRITE_PIDFILE, 0},
    { "PIDFILE", PIDFILE, 0},
    { "MOTD-FILE", MOTD_FILE, 0},
    { "AUTO-SERVER", AUTOSERVER, 0},

    { "ENABLE-OUTGOING-DCC-PROXYING", ENABLE_OUTGOING_DCC_PROXYING, 0},
    { "ENABLE-INCOMING-DCC-PROXYING", ENABLE_INCOMING_DCC_PROXYING, 0},
    { "DCC-LISTEN-PORT-RANGE", DCC_LISTEN_PORT_RANGE, 0},
    
    /* Logging options */
    { "ENABLE-PRIVATE-LOGGING", ENABLE_PRIVATE_LOGGING, 0},
    { "ENABLE-PUBLIC-LOGGING", ENABLE_PUBLIC_LOGGING, 0},
    { "ENABLE-SEPERATE-LOGGING", ENABLE_SEPERATE_LOGGING, 0},
    { "LOG-DIR", LOG_DIR, 0},
    { "DEFAULT-LOG-OPTIONS", DEFAULT_LOG_OPTIONS, 0},
    { "MAX-LOGFILE-SIZE", MAX_LOGFILE_SIZE, 0},
    
    /* Input queue sizes */
    { "MAX-CLIENT-INPUT-QUEUE", MAX_CLIENT_IQ, 0},
    { "MAX-SERVER-INPUT-QUEUE", MAX_SERVER_IQ, 0},
    { "MIN-SERVER-INPUT-QUEUE", MIN_SERVER_IQ, 0},
    { "MIN-CLIENT-INPUT-QUEUE", MIN_CLIENT_IQ, 0}


};

static inline void ERROR(const char *file, unsigned line, const char *error)
{
    fprintf(stderr, "%s:%ul: %s", file, line, error);
}
/*
 *   Loads configuration file.
 *  Return values:
 *     -1    : error parsing.
 *     0     : error loading.
 *     1     : success
 */
int load_config_file(struct config_struct *pconf, const char *file)
{
    int fd = open(file, O_RDONLY), q;
    char line[1024];
    dynbuff db(4096, 0);
    struct config_struct cf;
    unsigned curline = 0;
    struct stat st;
    obj_set<rule_set> rs;      /* we keep a private list of rulesets we created,
                                     in case loading config file fails we can clean them up */
    const struct cfg_command_set config_commands[] = {
        { "SET", SET, 0},
        { "ALLOW", ALLOW, 0},
        { "DENY", DENY, 0},
        { "LISTEN", LISTEN, 0},
        { "ADMIN", ADMIN, 0},
        { "VHOSTS", VHOSTS, 0}

    };
    /* find how big file is */
    stat(file, &st);
    if (!st.st_size || fd < 0)
        return 0;
    if ((q = db.add(fd, st.st_size)) <= 0)
        return (q == 0) ? 0 : -1;
    close (fd);

    /* clear out config and fill in some defaults first */
    memset(&cf, 0, sizeof(cf));
    cf.min_serverQ    = DEF_MIN_SERVER_IQ;
    cf.max_serverQ    = DEF_MAX_SERVER_IQ;
    cf.min_clientQ    = DEF_MIN_CLIENT_IQ;
    cf.max_clientQ    = DEF_MAX_CLIENT_IQ;
    cf.flags               |= KILL_WHEN_FULL;
    cf.flags               |= ENABLE_DETACH_COMMAND;
    cf.flags               |= ENABLE_VHOST_COMMAND;
    cf.max_idle             = DEF_MAX_IDLE_TIME;
    cf.max_register_time    = DEF_MAX_REGISTER_TIME;
    cf.max_dns_wait_time    = DEF_MAX_DNS_WAIT_TIME;
    cf.flags               &= ~(DROP_ON_DISCONNECT);
    cf.max_failed_passwords =  3;
    cf.max_failed_admins    = 0;
    cf.flags |= PREVENT_SELF_CONNECTS;
    for (unsigned x = 0; x < (sizeof(set_commands) / sizeof(struct cfg_command_set)); x++)
        set_commands[x].set = 0;

    while ((q = db.get_line(0,line,sizeof(line),0)) != -1)
    {
        char *rest;
        char command[40];
        const struct cfg_command_set *pc = NULL;
        curline++;

        remove_tabs(line);        
        
        if (!(gettok(line, command, 8, ' ', 1))    /* Isolate the command */
            || (*command == '#'))                /* Make sure its not a comment */
            continue;
        else                                     /* See if it's valid */
        {
            for (unsigned x = 0; x < (sizeof(config_commands) / sizeof(struct cfg_command_set)); x++)
                if (strcasecmp(config_commands[x].command, command) == 0)
                {
                    pc = &config_commands[x];
                    goto ugly_goto_but_needed;
                }
            ERROR(file, curline, "Unrecongnized command\n");
            return -1;
        }
ugly_goto_but_needed:

        rest = line + strlen(command) + 1;
        switch (pc->id)
        {
        case LISTEN:
            if (!gettok(rest, NULL, 0, ' ',1))
                ERROR(file, curline, "no ports given for listen command.\n");
            else if (!cf.ports)
                cf.ports = my_strdup(rest);
            else if (cf.ports)
                ERROR(file,curline, "only one listen command per file allowed. Ignoring this one\n");
            break;

        case SET:
            char value[100];
            if (!gettok(rest, command,  sizeof(command), ' ', 1) 
                || !gettok(rest, value, sizeof(value), ' ', 2, 1))
            {
                ERROR(file,curline,"set command is incomplete - ignoring.\n");
                continue;
            }
            q = do_set_command(&cf, command,value);
            if (q == -1) {
                fprintf(stderr, "%s:%d: error: unknown 'set' variable \"%s\"\n",
                        file, curline, command);
                        
                return -1;
            }
            else if (q == -2)
                ERROR(file, curline, "this particular variable was already set in the file -- ignoring.\n");
            break;

            /* First we find the { after the DENY or ALLOW and then 
             *  find the ending },
             *  then it is all sent to do_rule_set() 
             */
        case ALLOW:
        case DENY: 
        case ADMIN:
        case VHOSTS:
            {
                if (!strchr(line, '{'))
                {
                    /* check if there is actually a line after */
                    if (db.get_line(0, line, sizeof(line), 0) < 0)   /* force removal as well */
                    {
                        ERROR(file, curline, "Unexpected EOF\n");
                        return -1;
                    }   /* Check if there is actually a { */
                    else if (!strchr(line, '{'))
                    {
                        ERROR(file,curline,"'{' must be on same line or below block command\n");
                        return -1;
                    }
                    curline++;
                }
                
                /* Now find the } */
                u_long len = db.get_pos('}');
                char reason[200];
                if ((signed) len < 0) {
                    ERROR(file,curline, "Can't find matching '}'");
                    return -1;
                }

                rest = new char[len + 1];            /* just reusing rest variable */
                unsigned orig = db.have_char('\n');
                db.get(rest, len - 1);               /* we don't want the '}' to be included */
                rest[len - 1] = 0;
                db.get_line(0, command, sizeof(command),0); /* get rid of line after */
                if (((pc->id == ALLOW || pc->id == DENY) && do_rule_set(&cf, &rs, pc->id, rest, reason,sizeof(reason)) < 0)
                    || ((pc->id == VHOSTS || pc->id == ADMIN) && do_misc_block(&cf, pc->id, rest, reason, sizeof(reason)) < 0))
                {
                    ERROR(file, curline, reason);
                    delete[] rest;
                    return -1;
                }
                curline += (orig - db.have_char('\n'));
                delete[] rest;
            }
            break;

        default:
            ERROR(line, curline, "Impossible error 19!\n");
            break;
        }
    }

    /* 
     * validate configuration structure 
     *     if ok copy local config_struct to supplied pointer
     */
    printf("Validating configuration settings..");
   
    fflush(stdout); 
   
    if (!check_config(&cf, &rs))
        return -1;
    
    
    printf("..ok\n");
    delete pconf->admins;
    
    delete pconf->vhost_table;
    delete[] pconf->logfile;
    delete[] pconf->configfile;
    delete[] pconf->ports;
    delete[] pconf->dcc_ports;
    delete[] pconf->autoserver;
    
    memcpy(pconf, &cf, sizeof(cf));
    pconf->configfile = my_strdup(file);
    return 1;
}


static struct cfg_command_set *match_set_command(const char *command) 
{
    for (unsigned x = 0; x < (sizeof(set_commands) / sizeof(struct cfg_command_set)); x++)
        if (strcasecmp(set_commands[x].command, command) == 0)
            return &set_commands[x];
    return NULL;
}

/*
 *   Return values:
 *  1  -  success
 * -1  -  no such variable
 * -2  -  value for this has already been set..
 */
static int do_set_command(struct config_struct *cf, const char * command, const char * value)
{
    struct cfg_command_set *pc = NULL;
    if ((pc = match_set_command(command)))
    {
        unsigned long v = atoi(value);

        if (pc->set)
            return -2;
        pc->set = 1;
        switch (pc->id)
        {
        case NO_REVERSE_LOOKUPS:
        case ENABLE_VHOST_COMMAND:
        case ENABLE_DETACH_COMMAND:
        case PREVENT_SELF_CONNECTS:
        case DROP_ON_DISCONNECT:
        case KILL_WHEN_FULL:
        case ENABLE_FAKE_IDENTS:
        case AUTO_FAKE_IDENT:
        case ENABLE_OUTGOING_DCC_PROXYING:
        case ENABLE_INCOMING_DCC_PROXYING:
        case ENABLE_AUTO_DETACH:
        case WRITE_PIDFILE:
        case ENABLE_PRIVATE_LOGGING:
        case ENABLE_PUBLIC_LOGGING:
        case ENABLE_SEPERATE_LOGGING:
            if (v)                      
                cf->flags |= pc->id;
            else
                cf->flags &= ~(pc->id);

            return 1;

        /* Numeric Options */
        case MAX_FAILED_ADMINS:
            cf->max_failed_admins = v;
            return 1;
        case MAX_FAILED_PASSWORDS:
            cf->max_failed_passwords = v;
            return 1;
        case MAX_DNS_WAIT_TIME:
            cf->max_dns_wait_time = v;
            return 1;         
        case MIN_CLIENT_IQ:
            cf->min_clientQ = v;
            return 1;
        case MAX_CLIENT_IQ:
            cf->max_clientQ = v;
            return 1;
        case MAX_SERVER_IQ:
            cf->max_serverQ = v;
            return 1;
        case MIN_SERVER_IQ:
            cf->min_serverQ = v;
            return 1;
        case MAX_IDLE_TIME:
            cf->max_idle = (time_t) v;
            return 1;
        case MAX_REGISTER_TIME:
            cf->max_register_time = (time_t) v;
            return 1;

        /* String values */
        case LOGFILE:
            cf->logfile = my_strdup(value);
            return 1;
        case PIDFILE:
            cf->pidfile = my_strdup(value);
            return 1;
        case PASSWORD:
            cf->password = my_strdup(value);
            return 1;
        case LOG_DIR:
            cf->logdir = my_strdup(value);
            return 1;
        case MOTD_FILE:
            cf->motd_file = my_strdup(value);
            return 1;
        case MAX_LOGFILE_SIZE:
            cf->max_logsize = v;
            return 1;
        case DEFAULT_LOG_OPTIONS:
            cf->log_options = logfile::charflags_to_int(value);
            return 1;
        case DCC_LISTEN_PORT_RANGE:
            cf->dcc_ports = my_strdup(value);
            return 1;
        case AUTOSERVER:
            cf->autoserver = my_strdup(value);
            return 1;
        default:
            break;
        }
    }
    return -1;        
}

/* 
 *   Return values:
 *      1  -     Success and number of lines in the block
 *      0  -     Minor syntax error or something, but object was still created.   
 *     -1  -     Error in creating rule set. No rule set created. Reason given in 'reason'.
 *
 */  
static int do_rule_set(struct config_struct *, obj_set<rule_set> *prs, int id, 
                       const char *block, char * reason, u_long reason_max)
{
    dynbuff db(2048, 0);
    char line[512];
    unsigned success_tos = 0, success_froms = 0, curline = 0;
    rule_set * rs;

    if (id == ALLOW)
        rs = new allowed_rule_set;
    else if (id == DENY)
        rs = new denied_rule_set;
    else
        return -1;
    db.add(block);

    while (db.get_line(0, line, sizeof(line), 0) != -1)
    {
        int num = 0, type = 0, current = 1;          
        char address[256], ports[200], ban_reason[200], temp[24];

        curline++;
        /* Check for comment character # and blank lines.. */
        remove_tabs(line);
        if (!gettok(line,temp, sizeof(temp), ' ', 1)
            || *temp == '#')
            continue;

        /* 
         * Get second word first, which can be
         *  'from' or 'to' or the address depending on if a number was entered
         * before them.
         */
        gettok(line,temp, sizeof (temp), ' ', 2);
        if (strcasecmp(temp, "from") == 0)
            type = rule_set::FROM;
        else if (strcasecmp(temp, "to") == 0)
            type = rule_set::TO;

        /* Get first word, which will either be the number  or the type. */
        gettok(line, temp, sizeof(temp), ' ',1);
        if (type)
        {
            if ((num = atoi(temp)) == 0)
            {
                fprintf(stderr, "In line: %s\n'0' or text entered for number field, using unlimited.\n",
                        line);
                num = rule_set::UNLIMITED;
            }
            current = 3;
        }
        else {
            if (strcasecmp(temp, "from") == 0)
                type = rule_set::FROM;
            else if (strcasecmp(temp, "to") == 0)
                type = rule_set::TO;
            else {
                fprintf(stderr, "In line: %s\nFirst word must be 'from' or 'to'.\n", line);
                continue;
            }
            num = rule_set::UNLIMITED;
            current = 2;
        }

        /* Can now get address */
        if (!gettok(line, address, sizeof(address), ' ', current++))
        {
            fprintf(stderr, "In line: %s\nThis line is too short. Minimum requirements are <from/to> <address>\n", line);          
            continue;
        }
        if (type == rule_set::FROM)
            success_froms++;
        else
            success_tos++;

        /* Optional argument: port. Defaults to 'all'/
           Will also handle 'on' (quite messily?) */
        if ((!gettok(line,ports, sizeof(ports), ' ', current++))
            || (
               (strcasecmp(ports,"on") == 0) && 
               (!gettok(line,ports, sizeof(ports), ' ', current++)) &&
               (strcasecmp(ports,"on") == 0)
               ))
            safe_strcpy(ports,"all",sizeof(ports));

        /* Remaining text is the ban reason. Only useful when id == DENY.*/
        if ((id == DENY) &&
            (!gettok(line, ban_reason, sizeof(ban_reason), ' ', current++, 1)))
            safe_strcpy(ban_reason, "No reason was given!", sizeof(ban_reason));

        /* All done w/ this ruleset */
        if (type == rule_set::FROM)
            rs->add_host_from(address, ports, ban_reason, num);
        else
            rs->add_host_to(address, ports, ban_reason, num);

        
    }

    /* Is it valid rule set? */
    if ((id == ALLOW) && (success_tos == 0 || success_froms == 0))
    {
        safe_strcpy(reason, "Rule set of type 'allow' must contain at least one 'from' field and one 'to' field.\n", reason_max);
        delete rs;
        return -1;
    }
    else if (success_tos == 0 && success_froms == 0)
    {
        safe_strcpy(reason, "No valid 'from' or 'to' fields found in rule set. Rule set destroyed.\n", reason_max);
        delete rs;
        return -1;
    }
    /* yes it is valid */
    prs->add(rs);
    return 1;
}

/*
 * handle admin/vhosts blocks
 *
 * return: 1  - success
 *         0  - minor errors, but worked otherwise
 *        -1  - complete failure
 */
int do_misc_block(struct config_struct * cf, int type, const char * block,
                  char * reason, u_long reason_size)
{
    /* temporary storage */
    char admin_pass[PASSWORD_LENGTH] = "";
    char admin_name[NICKNAME_LENGTH] = "";
    
    /* check if an admin as already been defined */
    if (type == ADMIN)
    {
        if (cf->admins)
        {
            safe_strcpy(reason, "Only one admin block may be defined.\n", reason_size);
            return -1;
        }
        else
            cf->admins = new admin_block;
    }
    else if (type == VHOSTS)
    {
        if (cf->vhost_table)
        {
            safe_strcpy(reason, "Only one vhost table may be defined.\n", 
                        reason_size);
            return -1;
        }
        else
            cf->vhost_table = new misc_block;
    }

    dynbuff db(512,0);
    char line[512];
    db.add(block);

    while (db.get_line(0, line, sizeof(line), 0) != -1)
    {
        char word1[512], temp[512];

        remove_tabs(line);
        /* Check for comment character # and blank lines.. */
        if (!gettok(line,word1, sizeof(word1), ' ', 1)
            || *word1 == '#')
            continue;

        if (!gettok(line, temp, sizeof(temp), ' ', 2) && type == ADMIN)
        {
            fprintf(stderr, "While reading: %s: line is incomplete\n", line);
            continue;
        }

        if (type == VHOSTS)
            cf->vhost_table->add(word1);
        else 
        {
            if (strcasecmp("FROM", word1) == 0)
                cf->admins->add(temp);
            else if (strcasecmp("PASSWORD", word1) == 0)
                safe_strcpy(admin_pass, temp, sizeof(admin_pass));
            else if (strcasecmp("NAME", word1) == 0)
                safe_strcpy(admin_name, temp, sizeof(admin_name));
            else {
                delete cf->admins;
                cf->admins = NULL;
                safe_strcpy(reason, "Bad command in one of the lines in this block.\n"
                        "Maybe you used 'set password' or 'set name' -- get rid of the 'set'\n",
                        reason_size);
                return -1;
            }
        }
    }
    
    /* Check all of this */
    if (type == ADMIN)
    {
        if (cf->admins->num() < 1)
        {
            delete cf->admins;
            cf->admins = NULL;
            safe_strcpy(reason, "No addresses were defined in admin block.\n", reason_size);
            return -1;
        }
        else if (strcmp(admin_name, "") == 0 || strcmp(admin_pass, "") == 0) {
            delete cf->admins;
            cf->admins = NULL;
            safe_strcpy(reason, "No password and/or admin-name set in admin block.\n", reason_size);
            return -1;
        }
        /* Looks like it is ok */
        cf->admins->set_credentials(admin_name, admin_pass);
        return 1;
    }
    
    /* type == VHOSTS */
    if (cf->vhost_table->num() < 1)
    {
        delete cf->vhost_table;
        cf->vhost_table = NULL;
        safe_strcpy(reason, "Vhost block was empty.\n", reason_size);         
        return -1;
    }

    /* Can save some memory this way */
    if (cf->vhost_table->have("all"))
    {
        delete cf->vhost_table;
        cf->vhost_table = new misc_block;
        cf->vhost_table->add("all");
    }

    return 1;
}


static bool check_config(struct config_struct *cf, const obj_set<rule_set> *prs)
{
    bool err = 0;
    if ((cf->flags & WRITE_PIDFILE) && !(cf->pidfile))
    {
        fprintf(stderr,"\nPid-file writing requested but no pid-file specified (use: set pidfile ...)");
        err = 1;
    }
    if ((cf->pidfile) && !(cf->flags & WRITE_PIDFILE))
    {
        fprintf(stderr, "\nA pid file was specified but pid-file writing was not.. assuming you want to do that");
        cf->flags |= WRITE_PIDFILE;
    }
    if ((cf->flags & AUTO_FAKE_IDENT) && !(cf->flags & ENABLE_FAKE_IDENTS))
    {
        fprintf(stderr,"\nauto-fake-idents set to 1, but enable-fake-idents not set to 1. Please fix");
        err = 1;
    }
    if ((cf->flags & ENABLE_AUTO_DETACH) && !(cf->flags & ENABLE_DETACH_COMMAND))
    {
        fprintf(stderr,"\nenable-auto-detach enabled but detach/reattach commands are not.");
        err = 1;
    }
    if (!prs->num || !prs->objs)
    {
        fprintf(stderr,"\nNo rulesets defined");
        err = 1;   
    }
    if (!cf->logfile)
    {
        fprintf(stderr, "\nNo logfile defined");
        err = 1;
    }
    if (!cf->vhost_table)
    {
        fprintf(stderr, "\nNo vhost table was defined: creating a default one.");
        cf->vhost_table = new misc_block;
        cf->vhost_table->add("all");
    }
    if (!cf->admins)
    {
        fprintf(stderr, "\nMissing administrator information.");
        err =1;
    }
    if (!cf->password)
    {
        fprintf(stderr, "\nNo global user password -- not recommended but going on.");
        cf->password = my_strdup("");
    }
    if ((cf->flags & ENABLE_PUBLIC_LOGGING) || (cf->flags & ENABLE_PRIVATE_LOGGING))
    {
        if (!cf->logdir || access(cf->logdir, R_OK | W_OK | X_OK))
        {
            fprintf(stderr, "\nCannot access log file base dir '%s': %s", cf->logdir, strerror(errno));
            err = 1;
        }
        if (!cf->log_options)
        {
            fprintf(stderr, "\nNo log options were set: defaulting to log all");
            cf->log_options = logfile::LOG_ALL;
        }
    }

    if (err) {
        fputc('\n', stderr);
        return 0;
    }
    return 1;
}


bool admin_block::authorized(const char * uname, const char * pass, const char * addr)
{
    extern bool smart_match(const char *, const char *);
    /* see if name/pass is ok */
    if (strcmp(uname, admin_name) == 0 && strcmp(admin_password, pass) == 0)
        for (unsigned i = 0; i < stuff.num; i++)
            if (smart_match(addr, stuff.objs[i]))
                return 1;
    return 0;
}

/* Called when rehash command is issued or sighup is caught */
int reload_config_file(struct config_struct *cf, const char *)
{
    rule_set *rs = rule_set::first();
    obj_set<rule_set> ob_set, new_set;
    int yy;
    char * tmp = my_strdup(cf->configfile);
    
    /* To preserve these values through rehash */
    struct in_addr iface = cf->iface, iface_conn = cf->iface_conn;       

    /* Mark rulesets as obsolete */
    for (; rs; rs = rs->next())
        rs->obsolete = 1; 
    if ((yy = load_config_file(cf, tmp)) < 1)
    {
        delete[] tmp;
        for (rs = rule_set::first(); rs; rs=rs->next())
            rs->obsolete = 0;
        return yy;
    }
    delete[] tmp;

    /*
     *  Go through the new list of all rulesets:
     *   * if it's marked obsolete and there are no users, destroy it
     *   * if it's obsolete and there are users, add to ob_set
     *   * if it's not obsolete add it to new_set
     *  Then, compare each rule_set in new_set to each rule_set in ob_set.
     */
    rs = rule_set::first();
    while (rs)
    {
        if (rs->obsolete && !rs->num_registered_to && !rs->num_registered_from)
        {
            rule_set * tmp = rs->next();
            delete rs;
            rs = tmp;
            continue;
        }
        else if (rs->obsolete)
            ob_set.add(rs);
        else
            new_set.add(rs);
        rs = rs->next();
    }

    for (unsigned i = 0; i < new_set.num; ++i)
    {
        /*
         * Compare this new one to each of the obsolete ones..
         * If this new one is the same as the obsolete one, then we'll
         * keep the obsolete one and get rid of this one
         */
        for (unsigned x = 0; x < ob_set.num; ++x)
        {
            if ((ob_set.objs[x]->obsolete) && (*new_set.objs[i] == *ob_set.objs[x]))
            {
                ob_set.objs[x]->obsolete = 0;
                delete new_set.objs[i];
                break;
            }
        }
    }
    
    cf->iface      = iface;
    cf->iface_conn = iface_conn;
    return 1;
}

