// general.cpp
/*
 * (C) 1998, 1999, 2000 Murat Deligonul
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 *
 */

#include "autoconf.h"

#include "general.h"

#include <stdio.h>              
#define MARK fprintf(stderr, "%s: %s: %d: marked\n", __FILE__, __PRETTY_FUNCTION__, __LINE__)


#ifndef INADDR_NONE
#	define INADDR_NONE ((unsigned long) -1)
#endif 

static char tmpbuff[256];
static int max_dns_wait_time;

void set_dns_timeout(int x)
{
    max_dns_wait_time = x;
}

/* 
 *  Gets which'th token, seperated by sep
 *  return 1 on success 0 on failure.
 *  Last argument indicates whether.
 *  
 *  Null argument can be passed as buffer to indicate you only
 *  want to check the existance of a token.
 */

int gettok(const char *p, char *buffer, unsigned long maxsize, 
           char sep, int which, bool get_rest)
{
    int numfound = 1;
    /* remove leading seps */
    while (*p == sep)
        p++;
    do {
        if (numfound == which && *p)
        {
            char * next = strchr(p, sep);
            if ((get_rest && buffer) || (!next /* && *(p+1) */) )
            {
                if (buffer)
                    safe_strcpy(buffer, p, maxsize);
                return 1;
            }
            else if (next)
            {
                if (buffer)
                {
                    u_long bytes2copy = (next - p) + 1; /* how many bytes must be copied, including
                                            NULL character */
                    if (bytes2copy >= maxsize)
                        bytes2copy = maxsize;
                    strncpy(buffer, p, bytes2copy);
                    buffer[bytes2copy - 1] = 0;
                }
                return 1;
            }
            return 0;
        }
        /* check if this letter == sep, and the next letter is not sep */
        if (*p == sep && *(p+1) != sep)
            numfound++;
    } while (*++p);
    /* failed */
    return 0;
} 

/* 
 * Checks if orig has " :" or ":" or " " before it, removes it 
 *   if needed
 */
const char * no_leading(const char *orig)
{
    while (*orig == ':' || *orig == ' ')
        ++orig;
    return orig;
}
/* 
 *  Copies src to dest, pretends dest is maxsize long.
 *  will truncate by 1 if needs room to add \0 (bad?)
 */
int safe_strcpy(char *dest, const char *src, unsigned long maxsize)
{
    unsigned long len = strlen(src);
    
    if (len >=maxsize)
        len = maxsize - 1;
    memcpy(dest, src, len);
    dest[len] = 0;
    return 1;
}

bool resolve_address(const char *hostname, struct sockaddr_in *sin)
{
#ifdef __DEBUG__
    fprintf(stderr, "in resolver: looking up %s..", hostname);
#endif
    if (inet_addr(hostname) == INADDR_NONE)
    {
        struct hostent *phe = NULL;
        alarm(max_dns_wait_time); 
        if (!(phe = gethostbyname(hostname)))
        {
#ifdef __DEBUG__
            fprintf(stderr, "failed\n");
#endif
            return (alarm(0), 0);
        }
#ifdef __DEBUG__                         
        fprintf(stderr, "success\n");
#endif
        alarm(0); 
        memcpy((char *) &sin->sin_addr, phe->h_addr, phe->h_length);
    }
    else
        sin->sin_addr.s_addr = inet_addr(hostname);
#ifdef __DEBUG__   
    fprintf(stderr, "worked; ip.\n");
#endif
    return 1;
}

const char *timestamp(bool simple)
{
    time_t timex = time(NULL);
    if (simple)
    {
        /* Report in [hour:minute] format */
        struct tm *t;
        t = localtime(&timex);
        sprintf(tmpbuff, "[%02d:%02d]", t->tm_hour, t->tm_min);
        
    } else {
        safe_strcpy(tmpbuff,ctime(&timex), sizeof(tmpbuff));
        int q = strlen(tmpbuff);
        if (tmpbuff[q - 1] == '\n')
            tmpbuff[q - 1] = 0;
        strcat(tmpbuff, " ");
    }
    return tmpbuff;
}

int fdprintf(int fd, const char *format, ...)
{
    static char buffer[2048];
    buffer[0] = 0;
    va_list ap;
    va_start(ap, format); 
    vsprintf(buffer, format, ap);
    va_end(ap);
    return (write(fd, buffer, strlen(buffer)));
}

const char * strtrunc(const char * str, unsigned long numspaces)
{
    unsigned long len =  strlen(str);
    if (len > numspaces)
    {
        safe_strcpy(tmpbuff, str, numspaces + 1);
        return tmpbuff;
    }
    return str;
}


/* This is the super version that might be used later */
#ifdef SUPER_MY_STRDUP
char *my_strdup(const char *origstring, unsigned long len)
{
    if (!origstring) 
        return 0;

    len = len ? len : strlen(origstring);
    char *z = new char[len + 1];
    strncpy(z, origstring, len);
    z[len] = 0;
    return z;
}
#else
char *my_strdup(const char *origstring)
{
    if (!origstring) 
        return 0;

    unsigned long len = strlen(origstring);
    char *z = new char[len + 1];
    strncpy(z, origstring, len);
    z[len] = 0;
    return z;
}

#endif 

/* ...stolen from ircd-hybrid 5.2. Slightly modified */
/*
** _match()
** Iterative matching function, rather than recursive.
** Written by Douglas A Lewis (dalewis@acsu.buffalo.edu)
*/
int wild_match(const char *mask,const char *name)
{
    const u_char *m = (const u_char *) mask, *n = (const u_char *) name;
    const char   *ma = mask, *na = name;
    int  wild = 0, q = 0;

    while (1)
    {
        if (*m == '*')
        {
            while (*m == '*')
                m++;
            wild = 1;
            ma = (char *)m;
            na = (char *)n;
        }

        if (!*m)
        {
            if (!*n)
                return !(0);
            for (m--; (m > (u_char *)mask) && (*m == '?'); m--)
                ;
            if ((*m == '*') && (m > (u_char *)mask) &&
                (m[-1] != '\\'))
                return !(0);
            if (!wild)
                return !(1);
            m = (u_char *)ma;
            n = (u_char *)++na;
        }
        else if (!*n)
        {
            while (*m == '*')
                m++;
            return !(*m != 0);
        }
        if ((*m == '\\') && ((m[1] == '*') || (m[1] == '?')))
        {
            m++;
            q = 1;
        }
        else
            q = 0;

        if ((tolower(*m) != tolower(*n)) && ((*m != '?') || q))
        {
            if (!wild)
                return !(1);
            m = (u_char *)ma;
            n = (u_char *)++na;
        }
        else
        {
            if (*m)
                m++;
            if (*n)
                n++;
        }
    }
}

/* takes ip */
bool reverse_lookup(const char *host, char *buffer, u_long len)
{
    struct hostent *pheer;
    struct in_addr ins;
#ifdef __DEBUG__
    fprintf(stderr, "in reverse_lookup: %s..." , host);
#endif
    ins.s_addr = inet_addr(host);
    alarm(max_dns_wait_time); 
    if ((pheer = gethostbyaddr((char *) &ins, sizeof(ins), AF_INET)) == NULL)
    {
#ifdef __DEBUG__
        fprintf(stderr, "failed\n");
#endif
        alarm(0); 
        return 0;
    }
    safe_strcpy(buffer, pheer->h_name, len);
#ifdef __DEBUG__
    fprintf(stderr, "ok.\n");
#endif
    alarm(0);
    return 1;
} 

/*
 *  Used for filling in_addr structures for vhost-related code.
 *  Resolves if needed, makes a test bind, and returns:
 *  
 *  1  - Success
 *  0  - Failed, check errno for reason
 * -1  - name lookup failure 
 * 
 */
int fill_in_addr(const char *host , struct in_addr *pin)
{
    struct sockaddr_in sin;
    int s;
    memset(&sin, 0, sizeof sin);    /* Will fail on FreeBSD w/o this!!! */
    if (!resolve_address(host, &sin))
        return -1;
    /* test bind */
    s = socket(AF_INET, SOCK_STREAM, 0);
    sin.sin_port = htons(0);
    sin.sin_family = AF_INET;
    if (bind(s, (struct sockaddr *) &sin, sizeof sin) < 0)
        return close(s), 0;
    /* works */
    close(s);
    memcpy(pin, &sin.sin_addr, sizeof(sin.sin_addr));
    return 1;
}

/*
 * take seconds, turn to human readable form 
 * verbose  -     include days not just hours:min
 *
 * fixme:   size is ignored
 */
void duration(time_t howlong, bool verbose, char * buff, u_long )
{
    int hours = 0;
    howlong /= 60;
    for (; howlong > 59; howlong -=60)
        hours++;
    /* howlong now stores minutes */
    if (verbose && hours > 23)
    {
        int days = 0;
        for (; hours > 23; hours -=24)
            days++;
        sprintf(buff, "%d day(s) %02d:%02d", days, hours, (int) howlong);   
    }
    else
        sprintf(buff, "%02d:%02d", hours, (int) howlong);
}

/*
 * Return pid of child to parent.
 * 0 to child.
 * -1 on error.
 */
int daemon(bool cr)
{
    pid_t xx;
    switch ((xx = fork()))
    {
    case 0:
        /* Child */
        setsid();
        close(STDIN_FILENO);
        close(STDOUT_FILENO);
        close(STDERR_FILENO);
        if (cr)
            chroot("/");
        return 0;

    case -1:
        return -1;

    default:
        return xx;

    }
}


/*
 * Return: 
 * -1:      error, socket died
 *  0:      socket closed from eof 
 *  1:      socket ok.
 *  2:      socket of, stuff to read;
 */
int recv_test(int fd)
{
    char dummy;
    switch (recv(fd, &dummy, 1, MSG_PEEK))
    {
    case 0:
        return 0;
    case -1:
        if (errno == EAGAIN)
            return 1;
        return -1;
    }
    return 2;
}

const char * remove_path(const char * filename)
{
    const char * p;
    while ((p = strchr(filename, '/')))
    {
        filename = p+1;
    }
    return filename;
}
