/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/**********************************************************************
*
*  File: sqcurve3.c
*
*  Purpose: Does calculations needed for including square curvature
*              in energy. Linear model only.
*              Named quantity methods.
*
*        Vertex star approach.
*
*        Assumes complete star around vertex. v_info comes in with
*        central vertex first, followed by ring in order.  Sides
*        are in same order.
*/

#include "include.h"

/* some timing results:

    4-processor 4400 SGI (polyphemus)
    Old methods used processors in 2:1:1:1 ratio.  New are 1:1:1:1.o
    So would scale better to more procs.
    PLAIN:
    Star methods take about same time as old methods, but better load balance
    EFFECTIVE AREA, NORMAL_SQ
     New methods take same clock time, more total time.
    */  

/* local storage max */
#define MAXV 20

REAL star_sqcurve_method_all ARGS((struct qinfo*,int));
static int H0_flag; /* set to use (H - H_0)^2 */
static REAL h0_value;  /* value of H_0 */


/***********************************************************************
*
*  Function: star_sqcurve_method_init()
*
*  Purpose: Initializes data structures for square curvature.
*/

void star_sqcurve_method_init(mode,mi)
int mode; /* METHOD_VALUE or METHOD_GRADIENT */
struct method_instance *mi;
{ int k,n;
  struct gen_quant_method *gm;

  if ( web.dimension != 2 )
     kb_error(1610,"sq_mean_curvature method only for 2D facets.\n",RECOVERABLE);

  if ( web.modeltype != LINEAR )
     kb_error(1611,"sq_mean_curvature method method only for LINEAR model.\n",RECOVERABLE);

  if ( everything_quantities_flag )
     GEN_QUANTS[sq_mean_curv_quantity_num].modulus = 
        globals[square_curvature_param].value.real;

  /* see if using (H - H_0)^2 adjustment */
  H0_flag = 0;
  k = lookup_global("h_zero");
  if ( k >= 0 ) 
        { 
          H0_flag = 1;
          h0_value = globals[k].value.real;
        }
  else h0_value = 0.0;

  if ( star_sq_mean_curvature_mi < 0 )
  { /* see what method indices correspond to what methods */
     for ( n=0,gm = basic_gen_methods ; gm->name[0] != ' ' ; gm++,n++ )
     { if ( stricmp(gm->name,"star_sq_mean_curvature") == 0 ) 
            star_sq_mean_curvature_mi = n;
        if ( stricmp(gm->name,"star_eff_area_sq_mean_curvature") == 0 ) 
            star_eff_area_sq_mean_curvature_mi = n;
        if ( stricmp(gm->name,"star_normal_sq_mean_curvature") == 0 ) 
            star_normal_sq_mean_curvature_mi = n;
     }
  }
  if ( mi->gen_method == star_eff_area_sq_mean_curvature_mi )
      if ( SDIM != 3 )
             kb_error(1612,"star_eff_area_sq_mean_curvature method only for 3D space.\n",
                RECOVERABLE);
  if ( mi->gen_method == star_normal_sq_mean_curvature_mi )
      if ( SDIM != 3 )
             kb_error(1613,"star_normal_sq_mean_curvature method only for 3D space.\n",
                RECOVERABLE);

  if ( self_similar_flag )
  { int param = lookup_global(SELFSIM_NAME);
     kb_error(1614,"Cannot use self-similarity with star_sq_... methods.\n",RECOVERABLE);

      if ( param < 0 ) /* missing, so add */
        { param = add_global(SELFSIM_NAME);
          globals[param].value.real = 1.0;  /* default */
          globals[param].flags |=  ORDINARY_PARAM | SURFACE_PARAMETER;
        }
     if ( H0_flag == 0 ) { H0_flag = 1; h0_value = 0.0; }
  }
  if ( vedge_timestamp < top_timestamp )
     { make_vedge_lists();
        vedge_timestamp = top_timestamp;
     }
}

/***********************************************************************
*
*  function: star_sqcurve_method_all()
*
*  purpose:  value and gradient calculations using common code.
*
*/
#define TEST_SQ    0
#define PLAIN_SQ  1
#define EFF_SQ     2
#define NORMAL_SQ 3

REAL star_sqcurve_method_all(v_info,mode)
struct qinfo *v_info;
int mode; /* METHOD_VALUE, METHOD_GRADIENT, or METHOD_HESSIAN */
{
#ifndef MAC_CW
  int variety; /* PLAIN_SQ, EFF_SQ, NORMAL_SQ */
  int pairs;
  int i,j,k,ii,kk;
  REAL energy,ff,fn,nn,area,g,h;
  REAL dAdv[MAXCOORD], *a, *d, *s1s1, *s1s2, *s2s2;
  REAL vnorm[MAXCOORD];
  REAL **dAdv1,**dAdv2,***dvnorm1,***dvnorm2,**ds2;
  REAL ***ddAdv1dv1,***ddAdv2dv1,***ddAdv1dv2,***ddAdv2dv2;
  REAL ***ddss12,***ddss21,***ddss22;
  REAL **dfndv1,**dfndv2,**dffdv1,**dffdv2,**dnndv1,**dnndv2;
  REAL ***dfdv1,***dfdv2;
  REAL *s1,*s2;
  REAL **s = v_info->sides[0];
  REAL temp[MAXCOORD];
  REAL antisymnorm[MAXCOORD][MAXCOORD];
  REAL antisymf[MAXCOORD][MAXCOORD];
  REAL aa75,a75,ffaaa,ffaa,an75,ann75,fnnna,fnna,fn75,fnn75,ffn75;
  REAL afnfn,afffnfn,fffnfn,ffafnfnfn,afffffnfnfnfn,fffffnfnfna,fffffnfnfn,
        fffffnfn; /* common subexpressions */
  REAL aa[MAXV*5];
  MAT2D(ds1,12*MAXV,MAXCOORD);
  MAT3D(ddss11,16*MAXV,MAXCOORD,MAXCOORD);
  REAL hh0;  /* curvature adjusted for h-h0 */

  if ( METH_INST[v_info->method].gen_method == star_normal_sq_mean_curvature_mi )
     variety = NORMAL_SQ;
  else if ( METH_INST[v_info->method].gen_method == star_eff_area_sq_mean_curvature_mi )
     variety = EFF_SQ;
  else variety = PLAIN_SQ;

/* variety = TEST_SQ; */    /****    TESTING ONLY ******/

  pairs = (v_info->vcount - 1);
  if ( pairs <= 0 ) return 0.0;

  if ( v_info->vcount > MAXV )
  { a = (REAL*)mycalloc(5*pairs,sizeof(REAL));
     ds1 = dmatrix(0,12*v_info->vcount,0,SDIM-1);
  } else
  { memset((char*)aa,0,sizeof(REAL)*5*pairs);
     a = aa;
     memset((char*)ds1[0],0,sizeof(REAL)*12*v_info->vcount*MAXCOORD);
  }
  d = a+pairs; s1s1 = d + pairs; s1s2 = s1s1 + pairs; s2s2 = s1s2 + pairs;
  ds2 = ds1 + pairs; dAdv1 = ds2 + pairs; dAdv2 = dAdv1 + pairs;
  dfndv1 = dAdv2 + pairs; dfndv2 = dfndv1 + pairs; dffdv1 = dfndv2 + pairs;
  dffdv2 = dffdv1 + pairs; dnndv1 = dffdv2 + pairs; dnndv2 = dnndv1 + pairs;

  /* basic dot products */
  area = 0.0;
  for ( j = 0 ; j < SDIM ; j++ ) dAdv[j] = vnorm[j] = 0.0;
  for ( k = 0 ; k < pairs ; k++ )
  { s1 = s[k]; s2 = s[(k+1==pairs)?0:k+1];
     s1s1[k] = SDIM_dot(s1,s1);
     s1s2[k] = SDIM_dot(s1,s2);
     s2s2[k] = SDIM_dot(s2,s2);
     d[k] = s1s1[k]*s2s2[k] - s1s2[k]*s1s2[k];
     a[k] = 0.5*sqrt(d[k]);
     area += a[k];
     for ( j = 0 ; j < SDIM ; j++ )
     { ds1[k][j] = 2*(s2s2[k]*s1[j] - s1s2[k]*s2[j]);
        ds2[k][j] = 2*(s1s1[k]*s2[j] - s1s2[k]*s1[j]);
        dAdv1[k][j] = 0.125/a[k]*ds1[k][j];
        dAdv2[k][j] = 0.125/a[k]*ds2[k][j];
        dAdv[j] -= dAdv1[k][j] + dAdv2[k][j];
     }
     if ( variety != PLAIN_SQ )
     { cross_prod(s1,s2,temp);
        for ( j = 0 ; j < SDIM ; j++ )
          vnorm[j] += 0.5*temp[j];
     }
  }

  /* energy */
  ff = SDIM_dot(dAdv,dAdv);
  switch ( variety )
  { case TEST_SQ:
        nn = SDIM_dot(vnorm,vnorm);
        energy = nn; break; 
     case PLAIN_SQ: 
        energy = 0.75/area*ff;
        break;
     case EFF_SQ:
        nn = SDIM_dot(vnorm,vnorm);
        if  ( nn <= 0.0 ) return 0.0;
        energy = 0.75*area*ff/nn;
        break;
     case NORMAL_SQ:
        fn = SDIM_dot(dAdv,vnorm);
        if ( fn == 0.0 ) 
        { ff = 0.0; fn = 1e-6; }
        hh0 = ff/fn - 2*h0_value/3;
        energy = 0.75*area*hh0*hh0;
        break;
    }
    if ( get_vattr(v_info->id) & AXIAL_POINT )
    { if ( sym_flags & DOUBLE_AXIAL )
          energy /= rotorder/2;
      else energy /= rotorder;
      energy *= rotorder/v_info->axial_order;
    }
    if ( mode == METHOD_VALUE ) goto all_exit;

    if ( v_info->vcount > MAXV )
      ddss11 = dmatrix3(16*pairs,SDIM,SDIM);
    else memset((char*)ddss11[0][0],0,sizeof(REAL)*16*pairs*MAXCOORD*MAXCOORD);
    ddss12 = ddss11 + pairs; ddss21 = ddss12 + pairs; ddss22 = ddss21 + pairs;
    ddAdv1dv1 = ddss22 + pairs; ddAdv1dv2 = ddAdv1dv1 + pairs;
    ddAdv2dv1 = ddAdv1dv2 + pairs; ddAdv2dv2 = ddAdv2dv1 + pairs;
    dvnorm1 = ddAdv2dv2 + pairs; dvnorm2 = dvnorm1 + pairs;
    dfdv1 = dvnorm2 + pairs; dfdv2 = dfdv1 + pairs;

    /* first, some more common terms */
    for ( k = 0 ; k < pairs ; k++ )
    { s1 = s[k]; s2 = s[(k+1==pairs)?0:k+1];
      for ( i = 0 ; i < SDIM ; i++ )
      { ddss11[k][i][i] = 2*s2s2[k];
         ddss12[k][i][i] = -2*s1s2[k];
         ddss21[k][i][i] = -2*s1s2[k];
         ddss22[k][i][i] = 2*s1s1[k];
         for ( j = 0 ; j < SDIM ; j++ )
         { ddss11[k][i][j] -= 2*s2[i]*s2[j];
            ddss12[k][i][j] += 4*s1[i]*s2[j] - 2*s2[i]*s1[j];
            ddss21[k][i][j] += 4*s2[i]*s1[j] - 2*s1[i]*s2[j];
            ddss22[k][i][j] -= 2*s1[i]*s1[j];

            ddAdv1dv1[k][i][j] = -0.125/a[k]/a[k]*dAdv1[k][i]*ds1[k][j]
                      + 0.125/a[k]*ddss11[k][i][j];
            ddAdv1dv2[k][i][j] = -0.125/a[k]/a[k]*dAdv1[k][i]*ds2[k][j]
                      + 0.125/a[k]*ddss12[k][i][j];
            dfdv1[k][i][j] = -ddAdv1dv1[k][i][j] - ddAdv1dv2[k][i][j];
            ddAdv2dv1[k][i][j] = -0.125/a[k]/a[k]*dAdv2[k][i]*ds1[k][j]
                      + 0.125/a[k]*ddss21[k][i][j];
            ddAdv2dv2[k][i][j] = -0.125/a[k]/a[k]*dAdv2[k][i]*ds2[k][j]
                      + 0.125/a[k]*ddss22[k][i][j];
            dfdv2[k][i][j] = -ddAdv2dv1[k][i][j] - ddAdv2dv2[k][i][j];
         }
         dffdv1[k][i] = 2*SDIM_dot(dfdv1[k][i],dAdv);
         dffdv2[k][i] = 2*SDIM_dot(dfdv2[k][i],dAdv);
      }

      if ( variety == PLAIN_SQ ) continue;
      dvnorm1[k][0][1] = -0.5*s2[2];
      dvnorm1[k][0][2] =  0.5*s2[1];
      dvnorm1[k][1][0] =  0.5*s2[2];
      dvnorm1[k][1][2] = -0.5*s2[0];
      dvnorm1[k][2][0] = -0.5*s2[1];
      dvnorm1[k][2][1] =  0.5*s2[0];
      dvnorm2[k][0][1] =  0.5*s1[2];
      dvnorm2[k][0][2] = -0.5*s1[1];
      dvnorm2[k][1][0] = -0.5*s1[2];
      dvnorm2[k][1][2] =  0.5*s1[0];
      dvnorm2[k][2][0] =  0.5*s1[1];
      dvnorm2[k][2][1] = -0.5*s1[0];
      for ( i = 0 ; i < SDIM ; i++ )
      { dfndv1[k][i] = SDIM_dot(dfdv1[k][i],vnorm);
         dfndv1[k][i] += SDIM_dot(dAdv,dvnorm1[k][i]);
         dfndv2[k][i] = SDIM_dot(dfdv2[k][i],vnorm);
         dfndv2[k][i] += SDIM_dot(dAdv,dvnorm2[k][i]);
         dnndv1[k][i] = 2*SDIM_dot(vnorm,dvnorm1[k][i]);
         dnndv2[k][i] = 2*SDIM_dot(vnorm,dvnorm2[k][i]);
      }
    }

  /* now, the actual gradients */
  for ( k = 0 ; k < v_info->vcount ; k++ )
     for ( i = 0 ; i < SDIM ; i++ )
        v_info->grad[k][i] = 0.0;

  switch ( variety )
  { case TEST_SQ:
        for ( k = 0 ; k < pairs ; k++ )
        { REAL *grad2 = (k+1==pairs)?v_info->grad[1]:v_info->grad[k+2];
          for ( i = 0 ; i < SDIM ; i++ )
          { g = dnndv1[k][i];
             v_info->grad[k+1][i] += g;
             v_info->grad[0][i] -= g;
             g = dnndv2[k][i];
             grad2[i] += g;
             v_info->grad[0][i] -= g;
          }
         }
        break;

     case PLAIN_SQ: 
        for ( k = 0 ; k < pairs ; k++ )
        { REAL *grad2 = (k+1==pairs)?v_info->grad[1]:v_info->grad[k+2];
          for ( i = 0 ; i < SDIM ; i++ )
          { g = -0.75*ff/area/area*dAdv1[k][i] + 0.75/area*dffdv1[k][i];
             v_info->grad[k+1][i] += g;
             v_info->grad[0][i] -= g;
             g = -0.75*ff/area/area*dAdv2[k][i] + 0.75/area*dffdv2[k][i];
             grad2[i] += g;
             v_info->grad[0][i] -= g;
          }
         }
        break;
     case EFF_SQ:
        for ( k = 0 ; k < pairs ; k++ )
        { REAL *grad2 = (k+1==pairs)?v_info->grad[1]:v_info->grad[k+2];
          for ( i = 0 ; i < SDIM ; i++ )
          { g = 0.75*ff/nn*dAdv1[k][i] 
                    + 0.75*area/nn*dffdv1[k][i]
                      - 0.75*area/nn/nn*ff*dnndv1[k][i];
             v_info->grad[k+1][i] += g;
             v_info->grad[0][i] -= g;
             g = 0.75*ff/nn*dAdv2[k][i] 
                    + 0.75*area/nn*dffdv2[k][i]
                      - 0.75*area/nn/nn*ff*dnndv2[k][i];
             grad2[i] += g;
             v_info->grad[0][i] -= g;
          }
        } 
        break;
     case NORMAL_SQ:
      if ( fn != 0.0 )
        for ( k = 0 ; k < pairs ; k++ )
        { REAL *grad2 = (k+1==pairs)?v_info->grad[1]:v_info->grad[k+2];
          for ( i = 0 ; i < SDIM ; i++ )
          { g = 0.75*hh0*hh0*dAdv1[k][i] 
                    + 1.5*area/fn*hh0*dffdv1[k][i]
                    - 1.5*area*ff*hh0/fn/fn*dfndv1[k][i]; 
             v_info->grad[k+1][i] += g;
             v_info->grad[0][i] -= g;
             g = 0.75*hh0*hh0*dAdv2[k][i] 
                    + 1.5*area/fn*hh0*dffdv2[k][i]
                    - 1.5*area*ff*hh0/fn/fn*dfndv2[k][i]; 
             grad2[i] += g;
             v_info->grad[0][i] -= g;
            }
         }
        break;
    }

    if ( get_vattr(v_info->id) & AXIAL_POINT )
    { REAL fudge = 1./rotorder;
      if ( sym_flags & DOUBLE_AXIAL ) fudge *= 2;
      fudge *= rotorder/v_info->axial_order;
      for ( i = 0 ; i < v_info->vcount ; i++ )
         for ( j = 0 ; j < SDIM ; j++ )
          v_info->grad[i][j] *= fudge;
    }
    if ( mode == METHOD_GRADIENT ) goto all_exit;
#if defined(MAC_APP) && !defined(powerc)
  kb_error(1999,"hessian function code too big for Mac.\n",RECOVERABLE);
#else
    /* Hessian */
    if ( (variety == EFF_SQ) || (variety == TEST_SQ) )
    { for ( i = 0 ; i < SDIM ; i++ )
      { antisymnorm[i][(i+1)%SDIM] = vnorm[(i+2)%SDIM]/2;
         antisymnorm[(i+1)%SDIM][i] = -vnorm[(i+2)%SDIM]/2;
         antisymnorm[i][i] = 0.0;
      }
    }
    if ( (variety == NORMAL_SQ) || (variety == TEST_SQ) )
    { for ( i = 0 ; i < SDIM ; i++ )
      { antisymf[i][(i+1)%SDIM] = dAdv[(i+2)%SDIM]/2;
         antisymf[(i+1)%SDIM][i] = -dAdv[(i+2)%SDIM]/2;
         antisymf[i][i] = 0.0;
      }
    }

#define OPTIMIZED_HESS

#ifdef OPTIMIZED_HESS
    
  if ( H0_flag && (variety != NORMAL_SQ) ) 
     kb_error(1615,"Can only use h_zero with star_normal_sq_curve.\n",RECOVERABLE);


    switch ( variety )
    { case PLAIN_SQ:
          aa75 = 0.75/area/area;
          a75 = 0.75/area;
          ffaaa = 1.5*ff/area/area/area;
          ffaa = 0.75*ff/area/area;
          break;
      case EFF_SQ:
          an75 = 0.75*area/nn;
          ann75 = 0.75*area/nn/nn;
          fnnna = 1.5*ff/nn/nn/nn*area;
          fnna = 0.75*ff/nn/nn*area;
          fn75 = 0.75/nn;
          fnn75 = 0.75*ff/nn/nn;
          ffn75 = 0.75*ff/nn;
          break;
      case NORMAL_SQ:
          afnfn = 1.5*area/fn/fn;
          afffnfn = 1.5*area*hh0/fn;
          fffnfn = 1.5*hh0/fn;
          ffafnfnfn = 1.5*area*(ff/fn + hh0)/fn/fn;
          afffffnfnfnfn = 1.5*area*ff*(ff/fn+2*hh0)/fn/fn/fn;
          fffffnfnfna = 1.5*ff*hh0/fn/fn*area;
          fffffnfnfn  = 1.5*ff*hh0/fn/fn;
          fffffnfn = 0.75*hh0*hh0;

          break;
    }

    for ( k = 0 ; k < pairs ; k++ ) 
    { REAL co1 = 0.25/a[k]/a[k]/a[k];
      REAL co2 = 0.125/a[k]/a[k];
      REAL co3 = 0.125/a[k];
      REAL dssf = (SDIM_dot(ds1[k],dAdv)+SDIM_dot(ds2[k],dAdv));
      REAL s1f,s2f;
      REAL ddssf1[MAXCOORD],ddssf2[MAXCOORD];


      for ( i = 0 ; i < SDIM ; i++ )
      { ddssf1[i] = 
            (SDIM_dot(ddss11[k][i],dAdv)+SDIM_dot(ddss12[k][i],dAdv));
         ddssf2[i] =
            (SDIM_dot(ddss21[k][i],dAdv)+SDIM_dot(ddss22[k][i],dAdv));
      }
      s1 = s[k]; s2 = s[(k+1==pairs)?0:k+1];
      s1f = SDIM_dot(s1,dAdv);
      s2f = SDIM_dot(s2,dAdv);

      for ( kk = 0 ; kk < pairs ; kk++ )
        { REAL **h11 = v_info->hess[k+1][kk+1];
          REAL **h12 = (kk+1==pairs)?v_info->hess[k+1][1]:v_info->hess[k+1][kk+2];
          REAL **h21 = (k+1==pairs)?v_info->hess[1][kk+1]:v_info->hess[k+2][kk+1];
          REAL **h22 = v_info->hess[((k+1)%pairs)+1][((kk+1)%pairs)+1];
          REAL **h00 = v_info->hess[0][0];
          REAL **h10 = v_info->hess[k+1][0];
          REAL **h02 = v_info->hess[0][((kk+1)%pairs)+1];
          REAL **h20 = v_info->hess[((k+1)%pairs)+1][0];
          REAL **h01 = v_info->hess[0][kk+1];
          REAL ddff,dddsf,ddnn,ddfn;

          for ( i = 0 ; i < SDIM ; i++ )
             for ( ii = 0 ; ii < SDIM ; ii++ )
             { /* 11 term */
                ddff = SDIM_dot(dfdv1[k][i],dfdv1[kk][ii]);
                if ( k == kk )
                {
                  ddff += (-co1*dAdv1[k][i]*dAdv1[k][ii] + co2*ddAdv1dv1[k][i][ii])
                              *dssf;
                  ddff += co2*dAdv1[k][i]*ddssf1[ii];
                  ddff += co2*dAdv1[kk][ii]*ddssf1[i];
                  dddsf = ((i==ii) ? 4*s2f : 0. )
                             - 2*s2[i]*dAdv[ii] - 2*s2[ii]*dAdv[i];
                  ddff -= co3*dddsf;
                }
                ddff *= 2;
              switch ( variety )
              { case TEST_SQ:
                      h = ddnn = 2*SDIM_dot(dvnorm1[k][i],dvnorm1[kk][ii]);
                     break;
                 case PLAIN_SQ:
                     h = -aa75*(dffdv1[k][i]*dAdv1[kk][ii]
                                 + dffdv1[kk][ii]*dAdv1[k][i])
                         + ffaaa*dAdv1[k][i]*dAdv1[kk][ii]
                         + a75*ddff;
                     if ( k == kk )
                        h -= ffaa*ddAdv1dv1[k][i][ii];
                     break;
                 case EFF_SQ:
                      ddnn = 2*SDIM_dot(dvnorm1[k][i],dvnorm1[kk][ii]);
                      h = an75*ddff
                         - ann75*dffdv1[k][i]*dnndv1[kk][ii]
                         - ann75*dffdv1[kk][ii]*dnndv1[k][i]
                         + fnnna*dnndv1[k][i]*dnndv1[kk][ii]
                         - fnna*ddnn
                         + fn75*dffdv1[k][i]*dAdv1[kk][ii]
                         + fn75*dffdv1[kk][ii]*dAdv1[k][i]
                         - fnn75*dAdv1[kk][ii]*dnndv1[k][i]
                         - fnn75*dnndv1[kk][ii]*dAdv1[k][i];
                      if ( k == kk )
                         h += ffn75*ddAdv1dv1[k][i][ii];
                      break;
                 case NORMAL_SQ:
                      ddfn = SDIM_dot(dfdv1[k][i],dvnorm1[kk][ii])
                              + SDIM_dot(dvnorm1[k][i],dfdv1[kk][ii]);
                      if ( k == kk )
                      {
                         ddfn += (-co1*dAdv1[k][i]*dAdv1[k][ii]
                                     + co2*ddAdv1dv1[k][i][ii])
                            *(SDIM_dot(ds1[k],vnorm)+SDIM_dot(ds2[k],vnorm));
                         ddfn += co2*dAdv1[k][i]*
                          (SDIM_dot(ddss11[kk][ii],vnorm)
                              +SDIM_dot(ddss12[kk][ii],vnorm));
                         ddfn += co2*dAdv1[kk][ii]*
                            (SDIM_dot(ddss11[k][i],vnorm)
                              +SDIM_dot(ddss12[k][i],vnorm));
                         dddsf = ((i==ii) ? 4*SDIM_dot(s2,vnorm) : 0. )
                                    - 2*s2[i]*vnorm[ii] - 2*s2[ii]*vnorm[i];
                         ddfn -= co3*dddsf;
                      }
                      h  = afnfn*dffdv1[k][i]*dffdv1[kk][ii]
                          + afffnfn*ddff
                          + fffnfn*dffdv1[k][i]*dAdv1[kk][ii]
                          + fffnfn*dAdv1[k][i]*dffdv1[kk][ii]
                          - ffafnfnfn*dffdv1[k][i]*dfndv1[kk][ii]
                          - ffafnfnfn*dfndv1[k][i]*dffdv1[kk][ii]
                          + afffffnfnfnfn*dfndv1[k][i]*dfndv1[kk][ii]
                          - fffffnfnfna*ddfn
                          - fffffnfnfn*dfndv1[k][i]*dAdv1[kk][ii]
                          - fffffnfnfn*dAdv1[k][i]*dfndv1[kk][ii];
                      if ( k == kk ) 
                         h += fffffnfn*ddAdv1dv1[k][i][ii];
                      break;
                }
                h11[i][ii] += h;
                h01[i][ii] -= h;
                h10[i][ii] -= h;
                h00[i][ii] += h;

                /* 12 term */
                ddff = SDIM_dot(dfdv1[k][i],dfdv2[kk][ii]);
                if ( k == kk )
                { ddff += (-co1*dAdv1[k][i]*dAdv2[k][ii] + co2*ddAdv1dv2[k][i][ii])
                              *dssf;
                  ddff += co2*dAdv1[k][i]*ddssf2[ii];
                  ddff += co2*dAdv2[kk][ii]*ddssf1[i];
                  dddsf = 4*dAdv[i]*s2[ii] -2*dAdv[ii]*s2[i] 
                              - 2*((i==ii) ? s2f : 0.0)
                          + 4*dAdv[ii]*s1[i] - 2*dAdv[i]*s1[ii]
                              - 2*((i==ii) ? s1f : 0.0);
                  ddff -= co3*dddsf;
                }
              ddff *= 2;
              switch ( variety )
              { case TEST_SQ:
                      ddnn = 2*SDIM_dot(dvnorm1[k][i],dvnorm2[kk][ii]);
                      if ( k == kk ) ddnn += 2*antisymnorm[i][ii];
                      h = ddnn;
                     break;
                 case PLAIN_SQ:
                     h = -aa75*dffdv1[k][i]*dAdv2[kk][ii]
                         - aa75*dffdv2[kk][ii]*dAdv1[k][i]
                         + ffaaa*dAdv1[k][i]*dAdv2[kk][ii]
                         + a75*ddff;
                     if ( k == kk )
                        h -= ffaa*ddAdv1dv2[k][i][ii];
                      break;
                 case EFF_SQ:
                      ddnn = 2*SDIM_dot(dvnorm1[k][i],dvnorm2[kk][ii]);
                      if ( k == kk ) ddnn += 2*antisymnorm[i][ii];
                      h = an75*ddff
                         - ann75*dffdv1[k][i]*dnndv2[kk][ii]
                         - ann75*dffdv2[kk][ii]*dnndv1[k][i]
                         + fnnna*dnndv1[k][i]*dnndv2[kk][ii]
                         - fnna*ddnn
                         + fn75*dffdv1[k][i]*dAdv2[kk][ii]
                         + fn75*dffdv2[kk][ii]*dAdv1[k][i]
                         - fnn75*dAdv2[kk][ii]*dnndv1[k][i]
                         - fnn75*dnndv2[kk][ii]*dAdv1[k][i];
                      if ( k == kk )
                         h += ffn75*ddAdv1dv2[k][i][ii];
                      break;
                 case NORMAL_SQ:
                      ddfn = SDIM_dot(dfdv1[k][i],dvnorm2[kk][ii])
                              + SDIM_dot(dvnorm1[k][i],dfdv2[kk][ii]);
                      if ( k == kk ) ddfn += antisymf[i][ii];
                      if ( k == kk )
                      {
                         ddfn += (-co1*dAdv1[k][i]*dAdv2[k][ii]
                                     + co2*ddAdv1dv2[k][i][ii])
                                     *(SDIM_dot(ds1[k],vnorm)+SDIM_dot(ds2[k],vnorm));
                         ddfn += co2*dAdv1[k][i]*
                          (SDIM_dot(ddss21[kk][ii],vnorm)
                              +SDIM_dot(ddss22[kk][ii],vnorm));
                         ddfn += co2*dAdv2[kk][ii]*
                            (SDIM_dot(ddss11[k][i],vnorm)
                              +SDIM_dot(ddss12[k][i],vnorm));
                         dddsf = 4*vnorm[i]*s2[ii] -2*vnorm[ii]*s2[i] 
                                     - 2*((i==ii) ? SDIM_dot(s2,vnorm) : 0.0)
                                 + 4*vnorm[ii]*s1[i] - 2*vnorm[i]*s1[ii]
                                     - 2*((i==ii) ? SDIM_dot(s1,vnorm) : 0.0);
                         ddfn -= co3*dddsf;
                      }
                      h  = afnfn*dffdv1[k][i]*dffdv2[kk][ii]
                          + afffnfn*ddff
                          - ffafnfnfn*dffdv1[k][i]*dfndv2[kk][ii]
                          + fffnfn*dffdv1[k][i]*dAdv2[kk][ii]
                          + fffnfn*dAdv1[k][i]*dffdv2[kk][ii]
                          - ffafnfnfn*dfndv1[k][i]*dffdv2[kk][ii]
                          + afffffnfnfnfn*dfndv1[k][i]*dfndv2[kk][ii]
                          - fffffnfnfna*ddfn
                          - fffffnfnfn*dfndv1[k][i]*dAdv2[kk][ii]
                          - fffffnfnfn*dAdv1[k][i]*dfndv2[kk][ii];
                      if ( k == kk ) 
                         h += fffffnfn*ddAdv1dv2[k][i][ii];
                      break;
                }
                h12[i][ii] += h;
                h02[i][ii] -= h;
                h10[i][ii] -= h;
                h00[i][ii] += h;

                /* 21 term */
                ddff = SDIM_dot(dfdv2[k][i],dfdv1[kk][ii]);
                if ( k == kk )
                { ddff += (-co1*dAdv2[k][i]*dAdv1[k][ii] + co2*ddAdv2dv1[k][i][ii])
                              *dssf;
                  ddff += co2*dAdv2[k][i]*ddssf1[ii];
                  ddff += co2*dAdv1[kk][ii]*ddssf2[i];
                  dddsf = 4*dAdv[i]*s1[ii] -2*dAdv[ii]*s1[i] 
                              - 2*((i==ii) ? s1f: 0.0)
                          + 4*dAdv[ii]*s2[i] - 2*dAdv[i]*s2[ii]
                              - 2*((i==ii) ? s2f : 0.0);
                  ddff -= co3*dddsf;
                }
              ddff *= 2;
              switch ( variety )
              { case TEST_SQ:
                      ddnn = 2*SDIM_dot(dvnorm2[k][i],dvnorm1[kk][ii]);
                      if ( k == kk ) ddnn += 2*antisymnorm[ii][i];
                      h = ddnn;
                     break;
                 case PLAIN_SQ:
                     h = -aa75*dffdv2[k][i]*dAdv1[kk][ii]
                         - aa75*dffdv1[kk][ii]*dAdv2[k][i]
                         + ffaaa*dAdv2[k][i]*dAdv1[kk][ii]
                         + a75*ddff;
                     if ( k == kk )
                        h -= ffaa*ddAdv2dv1[k][i][ii];
                      break;
                 case EFF_SQ:
                      ddnn = 2*SDIM_dot(dvnorm2[k][i],dvnorm1[kk][ii]);
                      if ( k == kk ) ddnn += 2*antisymnorm[ii][i];
                      h = an75*ddff
                         - ann75*dffdv2[k][i]*dnndv1[kk][ii]
                         - ann75*dffdv1[kk][ii]*dnndv2[k][i]
                         + fnnna*dnndv2[k][i]*dnndv1[kk][ii]
                         - fnna*ddnn
                         + fn75*dffdv2[k][i]*dAdv1[kk][ii]
                         + fn75*dffdv1[kk][ii]*dAdv2[k][i]
                         - fnn75*dAdv1[kk][ii]*dnndv2[k][i]
                         - fnn75*dnndv1[kk][ii]*dAdv2[k][i];
                      if ( k == kk )
                         h += ffn75*ddAdv2dv1[k][i][ii];
                      break;
                 case NORMAL_SQ:
                      ddfn = SDIM_dot(dfdv2[k][i],dvnorm1[kk][ii])
                              + SDIM_dot(dvnorm2[k][i],dfdv1[kk][ii]);
                      if ( k == kk ) ddfn += antisymf[ii][i];
                      if ( k == kk )
                      {
                         ddfn += (-co1*dAdv2[k][i]*dAdv1[k][ii]
                                     + co2*ddAdv2dv1[k][i][ii])
                                     *(SDIM_dot(ds1[k],vnorm)+SDIM_dot(ds2[k],vnorm));
                         ddfn += co2*dAdv2[k][i]*
                          (SDIM_dot(ddss11[kk][ii],vnorm)
                              +SDIM_dot(ddss12[kk][ii],vnorm));
                         ddfn += co2*dAdv1[kk][ii]*
                            (SDIM_dot(ddss21[k][i],vnorm)
                              +SDIM_dot(ddss22[k][i],vnorm));
                         dddsf = 4*vnorm[i]*s1[ii] -2*vnorm[ii]*s1[i] 
                                     - 2*((i==ii) ? SDIM_dot(s1,vnorm) : 0.0)
                                 + 4*vnorm[ii]*s2[i] - 2*vnorm[i]*s2[ii]
                                     - 2*((i==ii) ? SDIM_dot(s2,vnorm) : 0.0);
                         ddfn -= co3*dddsf;
                      }
                      h  = afnfn*dffdv2[k][i]*dffdv1[kk][ii]
                          + afffnfn*ddff
                          - ffafnfnfn*dffdv2[k][i]*dfndv1[kk][ii]
                          + fffnfn*dffdv2[k][i]*dAdv1[kk][ii]
                          - ffafnfnfn*dfndv2[k][i]*dffdv1[kk][ii]
                          + afffffnfnfnfn*dfndv2[k][i]*dfndv1[kk][ii]
                          - fffffnfnfna*ddfn
                          - fffffnfnfn*dfndv2[k][i]*dAdv1[kk][ii]
                          + fffnfn*dAdv2[k][i]*dffdv1[kk][ii]
                          - fffffnfnfn*dAdv2[k][i]*dfndv1[kk][ii];
                      if ( k == kk ) 
                         h += fffffnfn*ddAdv2dv1[k][i][ii];
                      break;
                }
                h21[i][ii] += h;
                h01[i][ii] -= h;
                h20[i][ii] -= h;
                h00[i][ii] += h;

                /* 22 term */
                ddff = SDIM_dot(dfdv2[k][i],dfdv2[kk][ii]);
                if ( k == kk )
                { ddff += (-co1*dAdv2[k][i]*dAdv2[k][ii] + co2*ddAdv2dv2[k][i][ii])
                              *dssf;
                  ddff += co2*dAdv2[k][i]*ddssf2[ii];
                  ddff += co2*dAdv2[kk][ii]*ddssf2[i];
                  dddsf = ((i==ii) ? 4*s1f : 0. )
                             - 2*s1[i]*dAdv[ii] - 2*s1[ii]*dAdv[i];
                  ddff -= co3*dddsf;
                }
              ddff *= 2;
              switch ( variety )
              { case TEST_SQ:
                      ddnn = 2*SDIM_dot(dvnorm2[k][i],dvnorm2[kk][ii]);
                      h = ddnn;
                     break;
                 case PLAIN_SQ:
                     h = -aa75*dffdv2[k][i]*dAdv2[kk][ii]
                         - aa75*dffdv2[kk][ii]*dAdv2[k][i]
                         + ffaaa*dAdv2[k][i]*dAdv2[kk][ii]
                         + a75*ddff;
                     if ( k == kk )
                        h -= ffaa*ddAdv2dv2[k][i][ii];
                      break;
                 case EFF_SQ:
                      ddnn = 2*SDIM_dot(dvnorm2[k][i],dvnorm2[kk][ii]);
                      h = an75*ddff
                         - ann75*dffdv2[k][i]*dnndv2[kk][ii]
                         - ann75*dffdv2[kk][ii]*dnndv2[k][i]
                         + fnnna*dnndv2[k][i]*dnndv2[kk][ii]
                         - fnna*ddnn
                         + fn75*dffdv2[k][i]*dAdv2[kk][ii]
                         + fn75*dffdv2[kk][ii]*dAdv2[k][i]
                         - fnn75*dAdv2[kk][ii]*dnndv2[k][i]
                         - fnn75*dnndv2[kk][ii]*dAdv2[k][i];
                      if ( k == kk )
                         h += ffn75*ddAdv2dv2[k][i][ii];
                      break;
                 case NORMAL_SQ:
                      ddfn = SDIM_dot(dfdv2[k][i],dvnorm2[kk][ii])
                              + SDIM_dot(dvnorm2[k][i],dfdv2[kk][ii]);
                      if ( k == kk )
                      {
                         ddfn += (-co1*dAdv2[k][i]*dAdv2[k][ii]
                                     + co2*ddAdv2dv2[k][i][ii])
                                     *(SDIM_dot(ds1[k],vnorm)+SDIM_dot(ds2[k],vnorm));
                         ddfn += co2*dAdv2[k][i]*
                          (SDIM_dot(ddss21[kk][ii],vnorm)
                              +SDIM_dot(ddss22[kk][ii],vnorm));
                         ddfn += co2*dAdv2[kk][ii]*
                            (SDIM_dot(ddss21[k][i],vnorm)
                              +SDIM_dot(ddss22[k][i],vnorm));
                         dddsf = ((i==ii) ? 4*SDIM_dot(s1,vnorm) : 0. )
                                    - 2*s1[i]*vnorm[ii] - 2*s1[ii]*vnorm[i];
                         ddfn -= co3*dddsf;
                      }
                      h  = afnfn*dffdv2[k][i]*dffdv2[kk][ii]
                          + afffnfn*ddff
                          - ffafnfnfn*dffdv2[k][i]*dfndv2[kk][ii]
                          + fffnfn*dffdv2[k][i]*dAdv2[kk][ii]
                          - ffafnfnfn*dfndv2[k][i]*dffdv2[kk][ii]
                          + afffffnfnfnfn*dfndv2[k][i]*dfndv2[kk][ii]
                          - fffffnfnfna*ddfn
                          - fffffnfnfn*dfndv2[k][i]*dAdv2[kk][ii]
                          + fffnfn*dAdv2[k][i]*dffdv2[kk][ii]
                          - fffffnfnfn*dAdv2[k][i]*dfndv2[kk][ii];
                      if ( k == kk ) 
                         h += fffffnfn*ddAdv2dv2[k][i][ii];
                      break;
                }
                h22[i][ii] += h;
                h02[i][ii] -= h;
                h20[i][ii] -= h;
                h00[i][ii] += h;

             }
        }
     } /* end for k */
#endif
#ifdef ORIGINAL_HESS
  if ( H0_flag ) 
     kb_error(1616,"Can't use h_zero with star_sq_curve_all().\n",RECOVERABLE);

    for ( k = 0 ; k < pairs ; k++ )
      for ( kk = 0 ; kk < pairs ; kk++ )
        { REAL **h11 = v_info->hess[k+1][kk+1];
          REAL **h12 = (kk+1==pairs)?v_info->hess[k+1][1]:v_info->hess[k+1][kk+2];
          REAL **h21 = (k+1==pairs)?v_info->hess[1][kk+1]:v_info->hess[k+2][kk+1];
          REAL **h22 = v_info->hess[((k+1)%pairs)+1][((kk+1)%pairs)+1];
          REAL **h00 = v_info->hess[0][0];
          REAL **h10 = v_info->hess[k+1][0];
          REAL **h02 = v_info->hess[0][((kk+1)%pairs)+1];
          REAL **h20 = v_info->hess[((k+1)%pairs)+1][0];
          REAL **h01 = v_info->hess[0][kk+1];
          REAL ddff,dddsf,ddnn,ddfn;
          s1 = s[k]; s2 = s[(k+1==pairs)?0:k+1];

          for ( i = 0 ; i < SDIM ; i++ )
             for ( ii = 0 ; ii < SDIM ; ii++ )
             { /* 11 term */
                ddff = SDIM_dot(dfdv1[k][i],dfdv1[kk][ii]);
                if ( k == kk )
                {
                  ddff += (-0.25/a[k]/a[k]/a[k]*dAdv1[k][i]*dAdv1[k][ii]
                              + 0.125/a[k]/a[k]*ddAdv1dv1[k][i][ii])
                              *(SDIM_dot(ds1[k],dAdv)+SDIM_dot(ds2[k],dAdv));
                  ddff += 0.125/a[k]/a[k]*dAdv1[k][i]*
                    (SDIM_dot(ddss11[kk][ii],dAdv)+SDIM_dot(ddss12[kk][ii],dAdv));
                  ddff += 0.125/a[k]/a[k]*dAdv1[kk][ii]*
                     (SDIM_dot(ddss11[k][i],dAdv)+SDIM_dot(ddss12[k][i],dAdv));
                  dddsf = ((i==ii) ? 4*SDIM_dot(s2,dAdv) : 0 )
                             - 2*s2[i]*dAdv[ii] - 2*s2[ii]*dAdv[i];
                  ddff -= 0.125/a[k]*dddsf;
                }
                ddff *= 2;
              switch ( variety )
              { case TEST_SQ:
                      h = ddnn = 2*SDIM_dot(dvnorm1[k][i],dvnorm1[kk][ii]);
                     break;
                 case PLAIN_SQ:
                     h = -0.75/area/area*dffdv1[k][i]*dAdv1[kk][ii]
                         - 0.75/area/area*dffdv1[kk][ii]*dAdv1[k][i]
                         + 1.5*ff/area/area/area*dAdv1[k][i]*dAdv1[kk][ii]
                         + 0.75/area*ddff;
                     if ( k == kk )
                        h -= 0.75*ff/area/area*ddAdv1dv1[k][i][ii];
                     break;
                 case EFF_SQ:
                      ddnn = 2*SDIM_dot(dvnorm1[k][i],dvnorm1[kk][ii]);
                      h = 0.75*ddff*area/nn 
                         - 0.75*dffdv1[k][i]/nn/nn*area*dnndv1[kk][ii]
                         - 0.75*dffdv1[kk][ii]/nn/nn*area*dnndv1[k][i]
                         + 1.5*ff/nn/nn/nn*area*dnndv1[k][i]*dnndv1[kk][ii]
                         - 0.75*ff/nn/nn*area*ddnn
                         + 0.75*dffdv1[k][i]/nn*dAdv1[kk][ii]
                         + 0.75*dffdv1[kk][ii]/nn*dAdv1[k][i]
                         - 0.75*ff/nn/nn*dAdv1[kk][ii]*dnndv1[k][i]
                         - 0.75*ff/nn/nn*dnndv1[kk][ii]*dAdv1[k][i];
                      if ( k == kk )
                         h += 0.75*ff/nn*ddAdv1dv1[k][i][ii];
                      break;
                 case NORMAL_SQ:
                      ddfn = SDIM_dot(dfdv1[k][i],dvnorm1[kk][ii])
                              + SDIM_dot(dvnorm1[k][i],dfdv1[kk][ii]);
                      if ( k == kk )
                      {
                         ddfn += (-0.25/a[k]/a[k]/a[k]*dAdv1[k][i]*dAdv1[k][ii]
                                     + 0.125/a[k]/a[k]*ddAdv1dv1[k][i][ii])
                                     *(SDIM_dot(ds1[k],vnorm)+SDIM_dot(ds2[k],vnorm));
                         ddfn += 0.125/a[k]/a[k]*dAdv1[k][i]*
                          (SDIM_dot(ddss11[kk][ii],vnorm)
                              +SDIM_dot(ddss12[kk][ii],vnorm));
                         ddfn += 0.125/a[k]/a[k]*dAdv1[kk][ii]*
                            (SDIM_dot(ddss11[k][i],vnorm)
                              +SDIM_dot(ddss12[k][i],vnorm));
                         dddsf = ((i==ii) ? 4*SDIM_dot(s2,vnorm) : 0 )
                                    - 2*s2[i]*vnorm[ii] - 2*s2[ii]*vnorm[i];
                         ddfn -= 0.125/a[k]*dddsf;
                      }
                      h  = 1.5*area/fn/fn*dffdv1[k][i]*dffdv1[kk][ii]
                          + 1.5*area*ff/fn/fn*ddff
                          + 1.5*ff/fn/fn*dffdv1[k][i]*dAdv1[kk][ii]
                          + 1.5*ff/fn/fn*dAdv1[k][i]*dffdv1[kk][ii]
                          - 3*ff*area/fn/fn/fn*dffdv1[k][i]*dfndv1[kk][ii]
                          - 3*ff*area/fn/fn/fn*dfndv1[k][i]*dffdv1[kk][ii]
                          + 4.5*area*ff*ff/fn/fn/fn/fn*dfndv1[k][i]*dfndv1[kk][ii]
                          - 1.5*ff*ff/fn/fn/fn*area*ddfn
                          - 1.5*ff*ff/fn/fn/fn*dfndv1[k][i]*dAdv1[kk][ii]
                          - 1.5*ff*ff/fn/fn/fn*dAdv1[k][i]*dfndv1[kk][ii];
                      if ( k == kk ) 
                         h += 0.75*ff*ff/fn/fn*ddAdv1dv1[k][i][ii];
                      break;
                }
                h11[i][ii] += h;
                h01[i][ii] -= h;
                h10[i][ii] -= h;
                h00[i][ii] += h;

                /* 12 term */
                ddff = SDIM_dot(dfdv1[k][i],dfdv2[kk][ii]);
                if ( k == kk )
                { ddff += (-0.25/a[k]/a[k]/a[k]*dAdv1[k][i]*dAdv2[k][ii]
                              + 0.125/a[k]/a[k]*ddAdv1dv2[k][i][ii])
                              *(SDIM_dot(ds1[k],dAdv)+SDIM_dot(ds2[k],dAdv));
                  ddff += 0.125/a[k]/a[k]*dAdv1[k][i]*
                    (SDIM_dot(ddss21[kk][ii],dAdv)+SDIM_dot(ddss22[kk][ii],dAdv));
                  ddff += 0.125/a[k]/a[k]*dAdv2[kk][ii]*
                     (SDIM_dot(ddss11[k][i],dAdv)+SDIM_dot(ddss12[k][i],dAdv));
                  dddsf = 4*dAdv[i]*s2[ii] -2*dAdv[ii]*s2[i] 
                              - 2*((i==ii) ? SDIM_dot(s2,dAdv) : 0.0)
                          + 4*dAdv[ii]*s1[i] - 2*dAdv[i]*s1[ii]
                              - 2*((i==ii) ? SDIM_dot(s1,dAdv) : 0.0);
                  ddff -= 0.125/a[k]*dddsf;
                }
              ddff *= 2;
              switch ( variety )
              { case TEST_SQ:
                      ddnn = 2*SDIM_dot(dvnorm1[k][i],dvnorm2[kk][ii]);
                      if ( k == kk ) ddnn += 2*antisymnorm[i][ii];
                      h = ddnn;
                     break;
                 case PLAIN_SQ:
                     h = -0.75/area/area*dffdv1[k][i]*dAdv2[kk][ii]
                         - 0.75/area/area*dffdv2[kk][ii]*dAdv1[k][i]
                         + 1.5*ff/area/area/area*dAdv1[k][i]*dAdv2[kk][ii]
                         + 0.75/area*ddff;
                     if ( k == kk )
                        h -= 0.75*ff/area/area*ddAdv1dv2[k][i][ii];
                      break;
                 case EFF_SQ:
                      ddnn = 2*SDIM_dot(dvnorm1[k][i],dvnorm2[kk][ii]);
                      if ( k == kk ) ddnn += 2*antisymnorm[i][ii];
                      h = 0.75*ddff*area/nn 
                         - 0.75*dffdv1[k][i]/nn/nn*area*dnndv2[kk][ii]
                         - 0.75*dffdv2[kk][ii]/nn/nn*area*dnndv1[k][i]
                         + 1.5*ff/nn/nn/nn*area*dnndv1[k][i]*dnndv2[kk][ii]
                         - 0.75*ff/nn/nn*area*ddnn
                         + 0.75*dffdv1[k][i]/nn*dAdv2[kk][ii]
                         + 0.75*dffdv2[kk][ii]/nn*dAdv1[k][i]
                         - 0.75*ff/nn/nn*dAdv2[kk][ii]*dnndv1[k][i]
                         - 0.75*ff/nn/nn*dnndv2[kk][ii]*dAdv1[k][i];
                      if ( k == kk )
                         h += 0.75*ff/nn*ddAdv1dv2[k][i][ii];
                      break;
                 case NORMAL_SQ:
                      ddfn = SDIM_dot(dfdv1[k][i],dvnorm2[kk][ii])
                              + SDIM_dot(dvnorm1[k][i],dfdv2[kk][ii]);
                      if ( k == kk ) ddfn += antisymf[i][ii];
                      if ( k == kk )
                      {
                         ddfn += (-0.25/a[k]/a[k]/a[k]*dAdv1[k][i]*dAdv2[k][ii]
                                     + 0.125/a[k]/a[k]*ddAdv1dv2[k][i][ii])
                                     *(SDIM_dot(ds1[k],vnorm)+SDIM_dot(ds2[k],vnorm));
                         ddfn += 0.125/a[k]/a[k]*dAdv1[k][i]*
                          (SDIM_dot(ddss21[kk][ii],vnorm)
                              +SDIM_dot(ddss22[kk][ii],vnorm));
                         ddfn += 0.125/a[k]/a[k]*dAdv2[kk][ii]*
                            (SDIM_dot(ddss11[k][i],vnorm)
                              +SDIM_dot(ddss12[k][i],vnorm));
                         dddsf = 4*vnorm[i]*s2[ii] -2*vnorm[ii]*s2[i] 
                                     - 2*((i==ii) ? SDIM_dot(s2,vnorm) : 0.0)
                                 + 4*vnorm[ii]*s1[i] - 2*vnorm[i]*s1[ii]
                                     - 2*((i==ii) ? SDIM_dot(s1,vnorm) : 0.0);
                         ddfn -= 0.125/a[k]*dddsf;
                      }
                      h  = 1.5*area/fn/fn*dffdv1[k][i]*dffdv2[kk][ii]
                          + 1.5*area*ff/fn/fn*ddff
                          - 3*ff*area/fn/fn/fn*dffdv1[k][i]*dfndv2[kk][ii]
                          + 1.5*ff/fn/fn*dffdv1[k][i]*dAdv2[kk][ii]
                          - 3*ff*area/fn/fn/fn*dfndv1[k][i]*dffdv2[kk][ii]
                          + 4.5*area*ff*ff/fn/fn/fn/fn*dfndv1[k][i]*dfndv2[kk][ii]
                          - 1.5*ff*ff/fn/fn/fn*area*ddfn
                          - 1.5*ff*ff/fn/fn/fn*dfndv1[k][i]*dAdv2[kk][ii]
                          + 1.5*ff/fn/fn*dAdv1[k][i]*dffdv2[kk][ii]
                          - 1.5*ff*ff/fn/fn/fn*dAdv1[k][i]*dfndv2[kk][ii];
                      if ( k == kk ) 
                         h += 0.75*ff*ff/fn/fn*ddAdv1dv2[k][i][ii];
                      break;
                }
                h12[i][ii] += h;
                h02[i][ii] -= h;
                h10[i][ii] -= h;
                h00[i][ii] += h;

                /* 21 term */
                ddff = SDIM_dot(dfdv2[k][i],dfdv1[kk][ii]);
                if ( k == kk )
                { ddff += (-0.25/a[k]/a[k]/a[k]*dAdv2[k][i]*dAdv1[k][ii]
                              + 0.125/a[k]/a[k]*ddAdv2dv1[k][i][ii])
                              *(SDIM_dot(ds1[k],dAdv)+SDIM_dot(ds2[k],dAdv));
                  ddff += 0.125/a[k]/a[k]*dAdv2[k][i]*
                    (SDIM_dot(ddss11[kk][ii],dAdv)+SDIM_dot(ddss12[kk][ii],dAdv));
                  ddff += 0.125/a[k]/a[k]*dAdv1[kk][ii]*
                     (SDIM_dot(ddss21[k][i],dAdv)+SDIM_dot(ddss22[k][i],dAdv));
                  dddsf = 4*dAdv[i]*s1[ii] -2*dAdv[ii]*s1[i] 
                              - 2*((i==ii) ? SDIM_dot(s1,dAdv) : 0.0)
                          + 4*dAdv[ii]*s2[i] - 2*dAdv[i]*s2[ii]
                              - 2*((i==ii) ? SDIM_dot(s2,dAdv) : 0.0);
                  ddff -= 0.125/a[k]*dddsf;
                }
              ddff *= 2;
              switch ( variety )
              { case TEST_SQ:
                      ddnn = 2*SDIM_dot(dvnorm2[k][i],dvnorm1[kk][ii]);
                      if ( k == kk ) ddnn += 2*antisymnorm[ii][i];
                      h = ddnn;
                     break;
                 case PLAIN_SQ:
                     h = -0.75/area/area*dffdv2[k][i]*dAdv1[kk][ii]
                         - 0.75/area/area*dffdv1[kk][ii]*dAdv2[k][i]
                         + 1.5*ff/area/area/area*dAdv2[k][i]*dAdv1[kk][ii]
                         + 0.75/area*ddff;
                     if ( k == kk )
                        h -= 0.75*ff/area/area*ddAdv2dv1[k][i][ii];
                      break;
                 case EFF_SQ:
                      ddnn = 2*SDIM_dot(dvnorm2[k][i],dvnorm1[kk][ii]);
                      if ( k == kk ) ddnn += 2*antisymnorm[ii][i];
                      h = 0.75*ddff*area/nn 
                         - 0.75*dffdv2[k][i]/nn/nn*area*dnndv1[kk][ii]
                         - 0.75*dffdv1[kk][ii]/nn/nn*area*dnndv2[k][i]
                         + 1.5*ff/nn/nn/nn*area*dnndv2[k][i]*dnndv1[kk][ii]
                         - 0.75*ff/nn/nn*area*ddnn
                         + 0.75*dffdv2[k][i]/nn*dAdv1[kk][ii]
                         + 0.75*dffdv1[kk][ii]/nn*dAdv2[k][i]
                         - 0.75*ff/nn/nn*dAdv1[kk][ii]*dnndv2[k][i]
                         - 0.75*ff/nn/nn*dnndv1[kk][ii]*dAdv2[k][i];
                      if ( k == kk )
                         h += 0.75*ff/nn*ddAdv2dv1[k][i][ii];
                      break;
                 case NORMAL_SQ:
                      ddfn = SDIM_dot(dfdv2[k][i],dvnorm1[kk][ii])
                              + SDIM_dot(dvnorm2[k][i],dfdv1[kk][ii]);
                      if ( k == kk ) ddfn += antisymf[ii][i];
                      if ( k == kk )
                      {
                         ddfn += (-0.25/a[k]/a[k]/a[k]*dAdv2[k][i]*dAdv1[k][ii]
                                     + 0.125/a[k]/a[k]*ddAdv2dv1[k][i][ii])
                                     *(SDIM_dot(ds1[k],vnorm)+SDIM_dot(ds2[k],vnorm));
                         ddfn += 0.125/a[k]/a[k]*dAdv2[k][i]*
                          (SDIM_dot(ddss11[kk][ii],vnorm)
                              +SDIM_dot(ddss12[kk][ii],vnorm));
                         ddfn += 0.125/a[k]/a[k]*dAdv1[kk][ii]*
                            (SDIM_dot(ddss21[k][i],vnorm)
                              +SDIM_dot(ddss22[k][i],vnorm));
                         dddsf = 4*vnorm[i]*s1[ii] -2*vnorm[ii]*s1[i] 
                                     - 2*((i==ii) ? SDIM_dot(s1,vnorm) : 0.0)
                                 + 4*vnorm[ii]*s2[i] - 2*vnorm[i]*s2[ii]
                                     - 2*((i==ii) ? SDIM_dot(s2,vnorm) : 0.0);
                         ddfn -= 0.125/a[k]*dddsf;
                      }
                      h  = 1.5*area/fn/fn*dffdv2[k][i]*dffdv1[kk][ii]
                          + 1.5*area*ff/fn/fn*ddff
                          - 3*ff*area/fn/fn/fn*dffdv2[k][i]*dfndv1[kk][ii]
                          + 1.5*ff/fn/fn*dffdv2[k][i]*dAdv1[kk][ii]
                          - 3*ff*area/fn/fn/fn*dfndv2[k][i]*dffdv1[kk][ii]
                          + 4.5*area*ff*ff/fn/fn/fn/fn*dfndv2[k][i]*dfndv1[kk][ii]
                          - 1.5*ff*ff/fn/fn/fn*area*ddfn
                          - 1.5*ff*ff/fn/fn/fn*dfndv2[k][i]*dAdv1[kk][ii]
                          + 1.5*ff/fn/fn*dAdv2[k][i]*dffdv1[kk][ii]
                          - 1.5*ff*ff/fn/fn/fn*dAdv2[k][i]*dfndv1[kk][ii];
                      if ( k == kk ) 
                         h += 0.75*ff*ff/fn/fn*ddAdv2dv1[k][i][ii];
                      break;
                }
                h21[i][ii] += h;
                h01[i][ii] -= h;
                h20[i][ii] -= h;
                h00[i][ii] += h;

                /* 22 term */
                ddff = SDIM_dot(dfdv2[k][i],dfdv2[kk][ii]);
                if ( k == kk )
                { ddff += (-0.25/a[k]/a[k]/a[k]*dAdv2[k][i]*dAdv2[k][ii]
                              + 0.125/a[k]/a[k]*ddAdv2dv2[k][i][ii])
                              *(SDIM_dot(ds1[k],dAdv)+SDIM_dot(ds2[k],dAdv));
                  ddff += 0.125/a[k]/a[k]*dAdv2[k][i]*
                    (SDIM_dot(ddss21[kk][ii],dAdv)+SDIM_dot(ddss22[kk][ii],dAdv));
                  ddff += 0.125/a[k]/a[k]*dAdv2[kk][ii]*
                     (SDIM_dot(ddss21[k][i],dAdv)+SDIM_dot(ddss22[k][i],dAdv));
                  dddsf = ((i==ii) ? 4*SDIM_dot(s1,dAdv) : 0 )
                             - 2*s1[i]*dAdv[ii] - 2*s1[ii]*dAdv[i];
                  ddff -= 0.125/a[k]*dddsf;
                }
              ddff *= 2;
              switch ( variety )
              { case TEST_SQ:
                      ddnn = 2*SDIM_dot(dvnorm2[k][i],dvnorm2[kk][ii]);
                      h = ddnn;
                     break;
                 case PLAIN_SQ:
                     h = -0.75/area/area*dffdv2[k][i]*dAdv2[kk][ii]
                         - 0.75/area/area*dffdv2[kk][ii]*dAdv2[k][i]
                         + 1.5*ff/area/area/area*dAdv2[k][i]*dAdv2[kk][ii]
                         + 0.75/area*ddff;
                     if ( k == kk )
                        h -= 0.75*ff/area/area*ddAdv2dv2[k][i][ii];
                      break;
                 case EFF_SQ:
                      ddnn = 2*SDIM_dot(dvnorm2[k][i],dvnorm2[kk][ii]);
                      h = 0.75*ddff*area/nn 
                         - 0.75*dffdv2[k][i]/nn/nn*area*dnndv2[kk][ii]
                         - 0.75*dffdv2[kk][ii]/nn/nn*area*dnndv2[k][i]
                         + 1.5*ff/nn/nn/nn*area*dnndv2[k][i]*dnndv2[kk][ii]
                         - 0.75*ff/nn/nn*area*ddnn
                         + 0.75*dffdv2[k][i]/nn*dAdv2[kk][ii]
                         + 0.75*dffdv2[kk][ii]/nn*dAdv2[k][i]
                         - 0.75*ff/nn/nn*dAdv2[kk][ii]*dnndv2[k][i]
                         - 0.75*ff/nn/nn*dnndv2[kk][ii]*dAdv2[k][i];
                      if ( k == kk )
                         h += 0.75*ff/nn*ddAdv2dv2[k][i][ii];
                      break;
                 case NORMAL_SQ:
                      ddfn = SDIM_dot(dfdv2[k][i],dvnorm2[kk][ii])
                              + SDIM_dot(dvnorm2[k][i],dfdv2[kk][ii]);
                      if ( k == kk )
                      {
                         ddfn += (-0.25/a[k]/a[k]/a[k]*dAdv2[k][i]*dAdv2[k][ii]
                                     + 0.125/a[k]/a[k]*ddAdv2dv2[k][i][ii])
                                     *(SDIM_dot(ds1[k],vnorm)+SDIM_dot(ds2[k],vnorm));
                         ddfn += 0.125/a[k]/a[k]*dAdv2[k][i]*
                          (SDIM_dot(ddss21[kk][ii],vnorm)
                              +SDIM_dot(ddss22[kk][ii],vnorm));
                         ddfn += 0.125/a[k]/a[k]*dAdv2[kk][ii]*
                            (SDIM_dot(ddss21[k][i],vnorm)
                              +SDIM_dot(ddss22[k][i],vnorm));
                         dddsf = ((i==ii) ? 4*SDIM_dot(s1,vnorm) : 0 )
                                    - 2*s1[i]*vnorm[ii] - 2*s1[ii]*vnorm[i];
                         ddfn -= 0.125/a[k]*dddsf;
                      }
                      h  = 1.5*area/fn/fn*dffdv2[k][i]*dffdv2[kk][ii]
                          + 1.5*area*ff/fn/fn*ddff
                          - 3*ff*area/fn/fn/fn*dffdv2[k][i]*dfndv2[kk][ii]
                          + 1.5*ff/fn/fn*dffdv2[k][i]*dAdv2[kk][ii]
                          - 3*ff*area/fn/fn/fn*dfndv2[k][i]*dffdv2[kk][ii]
                          + 4.5*area*ff*ff/fn/fn/fn/fn*dfndv2[k][i]*dfndv2[kk][ii]
                          - 1.5*ff*ff/fn/fn/fn*area*ddfn
                          - 1.5*ff*ff/fn/fn/fn*dfndv2[k][i]*dAdv2[kk][ii]
                          + 1.5*ff/fn/fn*dAdv2[k][i]*dffdv2[kk][ii]
                          - 1.5*ff*ff/fn/fn/fn*dAdv2[k][i]*dfndv2[kk][ii];
                      if ( k == kk ) 
                         h += 0.75*ff*ff/fn/fn*ddAdv2dv2[k][i][ii];
                      break;
                }
                h22[i][ii] += h;
                h02[i][ii] -= h;
                h20[i][ii] -= h;
                h00[i][ii] += h;

             }
        }
#endif

    if ( get_vattr(v_info->id) & AXIAL_POINT )
    { REAL fudge = 1./rotorder;
      if ( sym_flags & DOUBLE_AXIAL ) fudge *= 2;
      fudge *= rotorder/v_info->axial_order;
      for ( i = 0 ; i < v_info->vcount ; i++ )
        for ( ii = 0 ; ii < v_info->vcount ; ii++ )
         for ( k = 0 ; k < SDIM ; k++ )
          for ( kk = 0 ; kk < SDIM ; kk++ )
             v_info->hess[i][ii][k][kk] *= fudge;
    }
#endif

all_exit:

  if ( v_info->vcount > MAXV )
  { myfree((char*)a);
     free_matrix(ds1);
     if ( ddss11 ) free_matrix3(ddss11);
  }
  return energy;
#else
  kb_error(7643,"star_sq_mean_curvature not implemented on Mac 68K version.\n",WARNING);
  return 0.0;
#endif
}

/*************************************************************************
*
*  function: star_sqcurve_method_value()
*
*  purpose: calculate squared mean curvature of given vertex
*              from precalculated data.
*
*/

REAL star_sqcurve_method_value(v_info)
struct qinfo *v_info;
{

  /* following is streamlined for just energy */

  int variety; /* PLAIN_SQ, EFF_SQ, NORMAL_SQ */
  int pairs;
  int j,k;
  REAL energy,ff,fn,nn,area;
  REAL dAdv[MAXCOORD], a, d, s1s1, s1s2, s2s2;
  REAL vnorm[MAXCOORD];
  REAL ds1,ds2,coeff;
  REAL *s1,*s2;
  REAL **s = v_info->sides[0];
  REAL temp[MAXCOORD];

#ifdef MAC_APP
  return star_sqcurve_method_all(v_info,METHOD_VALUE);  
#else
  if ( METH_INST[v_info->method].gen_method == star_normal_sq_mean_curvature_mi )
     variety = NORMAL_SQ;
  else 
  { if ( H0_flag ) 
        kb_error(1617,"Can only use star_normal_sq_mean_curvature with h_zero.\n",
                RECOVERABLE);
     if ( METH_INST[v_info->method].gen_method == star_eff_area_sq_mean_curvature_mi )
        variety = EFF_SQ;
     else variety = PLAIN_SQ;
  }
  pairs = (v_info->vcount - 1);
  if ( pairs <= 0 ) return 0.0;

  /* basic dot products */
  area = 0.0;
  for ( j = 0 ; j < SDIM ; j++ ) dAdv[j] = vnorm[j] = 0.0;
  for ( k = 0 ; k < pairs ; k++ )
  { s1 = s[k]; s2 = s[(k+1==pairs)?0:k+1];
     s1s1 = SDIM_dot(s1,s1);
     s1s2 = SDIM_dot(s1,s2);
     s2s2 = SDIM_dot(s2,s2);
     d = s1s1*s2s2 - s1s2*s1s2;
     a = sqrt(d);
     area += a;
     coeff = 0.5/a;
     for ( j = 0 ; j < SDIM ; j++ )
     { ds1 = (s2s2*s1[j] - s1s2*s2[j]);
        ds2 = (s1s1*s2[j] - s1s2*s1[j]);
        dAdv[j] -= coeff*(ds1 + ds2);
     }
     if ( variety != PLAIN_SQ )
     { cross_prod(s1,s2,temp);
        for ( j = 0 ; j < SDIM ; j++ )
          vnorm[j] += 0.5*temp[j];
     }
  }
  area *= 0.5;

  /* energy */
  ff = SDIM_dot(dAdv,dAdv);
  switch ( variety )
  { case PLAIN_SQ: 
        energy = 0.75/area*ff;
        break;
     case EFF_SQ:
        nn = SDIM_dot(vnorm,vnorm);
        energy = 0.75*area*ff/nn;
        break;
     case NORMAL_SQ:
        fn = SDIM_dot(dAdv,vnorm);
        if ( fn == 0.0 ) energy = 0.0;
        else energy = 0.75*area*(ff/fn-2*h0_value/3)*(ff/fn-2*h0_value/3);
        break;
    }

  if ( get_vattr(v_info->id) & AXIAL_POINT )
  { energy /= rotorder;
     if ( sym_flags & DOUBLE_AXIAL ) energy *= 2;
     energy *= rotorder/v_info->axial_order;
  }

  return energy;
#endif
}

/*************************************************************************
*
*  function: star_sqcurve_method_grad()
*
*  purpose:  Convert square curvature data into forces for a vertex.
*
*/

REAL star_sqcurve_method_grad(v_info)
struct qinfo *v_info;
{

 /* version optimized for gradient */
 
  int variety; /* PLAIN_SQ, EFF_SQ, NORMAL_SQ */
  int pairs;
  int i,j,k;
  REAL energy,ff,fn,nn,area,g;
  REAL dAdv[MAXCOORD], *a=NULL, *d, *s1s1, *s1s2, *s2s2;
  REAL vnorm[MAXCOORD];
  REAL **dAdv1,**dAdv2,dvnorm1[MAXCOORD][MAXCOORD]
      ,dvnorm2[MAXCOORD][MAXCOORD],**ds2;
  REAL *s1,*s2;
  REAL **s = v_info->sides[0];
  REAL temp[MAXCOORD],ea,a15,ef1,ef2,ns1,ns2;
  REAL aa[MAXV*5];
  MAT2D(ds1,12*MAXV,MAXCOORD);
  REAL h0adj = 2./3*h0_value;

#ifdef MAC_APP
/* just to save a little space */
  return star_sqcurve_method_all(v_info,METHOD_GRADIENT);    
#else

  for ( i = 0 ; i < SDIM ; i++ )
    dvnorm1[i][i] = dvnorm2[i][i] = 0.0;

  if ( METH_INST[v_info->method].gen_method == star_normal_sq_mean_curvature_mi )
     variety = NORMAL_SQ;
  else if ( METH_INST[v_info->method].gen_method == star_eff_area_sq_mean_curvature_mi )
     variety = EFF_SQ;
  else variety = PLAIN_SQ;

  pairs = (v_info->vcount - 1);
  if ( pairs <= 0 ) return 0.0;

  if ( v_info->vcount > MAXV )
  { a = (REAL*)mycalloc(5*pairs,sizeof(REAL));
     ds1 = dmatrix(0,12*v_info->vcount,0,SDIM-1);
  } else
  { memset((char*)aa,0,sizeof(REAL)*5*pairs);
     a = aa;
     memset((char*)ds1[0],0,sizeof(REAL)*12*v_info->vcount*MAXCOORD);
  }
  d = a+pairs; s1s1 = d + pairs; s1s2 = s1s1 + pairs; s2s2 = s1s2 + pairs;
  ds2 = ds1 + pairs; dAdv1 = ds2 + pairs; dAdv2 = dAdv1 + pairs;

  /* basic dot products */
  area = 0.0;
  for ( j = 0 ; j < SDIM ; j++ ) dAdv[j] = vnorm[j] = 0.0;
  for ( k = 0 ; k < pairs ; k++ )
  { s1 = s[k]; s2 = s[(k+1==pairs)?0:k+1];
     s1s1[k] = SDIM_dot(s1,s1);
     s1s2[k] = SDIM_dot(s1,s2);
     s2s2[k] = SDIM_dot(s2,s2);
     d[k] = s1s1[k]*s2s2[k] - s1s2[k]*s1s2[k];
     a[k] = 0.5*sqrt(d[k]);
     area += a[k];
     for ( j = 0 ; j < SDIM ; j++ )
     { ds1[k][j] = 2*(s2s2[k]*s1[j] - s1s2[k]*s2[j]);
        ds2[k][j] = 2*(s1s1[k]*s2[j] - s1s2[k]*s1[j]);
        dAdv1[k][j] = 0.125/a[k]*ds1[k][j];
        dAdv2[k][j] = 0.125/a[k]*ds2[k][j];
        dAdv[j] -= dAdv1[k][j] + dAdv2[k][j];
     }
     if ( variety != PLAIN_SQ )
     { cross_prod(s1,s2,temp);
        for ( j = 0 ; j < SDIM ; j++ )
          vnorm[j] += 0.5*temp[j];
     }
  }

  /* energy */
  ff = SDIM_dot(dAdv,dAdv);
  switch ( variety )
  { case PLAIN_SQ: 
        energy = 0.75/area*ff;
        a15 = 1.5/area;  /* coefficient for later */
        break;
     case EFF_SQ:
        nn = SDIM_dot(vnorm,vnorm);
        energy = 0.75*area*ff/nn;
        ef1 = 1.5*area/nn;    /* coefficients for later */
        ef2 = 1.5*area/nn/nn*ff;
        break;
     case NORMAL_SQ:
        fn = SDIM_dot(dAdv,vnorm);
        if ( fn == 0.0 ) energy = 0.0;
        else
        { energy = 0.75*area*(ff/fn-h0adj)*(ff/fn-h0adj);
          ns1 = 3*area/fn*(ff/fn-h0adj); /* coefficients for later */
          ns2 = 1.5*area*(ff/fn-h0adj)*ff/fn/fn;
        }
        break;
    }

    /* gradient */
    ea = energy/area;
    for ( k = 0 ; k < pairs ; k++ )
    { REAL co = 0.25/a[k];
      REAL coco = 0.5*co/a[k];
      REAL *grad2 = (k+1==pairs)?v_info->grad[1]:v_info->grad[k+2];
      s1 = s[k]; s2 = s[(k+1==pairs)?0:k+1];
      if ( variety != PLAIN_SQ ) 
      { dvnorm1[0][1] = -0.5*s2[2];
         dvnorm1[0][2] =  0.5*s2[1];
         dvnorm1[1][0] =  0.5*s2[2];
         dvnorm1[1][2] = -0.5*s2[0];
         dvnorm1[2][0] = -0.5*s2[1];
         dvnorm1[2][1] =  0.5*s2[0];
         dvnorm2[0][1] =  0.5*s1[2];
         dvnorm2[0][2] = -0.5*s1[1];
         dvnorm2[1][0] = -0.5*s1[2];
         dvnorm2[1][2] =  0.5*s1[0];
         dvnorm2[2][0] =  0.5*s1[1];
         dvnorm2[2][1] = -0.5*s1[0];
      }
      for ( i = 0 ; i < SDIM ; i++ )
      { REAL coeff1 = coco*dAdv1[k][i];
         REAL coeff2 = coco*dAdv2[k][i];
         REAL dAdn1,dAdn2;
         REAL ddAdv1dv,ddAdv2dv;
         dAdn1 = dAdn2 = ddAdv1dv = ddAdv2dv = 0.0;
         for ( j = 0 ; j < SDIM ; j++ )
         { 
            REAL ddss11,ddss12,ddss21,ddss22;
            REAL t1,t2;
            ddss11 = -s2[i]*s2[j];
            ddss12 = 2*s1[i]*s2[j] - s2[i]*s1[j];
            ddss21 = 2*s2[i]*s1[j] - s1[i]*s2[j];
            ddss22 = -s1[i]*s1[j];
            if ( i == j )
            { ddss11 += s2s2[k];
              ddss12 -= s1s2[k];
              ddss21 -= s1s2[k];
              ddss22 += s1s1[k];
            }
            t1 = (coeff1*(ds1[k][j]+ds2[k][j]) - co*(ddss11 + ddss12));
            ddAdv1dv += dAdv[j]*t1;
            t2 = (coeff2*(ds1[k][j]+ds2[k][j]) - co*(ddss21 + ddss22));
            ddAdv2dv += dAdv[j]*t2;
            if ( variety == NORMAL_SQ )
            { dAdn1 += vnorm[j]*t1; dAdn2 += vnorm[j]*t2; }
         }

         switch ( variety )
         { case PLAIN_SQ: 
             g = a15*ddAdv1dv - ea*dAdv1[k][i] ;
             v_info->grad[k+1][i] += g;
             v_info->grad[0][i] -= g;
             g = a15*ddAdv2dv - ea*dAdv2[k][i];
             grad2[i] += g;
             v_info->grad[0][i] -= g;
             break;
            case EFF_SQ:
             g = ea*dAdv1[k][i] + ef1*ddAdv1dv
                      - ef2*SDIM_dot(dvnorm1[i],vnorm);
             v_info->grad[k+1][i] += g;
             v_info->grad[0][i] -= g;
             g = ea*dAdv2[k][i] + ef1*ddAdv2dv
                      - ef2*SDIM_dot(dvnorm2[i],vnorm);
             grad2[i] += g;
             v_info->grad[0][i] -= g;
             break;
            case NORMAL_SQ:
             if ( fn != 0.0 )
              { g = ea*dAdv1[k][i] + ns1*ddAdv1dv
                    - ns2*(dAdn1 + SDIM_dot(dvnorm1[i],dAdv)); 
                 v_info->grad[k+1][i] += g;
                 v_info->grad[0][i] -= g;
                 g = ea*dAdv2[k][i] + ns1*ddAdv2dv
                    - ns2*(dAdn2 + SDIM_dot(dvnorm2[i],dAdv)); 
                 grad2[i] += g;
                 v_info->grad[0][i] -= g;
                }
             break;
         }
      }
    }

  if ( get_vattr(v_info->id) & AXIAL_POINT )
  { REAL fudge = 1./rotorder;
     if ( sym_flags & DOUBLE_AXIAL ) fudge *= 2;
     fudge *= rotorder/v_info->axial_order;
     energy *= fudge;
     for ( i = 0 ; i < v_info->vcount ; i++ )
        for ( j = 0 ; j < SDIM ; j++ )
          v_info->grad[i][j] *= fudge;
  }

  if ( v_info->vcount > MAXV )
  { myfree((char*)a);
     free_matrix(ds1);
  }
  return energy;
#endif
}

/*************************************************************************
*
*  function: star_sqcurve_method_hess()
*
*
*/

REAL star_sqcurve_method_hess(v_info)
struct qinfo *v_info;
{
    return star_sqcurve_method_all(v_info,METHOD_HESSIAN);  
}
