/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: bbroptions.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "bbr.h"
#include "bbroptions.h"


/*
 *  Returns count of options for specified task
 */
int BBR_GetOptionCount(task_context_t * task)
{
    int count = -1;

    LOG_ENTRY();

    switch (task->action) {

    case EVMS_Task_Expand:
        count = BBR_EXPAND_OPTION_COUNT;
        break;

    case EVMS_Task_Shrink:
        count = BBR_SHRINK_OPTION_COUNT;
        break;

    default:
        break;
    }

    LOG_EXIT_INT(count);
    return count;
}


static int allocate_shrink_option_descriptors( task_context_t * context )
{
    int rc = EINVAL;


    LOG_ENTRY();

    //
    //  If things look Ok it means we have a default size for the create operation and now
    //  simply need to initialize options.
    //
    if ( context ) {

        context->option_descriptors->count = BBR_SHRINK_OPTION_COUNT ;      // must set count

        // Block replacement count
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].constraint.list = NULL;
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].constraint.range = EngFncs->engine_alloc( sizeof(value_range_t) );
        if (context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].constraint.range==NULL) {
            LOG_EXIT_INT(ENOMEM);
            return ENOMEM;
        }
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].constraint_type                  = EVMS_Collection_Range;
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].constraint.range->min.ui64       = BBR_MIN_REPLACEMENT_BLOCKS;
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].constraint.range->max.ui64       = BBR_MAX_REPLACEMENT_BLOCKS;
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].constraint.range->increment.ui64 = 1;


        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].flags = 0;
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].help = NULL;
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].name = EngFncs->engine_strdup( BBR_SHRINK_BLOCKCOUNT_NAME );
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].tip = EngFncs->engine_strdup( _("The number of replacement blocks you wish to use for the expanded object.") );
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].title = EngFncs->engine_strdup( _("Count") );
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].type = EVMS_Type_Unsigned_Int64;
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].unit = EVMS_Unit_None;
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].value.ui64 = 0;

    }


    LOG_EXIT_INT(rc);
    return rc;
}

static int initialize_shrink_option_descriptors( storage_object_t *obj, task_context_t * context )
{
    int rc = EINVAL;
    u_int64_t  blocks=0;



    LOG_ENTRY();

    // there should be exactly BBR_SHRINK_OPTION_COUNT options
    if ( context->option_descriptors->count != BBR_SHRINK_OPTION_COUNT) {
        LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);
        LOG_EXIT_INT(rc);
        return rc;
    }


    // check that we claimed this storage object
    if (obj->plugin == my_plugin_record ) {

        if ( obj->private_data ) {

            // test for BBR signature
            if ( ((BBR_Private_Data *)obj->private_data)->signature == EVMS_BBR_SIGNATURE ) {

                blocks = ((BBR_Private_Data *)obj->private_data)->replacement_blocks_needed;

                rc = 0;
            }
        }

    }


    //
    //  initialize options.
    //
    if ( rc == 0 ) {
        // Block replacement count
        context->option_descriptors->option[BBR_SHRINK_BLOCKCOUNT_INDEX].value.ui64 = blocks;

    }


    LOG_EXIT_INT(rc);
    return rc;
}




/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * list.  Also, for any object which is removed from the selected list,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int set_shrink_object( task_context_t * context,
                              list_anchor_t    declined_objects,
                              task_effect_t  * effect )
{
    int rc = EINVAL;
    storage_object_t  *obj;

    LOG_ENTRY();

    obj = EngFncs->first_thing(context->selected_objects, NULL);
    if (obj) {
        rc = initialize_shrink_option_descriptors(obj, context);
        if (!rc) {
            *effect |=  EVMS_Effect_Reload_Options;
        }
    }

    LOG_EXIT_INT(rc);
    return rc;
}




/*
 * Validate the objects in the selected_objects list in the task context.
 * Remove from the selected objects lists any objects which are not
 * acceptable.
 *
 * For unacceptable objects, create a declined_handle_t structure with the
 * reason why it is not acceptable, and add it to the declined_objects list.
 * Modify the accepatble_objects list in the task context as necessary
 * based on the selected objects and the current settings of the options.
 *
 * Modify any option settings as necessary based on the selected objects.
 * Return the appropriate task_effect_t settings if the object list(s),
 * minimum or maximum objects selected, or option settings have changed.
 */
int BBR_SetObjects( task_context_t * context,
                    list_anchor_t    declined_objects,
                    task_effect_t  * effect )
{

    int rc = EINVAL;

    LOG_ENTRY();

    if (context) {

        switch (context->action) {

        case EVMS_Task_Expand:

            // BBR is never an expansion point. It passes expands down to
            // child objects.
            rc = EINVAL;
            break;

        case EVMS_Task_Shrink:

            rc = set_shrink_object( context, declined_objects, effect );
            break;

        default:

            LOG_ERROR("context->action is unknown or unsupported\n");
            break;
        }
    }

    LOG_EXIT_INT(rc);
    return rc;
}




static int SetShrinkOption( task_context_t * context,
                            u_int32_t        index,
                            value_t        * value,
                            u_int32_t      * info )
{
    int rc = EINVAL;

    LOG_ENTRY();

    if (index == BBR_SHRINK_BLOCKCOUNT_INDEX ) {

        context->option_descriptors->option[index].value.ui64 = value->ui64;
        rc = 0;

    }


    LOG_EXIT_INT(rc);
    return rc;
}


/*
 *
 */
int BBR_SetOption( task_context_t * context,
                   u_int32_t        index,
                   value_t        * value,
                   u_int32_t      * info )
{
    int rc=EINVAL;

    LOG_ENTRY();

    // a measure of protection ...
    if (context == NULL) {
        LOG_EXIT_INT(rc);
        return rc;
    }

    switch (context->action) {

    case EVMS_Task_Expand:
        // BBR is never an expansion point ... so no options to set.
        rc = EINVAL;
        break;

    case EVMS_Task_Shrink:
        rc = SetShrinkOption( context, index, value, info );
        break;

    default:
        rc = EINVAL;
        break;
    }


    LOG_EXIT_INT(rc);

    return rc;
}

/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * list.  Also, for any object which is removed from the selected list,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int get_acceptable_shrink_objects( task_context_t * context  )
{
    storage_object_t * object, * child;
    list_anchor_t shrink_points;
    list_element_t itr1, itr2;
    int count, rc = EINVAL;

    LOG_ENTRY();

    count = EngFncs->list_count(context->acceptable_objects);
    if (count == 0) {

        shrink_points = EngFncs->allocate_list();
        if (shrink_points) {
            rc = EngFncs->get_object_list( EVMS_OBJECT, DATA_TYPE, my_plugin_record,
                                           NULL, VALID_INPUT_OBJECT,
                                           &context->acceptable_objects );
            if (!rc) {

                LIST_FOR_EACH_SAFE(context->acceptable_objects, itr1, itr2, object) {
                    child = GET_BBR_CHILD( object );
                    if (child) {
                        rc = child->plugin->functions.plugin->can_shrink(child, child->size,
                                                                         shrink_points);
                        if (rc) {
                            EngFncs->delete_element(itr1);
                        }
                        EngFncs->delete_all_elements(shrink_points);
                     }
	        }
            }

            EngFncs->destroy_list(shrink_points);
        }
	else {
            LOG_ERROR("Cannot allocate shrink-points list.\n");
        }
    }
    else {
        LOG_ERROR("context already has acceptable objects\n");
    }

    LOG_EXIT_INT(rc);
    return rc;
}



/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * list.  Also, for any object which is removed from the selected list,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
int BBR_InitTask( task_context_t * context )
{
    int rc = EINVAL;

    LOG_ENTRY();


    if (context) {

        switch (context->action) {

        case EVMS_Task_Expand:

            // BBR is never an expansion point. It always passes expand
            // requests down to child objects who become expansion points.
            break;

        case EVMS_Task_Shrink:

            context->min_selected_objects = 1;
            context->max_selected_objects = 1;

            rc = allocate_shrink_option_descriptors( context );
            if (rc == 0) {
                rc = get_acceptable_shrink_objects( context );
            }
            break;

        default:
               LOG_ERROR("context->action is unknown or unsupported\n");
               break;
        }
    }


    LOG_EXIT_INT(rc);
    return rc;
}


/*
 * Returns information about a BBR object
 */
int BBR_GetInfo( storage_object_t  * object, char *name,  extended_info_array_t * * info)
{
    int rc = EINVAL;
    extended_info_array_t   *Info;
    BBR_Private_Data        *pdata = (BBR_Private_Data *) object->private_data;


    LOG_ENTRY();

    // a measure of protection ...
    if ((info == NULL)||(pdata->signature != EVMS_BBR_SIGNATURE)) {
        LOG_EXIT_INT(rc);
        return rc;
    }

    rc    = ENOMEM;  // init to failed calloc
    *info = NULL;     // init to no info returned

    if ( object->object_type == EVMS_OBJECT ) {

        Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + (BBR_INFO_COUNT * sizeof(extended_info_t) ) );
        if (Info) {

            Info->count = BBR_INFO_COUNT;

            Info->info[BBR_INFO_NAME_INDEX].name = EngFncs->engine_strdup( "Name" );
            Info->info[BBR_INFO_NAME_INDEX].title = EngFncs->engine_strdup( _("Name") );
            Info->info[BBR_INFO_NAME_INDEX].desc = EngFncs->engine_strdup( _("This is the name given to the storage object. It must be unique on the system.") );
            Info->info[BBR_INFO_NAME_INDEX].type               = EVMS_Type_String;
            Info->info[BBR_INFO_NAME_INDEX].unit               = EVMS_Unit_None;
            Info->info[BBR_INFO_NAME_INDEX].value.s = EngFncs->engine_strdup( object->name );
            Info->info[BBR_INFO_NAME_INDEX].collection_type    = EVMS_Collection_None;
            memset( &Info->info[BBR_INFO_NAME_INDEX].group, 0, sizeof(group_info_t));

            Info->info[BBR_INFO_SIZE_INDEX].name = EngFncs->engine_strdup( "Size" );
            Info->info[BBR_INFO_SIZE_INDEX].title = EngFncs->engine_strdup( _("Size") );
            Info->info[BBR_INFO_SIZE_INDEX].desc = EngFncs->engine_strdup( _("This is the size in sectors of the storage object after reserving space for metadata.") );
            Info->info[BBR_INFO_SIZE_INDEX].type               = EVMS_Type_Unsigned_Int64;
            Info->info[BBR_INFO_SIZE_INDEX].unit               = EVMS_Unit_Sectors;
            Info->info[BBR_INFO_SIZE_INDEX].value.ui64         = object->size;
            Info->info[BBR_INFO_SIZE_INDEX].collection_type    = EVMS_Collection_None;
            memset( &Info->info[BBR_INFO_SIZE_INDEX].group, 0, sizeof(group_info_t));
            Info->info[BBR_INFO_SIZE_INDEX].flags |= EVMS_EINFO_FLAGS_NO_UNIT_CONVERSION;

            Info->info[BBR_INFO_BLKS_INDEX].name = EngFncs->engine_strdup( "Reserve Blocks" );
            Info->info[BBR_INFO_BLKS_INDEX].title = EngFncs->engine_strdup( _("Blocks") );
            Info->info[BBR_INFO_BLKS_INDEX].desc = EngFncs->engine_strdup( _("This is the number of replacement blocks BBR is reserving for this storage object.") );
            Info->info[BBR_INFO_BLKS_INDEX].type               = EVMS_Type_Unsigned_Int64;
            Info->info[BBR_INFO_BLKS_INDEX].unit               = EVMS_Unit_None;
            Info->info[BBR_INFO_BLKS_INDEX].value.ui64         = pdata->replacement_blocks_needed;
            Info->info[BBR_INFO_BLKS_INDEX].collection_type    = EVMS_Collection_None;
            memset( &Info->info[BBR_INFO_BLKS_INDEX].group, 0, sizeof(group_info_t));

            Info->info[BBR_INFO_BLKSZ_INDEX].name = EngFncs->engine_strdup( "Block Size" );
            Info->info[BBR_INFO_BLKSZ_INDEX].title = EngFncs->engine_strdup( _("Block Size") );
            Info->info[BBR_INFO_BLKSZ_INDEX].desc = EngFncs->engine_strdup( _("This value tells you the size of a replacement block.") );
            Info->info[BBR_INFO_BLKSZ_INDEX].type               = EVMS_Type_Unsigned_Int64;
            Info->info[BBR_INFO_BLKSZ_INDEX].unit               = EVMS_Unit_None;
            Info->info[BBR_INFO_BLKSZ_INDEX].value.ui64         = pdata->block_size;
            Info->info[BBR_INFO_BLKSZ_INDEX].collection_type    = EVMS_Collection_None;
            memset( &Info->info[BBR_INFO_BLKSZ_INDEX].group, 0, sizeof(group_info_t));

            Info->info[BBR_INFO_TBLSZ_INDEX].name = EngFncs->engine_strdup( "Table Size" );
            Info->info[BBR_INFO_TBLSZ_INDEX].title = EngFncs->engine_strdup( _("Size of BBR table") );
            Info->info[BBR_INFO_TBLSZ_INDEX].desc = EngFncs->engine_strdup( _("This is the number of sectors being used by the BBR remapping table.") );
            Info->info[BBR_INFO_TBLSZ_INDEX].type               = EVMS_Type_Unsigned_Int64;
            Info->info[BBR_INFO_TBLSZ_INDEX].unit               = EVMS_Unit_Sectors;
            Info->info[BBR_INFO_TBLSZ_INDEX].value.ui64         = pdata->bbr_table_size_in_sectors;
            Info->info[BBR_INFO_TBLSZ_INDEX].collection_type    = EVMS_Collection_None;
            memset( &Info->info[BBR_INFO_TBLSZ_INDEX].group, 0, sizeof(group_info_t));
            Info->info[BBR_INFO_TBLSZ_INDEX].flags |= EVMS_EINFO_FLAGS_NO_UNIT_CONVERSION;

            *info = Info;

            rc = 0;
        }
        else {
            LOG_ERROR("unable to malloc memory for extended info array\n");
        }
    }

    LOG_EXIT_INT(rc);
    return rc;
}


/*
 *
 */
int BBR_GetPluginInfo( char * descriptor_name, extended_info_array_t * * info )
{
    int rc = EINVAL;
    extended_info_array_t   *Info;
    char                     version_string[64];
    char                     required_engine_api_version_string[64];
    char                     required_plugin_api_version_string[64];


    LOG_ENTRY();

    // a measure of protection ...
    if (info == NULL) {
        LOG_EXIT_INT(rc);
        return rc;
    }

    rc    = ENOMEM;  // init to failed calloc
    *info = NULL;     // init to no info returned

    Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + (BBR_PLUGIN_INFO_COUNT*sizeof(extended_info_t))  );
    if (Info) {

        Info->count = BBR_PLUGIN_INFO_COUNT;

        sprintf(version_string, "%d.%d.%d",
                MAJOR_VERSION,
                MINOR_VERSION,
                PATCH_LEVEL );

        sprintf(required_engine_api_version_string, "%d.%d.%d",
                my_plugin_record->required_engine_api_version.major,
                my_plugin_record->required_engine_api_version.minor,
                my_plugin_record->required_engine_api_version.patchlevel );

        sprintf(required_plugin_api_version_string, "%d.%d.%d",
                my_plugin_record->required_plugin_api_version.plugin.major,
                my_plugin_record->required_plugin_api_version.plugin.minor,
                my_plugin_record->required_plugin_api_version.plugin.patchlevel );

        Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].name = EngFncs->engine_strdup( "Short Name" );
        Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].title = EngFncs->engine_strdup( _("Short Name") );
        Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].desc = EngFncs->engine_strdup( _("A short name given to this plug-in") );
        Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].type               = EVMS_Type_String;
        Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].unit               = EVMS_Unit_None;
        Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].value.s = EngFncs->engine_strdup( my_plugin_record->short_name );
        Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].group, 0, sizeof(group_info_t));

        Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].name = EngFncs->engine_strdup( "Long Name" );
        Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].title = EngFncs->engine_strdup( _("Long Name") );
        Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].desc = EngFncs->engine_strdup( _("A longer, more descriptive name for this plug-in") );
        Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].type               = EVMS_Type_String;
        Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].unit               = EVMS_Unit_None;
        Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].value.s = EngFncs->engine_strdup( my_plugin_record->long_name );
        Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].group, 0, sizeof(group_info_t));

        Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].name = EngFncs->engine_strdup( "Type" );
        Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].title = EngFncs->engine_strdup( _("Plug-in Type") );
        Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].desc = EngFncs->engine_strdup( _("There are various types of plug-ins, each responsible for some kind of storage object or logical volume.") );
        Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].type               = EVMS_Type_String;
        Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].unit               = EVMS_Unit_None;
        Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].value.s = EngFncs->engine_strdup( _("EVMS Feature") );
        Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].group, 0, sizeof(group_info_t));

        Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].name = EngFncs->engine_strdup( "Version" );
        Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].title = EngFncs->engine_strdup( _("Plug-in Version") );
        Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].desc = EngFncs->engine_strdup( _("This is the version number of the plug-in.") );
        Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].type               = EVMS_Type_String;
        Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].unit               = EVMS_Unit_None;
        Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].value.s = EngFncs->engine_strdup( version_string );
        Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].group, 0, sizeof(group_info_t));

        Info->info[BBR_PLUGIN_INFO_REQENGSVCVERSION_INDEX].name = EngFncs->engine_strdup( "Required Engine Services Version" );
        Info->info[BBR_PLUGIN_INFO_REQENGSVCVERSION_INDEX].title = EngFncs->engine_strdup( _("Required Engine Services Version") );
        Info->info[BBR_PLUGIN_INFO_REQENGSVCVERSION_INDEX].desc = EngFncs->engine_strdup( _("This is the version of the Engine services that this plug-in requires.  "
											    "It will not run on older versions of the Engine services.") );
        Info->info[BBR_PLUGIN_INFO_REQENGSVCVERSION_INDEX].type               = EVMS_Type_String;
        Info->info[BBR_PLUGIN_INFO_REQENGSVCVERSION_INDEX].unit               = EVMS_Unit_None;
        Info->info[BBR_PLUGIN_INFO_REQENGSVCVERSION_INDEX].value.s = EngFncs->engine_strdup( required_engine_api_version_string );
        Info->info[BBR_PLUGIN_INFO_REQENGSVCVERSION_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[BBR_PLUGIN_INFO_REQENGSVCVERSION_INDEX].group, 0, sizeof(group_info_t));

        Info->info[BBR_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].name = EngFncs->engine_strdup( "Required Engine Plug-in API Version" );
        Info->info[BBR_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].title = EngFncs->engine_strdup( _("Required Engine Plug-in API Version") );
        Info->info[BBR_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].desc = EngFncs->engine_strdup( _("This is the version of the Engine plug-in API that this plug-in requires.  "
											     "It will not run on older versions of the Engine plug-in API.") );
        Info->info[BBR_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].type               = EVMS_Type_String;
        Info->info[BBR_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].unit               = EVMS_Unit_None;
        Info->info[BBR_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].value.s = EngFncs->engine_strdup( required_plugin_api_version_string );
        Info->info[BBR_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[BBR_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].group, 0, sizeof(group_info_t));

        *info = Info;

        rc = 0;
    }


    LOG_EXIT_INT(rc);
    return rc;
}
