/*
 *
 *   (C) Copyright IBM Corp. 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libmac.so
 *
 *   File: commit.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "mac_plugin.h"


int do_mac_commit( LOGICALDISK *ld, DISKSEG *md,
                   DISKSEG *parent, boolean backup )
{
	mac_partition_t            *p = NULL;
	char                       *pmap=NULL;
	mac_disklabel_t            *disklabel=NULL;
	DISKSEG                    *seg;
	char                       *buffer;
	disk_private_data_t        *disk_pdata;
	int                         rc = 0, i;
	seg_private_data_t         *pdata;
	int                         map_count=0;
	int                         count=0;
	int                         vsectors_per_block;
	sector_count_t              max_map_count=0;
	char                        pname[PMAP_STRING_SIZE+1];
	char                        ptype[PMAP_STRING_SIZE+1];
	list_element_t              iter;


	LOG_ENTRY();

	REQUIRE(ld);
	REQUIRE(md);

	disk_pdata = get_mac_disk_private_data(ld);

	REQUIRE(disk_pdata);

	buffer = malloc(md->size * EVMS_VSECTOR_SIZE);

	REQUIRE(buffer != NULL);

	disklabel = (mac_disklabel_t *) buffer; 

	// read mac info & verify we have valid disk label
	rc = READ( ld, md->start, md->size, buffer );
	if (!rc) {
		if ( MAC_DISK_TO_CPU16(disklabel->signature) != MAC_DISK_MAGIC) {
			rc = EINVAL;
		}
	}

	if (!rc) {
		vsectors_per_block = MAC_DISK_TO_CPU16(disklabel->block_size)>>EVMS_VSECTOR_SIZE_SHIFT;  
		pmap = buffer + (vsectors_per_block*EVMS_VSECTOR_SIZE);
		max_map_count = md->size - vsectors_per_block;
	}
	else {
		free(buffer);
		LOG_EXIT_INT(rc);
		return rc;
	}        

	// walk through the partition table, removing all linux partition records                
	// by replacing them with Apple Scratch records
	LOG_DEBUG("there should be %d partition records\n", disk_pdata->pcount);               
	for ( i=0; (i<max_map_count)&&(rc==0); i++ ) {

		p = (mac_partition_t *) (pmap + (i*vsectors_per_block*EVMS_VSECTOR_SIZE));

		strncpy(pname, p->name, PMAP_STRING_SIZE);
		strncpy(ptype, p->type, PMAP_STRING_SIZE);

		LOG_DEBUG( "   Index (%d): type: %s   name: %s\n", i, ptype, pname );

		if ( MAC_DISK_TO_CPU32(p->status) & MAC_STATUS_VALID ) {

			if ( isa_mac_data_partition(p) == TRUE ) {
				LOG_DEBUG("     clearing the entry\n");
				memset( (void *) p, 0, sizeof(mac_partition_t) );  
			}
			else {
				LOG_DEBUG("     not clearing entry\n");
			}

		}
		else {	// we have run off the end of the partition table
			LOG_DEBUG("oops ... invalid partition record\n");
			break;
		}
	}


	// now walk through the segment list on this drive and add each 
	// segment to the PMAP
	LIST_FOR_EACH( ld->parent_objects, iter, seg ) {

		if (seg->data_type == DATA_TYPE) {

			LOG_DEBUG("     adding segment %s to the pmap\n", seg->name);

			pdata = (seg_private_data_t *) seg->private_data;

			for ( i=0,rc=ENOSPC; (rc!=0) && (i<max_map_count); i++ ) {

				p = (mac_partition_t *) (pmap + (i*vsectors_per_block*EVMS_VSECTOR_SIZE));

				if ( isa_mac_null_partition(p)==TRUE) {
					rc = 0;                                         
				}
			}

			if (!rc) {
				memcpy( (void *) p, (void *) &pdata->p_record, sizeof(mac_partition_t) );                        
			}

		}

	}

	// tally up the partitions to get a current map count
	map_count = 0;                          
	for ( i=0; i<max_map_count; i++) {
		p = (mac_partition_t *) (pmap + (i*vsectors_per_block*EVMS_VSECTOR_SIZE));
		if ( MAC_DISK_TO_CPU32(p->status) & MAC_STATUS_VALID ) {
			++map_count;
		}
	}               

	// keep map count current in the disk private data
	disk_pdata->pcount = map_count;

	// update the map_count in each partition record        
	count = 0;              
	for ( i=0; i<max_map_count && count<map_count; i++) {
		p = (mac_partition_t *) (pmap + (i*vsectors_per_block*EVMS_VSECTOR_SIZE));
		if ( MAC_DISK_TO_CPU32(p->status) & MAC_STATUS_VALID ) {
			p->map_count = CPU_TO_MAC_DISK32(map_count);  
			++count;
		}
	}

	// commit changes
	if (backup) {
		rc = EngFncs->save_metadata( parent->name, ld->name,
					     md->start, md->size, buffer );
	} else {
		rc = WRITE( ld, md->start, md->size, buffer );
	}


	LOG_EXIT_INT(rc);
	return rc;
}



/*
 *  Function:  commit_mac_metadata
 *
 *  Called to commit the mac disk label information.
 */
int  commit_mac_segments( DISKSEG *seg, LOGICALDISK *ld,  uint commit_phase )
{
	int rc=0;

	LOG_ENTRY();

	REQUIRE(ld != NULL);
	REQUIRE(seg != NULL);

	if (seg->data_type == META_DATA_TYPE) {

		rc = do_mac_commit( ld, seg, NULL, FALSE );
		if (!rc) {
			seg->flags &= ~SOFLAG_DIRTY;
		}

	}
	else {
		seg->flags &= ~SOFLAG_DIRTY;
	}


	LOG_EXIT_INT(rc);
	return rc;
}

