/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-thread-manager.c,v 1.44 2005/02/06 01:10:00 hoa Exp $
 */

#include "etpan-thread-manager.h"
#include "etpan-thread-manager-types.h"

#include <stdlib.h>
#include "etpan-app.h"
#include "etpan-msg-params.h"
#include <unistd.h>
#include "etpan-errors.h"
#include "etpan-cfg-vfolder.h"
#include "etpan-msg-renderer.h"
#include "etpan-tools.h"
#include <string.h>
#include <paths.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include "etpan-imf-helper.h"
#include "etpan-msg-new.h"
#include "etpan-msg-reply.h"
#include "etpan-cfg-mime.h"
#include "etpan-mime-tools.h"
#include <sys/socket.h>
#include "etpan-nntp.h"
#include "etpan-imap.h"
#include <libetpan/libetpan.h>
#include "etpan-config.h"
#include "etpan-abook-driver.h"
#include "etpan-folder-discover.h"
#include "etpan-cfg-sender.h"
#include "etpan-sender.h"

/*
  these 3 headers MUST be included before <sys/select.h>
  to insure compatibility with Mac OS X (this is true for 10.2)
*/
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/select.h>

#define THREAD_STACK_SIZE (2 * 1024 * 1024)

enum {
  THREAD_STATE_DETACHED,
  THREAD_STATE_DETACHED_EXECUTING,
  THREAD_STATE_DETACHED_FINISHED,
  THREAD_STATE_NOT_STARTED,
  THREAD_STATE_NOT_STARTED_CANCELLED,
  THREAD_STATE_EXECUTING,
  THREAD_STATE_EXECUTING_CANCELLED,
  THREAD_STATE_WAIT,
  THREAD_STATE_WAIT_CANCELLED,
  THREAD_STATE_NOTIFIED
};


struct thread_info {
  int cmd;
  
  /* name is for debug purpose */
  char * name;
  
  /* run the operation */
  int (* run)(struct etpan_thread_op *);
  
  /* clean on cancel or detach */
  void (* clean_arg)(struct etpan_thread_op *);
  void (* clean_result)(struct etpan_thread_op *);
  
  int need_connect;
  int can_cancel;
};

static void lock(struct etpan_thread_manager * manager);
static void unlock(struct etpan_thread_manager * manager);

static void folder_status_clean_result(struct etpan_thread_op * op);
static int folder_status(struct etpan_thread_op * op);

static int folder_check(struct etpan_thread_op * op);

static void folder_get_msg_list_clean_arg(struct etpan_thread_op * op);
static void folder_get_msg_list_clean_result(struct etpan_thread_op * op);
static int folder_get_msg_list(struct etpan_thread_op * op);

static void folder_free_msg_list_clean_arg(struct etpan_thread_op * op);
static int folder_free_msg_list(struct etpan_thread_op * op);

static int message_get_bodystructure(struct etpan_thread_op * op);

static int message_flush(struct etpan_thread_op * op);

static int message_check(struct etpan_thread_op * op);

static int folder_expunge(struct etpan_thread_op * op);

static void message_fetch_clean_result(struct etpan_thread_op * op);
static int message_fetch(struct etpan_thread_op * op);
static int message_fetch_header(struct etpan_thread_op * op);
static int message_fetch_section(struct etpan_thread_op * op);
static int message_fetch_section_header(struct etpan_thread_op * op);
static int message_fetch_section_mime(struct etpan_thread_op * op);
static int message_fetch_result_free(struct etpan_thread_op * op);

static void message_render_mime_clean_result(struct etpan_thread_op * op);
static int message_render_mime(struct etpan_thread_op * op);

static void message_render_reply_mime_clean_arg(struct etpan_thread_op * op);
static void
message_render_reply_mime_clean_result(struct etpan_thread_op * op);
static int message_render_reply_mime(struct etpan_thread_op * op);

static void folder_append_message_clean_arg(struct etpan_thread_op * op);
static int folder_append_message(struct etpan_thread_op * op);

#if 0
static void message_send_clean_arg(struct etpan_thread_op * op);
static int message_send(struct etpan_thread_op * op);
#endif

#if 0
static int message_free(struct etpan_thread_op * op);
#endif

static void msglist_check_clean_arg(struct etpan_thread_op * op);
static int msglist_check(struct etpan_thread_op * op);

static void msglist_move_clean_arg(struct etpan_thread_op * op);
static int msglist_move(struct etpan_thread_op * op);

static void msglist_copy_clean_arg(struct etpan_thread_op * op);
static int msglist_copy(struct etpan_thread_op * op);

static void msglist_spam_clean_arg(struct etpan_thread_op * op);
static int msglist_spam(struct etpan_thread_op * op);

static void message_mime_copy_clean_result(struct etpan_thread_op * op);
static int message_mime_copy(struct etpan_thread_op * op);

static int ref_message(struct etpan_thread_op * op);

static int unref_message(struct etpan_thread_op * op);

static void nntp_list_groups_clean_result(struct etpan_thread_op * op);
static void nntp_list_groups_clean_arg(struct etpan_thread_op * op);
static int nntp_list_groups(struct etpan_thread_op * op);

static void imap_list_mailboxes_clean_result(struct etpan_thread_op * op);
static void imap_list_mailboxes_clean_arg(struct etpan_thread_op * op);
static int imap_list_mailboxes(struct etpan_thread_op * op);

static void imap_create_clean_arg(struct etpan_thread_op * op);
static int imap_create(struct etpan_thread_op * op);

static void imap_select_clean_arg(struct etpan_thread_op * op);
static int imap_select(struct etpan_thread_op * op);

static int stop_storage_thread(struct etpan_thread_op * op);

static void done_config_clean_arg(struct etpan_thread_op * op);
static int done_config(struct etpan_thread_op * op);

static int storage_disconnect(struct etpan_thread_op * op);

static int abook_disconnect(struct etpan_thread_op * op);

static void abook_lookup_clean_result(struct etpan_thread_op * op);
static void abook_lookup_clean_arg(struct etpan_thread_op * op);
static int abook_lookup(struct etpan_thread_op * op);

static void sendmail_file_clean_arg(struct etpan_thread_op * op);
static int sendmail_file(struct etpan_thread_op * op);

static int discover_folder(struct etpan_thread_op * op);

enum {
  CONNECT_NONE = 0,
  CONNECT_MAILACCESS,
  CONNECT_ABOOK,
};

static struct thread_info thread_info_tab[] = {
  {
    .cmd = ETPAN_THREAD_END,
    .name = "end",
    .run = NULL,
    .clean_arg = NULL,
    .clean_result = NULL,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 0,
  },
  
  {
    .cmd = ETPAN_THREAD_FOLDER_GET_MSG_LIST,
    .name = "folder-get-msg-list",
    .run = folder_get_msg_list,
    .clean_arg = folder_get_msg_list_clean_arg,
    .clean_result = folder_get_msg_list_clean_result,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 1,
  },

  {
    .cmd = ETPAN_THREAD_FOLDER_FREE_MSG_LIST,
    .name = "folder-free-msg-list",
    .run = folder_free_msg_list,
    .clean_arg = folder_free_msg_list_clean_arg,
    .clean_result = NULL,
    .need_connect = CONNECT_NONE,
    .can_cancel = 0,
  },
  
  {
    .cmd = ETPAN_THREAD_FOLDER_CHECK,
    .name = "folder-check",
    .run = folder_check,
    .clean_arg = NULL,
    .clean_result = NULL,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 0,
  },

  {
    .cmd = ETPAN_THREAD_FOLDER_STATUS,
    .name = "folder-status",
    .run = folder_status,
    .clean_arg = NULL,
    .clean_result = folder_status_clean_result,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 1,
  },

  {
    .cmd = ETPAN_THREAD_FOLDER_EXPUNGE,
    .name = "folder-expunge",
    .run = folder_expunge,
    .clean_arg = NULL,
    .clean_result = NULL,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 0,
  },

  {
    .cmd = ETPAN_THREAD_MESSAGE_GET_BODYSTRUCTURE,
    .name = "message-get-bodystructure",
    .run = message_get_bodystructure,
    .clean_arg = NULL,
    .clean_result = NULL,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 1,
  },

  {
    .cmd = ETPAN_THREAD_MESSAGE_FLUSH,
    .name = "message-flush",
    .run = message_flush,
    .clean_arg = NULL,
    .clean_result = NULL,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 0,
  },

  {
    .cmd = ETPAN_THREAD_MESSAGE_CHECK,
    .name = "message-check",
    .run = message_check,
    .clean_arg = NULL,
    .clean_result = NULL,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 0,
  }, 

  {
    .cmd = ETPAN_THREAD_MESSAGE_FETCH,
    .name = "message-fetch",
    .run = message_fetch,
    .clean_arg = NULL,
    .clean_result = message_fetch_clean_result,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 1,
  },

  { .cmd = ETPAN_THREAD_MESSAGE_FETCH_HEADER,
    .name = "message-fetch-header",
    .run = message_fetch_header,
    .clean_arg = NULL,
    .clean_result = message_fetch_clean_result,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 1,
  },

  {
    .cmd = ETPAN_THREAD_MESSAGE_FETCH_SECTION,
    .name = "message-fetch-section",
    .run = message_fetch_section,
    .clean_arg = NULL,
    .clean_result = message_fetch_clean_result,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 1,
  },

  { .cmd = ETPAN_THREAD_MESSAGE_FETCH_SECTION_MIME,
    .name = "message-fetch-section-mime",
    .run = message_fetch_section_mime,
    .clean_arg = NULL,
    .clean_result = message_fetch_clean_result,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 1,
  },

  { .cmd = ETPAN_THREAD_MESSAGE_FETCH_SECTION_HEADER,
    .name = "message-fetch-section-header",
    .run = message_fetch_section_header,
    .clean_arg = NULL,
    .clean_result = message_fetch_clean_result,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 1,
  },

  {
    .cmd = ETPAN_THREAD_MESSAGE_FETCH_RESULT_FREE,
    .name = "message-fetch-result-free",
    .run = message_fetch_result_free,
    .clean_arg = NULL,
    .clean_result = NULL,
    .need_connect = CONNECT_NONE,
    .can_cancel = 0,
  },

  {
    .cmd = ETPAN_THREAD_MESSAGE_RENDER_MIME,
    .name = "message-render-mime",
    .run = message_render_mime,
    .clean_arg = NULL,
    .clean_result = message_render_mime_clean_result,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 1,
  },

  {
    .cmd = ETPAN_THREAD_MESSAGE_RENDER_REPLY_MIME,
    .name = "message-render-reply-mime",
    .run = message_render_reply_mime,
    .clean_arg = message_render_reply_mime_clean_arg,
    .clean_result = message_render_reply_mime_clean_result,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 1,
  },

  {
    .cmd = ETPAN_THREAD_FOLDER_APPEND_MESSAGE,
    .name = "folder-append-message",
    .run = folder_append_message,
    .clean_arg = folder_append_message_clean_arg,
    .clean_result = NULL,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 0,
  },

#if 0  
  {
    .cmd = ETPAN_THREAD_MESSAGE_SEND,
    .name = "message-send",
    .run = message_send,
    .clean_arg = message_send_clean_arg,
    .clean_result = NULL,
    .need_connect = CONNECT_NONE,
    .can_cancel = 0,
  },
#endif

#if 0
  {
    .cmd = ETPAN_THREAD_MESSAGE_FREE,
    .name = "message-free",
    .run = message_free,
    .clean_arg = NULL,
    .clean_result = NULL,
    .need_connect = CONNECT_NONE,
    .can_cancel = 0
  },
#endif

  {
    .cmd = ETPAN_THREAD_MSGLIST_CHECK,
    .name = "msglist-check",
    .run = msglist_check,
    .clean_arg = msglist_check_clean_arg,
    .clean_result = NULL,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 0
  },

  {
    .cmd = ETPAN_THREAD_MSGLIST_COPY,
    .name = "msglist-copy",
    .run = msglist_copy,
    .clean_arg = msglist_copy_clean_arg,
    .clean_result = NULL,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 0
  },

  {
    .cmd = ETPAN_THREAD_MSGLIST_MOVE,
    .name = "msglist-move",
    .run = msglist_move,
    .clean_arg = msglist_move_clean_arg,
    .clean_result = NULL,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 0
  },

  {
    .cmd = ETPAN_THREAD_MSGLIST_SPAM,
    .name = "msglist-spam",
    .run = msglist_spam,
    .clean_arg = msglist_spam_clean_arg,
    .clean_result = NULL,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 0
  },

  {
    .cmd = ETPAN_THREAD_MESSAGE_MIME_COPY,
    .name = "message-mime-copy",
    .run = message_mime_copy,
    .clean_arg = NULL,
    .clean_result = message_mime_copy_clean_result,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 1,
  },

  {
    .cmd = ETPAN_THREAD_MESSAGE_REF,
    .name = "message-ref",
    .run = ref_message,
    .clean_arg = NULL,
    .clean_result = NULL,
    .need_connect = CONNECT_NONE,
    .can_cancel = 0,
  },

  {
    .cmd = ETPAN_THREAD_MESSAGE_UNREF,
    .name = "message-unref",
    .run = unref_message,
    .clean_arg = NULL,
    .clean_result = NULL,
    .need_connect = CONNECT_NONE,
    .can_cancel = 0,
  },

  {
    .cmd = ETPAN_THREAD_NNTP_LIST_GROUPS,
    .name = "nntp-list-groups",
    .run = nntp_list_groups,
    .clean_arg = nntp_list_groups_clean_arg,
    .clean_result = nntp_list_groups_clean_result,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 1,
  },

  {
    .cmd = ETPAN_THREAD_IMAP_LIST_MAILBOXES,
    .name = "imap-list-mailboxes",
    .run = imap_list_mailboxes,
    .clean_arg = imap_list_mailboxes_clean_arg,
    .clean_result = imap_list_mailboxes_clean_result,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 1,
  },

  {
    .cmd = ETPAN_THREAD_IMAP_CREATE,
    .name = "imap-create",
    .run = imap_create,
    .clean_arg = imap_create_clean_arg,
    .clean_result = NULL,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 1,
  },

  {
    .cmd = ETPAN_THREAD_IMAP_SELECT,
    .name = "imap-select",
    .run = imap_select,
    .clean_arg = imap_select_clean_arg,
    .clean_result = NULL,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 1,
  },

  {
    .cmd = ETPAN_THREAD_DONE_CONFIG,
    .name = "done-config",
    .run = done_config,
    .clean_arg = done_config_clean_arg,
    .clean_result = NULL,
    .need_connect = CONNECT_NONE,
    .can_cancel = 0,
  },

  {
    .cmd = ETPAN_THREAD_STOP_STORAGE_THREAD,
    .name = "stop-storage-thread",
    .run = stop_storage_thread,
    .clean_arg = NULL,
    .clean_result = NULL,
    .need_connect = CONNECT_NONE,
    .can_cancel = 0,
  },

  {
    .cmd = ETPAN_THREAD_STORAGE_DISCONNECT,
    .name = "storage-disconnect",
    .run = storage_disconnect,
    .clean_arg = NULL,
    .clean_result = NULL,
    .need_connect = CONNECT_NONE,
    .can_cancel = 0,
  },

  {
    .cmd = ETPAN_THREAD_ABOOK_LOOKUP,
    .name = "abook-lookup",
    .run = abook_lookup,
    .clean_arg = abook_lookup_clean_arg,
    .clean_result = abook_lookup_clean_result,
    .need_connect = CONNECT_ABOOK,
    .can_cancel = 1,
  },

  {
    .cmd = ETPAN_THREAD_ABOOK_DISCONNECT,
    .name = "abook-disconnect",
    .run = abook_disconnect,
    .clean_arg = NULL,
    .clean_result = NULL,
    .need_connect = CONNECT_NONE,
    .can_cancel = 0,
  },

  {
    .cmd = ETPAN_THREAD_SENDER_SENDMAIL,
    .name = "sendmail",
    .run = sendmail_file,
    .clean_arg = sendmail_file_clean_arg,
    .clean_result = NULL,
    .need_connect = CONNECT_NONE,
    .can_cancel = 1,
  },
  
  {
    .cmd = ETPAN_THREAD_DISCOVER_FOLDER,
    .name = "discover-folder",
    .run = discover_folder,
    .clean_arg = NULL,
    .clean_result = NULL,
    .need_connect = CONNECT_MAILACCESS,
    .can_cancel = 1,
  },
};

/*
  get_thread_info()
  
  return thread execution information given a command identifier.
  
  TODO : should use a hash
*/

static struct thread_info * get_thread_info(int cmd)
{
  unsigned int i;
  
  for(i = 0 ;
      i < (sizeof(thread_info_tab) / sizeof(thread_info_tab[0])) ;
      i ++) {
    if (thread_info_tab[i].cmd == cmd)
      return &thread_info_tab[i];
  }
  
  return NULL;
}


static struct etpan_thread_op *
thread_op_add(struct etpan_thread_data * data,
    int cmd,
    struct mailstorage * storage,
    struct mailfolder * folder,
    mailmessage * msg,
    struct mailmime * mime,
    void * arg, int detached);


static void log_op_info(struct etpan_app * app,
    char * prefix, int cmd,
    struct mailstorage * storage,
    struct mailfolder * folder);

static void debug_op_info(struct etpan_app * app,
    char * prefix, int cmd,
    struct mailstorage * storage,
    struct mailfolder * folder);

static void
delete_misc_thread_data(struct etpan_thread_manager * thread_manager);

#if 0
static void change_session_for_message(struct etpan_thread_data * thread_data,
    struct mailfolder * folder);

static int generic_mark_valid(chash * validity_table,
    void * data, void * valid_data)
{
  chashdatum key;
  
  key.data = (char *) &data;
  key.len = sizeof(data);
  
  if (valid_data != NULL) {
    int r;
    chashdatum value;
    
    value.data = valid_data;
    value.len = 0;
    
    r = chash_set(validity_table, &key, &value, NULL);
    if (r < 0)
      return ERROR_MEMORY;
  }
  else {
    chash_delete(validity_table, &key, NULL);
  }
  
  return NO_ERROR;
}

static int generic_is_valid(chash * validity_table, void * data)
{
  int r;
  chashdatum key;
  chashdatum result;
  
  key.data = (char *) &data;
  key.len = sizeof(data);
  
  r = chash_get(validity_table, &key, &result);
  if (r == 0)
    return 1;
  else
    return 0;
}
#endif

#if 0
static int etpan_ref_storage(struct etpan_thread_manager * thread_manager,
    struct mailstorage * storage)
{
  if (storage == NULL)
    return NO_ERROR;
  
  return generic_mark_valid(thread_manager->storage_validity,
      storage, storage);
}

static void etpan_unref_storage(struct etpan_thread_manager * thread_manager,
    struct mailstorage * storage)
{
  if (storage == NULL)
    return;
  
  generic_mark_valid(thread_manager->storage_validity,
      storage, NULL);
}
#endif

#if 0
static struct etpan_msg_params *
get_msg_params(struct etpan_thread_data * data, struct mailfolder * folder)
{
  struct etpan_msg_params * msg_ref;
  chashdatum key;
  chashdatum value;
  int r;

  if (folder == NULL)
    return NULL;
  
  key.data = (char *) &folder;
  key.len = sizeof(folder);
  r = chash_get(data->folder_validity, &key, &value);
  if (r < 0)
    return NULL;
  
  msg_ref = (struct etpan_msg_params *) value.data;

  return msg_ref;
}

static int folder_is_valid(struct etpan_thread_data * data,
    struct mailfolder * folder);

static int etpan_ref_folder(struct etpan_thread_data * data,
    struct mailfolder * folder)
{
  struct etpan_msg_params * msg_ref;
  int r;

  if (folder == NULL)
    return NO_ERROR;

  msg_ref = get_msg_params(data, folder);
  if (msg_ref != NULL) {
    ETPAN_APP_DEBUG((data->manager->app, "BUG detected, twice ref of folder"));
    return NO_ERROR;
  }
  
  msg_ref = etpan_msg_params_new();
  if (msg_ref == NULL)
    return ERROR_MEMORY;
  
  r = generic_mark_valid(data->folder_validity, folder, msg_ref);
  if (r != NO_ERROR) {
    etpan_msg_params_free(msg_ref);
    return r;
  }
  
  return NO_ERROR;
}

static void etpan_ref_message(struct etpan_thread_data * data,
    struct mailfolder * folder,
    mailmessage * msg);

static void etpan_unref_message(struct etpan_thread_data * data,
    struct mailfolder * folder,
    mailmessage * msg);

static int etpan_message_get_ref_count(struct etpan_thread_data * data,
    struct mailfolder * folder,
    mailmessage * msg);

static void etpan_unref_folder(struct etpan_thread_data * data,
    struct mailfolder * folder)
{
  struct etpan_msg_params * msg_ref;
  chashiter * cur;
  int r;
  carray * msg_to_unref;
  unsigned int i;
  int count;

  if (folder == NULL)
    return;
  
  msg_ref = get_msg_params(data, folder);
  
  msg_to_unref = carray_new(chash_count(msg_ref->hash));
  if (msg_to_unref == NULL) {
    ETPAN_APP_LOG((data->manager->app,
                      "disconnect folder - not enough memory"));
    return;
  }
  
  /* collect messages */
  
  for(cur = chash_begin(msg_ref->hash) ; cur != NULL ;
      cur = chash_next(msg_ref->hash, cur)) {
    chashdatum key;
    mailmessage * msg;
    
    chash_key(cur, &key);
    memcpy(&msg, key.data, sizeof(msg));
    
    carray_add(msg_to_unref, msg, NULL);
  }
  
  /* unref them */
  
  count = 0;
  for(i = 0 ; i < carray_count(msg_to_unref) ; i ++) {
    mailmessage * msg;
    
    msg = carray_get(msg_to_unref, i);
    
    etpan_unref_message(data, folder, msg);
    
    r = etpan_message_get_ref_count(data, folder, msg);
    if (r != -1) {
      count ++;
    }
  }
  if (count != 0) {
    ETPAN_APP_DEBUG((data->manager->app,
                        "messages : %i", count));
  }
  
  carray_free(msg_to_unref);
  
  etpan_msg_params_free(msg_ref);
  
  generic_mark_valid(data->folder_validity, folder, NULL);
}

static int folder_is_valid(struct etpan_thread_data * data,
    struct mailfolder * folder)
{
  if (folder->session != NULL) {
    return 1;
  }
  else {
    debug_op_info(data->manager->app,
        "BUG detected, folder is not valid",
        -1, NULL, folder);
    return 0;
  }
}


static void etpan_ref_message(struct etpan_thread_data * data,
    struct mailfolder * folder,
    mailmessage * msg)
{
  struct etpan_msg_params * msg_ref;
  int r;
  
  if (folder == NULL)
    return;
  
  msg_ref = get_msg_params(data, folder);
  r = etpan_msg_param_ref_message(msg_ref, msg);
#if 0
  ETPAN_APP_DEBUG((data->manager->app, "message ref %i", r));
#endif
  if (r == 1) {
    chashdatum key;
    chashdatum value;
    
    key.data = (char *) &msg;
    key.len = sizeof(msg);
    value.data = (char *) folder;
    value.len = 0;
    
    r = chash_set(data->manager->message_hash, &key, &value, NULL);
  }
}

static void etpan_unref_message(struct etpan_thread_data * data,
    struct mailfolder * folder,
    mailmessage * msg)
{
  struct etpan_msg_params * msg_ref;
  int r;
  struct etpan_msg_param * param;
  
  if (folder == NULL)
    return;
  
  msg_ref = get_msg_params(data, folder);
  param = etpan_msg_get_param(msg_ref, msg);
  if (param == NULL) {
    /*
      BAD - message was not referenced,
      used in case there is no more memory
    */
    mailmessage_free(msg);
    return;
  }
  
  r = etpan_msg_param_unref_message(msg_ref, msg);
#if 0
  ETPAN_APP_DEBUG((data->manager->app, "message ref %i", r));
#endif
  if (r <= 0) {
    chashdatum key;
    chashdatum value;
    
    if (r < 0) {
      /*
        BAD - message was not referenced,
        used in case there is no more memory
      */
    }
    
    key.data = (char *) &msg;
    key.len = sizeof(msg);
    
    r = chash_delete(data->manager->message_hash, &key, NULL);
    
    etpan_msg_params_remove(msg_ref, msg);
    mailmessage_free(msg);
  }
}


static int etpan_message_get_ref_count(struct etpan_thread_data * data,
    struct mailfolder * folder,
    mailmessage * msg)
{
  struct etpan_msg_params * msg_ref;
  struct etpan_msg_param * param;
  
  if (folder == NULL)
    return -1;
  
  msg_ref = get_msg_params(data, folder);
  param = etpan_msg_get_param(msg_ref, msg);
  if (param == NULL) {
    return -1;
  }
  
  return etpan_msg_param_message_get_ref_count(msg_ref, msg);
}


static int message_is_valid(struct etpan_thread_data * data,
    struct mailfolder * folder,
    mailmessage * msg)
{
  struct etpan_msg_params * msg_ref;
  chashdatum key;
  chashdatum value;
  int r;
  
  if (folder == NULL)
    return 1;
  
  key.data = (char *) &folder;
  key.len = sizeof(folder);
  r = chash_get(data->folder_validity, &key, &value);
  if (r < 0)
    goto not_valid;
  
  msg_ref = (struct etpan_msg_params *) value.data;
  
  if (etpan_msg_get_param(msg_ref, msg) == NULL)
    goto not_valid;
  
  return 1;
  
 not_valid:
  debug_op_info(data->manager->app,
        "BUG detected, message is not valid",
        -1, NULL, folder);
  return 0;
}

static int common_get_bodystructure(struct etpan_thread_data * data,
    struct mailfolder * folder, mailmessage * msg);

static int etpan_ref_msg_mime(struct etpan_thread_data * data,
    struct mailfolder * folder,
    mailmessage * msg)
{
  struct etpan_msg_params * msg_ref;
  int r;
  int count;
  
  if (folder == NULL)
    return NO_ERROR;

  etpan_ref_message(data, folder, msg);
  
  msg_ref = get_msg_params(data, folder);
  r = etpan_msg_param_ref_mime(msg_ref, msg);
  
  count = etpan_message_get_ref_count(data, folder, msg);
  ETPAN_APP_DEBUG((data->manager->app, "ref msg mime %i %i", r, count));
  
  if (r == 1) {
    r = common_get_bodystructure(data, folder, msg);
    if (r != NO_ERROR) {
      etpan_msg_param_unref_mime(msg_ref, msg);
      return r;
    }
  }
  
  return NO_ERROR;
}

static void recursive_unref_mime(struct etpan_thread_data * data,
    struct mailfolder * folder,
    mailmessage * msg,
    struct mailmime * mime);

static void etpan_unref_msg_mime(struct etpan_thread_data * data,
    struct mailfolder * folder,
    mailmessage * msg)
{
  struct etpan_msg_params * msg_ref;
  int r;
  int count;
  
  if (folder == NULL)
    return;
  
  msg_ref = get_msg_params(data, folder);
  r = etpan_msg_param_unref_mime(msg_ref, msg);
  count = etpan_message_get_ref_count(data, folder, msg);
  
  ETPAN_APP_DEBUG((data->manager->app, "ref msg mime %i %i", r, count));
  if (r == 0) {
    recursive_unref_mime(data, folder, msg, msg->mime);
    etpan_msg_flush(data->manager->app, folder, msg);
  }
  
  etpan_unref_message(data, folder, msg);
}

static int etpan_ref_mime(struct etpan_thread_data * data,
    struct mailfolder * folder,
    mailmessage * msg,
    struct mailmime * mime)
{
  if (folder == NULL)
    return ERROR_INVAL;
  
  return generic_mark_valid(data->mime_validity, mime, msg);
}

static void etpan_unref_mime(struct etpan_thread_data * data,
    struct mailfolder * folder,
    mailmessage * msg,
    struct mailmime * mime)
{
  if (folder == NULL)
    return;
  
  generic_mark_valid(data->mime_validity, mime, NULL);
}

static int mime_is_valid(struct etpan_thread_data * data,
    struct mailfolder * folder,
    mailmessage * msg,
    struct mailmime * mime)
{
  int value;
  
  if (folder == NULL)
    return 1;
  
  value = generic_is_valid(data->mime_validity, mime);
  if (!value) {
    debug_op_info(data->manager->app,
        "BUG detected, MIME section is not valid",
        -1, NULL, folder);
  }
  return value;
}
#endif

/*
  FOLDER_CHECK
  arg : storage, folder
  result : 0 if ok, != 0 on error
*/

/*
  FOLDER_STATUS
  arg : storage, folder
  result : 0 if ok, != 0 on error
*/


/*
  FOLDER_GET_MSG_LIST
  arg : storage, folder
  result : struct { env_tree env_list msg_params node_params }
*/


/*
  FOLDER_FREE_MSG_LIST
  arg : struct { env_tree env_list msg_params node_params }
*/


/*
  MESSAGE_GET_BODYSTRUCTURE
  arg : storage, folder, message
  result : mime
*/


/*
  MESSAGE_FETCH
  arg : storage, folder, message
  result : struct { int result ; char * content; }
            0 if ok, != 0 on error
*/

/*
  MESSAGE_FETCH_HEADER
  arg : storage, folder, message
  result : struct { int result ; char * content; }
            0 if ok, != 0 on error
*/

/*  
  MESSAGE_FETCH_SECTION,
  arg : storage, folder, message, mime
  result : struct { int result ; char * content; }
            0 if ok, != 0 on error
*/

/*
  MESSAGE_FETCH_SECTION_MIME,
  arg : storage, folder, message , mime
  result : struct { int result ; char * result; }
            0 if ok, != 0 on error
*/

/*
  MESSAGE_FETCH_SECTION_HEADER,
  arg : storage, folder, message, mime
  result : struct { int result ; char * content; }
            0 if ok, != 0 on error
*/


/*
  FOLDER_APPEND_MESSAGE
  sends a message
  arg is a filename
  file is removed at the end
            0 if ok, != 0 on error
*/

/*
  MESSAGE_SEND
  sends a message
  arg is a filename
  file is removed at the end
            0 if ok, != 0 on error
*/

/*
  MESSAGE_RENDER_MIME
  render a message, given a message and a mime part.
  result : struct { char * filename ; FILE * f ; carray * attr; }
            0 if ok, != 0 on error
*/

/*
  MESSAGE_RENDER_REPLY_MIME
  render a replied message, given a message, a mime part
  result : struct { char * filename ; FILE * f ; carray * attr; }
            0 if ok, != 0 on error
*/

/*
  MESSAGE_RENDER_FORWARD
  render a forwarded message, given a message, a mime part
  result : struct { char * filename ; FILE * f ; carray * attr; }
            0 if ok, != 0 on error
*/

/*
  MESSAGE_RENDER_FORWARD_AS_ATTACHMENT
  render a replied message, given a message, a mime part
  result : struct { char * filename ; FILE * f ; carray * attr; }
            0 if ok, != 0 on error
*/


/* ******************************************************************* */
/* begin of thread operation */

static int folder_check(struct etpan_thread_op * op)
{
  struct mailfolder * folder;
  int r;

  folder = op->data.mailaccess.folder;
  
#if 0
  if (!folder_is_valid(op->thread_data, folder))
    return ERROR_INVAL;
#endif

#if 0
  r = mailsession_check_folder(folder->fld_session);
#endif
  r = mailfolder_check(folder);
  if (r == MAIL_ERROR_STREAM) {
    log_op_info(op->thread_data->manager->app, "stream error",
        op->cmd, NULL, folder);
    
    return ERROR_STREAM;
  }
  else if (r != MAIL_NO_ERROR) {
    log_op_info(op->thread_data->manager->app, "failed",
        op->cmd, NULL, folder);
    
    return ERROR_CHECK;
  }
  
  return NO_ERROR;
}

static void folder_status_clean_result(struct etpan_thread_op * op)
{
  free(op->result);
  op->result = NULL;
}

static int folder_status(struct etpan_thread_op * op)
{
  struct mailfolder * folder;
  struct etpan_folder_status_result * result;
  int r;
  uint32_t number;
  uint32_t recent;
  uint32_t unseen;

  folder = op->data.mailaccess.folder;

#if 0  
  if (!folder_is_valid(op->thread_data, folder))
    return ERROR_INVAL;
#endif
  
#if 0
  r = mailsession_status_folder(folder->fld_session, folder->fld_pathname,
      &number, &recent, &unseen);
#endif
  r = mailfolder_status(folder, &number, &recent, &unseen);
  if (r == MAIL_ERROR_STREAM) {
    log_op_info(op->thread_data->manager->app, "stream error",
        op->cmd, NULL, folder);
    
    return ERROR_STREAM;
  }
  else if (r != MAIL_NO_ERROR) {
    log_op_info(op->thread_data->manager->app, "failed",
        op->cmd, NULL, folder);
    ETPAN_APP_DEBUG((op->thread_data->manager->app, "error code : %i - %s",
                        r, maildriver_strerror(r)));
    return ERROR_STATUS;
  }
  
  result = malloc(sizeof(* result));
  if (result == NULL) {
    log_op_info(op->thread_data->manager->app,
        "not enough memory", op->cmd, NULL, folder);
    return ERROR_MEMORY;
  }
  
  result->number = number;
  result->recent = recent;
  result->unseen = unseen;
  
  op->result = result;
  
  return NO_ERROR;
}


static int folder_expunge(struct etpan_thread_op * op)
{
  struct mailfolder * folder;
  int r;

  folder = op->data.mailaccess.folder;

#if 0  
  if (!folder_is_valid(op->thread_data, folder))
    return ERROR_INVAL;
#endif
  
#if 0
  r = mailsession_expunge_folder(folder->fld_session);
#endif
  r = mailfolder_expunge(folder);
  if (r == MAIL_ERROR_STREAM) {
    log_op_info(op->thread_data->manager->app, "stream error",
        op->cmd, NULL, folder);
    
    return ERROR_STREAM;
  }
  else if (r != MAIL_NO_ERROR) {
    log_op_info(op->thread_data->manager->app, "failed",
        op->cmd, NULL, folder);
    return ERROR_EXPUNGE;
  }
  
  return NO_ERROR;
}



static pthread_mutex_t sort_lock = PTHREAD_MUTEX_INITIALIZER;
static struct etpan_node_msg_params * node_params_comp = NULL;

static int score_comp(struct mailmessage_tree ** ptree1,
    struct mailmessage_tree ** ptree2)
{
  int32_t score1;
  int32_t score2;

  score1 = etpan_node_msg_get_score(node_params_comp, * ptree1);
  score2 = etpan_node_msg_get_score(node_params_comp, * ptree2);

  if (score1 == score2)
    return mailthread_tree_timecomp(ptree1, ptree2);
  else
    return score1 - score2;
}


static void folder_get_msg_list_clean_arg(struct etpan_thread_op * op)
{
  struct etpan_folder_get_msg_list_arg * arg;
  
  arg = op->arg;
  
  if (arg != NULL) {
    chash_free(arg->node_params);
    free(arg);
  }
  op->arg = NULL;
}

static void folder_get_msg_list_clean_result(struct etpan_thread_op * op)
{
  struct etpan_folder_get_msg_list_result * result;
  
  result = op->result;

  etpan_node_msg_params_free(result->node_params);
  mailmessage_tree_free_recursive(result->env_tree);
  
  free(result);
  
  op->result = NULL;
}

#define MAX_OUTPUT 80

/*
  synchronize_msg_list()
  
  keep old env_list and updates it with new_env_list
*/

#if 0
static int
synchronize_msg_list(struct etpan_thread_op * op,
    struct mailmessage_list * new_env_list)
{
  chash * msg_hash;
  uint32_t i;
  int r;
  int res;
  chashiter * cur;
  struct etpan_msg_params * msg_ref;
#if 0
  carray * msg_to_unref;
#endif
  
  msg_ref = get_msg_params(op->thread_data, op->folder);
  
  msg_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYVALUE);
  if (msg_hash == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  /* hash new list, if elements does not exist in new list, remove them */
  
  for(i = 0 ; i < carray_count(new_env_list->msg_tab) ; i ++) {
    mailmessage * msg;
    chashdatum key;
    chashdatum value;
    
    msg = carray_get(new_env_list->msg_tab, i);
    
    if (msg->uid == NULL)
      continue;
    
    key.data = msg->uid;
    key.len = strlen(msg->uid);
    value.data = (char *) &i;
    value.len = sizeof(i);
    
    r = chash_set(msg_hash, &key, &value, NULL);
    if (r < 0) {
      res = ERROR_MEMORY;
      goto free_hash;
    }
  }

#if 0  
  msg_to_unref = carray_new(chash_count(msg_ref->hash));
  if (msg_to_unref == NULL) {
    ETPAN_APP_LOG((op->thread_data->manager->app,
                      "get message list - not enough memory"));
    res = ERROR_MEMORY;
    goto free_hash;
  }
#endif
  
  for(cur = chash_begin(msg_ref->hash) ; cur != NULL ;
      cur = chash_next(msg_ref->hash, cur)) {
    mailmessage * msg;
    chashdatum key;
    chashdatum value;
#if 0
    int expired_message;
#endif
    
    chash_key(cur, &key);
    memcpy(&msg, key.data, sizeof(msg));
    
#if 0
    expired_message = 0;
#endif

    if ((msg->uid == NULL) || (msg->fields == NULL)) {
#if 0
      expired_message = 1;
#endif
    }
    else {
      key.data = msg->uid;
      key.len = strlen(msg->uid);
      
      r = chash_get(msg_hash, &key, &value);
      if (r < 0) {
#if 0
        expired_message = 1;
#endif
      }
      else {
        mailmessage * new_msg;
        
        i = * (uint32_t *) value.data;

        chash_delete(msg_hash, &key, NULL);

        new_msg = carray_get(new_env_list->msg_tab, i);
        /* modify index of old message */
        msg->index = new_msg->index;
        mailmessage_free(new_msg);

        /* replace new message with old */
        carray_set(new_env_list->msg_tab, i, msg);
      }
    }
#if 0
    if (expired_message) {
      /* this always succeed because we preallocated the table */
      carray_add(msg_to_unref, msg, NULL);
    }
#endif
  }
  
  /* elements that remains in msg_hash are new messages */
  ETPAN_APP_DEBUG((op->thread_data->manager->app,
                    "new messages %i", chash_count(msg_hash)));
  
#if 0
  for(i = 0 ; i < carray_count(msg_to_unref) ; i ++) {
    mailmessage * msg;
    
    msg = carray_get(msg_to_unref, i);

    etpan_unref_message(op->thread_data, op->folder, msg);
  }
  
  carray_free(msg_to_unref);
#endif
  
  chash_free(msg_hash);

  return NO_ERROR;

 free_hash:
  chash_free(msg_hash);
 err:
  return res;
}
#endif

/*
static int synchronize_node_params(struct etpan_node_msg_params * tree_params,
    struct etpan_msg_params * msg_params,
    chash * uid_node_params,
    struct mailmessage_tree * node)
*/
static int synchronize_node_params(struct etpan_node_msg_params * tree_params,
    chash * uid_node_params,
    struct mailmessage_tree * node)
{
  mailmessage * msg;
  int r;
  int res;
  unsigned int cur;
  
  msg = node->node_msg;
  if (msg != NULL) {
    if (msg->msg_uid != NULL) {
      chashdatum key;
      chashdatum value;
      struct etpan_node_msg_param * node_param;
      
      key.data = msg->msg_uid;
      key.len = strlen(msg->msg_uid);
      
      r = chash_get(uid_node_params, &key, &value);
      if (r == 0) {
        struct etpan_node_msg_param * new_node_param;
        
        node_param = value.data;
        
#if 0
        new_node_param = etpan_node_msg_param_new(msg_params, node);
#endif
        new_node_param = etpan_node_msg_param_new(NULL, node);
        if (new_node_param == NULL) {
          res = ERROR_MEMORY;
          goto err;
        }
        memcpy(new_node_param, node_param, sizeof(* node_param));

        key.data = &node;
        key.len = sizeof(node);
        value.data = new_node_param;
        value.len = 0;
        r = chash_set(tree_params->hash, &key, &value, NULL);
        if (r < 0) {
          etpan_node_msg_param_free(new_node_param);
          res = ERROR_MEMORY;
          goto err;
        }
      }
    }
  }
  
  for(cur = 0 ; cur < carray_count(node->node_children) ; cur ++) {
#if 0
    r = synchronize_node_params(tree_params, msg_params,
        uid_node_params, carray_get(node->children, cur));
#endif
    r = synchronize_node_params(tree_params,
        uid_node_params, carray_get(node->node_children, cur));
    if (r != NO_ERROR) {
      res = r;
      goto err;
    }
  }
  
  return NO_ERROR;
  
 err:
  return ERROR_MEMORY;
}

static int is_cancelled(struct etpan_thread_op * op);

#if 0
static int get_msg_list(struct etpan_thread_op * op,
    struct mailmessage_list ** result)
{
  int r;
  struct mailfolder * folder;
  int res;
  struct mailmessage_list * env_list;
  
  folder =op->folder;
  
  r = mailsession_get_messages_list(folder->session, &env_list);
  if (r != MAIL_NO_ERROR) {
    log_op_info(op->thread_data->manager->app, "failed",
        op->cmd, NULL, folder);
    res = ERROR_ENV_LIST;
    goto err;
  }
  
  * result = env_list;
  
  return NO_ERROR;
  
 err:
  return res;
}
#endif

/*
static void recursive_ref_nodes(struct etpan_thread_data * data,
    struct mailfolder * folder,
    struct mailmessage_tree * node)
*/
static void recursive_ref_nodes(struct etpan_thread_data * data,
    struct mailmessage_tree * node)
{
  unsigned int cur;
  
  if (node->node_msg != NULL) {
#if 0
    etpan_ref_message(data, folder, node->msg);
#endif
    libetpan_message_ref(data->manager->engine, node->node_msg);
  }
  
  for(cur = 0 ; cur < carray_count(node->node_children) ; cur ++) {
#if 0
    recursive_ref_nodes(data, folder, carray_get(node->children, cur));
#endif
    recursive_ref_nodes(data, carray_get(node->node_children, cur));
  }
}


/*
  folder_get_msg_list()
*/

static int folder_get_msg_list(struct etpan_thread_op * op)
{
  struct mailfolder * folder;
  struct mailmessage_list * env_list;
  struct mailmessage_list * lost_msg_list;
  struct mailmessage_tree * env_tree;
  struct etpan_node_msg_params * node_params;
#if 0
  unsigned int i;
#endif
  int res;
  int r;
  struct etpan_folder_get_msg_list_result * result;
  char output[MAX_OUTPUT];
  struct etpan_folder_get_msg_list_arg * arg;
  int update_requested;
  struct mailengine * engine;
  int thread_type;
  
  engine = op->thread_data->manager->engine;
  
  arg = op->arg;

  if (arg != NULL)
    update_requested = 1;
  else
    update_requested = 0;
  
  folder = op->data.mailaccess.folder;

#if 0  
  if (!folder_is_valid(op->thread_data, folder)) {
    res = ERROR_INVAL;
    goto err;
  }
#endif

  if (is_cancelled(op)) {
    res = ERROR_CANCELLED;
    goto err;
  }
  
  log_op_info(op->thread_data->manager->app,
      "get message list", -1, folder->fld_storage, folder);

  if ((strcasecmp(folder->fld_session->sess_driver->sess_name, "nntp") == 0) ||
      (strcasecmp(folder->fld_session->sess_driver->sess_name,
          "nntp-cached") == 0)) {
    uint32_t max;
    
    max = etpan_vfolder_get_max(op->thread_data->manager->app->config.vfolder_config, folder);
    
    etpan_nntp_set_max_articles(folder->fld_session, max);
  }

#if 0  
  r = get_msg_list(op, &env_list);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
#endif
  r = libetpan_folder_get_msg_list(engine, op->data.mailaccess.folder,
      &env_list, &lost_msg_list);
  if (r == MAIL_ERROR_STREAM) {
    log_op_info(op->thread_data->manager->app, "stream error",
        op->cmd, NULL, folder);
    
    res = ERROR_STREAM;
    goto err;
  }
  else if (r != MAIL_NO_ERROR) {
    res = ERROR_ENV_LIST;
    goto err;
  }
  
  if (is_cancelled(op)) {
#if 0
    libetpan_folder_free_msg_list(engine, op->data.mailaccess.folder,
        lost_msg_list);
    libetpan_folder_free_msg_list(engine, op->data.mailaccess.folder,
        env_list);
#endif
#if 0
    mailmessage_list_free(env_list);
#endif
    res = ERROR_CANCELLED;
    goto free_env_list;
  }
  
#if 0
  /* synchronize list if required */
  
  r = synchronize_msg_list(op, env_list);
  if (r != NO_ERROR) {
    log_op_info(op->thread_data->manager->app, "not enough memory",
        op->cmd, NULL, folder);
    res = r;
    goto free_env_list;
  }
#endif

  snprintf(output, sizeof(output),
      "get envelopes list - %i messages",
      carray_count(env_list->msg_tab));
  log_op_info(op->thread_data->manager->app, output, -1,
      folder->fld_storage, folder);
  
#if 0
  r = mailsession_get_envelopes_list(folder->session, env_list);
#endif
  r = libetpan_folder_fetch_env_list(engine, folder, env_list);
  if (r == MAIL_ERROR_STREAM) {
    log_op_info(op->thread_data->manager->app, "stream error",
        op->cmd, NULL, folder);
    
    res = ERROR_STREAM;
    goto free_env_list;
  }
  else if (r != MAIL_NO_ERROR) {
    log_op_info(op->thread_data->manager->app, "failed",
        op->cmd, NULL, folder);
    res = ERROR_ENV_LIST;
    goto free_env_list;
  }
  
  /* build message thread */
  
  if (is_cancelled(op)) {
    res = ERROR_CANCELLED;
    goto free_env_list;
  }
  
  snprintf(output, sizeof(output),
      "build message thread - %i messages",
      carray_count(env_list->msg_tab));
  log_op_info(op->thread_data->manager->app, output, -1,
      folder->fld_storage, folder);
  
  switch (op->thread_data->manager->app->config.global_config->thread_type) {
  case ETPAN_THREAD_TYPE_BY_REFERENCES:
    thread_type = MAIL_THREAD_REFERENCES;
    break;
  case ETPAN_THREAD_TYPE_BY_REFERENCES_NO_SUBJECT:
    thread_type = MAIL_THREAD_REFERENCES_NO_SUBJECT;
    break;
  case ETPAN_THREAD_TYPE_NONE:
  default:
    thread_type = MAIL_THREAD_NONE;
    break;
  }
  
  r = mail_build_thread(thread_type,
      op->thread_data->manager->app->config.global_config->message_charset,
      env_list, &env_tree, NULL);
  if (r != MAIL_NO_ERROR) {
    log_op_info(op->thread_data->manager->app, "not enough memory",
        op->cmd, NULL, folder);
    res = ERROR_MEMORY;
    goto free_env_list;
  }
  
  if (is_cancelled(op)) {
    res = ERROR_CANCELLED;
    goto free_env_tree;
  }
  
  /* references list */
  
#if 0
  msg_ref = get_msg_params(op->thread_data, op->folder);
  
  r = etpan_msg_params_add_list(msg_ref, env_list->msg_tab);
  if (r != NO_ERROR) {
    res = r;
    goto free_env_tree;
  }
#endif
  
#if 0
  recursive_ref_nodes(op->thread_data, folder, env_tree);
#endif
  
  node_params = etpan_node_msg_params_new();
  if (node_params == NULL) {
    log_op_info(op->thread_data->manager->app, "not enough memory",
        op->cmd, NULL, folder);
    res = ERROR_CONNECT;
    goto free_env_tree;
  }

  if (update_requested) {
#if 0
    r = synchronize_node_params(node_params, msg_ref,
        arg->node_params, env_tree);
#endif
    r = synchronize_node_params(node_params,
        arg->node_params, env_tree);
    if (r != NO_ERROR) {
      res = r;
      goto free_node_params;
    }
  }

#if 0  
  r = etpan_node_msg_params_add_recursive(node_params, msg_ref, env_tree);
#endif
  r = etpan_node_msg_params_add_recursive(node_params, NULL, env_tree);
  if (r != NO_ERROR) {
    log_op_info(op->thread_data->manager->app, "not enough memory",
        op->cmd, NULL, folder);
    res = ERROR_MEMORY;
    goto free_node_params;
  }
  
  if (is_cancelled(op)) {
    res = ERROR_CANCELLED;
    goto free_node_params;
  }
  
  snprintf(output, sizeof(output),
      "sort messages - %i messages",
      carray_count(env_list->msg_tab));
  log_op_info(op->thread_data->manager->app, output, -1,
      folder->fld_storage, folder);
  
  pthread_mutex_lock(&sort_lock);
  node_params_comp = node_params;
  r = mail_thread_sort(env_tree, score_comp, TRUE);
  pthread_mutex_unlock(&sort_lock);
  if (r != MAIL_NO_ERROR) {
    log_op_info(op->thread_data->manager->app, "not enough memory",
        op->cmd, NULL, folder);
    res = ERROR_MEMORY;
    goto free_node_params;
  }
  
  if (is_cancelled(op)) {
    res = ERROR_CANCELLED;
    goto free_node_params;
  }
  
  result = malloc(sizeof(* result));
  if (result == NULL) {
    log_op_info(op->thread_data->manager->app, "not enough memory",
        op->cmd, NULL, folder);
    res = ERROR_MEMORY;
    goto free_node_params;
  }
  
  recursive_ref_nodes(op->thread_data, env_tree);
  
  snprintf(output, sizeof(output),
      "completed - %i messages",
      carray_count(env_list->msg_tab));
  log_op_info(op->thread_data->manager->app, output, -1,
      folder->fld_storage, folder);
  
#if 0
  libetpan_folder_free_msg_list(engine, op->data.mailaccess.folder,
      lost_msg_list);
#endif
  carray_set_size(lost_msg_list->msg_tab, 0);
  mailmessage_list_free(lost_msg_list);
  libetpan_folder_free_msg_list(engine, op->data.mailaccess.folder, env_list);
  
  result->env_tree = env_tree;
  result->node_params = node_params;
  
  op->result = result;

#if 0
  carray_set_size(env_list->msg_tab, 0);
  mailmessage_list_free(env_list);
#endif
  
  return NO_ERROR;
  
 free_node_params:
  etpan_node_msg_params_free(node_params);
 free_env_tree:
  mailmessage_tree_free_recursive(env_tree);
 free_env_list:
#if 0
  for(i = 0 ; i < carray_count(env_list->msg_tab) ; i ++) {
    mailmessage * msg;
    int r;
    
    msg = carray_get(env_list->msg_tab, i);
    r = etpan_message_get_ref_count(op->thread_data, folder, msg);
    if (r <= 0)
      etpan_unref_message(op->thread_data, op->folder, msg);
  }
  carray_set_size(env_list->msg_tab, 0);
  mailmessage_list_free(env_list);
#endif
  carray_set_size(lost_msg_list->msg_tab, 0);
  libetpan_folder_free_msg_list(engine, op->data.mailaccess.folder,
      lost_msg_list);
  libetpan_folder_free_msg_list(engine, op->data.mailaccess.folder, env_list);
 err:
  return res;
}

static void folder_free_msg_list_clean_arg(struct etpan_thread_op * op)
{
  free(op->arg);
  op->arg = NULL;
}

/*
static void recursive_unref_nodes(struct etpan_thread_data * data,
    struct mailfolder * folder,
    struct mailmessage_tree * node)
*/
static void recursive_unref_nodes(struct etpan_thread_data * data,
    struct mailmessage_tree * node)
{
  unsigned int cur;
  
  if (node->node_msg != NULL) {
    libetpan_message_unref(data->manager->engine, node->node_msg);
#if 0
    etpan_unref_message(data, folder, node->msg);
#endif
  }
  
  for(cur = 0 ; cur < carray_count(node->node_children) ; cur ++) {
#if 0
    recursive_unref_nodes(data, folder, carray_get(node->children, cur));
#endif
    recursive_unref_nodes(data, carray_get(node->node_children, cur));
  }
}

static int folder_free_msg_list(struct etpan_thread_op * op)
{
  struct etpan_folder_free_msg_list_arg * arg;
  
  arg = op->arg;
  
  if (arg->node_params != NULL)
    etpan_node_msg_params_free(arg->node_params);
#if 0
  recursive_unref_nodes(op->thread_data, op->folder, arg->env_tree);
#endif
  recursive_unref_nodes(op->thread_data, arg->env_tree);
  mailmessage_tree_free_recursive(arg->env_tree);
  
  return NO_ERROR;
}

#if 0
static void recursive_unref_mime(struct etpan_thread_data * data,
    struct mailfolder * folder,
    mailmessage * msg,
    struct mailmime * mime)
{
  clistiter * cur;
  
  etpan_unref_mime(data, folder, msg, mime);
  switch (mime->type) {
  case MAILMIME_MULTIPLE:
    for(cur = clist_begin(mime->list) ; cur != NULL ; cur = cur->next) {
      recursive_unref_mime(data, folder, msg, cur->data);
    }
    break;
  case MAILMIME_MESSAGE:
    if (mime->msg_mime != NULL)
      recursive_unref_mime(data, folder, msg, mime->msg_mime);
    break;
  }
}

static int recursive_ref_mime(struct etpan_thread_data * data,
    struct mailfolder * folder,
    mailmessage * msg,
    struct mailmime * mime)
{
  clistiter * cur;
  int r;
  
  r = etpan_ref_mime(data, folder, msg, mime);
  if (r != NO_ERROR)
    return r;
  
  switch (mime->type) {
  case MAILMIME_MULTIPLE:
    for(cur = clist_begin(mime->list) ; cur != NULL ; cur = cur->next) {
      r = recursive_ref_mime(data, folder, msg, cur->data);
      if (r != NO_ERROR)
        return r;
    }
    break;
  case MAILMIME_MESSAGE:
    if (mime->msg_mime != NULL) {
      r = recursive_ref_mime(data, folder, msg, mime->msg_mime);
      if (r != NO_ERROR)
        return r;
    }
    break;
  }
  
  return NO_ERROR;
}
#endif

#if 0
static int common_get_bodystructure(struct etpan_thread_data * data,
    struct mailfolder * folder, mailmessage * msg)
{
  int r;
  struct mailmime * mime;

  r = etpan_msg_get_bodystructure(data->manager->app, folder, msg, &mime);
  if (r != MAIL_NO_ERROR) {
    log_op_info(data->manager->app, "failed",
        ETPAN_THREAD_MESSAGE_GET_BODYSTRUCTURE, NULL, folder);
    return ERROR_FETCH;
  }
  
  r = recursive_ref_mime(data, folder, msg, msg->mime);
  if (r != NO_ERROR) {
    recursive_unref_mime(data, folder, msg, msg->mime);
    etpan_msg_flush(data->manager->app, folder, msg);
    return r;
  }

  return NO_ERROR;
}
#endif

static int message_get_bodystructure(struct etpan_thread_op * op)
{
  struct mailfolder * folder;
  mailmessage * msg;
  int r;
  
  folder = op->data.mailaccess.folder;
  msg = op->data.mailaccess.msg;
  
  r = libetpan_message_mime_ref(op->thread_data->manager->engine, msg);
  if (r <= 0) {
    r = -r;
    if (r == MAIL_ERROR_STREAM)
      return ERROR_STREAM;
    else
      return ERROR_FETCH;
  }
  
#if 0  
  if (!message_is_valid(op->thread_data, folder, msg))
    return ERROR_INVAL;

  r = etpan_ref_msg_mime(op->thread_data, folder, msg);
  if (r != NO_ERROR) {
    log_op_info(op->thread_data->manager->app, "failed",
        op->cmd, NULL, folder);
    return r;
  }
#endif
  
  op->result = msg->msg_mime;

  return NO_ERROR;
}

static int message_flush(struct etpan_thread_op * op)
{
  mailmessage * msg;
  struct mailfolder * folder;
  
  folder = op->data.mailaccess.folder;
  msg = op->data.mailaccess.msg;

#if 0  
  if (!message_is_valid(op->thread_data, folder, msg))
    return ERROR_INVAL;
  
  if (msg->mime != NULL) {
    etpan_unref_msg_mime(op->thread_data, folder, msg);
  }
#endif
  libetpan_message_mime_unref(op->thread_data->manager->engine, msg);
  
  return NO_ERROR;
}

static int message_check(struct etpan_thread_op * op)
{
  mailmessage * msg;
  struct mailfolder * folder;
  
  folder = op->data.mailaccess.folder;
  msg = op->data.mailaccess.msg;
  
#if 0
  if (!message_is_valid(op->thread_data, folder, msg))
    return ERROR_INVAL;
#endif
  
  mailmessage_check(msg);
  
  return NO_ERROR;
}

static void msglist_check_clean_arg(struct etpan_thread_op * op)
{
  struct etpan_msglist_check_arg * arg;
  
  arg = op->arg;
  
  carray_free(arg->msglist);
  free(arg);
  op->arg = NULL;
}

static int msglist_check(struct etpan_thread_op * op)
{
  struct etpan_msglist_check_arg * arg;
  struct mailfolder * folder;
  unsigned int i;

  folder = op->data.mailaccess.folder;
  arg = op->arg;
  
  for(i = 0 ; i < carray_count(arg->msglist) ; i ++) {
    mailmessage * msg;
    
    msg = carray_get(arg->msglist, i);

#if 0    
    if (!message_is_valid(op->thread_data, folder, msg))
      return ERROR_INVAL;
#endif
    
    mailmessage_check(msg);
  }
  
  return NO_ERROR;
}


static void message_fetch_clean_result(struct etpan_thread_op * op)
{
  struct etpan_message_fetch_result * result;
  struct mailprivacy * privacy;
  
  result = op->result;
  
#if 0
  etpan_msg_fetch_result_free(op->thread_data->manager->app,
      op->folder, op->msg, result->content);
#endif
  
  privacy = libetpan_engine_get_privacy(op->thread_data->manager->engine);
  
  mailprivacy_msg_fetch_result_free(privacy, op->data.mailaccess.msg,
      result->content);
  op->result = NULL;
}

/*
static int message_fetch_generic(struct etpan_thread_op * op,
    int (* fetch)(struct etpan_app *,
        struct mailfolder *, mailmessage *, char **, size_t *))
*/
static int message_fetch_generic(struct etpan_thread_op * op,
    int (* fetch)(struct mailprivacy *,
        mailmessage *, char **, size_t *))
{
  char * content;
  size_t content_len;
  struct etpan_message_fetch_result * result;
  mailmessage * msg;
  struct mailfolder * folder;
  int r;
  struct mailprivacy * privacy;
  
  folder = op->data.mailaccess.folder;
  msg = op->data.mailaccess.msg;

  privacy = libetpan_engine_get_privacy(op->thread_data->manager->engine);

#if 0  
  if (!message_is_valid(op->thread_data, folder, msg))
    return ERROR_INVAL;
#endif

#if 0  
  r = fetch(op->thread_data->manager->app,
      folder, msg, &content, &content_len);
#endif
  r = fetch(privacy, msg, &content, &content_len);
  if (r == MAIL_ERROR_STREAM) {
    log_op_info(op->thread_data->manager->app, "stream error",
        op->cmd, NULL, folder);
    
    return ERROR_STREAM;
  }
  else if (r != MAIL_NO_ERROR) {
    log_op_info(op->thread_data->manager->app, "failed",
        op->cmd, NULL, folder);
    return ERROR_FETCH;
  }
  
  result = malloc(sizeof(* result));
  if (result == NULL) {
    log_op_info(op->thread_data->manager->app, "not enough memory",
        op->cmd, NULL, folder);
#if 0
    etpan_msg_fetch_result_free(op->thread_data->manager->app,
        folder, msg, content);
#endif
    mailprivacy_msg_fetch_result_free(privacy, msg, content);
    return ERROR_MEMORY;
  }
  
  result->content = content;
  result->len = content_len;
  
  op->result = result;
  
  return NO_ERROR;
}

static int message_fetch(struct etpan_thread_op * op)
{
#if 0
  return message_fetch_generic(op, etpan_msg_fetch);
#endif
  return message_fetch_generic(op, mailprivacy_msg_fetch);
}

static int message_fetch_header(struct etpan_thread_op * op)
{
#if 0
  return message_fetch_generic(op, etpan_msg_fetch_header);
#endif
  return message_fetch_generic(op, mailprivacy_msg_fetch_header);
}

/*
static int message_fetch_section_generic(struct etpan_thread_op * op,
    int (* fetch)(struct etpan_app *,
        struct mailfolder *, mailmessage *, struct mailmime *,
        char **, size_t *))
*/
static int message_fetch_section_generic(struct etpan_thread_op * op,
    int (* fetch)(struct mailprivacy *,
        mailmessage *, struct mailmime *,
        char **, size_t *))
{
  struct mailmime * mime;
  mailmessage * msg;
  struct mailfolder * folder;
  char * content;
  size_t content_len;
  struct etpan_message_fetch_result * result;
  int r;
  struct mailprivacy * privacy;

  folder = op->data.mailaccess.folder;
  msg = op->data.mailaccess.msg;
  mime = op->data.mailaccess.mime;

#if 0  
  if (!mime_is_valid(op->thread_data, folder, msg, mime)) {
    return ERROR_INVAL;
  }
#endif
  
  privacy = libetpan_engine_get_privacy(op->thread_data->manager->engine);
  
#if 0
  r = fetch(op->thread_data->manager->app,
      folder, msg, mime, &content, &content_len);
#endif
  r = fetch(privacy, msg, mime, &content, &content_len);
  if (r == MAIL_ERROR_STREAM) {
    log_op_info(op->thread_data->manager->app, "stream error",
        op->cmd, NULL, folder);
    
    return ERROR_STREAM;
  }
  else if (r != MAIL_NO_ERROR) {
    log_op_info(op->thread_data->manager->app, "failed",
        op->cmd, NULL, folder);
    return ERROR_FETCH;
  }
  
  result = malloc(sizeof(* result));
  if (result == NULL) {
    log_op_info(op->thread_data->manager->app, "not enough memory",
        op->cmd, NULL, folder);
#if 0
    etpan_msg_fetch_result_free(op->thread_data->manager->app,
        folder, msg, content);
#endif
    
    mailprivacy_msg_fetch_result_free(privacy, msg, content);
    return ERROR_MEMORY;
  }
  
  result->content = content;
  result->len = content_len;
  
  op->result = result;
  
  return NO_ERROR;
}


static int message_fetch_section(struct etpan_thread_op * op)
{
#if 0
  return message_fetch_section_generic(op, etpan_msg_fetch_section);
#endif
  return message_fetch_section_generic(op,
      mailprivacy_msg_fetch_section);
}

static int message_fetch_section_header(struct etpan_thread_op * op)
{
#if 0
  return message_fetch_section_generic(op, etpan_msg_fetch_section_header);
#endif
  return message_fetch_section_generic(op,
      mailprivacy_msg_fetch_section_header);
}

static int message_fetch_section_mime(struct etpan_thread_op * op)
{
#if 0
  return message_fetch_section_generic(op, etpan_msg_fetch_section_mime);
#endif
  return message_fetch_section_generic(op,
      mailprivacy_msg_fetch_section_mime);
}

static int message_fetch_result_free(struct etpan_thread_op * op)
{
  mailmessage * msg;
  char * content;
  struct mailprivacy * privacy;
  
  msg = op->data.mailaccess.msg;
  content = op->arg;

  privacy = libetpan_engine_get_privacy(op->thread_data->manager->engine);

#if 0  
  etpan_msg_fetch_result_free(op->thread_data->manager->app,
      op->folder, msg, content);
#endif
  mailprivacy_msg_fetch_result_free(privacy, msg, content);
  
  return NO_ERROR;
}



static void message_render_mime_clean_result(struct etpan_thread_op * op)
{
  struct etpan_message_render_result * result;
  
  result = op->result;
  
  fclose(result->f);
  if (result->attr != NULL)
    etpan_text_prop_array_free(result->attr);
  unlink(result->filename);
  free(result->filename);
  free(result);
  
  op->result = NULL;
}

/*
  message_render_mime
  
  WARNING, this will leave the bodystructure inside the message,
  it is not flushed because it can be used by some other etpan_subapp
  
  all references to MIME parts should be checked before flushing the
  message.
*/

static int message_render_mime(struct etpan_thread_op * op)
{
  struct mailmime * mime;
  mailmessage * msg;
  struct mailfolder * folder;
  int r;
  carray * attr;
#if 0
  struct etpan_account_info * account;
#endif
  int res;
  struct etpan_message_render_result * result;
  FILE * f;
  char filename[PATH_MAX];
  struct mailengine * engine;
  struct etpan_app * app;
  
  folder = op->data.mailaccess.folder;
  msg = op->data.mailaccess.msg;
  mime = op->data.mailaccess.mime;
  app = op->thread_data->manager->app;

#if 0  
  if (mime != NULL) {
    if (!mime_is_valid(op->thread_data, folder, msg, mime)) {
      res = ERROR_INVAL;
      goto err;
    }
  }
#endif

  engine = op->thread_data->manager->engine;
  
  r = libetpan_message_mime_ref(engine, msg);
  if (r <= 0) {
    log_op_info(app, "failed", op->cmd, NULL, folder);
    r = -r;
    if (r == MAIL_ERROR_STREAM)
      res = ERROR_STREAM;
    else
      res = ERROR_FETCH;
    goto err;
  }

#if 0
  /* if get_bodystructure was not called */
  if (mime == NULL) {
    if (!message_is_valid(op->thread_data, folder, msg)) {
      res = ERROR_INVAL;
      goto err;
    }
    
    r = etpan_ref_msg_mime(op->thread_data, folder, msg);
    if (r != NO_ERROR) {
      log_op_info(op->thread_data->manager->app, "failed",
          op->cmd, NULL, folder);
      res = r;
      goto err;
    }
    mime = msg->mime;
  }
#endif

  if (mime == NULL)
    mime = msg->msg_mime;

  f = etpan_get_tmp_file(filename, PATH_MAX);
  if (f == NULL) {
    log_op_info(app, "failed", op->cmd, NULL, folder);
    res = ERROR_FILE;
    goto unref_msg_mime;
  }

  attr = carray_new(16);
  if (attr == NULL) {
    log_op_info(app, "not enough memory", op->cmd, NULL, folder);
    res = ERROR_MEMORY;
    goto close;
  }

#if 0  
  account =
    etpan_vfolder_get_account(op->thread_data->manager->app->vfolder_config,
        op->thread_data->manager->app->account_config,
        folder);
#endif

  r = etpan_render_mime(f, attr, op->thread_data->manager->app,
      msg, mime, ETPAN_RENDER_FLAGS_MAIN_HEADER);
  if (r == ERROR_STREAM) {
    log_op_info(app, "stream error", op->cmd, NULL, folder);
    
    res = ERROR_STREAM;
    goto free_attr;
  }
  else if (r != NO_ERROR) {
    log_op_info(app, "failed", op->cmd, NULL, folder);
    res = r;
    goto free_attr;
  }

  result = malloc(sizeof(* result));
  if (result == NULL) {
    log_op_info(app, "not enough memory", op->cmd, NULL, folder);
    res = ERROR_MEMORY;
    goto free_attr;
  }

  result->filename = strdup(filename);
  if (result->filename == NULL) {
    log_op_info(app, "not enough memory", op->cmd, NULL, folder);
    free(result);
    res = ERROR_MEMORY;
    goto free_attr;
  }

  result->f = f;
  result->attr = attr;
  
  op->result = result;

#if 0
  if (op->mime == NULL)
    etpan_unref_msg_mime(op->thread_data, folder, msg);
#endif
  libetpan_message_mime_unref(engine, msg);
  
  return NO_ERROR;
  
 free_attr:
  free(attr);
 close:
  fclose(f);
  unlink(filename);
 unref_msg_mime:
  libetpan_message_mime_unref(engine, msg);
#if 0
  if (op->mime == NULL)
    etpan_unref_msg_mime(op->thread_data, folder, msg);
#endif
 err:
  return res;
}



static void message_render_reply_mime_clean_arg(struct etpan_thread_op * op)
{
  struct etpan_message_render_reply_arg * arg;
  
  arg = op->arg;
  
  if (arg->dest_addr != NULL)
    mailimf_address_list_free(arg->dest_addr);
  free(arg);
  op->arg = NULL;
}

static void
message_render_reply_mime_clean_result(struct etpan_thread_op * op)
{
  struct etpan_message_render_reply_result * result;
  
  result = op->result;
  
  etpan_message_mime_clear(result->mime);
  mailmime_free(result->mime);
  free(result);
  op->result = NULL;
}

#if 0

/*
  etpan_mime_data_replace()
  
  write a mime part to a file and change the reference of mailmime_data
  to the file.
*/

static int etpan_mime_data_replace(int encoding_type,
    struct mailmime_data * data)
{
  char filename[PATH_MAX];
  FILE * f;
  size_t written;
  char * dup_filename;
  int res;
  int r;
  int decoded;

  if (data->type != MAILMIME_DATA_TEXT) {
    res = NO_ERROR;
    goto err;
  }
  
  f = etpan_get_mime_tmp_file(filename, sizeof(filename));
  if (f == NULL) {
    res = ERROR_FILE;
    goto err;
  }
  
  decoded = 0;
  if (encoding_type != -1) {
    char * content;
    size_t content_len;
    size_t cur_token;
    
    cur_token = 0;
    r = mailmime_part_parse(data->data, data->length,
        &cur_token, encoding_type, &content, &content_len);
    
    if (r == MAILIMF_NO_ERROR) {
      /* write decoded */
      written = fwrite(content, 1, content_len, f);
      if (written != content_len) {
        fclose(f);
        unlink(filename);
        res = ERROR_FILE;
        goto err;
      }
      mmap_string_unref(content);
      
      decoded = 1;
      data->encoded = 0;
    }
  }
  
  if (!decoded) {
    written = fwrite(data->data, 1, data->length, f);
    if (written != data->length) {
      fclose(f);
      unlink(filename);
      res = ERROR_FILE;
      goto err;
    }
  }
  
  fclose(f);
  
  dup_filename = strdup(filename);
  if (dup_filename == NULL) {
    unlink(filename);
    res = ERROR_MEMORY;
    goto err;
  }
  
  data->type = MAILMIME_DATA_FILE;
  data->data = NULL;
  data->length = 0;
  data->filename = dup_filename;
  
  return NO_ERROR;
  
 err:
  return res;
}


/*
  recursive_replace_single_parts()
  
  write all parts of the given mime part to file.
*/

static int recursive_replace_single_parts(struct mailmime * mime)
{
  int r;
  int res;
  clistiter * cur;
  
  mime->mime_start = NULL;
  
  switch(mime->type) {
  case MAILMIME_SINGLE:
    if (mime->body != NULL) {
      int encoding_type;
      struct mailmime_single_fields single_fields;
      
      mailmime_single_fields_init(&single_fields, mime->mime_fields,
          mime->content_type);
      
      if (single_fields.encoding != NULL)
        encoding_type = single_fields.encoding->type;
      else
        encoding_type = -1;
      
      r = etpan_mime_data_replace(encoding_type, mime->body);
      if (r != NO_ERROR) {
        res = r;
        goto err;
      }
    }
    break;
    
  case MAILMIME_MULTIPLE:
    if (mime->preamble != NULL) {
      r = etpan_mime_data_replace(-1, mime->preamble);
      if (r != NO_ERROR) {
        res = r;
        goto err;
      }
    }
    
    if (mime->epilogue != NULL) {
      r = etpan_mime_data_replace(-1, mime->epilogue);
      if (r != NO_ERROR) {
        res = r;
        goto err;
      }
    }
    
    for(cur = clist_begin(mime->list) ; cur != NULL ; cur = cur->next) {
      struct mailmime * child;
      
      child = cur->data;
      
      r = recursive_replace_single_parts(child);
      if (r != NO_ERROR) {
        res = r;
        goto err;
      }
    }
    
    break;
    
  case MAILMIME_MESSAGE:
    if (mime->msg_mime != NULL) {
      r = recursive_replace_single_parts(mime->msg_mime);
      if (r != NO_ERROR) {
        res = r;
        goto err;
      }
    }
    
    if (mime->msg_content != NULL) {
      r = etpan_mime_data_replace(-1, mime->msg_content);
      if (r != NO_ERROR) {
        res = r;
        goto err;
      }
    }
    
    break;
  }
  
  return NO_ERROR;
  
 err:
  return res;
}
#endif

/*
  get_message_mime_copy()

  fetch all parts
*/

static int get_message_mime_copy(struct etpan_thread_op * op,
    struct mailmime ** pattached_mime)
{
  char * content;
  size_t content_len;
  struct mailmime * attached_mime;
  int r;
  int res;
  struct mailfolder * folder;
  mailmessage * msg;
  struct mailmime * mime;
  struct mailengine * engine;
  struct mailprivacy * privacy;

  folder = op->data.mailaccess.folder;
  msg = op->data.mailaccess.msg;
  mime = op->data.mailaccess.mime;
  
  /*
    use message data driver, get bodystructure and
    convert all the data part in MAILMIME_SINGLE to files.
  */

  engine = op->thread_data->manager->engine;
  privacy = libetpan_engine_get_privacy(engine);

  r = libetpan_message_mime_ref(engine, msg);
  if (r <= 0) {
    r = -r;
    if (r == MAIL_ERROR_STREAM)
      res = ERROR_STREAM;
    else
      res = ERROR_FETCH;
    goto err;
  }
  if (mime ==  NULL)
    mime = msg->msg_mime;
#if 0
  if (mime == NULL) {
    r = etpan_ref_msg_mime(op->thread_data, folder, msg);
    if (r != NO_ERROR) {
      res = r;
      goto err;
    }
    mime = msg->mime;
  }
#endif

#if 0  
  r = etpan_msg_fetch_section(op->thread_data->manager->app,
      folder, msg, mime, &content, &content_len);
#endif
  r = mailprivacy_msg_fetch_section(privacy,
      msg, mime, &content, &content_len);
  if (r == MAIL_ERROR_STREAM) {
    log_op_info(op->thread_data->manager->app, "stream error",
        op->cmd, NULL, folder);
    
    res = ERROR_STREAM;
    goto unref_msg_mime;
  }
  else if (r != MAIL_NO_ERROR) {
    res = ERROR_FETCH;
    goto unref_msg_mime;
  }

#if 0  
  r = etpan_get_mime(op->thread_data->manager->app, folder,
      content, content_len, 0, &attached_mime);
#endif
  r = mailprivacy_get_mime(privacy, 0, content, content_len, &attached_mime);
  
#if 0
  etpan_msg_fetch_result_free(op->thread_data->manager->app,
      folder, msg, content);
#endif
  mailprivacy_msg_fetch_result_free(privacy,
      msg, content);
  
  if (r != MAIL_NO_ERROR) {
    res = ERROR_MEMORY;
    goto unref_msg_mime;
  }
  
#if 0
  if (op->mime == NULL)
    etpan_unref_msg_mime(op->thread_data, folder, msg);
#endif
  libetpan_message_mime_unref(engine, msg);
  
  * pattached_mime = attached_mime;
  
  return NO_ERROR;
  
 unref_msg_mime:
#if 0
  if (op->mime == NULL)
    etpan_unref_msg_mime(op->thread_data, folder, msg);
#endif
  libetpan_message_mime_unref(engine, msg);
 err:
  return res;
}


static int attach_file(struct etpan_app * app,
    struct mailmime * mime,
    struct mailmime * attach_mime, char * attach_filename)
{
  struct mailmime_single_fields single_fields;
  int encoding_type;
  char * filename;
  char * description;
  struct etpan_mime_info * mime_info;
  struct mailmime_content * content;
  struct mailmime * new_part;
  struct mailmime_disposition * dsp;
  struct mailmime_mechanism * encoding;
  struct mailmime_fields * mime_fields;
  int r;
  int res;
  
  mailmime_single_fields_init(&single_fields, attach_mime->mm_mime_fields,
      attach_mime->mm_content_type);
  
  if (single_fields.fld_encoding != NULL)
    encoding_type = single_fields.fld_encoding->enc_type;
  else 
    encoding_type = MAILMIME_MECHANISM_8BIT;
  
  filename = single_fields.fld_disposition_filename;
  if (filename == NULL)
    filename = single_fields.fld_content_name;
  
  description = single_fields.fld_description;
  
  /* Content-Disposition */
  if (filename != NULL) {
    filename = strdup(filename);
    if (filename == NULL) {
      res = ERROR_MEMORY;
      goto err;
    }
    
    dsp =
      mailmime_disposition_new_with_data(MAILMIME_DISPOSITION_TYPE_ATTACHMENT,
          filename, NULL, NULL, NULL, (size_t) -1);
    if (dsp == NULL) {
      free(filename);
      res = ERROR_MEMORY;
      goto err;
    }
  }
  else {
    dsp = NULL;
  }
  
  /* Content-Transfer-Encoding */
  encoding = mailmime_mechanism_new(encoding_type, NULL);
  if (encoding == NULL) {
    res = ERROR_MEMORY;
    goto free_dsp;
  }
  
  /* Content-Description */
  if (description != NULL) {
    description = strdup(description);
    if (description == NULL) {
      res = ERROR_MEMORY;
      goto free_encoding;
    }
  }
  
  mime_fields = mailmime_fields_new_with_data(encoding, NULL,
      description, dsp, NULL);
  if (mime_fields == NULL) {
    res = ERROR_MEMORY;
    goto free_description;
  }
  
  /* Content-Type */
  content = NULL;
  if (filename != NULL) {
    mime_info = etpan_get_mime_by_name(app->config.mime_config, filename);
    if (mime_info != NULL) {
      content = mailmime_content_new_with_str(mime_info->type);
      if (content == NULL) {
        res = ERROR_MEMORY;
        goto free_fields;
      }
    }
  }
  
  if (content == NULL) {
    if (attach_mime->mm_content_type != NULL) {
      char * content_str;
      
      /* TODO - could copy the content-type without generating a string */
           
      content_str = etpan_get_content_type_str(attach_mime->mm_content_type);
      if (content_str != NULL) {
        content = mailmime_content_new_with_str(content_str);
        free(content_str);
        if (content == NULL) {
          res = ERROR_MEMORY;
          goto free_fields;
        }
      }
    }
  }
  
  /* MIME part */
  new_part = mailmime_new_empty(content, mime_fields);
  if (new_part == NULL) {
    res = ERROR_MEMORY;
    goto free_content;
  }
  
  /* set file as body of MIME part */
  attach_filename = strdup(attach_filename);
  if (attach_filename == NULL) {
    res = ERROR_MEMORY;
    goto free_mime;
  }
  r = mailmime_set_body_file(new_part, attach_filename);
  if (r != MAILIMF_NO_ERROR) {
    free(attach_filename);
    res = ERROR_MEMORY;
    goto free_mime;
  }
  
  r = mailmime_smart_add_part(mime, new_part);
  if (r != MAILIMF_NO_ERROR) {
    res = ERROR_MEMORY;
    goto free_mime;
  }
  
  return NO_ERROR;
  
 free_mime:
  mailmime_free(new_part);
  goto err;
 free_content:
  mailmime_content_free(content);
 free_fields:
  mailmime_fields_free(mime_fields);
  goto err;
 free_description:
  if (description != NULL)
    free(description);
 free_encoding:
  mailmime_mechanism_free(encoding);
 free_dsp:
  if (dsp != NULL)
    mailmime_disposition_free(dsp);
 err:
  return res;
}

static int attach_content(struct etpan_app * app,
    struct mailmime * mime,
    struct mailmime * attach_mime,
    char * content, size_t content_len)
{
  FILE * f;
  char filename[PATH_MAX];
  size_t written;
  int res;
  int r;
  
  /* write file */
  f = etpan_get_mime_tmp_file(filename, sizeof(filename));
  if (f == NULL) {
    res = ERROR_FILE;
    goto err;
  }
  
  written = fwrite(content, 1, content_len, f);
  if (written != content_len) {
    res = ERROR_FILE;
    goto close;
  }
  
  fclose(f);
  
  ETPAN_APP_DEBUG((app, "attach file %s", filename));
  r = attach_file(app, mime, attach_mime, filename);
  if (r != NO_ERROR) {
    res = r;
    unlink(filename);
    goto err;
  }
  
  return NO_ERROR;
  
 close:
  fclose(f);
 err:
  return res;
}

/*
static int recursive_attach_parts(struct etpan_app * app,
    struct mailmime * mime,
    struct mailfolder * attach_folder,
    mailmessage * attach_msg,
    struct mailmime * attach_mime)
*/
static int recursive_attach_parts(struct etpan_app * app,
    struct mailprivacy * privacy,
    struct mailmime * mime,
    mailmessage * attach_msg,
    struct mailmime * attach_mime)
{
  int r;
  int res;
  clistiter * cur;
  char * content;
  size_t content_len;
  size_t cur_token;
  char * decoded_content;
  size_t decoded_content_len;
  struct mailmime_single_fields single_fields;
  int enc_type;
  
  switch(attach_mime->mm_type) {
  case MAILMIME_SINGLE:
    if (etpan_mime_is_text(attach_mime))
      return NO_ERROR;

#if 0    
    r = etpan_msg_fetch_section(app,
        attach_folder, attach_msg, attach_mime,
        &content, &content_len);
#endif
    r = mailprivacy_msg_fetch_section(privacy, attach_msg, attach_mime,
        &content, &content_len);
    if (r == MAIL_ERROR_STREAM) {
      res = ERROR_STREAM;
      goto err;
    }
    else if (r != MAIL_NO_ERROR) {
      res = ERROR_FETCH;
      goto err;
    }
    
    mailmime_single_fields_init(&single_fields,
        attach_mime->mm_mime_fields,
        attach_mime->mm_content_type);
   
    cur_token = 0;
    if (single_fields.fld_encoding == NULL)
      enc_type = MAILMIME_MECHANISM_8BIT;
    else
      enc_type = single_fields.fld_encoding->enc_type;
    
    r = mailmime_part_parse(content, content_len,
        &cur_token, enc_type,
        &decoded_content, &decoded_content_len);
    
    mailprivacy_msg_fetch_result_free(privacy, attach_msg, content);
    
    if (r != NO_ERROR) {
      res = r;
      goto err;
    }
    
    r = attach_content(app, mime, attach_mime,
        decoded_content, decoded_content_len);
    if (r != NO_ERROR) {
      mmap_string_unref(decoded_content);
      res = r;
      goto err;
    }
    
#if 0    
    etpan_msg_fetch_result_free(app, attach_folder, attach_msg, content);
#endif
    mmap_string_unref(decoded_content);
    break;
    
  case MAILMIME_MULTIPLE:
    for(cur = clist_begin(attach_mime->mm_data.mm_multipart.mm_mp_list) ;
        cur != NULL ; cur = clist_next(cur)) {
      struct mailmime * child;
      
      child = cur->data;
      
#if 0
      r = recursive_attach_parts(app, mime,
          attach_folder, attach_msg, child);
#endif
      r = recursive_attach_parts(app, privacy, mime,
          attach_msg, child);
      if (r != NO_ERROR) {
        res = r;
        goto err;
      }
    }
    
    break;
    
  case MAILMIME_MESSAGE:
    
    if (attach_mime->mm_data.mm_message.mm_msg_mime != NULL) {
#if 0
      r = recursive_attach_parts(app, mime,
          attach_folder, attach_msg, attach_mime->msg_mime);
#endif
      r = recursive_attach_parts(app, privacy, mime,
          attach_msg, attach_mime->mm_data.mm_message.mm_msg_mime);
      if (r != NO_ERROR) {
        res = r;
        goto err;
      }
    }
    break;
  }
  
  return NO_ERROR;
  
 err:
  return res;
}


static int
get_message_mime_non_text_plain(struct etpan_thread_op * op,
    struct mailmime * dest_mime)
{
  struct mailmime * attached_mime;
  int r;
  int res;
#if 0
  struct mailfolder * folder;
#endif
  mailmessage * msg;
  struct etpan_app * app;
  struct mailengine * engine;
  struct mailprivacy * privacy;
  
  app = op->thread_data->manager->app;
#if 0
  folder = op->folder;
#endif
  msg = op->data.mailaccess.msg;
  attached_mime = op->data.mailaccess.mime;
  
  engine = op->thread_data->manager->engine;
  privacy = libetpan_engine_get_privacy(engine);
  
#if 0
  if (attached_mime == NULL) {
    r = etpan_ref_msg_mime(op->thread_data, folder, msg);
    if (r != NO_ERROR) {
      res = r;
      goto err;
    }
    attached_mime = msg->mime;
  }
#endif
  
  r = libetpan_message_mime_ref(engine, msg);
  if (r <= 0) {
    r = -r;
    if (r == MAIL_ERROR_STREAM)
      res = ERROR_STREAM;
    else
      res = ERROR_FETCH;
    goto err;
  }
  if (attached_mime == NULL)
    attached_mime = msg->msg_mime;
  
#if 0  
  r = recursive_attach_parts(app, dest_mime, folder, msg, attached_mime);
#endif
  r = recursive_attach_parts(app, privacy, dest_mime, msg, attached_mime);
  if (r != NO_ERROR) {
    res = r;
    goto unref_msg_mime;
  }
  
#if 0
  if (op->mime == NULL)
    etpan_unref_msg_mime(op->thread_data, folder, msg);
#endif
  libetpan_message_mime_unref(engine, msg);
  
  return NO_ERROR;

 unref_msg_mime:
#if 0
  if (op->mime == NULL)
    etpan_unref_msg_mime(op->thread_data, folder, msg);
#endif
  libetpan_message_mime_unref(engine, msg);
 err:
  return res;
}




/*
  message_render_reply_mime
  
  WARNING, this will leave the bodystructure inside the message,
  it is not flushed because it can be used by some other etpan_subapp
  
  all references to MIME parts should be checked before flushing the
  message.
*/

#define QUOTE_BUF_SIZE 1024

static int message_render_reply_mime(struct etpan_thread_op * op)
{
  struct mailmime * mime;
  mailmessage * msg;
  struct mailfolder * folder;
  int r;
  struct etpan_account_info * account;
  int res;
  struct etpan_message_render_reply_result * result;
  FILE * f;
  char filename[PATH_MAX];
  char quoted_filename[PATH_MAX];
  FILE * quoted_f;
  char str[QUOTE_BUF_SIZE];
  char * from;
  char * decoded_from;
  struct etpan_message_render_reply_arg * arg;
  char * content;
  size_t content_len;
  struct mailimf_fields * fields;
  struct mailimf_fields * new_fields;
  struct etpan_app * app;
  size_t cur_token;
  struct mailmime * attached_mime;
  struct mailmime * edit_part;
  int line;
  int show_headers;
  struct mailengine * engine;
  struct mailprivacy * privacy;
  struct etpan_app_config * config;
  int first_line_part;
  int skip_line;

  folder = op->data.mailaccess.folder;
  msg = op->data.mailaccess.msg;
  mime = op->data.mailaccess.mime;
  arg = op->arg;
  app = op->thread_data->manager->app;
  config = &app->config;
  
#if 0
  if (mime != NULL) {
    if (!mime_is_valid(op->thread_data, folder, msg, mime)) {
      res = ERROR_INVAL;
      goto err;
    }
  }
#endif
  
#if 0
  /* if get_bodystructure was not called */
  if (mime == NULL) {
    if (!message_is_valid(op->thread_data, folder, msg)) {
      res = ERROR_INVAL;
      goto err;
    }

    r = etpan_ref_msg_mime(op->thread_data, folder, msg);
    if (r != NO_ERROR) {
      log_op_info(op->thread_data->manager->app, "failed",
          op->cmd, NULL, folder);
      res = r;
      goto err;
    }
    mime = msg->mime;
  }
#endif

  engine = op->thread_data->manager->engine;
  privacy = libetpan_engine_get_privacy(engine);
  
  r = libetpan_message_mime_ref(engine, msg);
  if (r == 0) {
    r = -r;
    if (r == MAIL_ERROR_STREAM)
      res = ERROR_STREAM;
    else
      res = ERROR_FETCH;
    goto err;
  }
  if (mime == NULL)
    mime = msg->msg_mime;
  
  account = etpan_vfolder_get_account(config->vfolder_config,
      config->account_config, folder);

  /* write quoted message */
  
  quoted_f = etpan_get_tmp_file(quoted_filename, PATH_MAX);
  if (quoted_f == NULL) {
    log_op_info(app, "failed", op->cmd, NULL, folder);
    res = ERROR_FILE;
    goto unref_msg_mime;
  }
  
  line = 1;
  
  /* header of the message text */
  switch (arg->reply_type) {
  case ETPAN_THREAD_REPLY:
  case ETPAN_THREAD_REPLY_ALL:
  case ETPAN_THREAD_REPLY_FOLLOWUP_TO:
    
    decoded_from = NULL;
    etpan_get_from_value(&msg->msg_single_fields, &from, NULL);
    if (from != NULL) {
      decoded_from =
        etpan_decode_mime_header(app, from);
      if (decoded_from != NULL)
        from = decoded_from;
      else
        from = from;
    }
    
    if (from != NULL) {
      fprintf(quoted_f, "%s wrote :\r\n\r\n", from);
      if (decoded_from != NULL) {
        free(decoded_from);
      }
      line += 2;
    }
    
    break;
  }
  
  /*
    avoid buffer to be duplicated both in case the process fork in
    etpan_render_mime() (call to external renderers)
  */
  fflush(quoted_f);

  switch (arg->reply_type) {
  case ETPAN_THREAD_REPLY:
  case ETPAN_THREAD_REPLY_ALL:
  case ETPAN_THREAD_REPLY_FOLLOWUP_TO:
    show_headers = 0;
    break;
    
  case ETPAN_THREAD_REPLY_FORWARD:
    show_headers = 1;
    break;
    
  default:
    show_headers = 1;
    break;
  }  
  
  /* quote text */
  switch (arg->reply_type) {
  case ETPAN_THREAD_REPLY:
  case ETPAN_THREAD_REPLY_ALL:
  case ETPAN_THREAD_REPLY_FOLLOWUP_TO:
  case ETPAN_THREAD_REPLY_FORWARD:
    /* render original message */
    
    f = etpan_get_tmp_file(filename, PATH_MAX);
    if (f == NULL) {
    log_op_info(op->thread_data->manager->app, "failed",
        op->cmd, NULL, folder);
      res = ERROR_FILE;
      goto close_quoted;
    }
    
    if (show_headers)
      r = etpan_render_mime(f, NULL, app,
          msg, mime, ETPAN_RENDER_FLAGS_MAIN_HEADER);
    else
      r = etpan_render_mime(f, NULL, app,
          msg, mime, 0);
    if (r != NO_ERROR) {
      log_op_info(app, "failed", op->cmd, NULL, folder);
      fclose(f);
      unlink(filename);
      res = r;
      goto close_quoted;
    }
    
    /* return at beginning of the written file */
    rewind(f);
    
    skip_line = 0;
    first_line_part = 1;
    while (fgets(str, 1024, f)) {
      size_t len;
      
      /* XXX - should handle lines that were not completely read */
      
      /* end of file */
      if (* str == '\0')
        break;

      /* ignore signature */
      if (strcmp(str, "-- \r\n") == 0)
        break;
      if (strcmp(str, "-- \n") == 0)
        break;
      
      if (first_line_part) {
        /* ignore when quote is above the limit */
        if (app->config.global_config->reply_quote_limit != -1) {
          int count;
          char * p;
        
          count = 0;
          for(p = str ; * p != 0 ; p ++) {
            if (* p == '>')
              count ++;
            else if ((* p != ' ') && (* p != '\t'))
              break;
          }
        
          if (count >= app->config.global_config->reply_quote_limit) {
            /* skip line */
            skip_line = 1;
          }
        }
        
        if (!skip_line) {
          r = fputs("> ", quoted_f);
          if (r < 0) {
            log_op_info(op->thread_data->manager->app, "failed",
                op->cmd, NULL, folder);
            fclose(f);
            unlink(filename);
            res = ERROR_FILE;
            goto close_quoted;
          }
        }
        
        first_line_part = 0;
      }
      
      if (!skip_line) {
        r = fputs(str, quoted_f);
        if (r < 0) {
          log_op_info(op->thread_data->manager->app, "failed",
              op->cmd, NULL, folder);
          fclose(f);
          unlink(filename);
          res = ERROR_FILE;
          goto close_quoted;
        }
      }
      
      len = strlen(str);
      if (str[len - 1] == '\n') {
        skip_line = 0;
        first_line_part = 1;
      }
      
      line ++;
    }
    
    r = fputs("\r\n", quoted_f);
    if (r < 0) {
      log_op_info(op->thread_data->manager->app, "failed",
          op->cmd, NULL, folder);
      fclose(f);
      unlink(filename);
      res = ERROR_FILE;
      goto close_quoted;
    }
    line ++;
    
    fclose(f);
    unlink(filename);
    break;
  }
  
  r = fputs("\r\n", quoted_f);
  if (r < 0) {
    log_op_info(op->thread_data->manager->app, "failed",
        op->cmd, NULL, folder);
    res = ERROR_FILE;
    goto close_quoted;
  }
  
  r = etpan_add_signature(quoted_f, account);
  if (r != NO_ERROR) {
    log_op_info(op->thread_data->manager->app, "failed",
        op->cmd, NULL, folder);
    res = ERROR_FILE;
    goto close_quoted;
  }
  
  /* render reply fields */

  /* fetch original header */
#if 0
  r = etpan_msg_fetch_section_header(op->thread_data->manager->app,
      folder, msg, mime, &content, &content_len);
#endif
  r = mailprivacy_msg_fetch_section_header(privacy,
      msg, mime, &content, &content_len);
  if (r == MAIL_ERROR_STREAM) {
    log_op_info(op->thread_data->manager->app, "stream error",
        op->cmd, NULL, folder);
    
    res = ERROR_STREAM;
    goto close_quoted;
  }
  else if (r != MAIL_NO_ERROR) {
    /* new empty fields */
    r = etpan_new_get_fields(app, folder, NULL, &new_fields);
    if (r != NO_ERROR) {
      log_op_info(op->thread_data->manager->app, "failed",
          op->cmd, NULL, folder);
      res = r;
      goto close_quoted;
    }
  }
  else {
    cur_token = 0;
    r = mailimf_fields_parse(content, content_len, &cur_token, &fields);

#if 0    
    etpan_msg_fetch_result_free(op->thread_data->manager->app,
        folder, msg, content);
#endif
    mailprivacy_msg_fetch_result_free(privacy, msg, content);
    
    if (r != MAILIMF_NO_ERROR) {
      log_op_info(op->thread_data->manager->app, "failed",
          op->cmd, NULL, folder);
      res = ERROR_PARSE;
      goto close_quoted;
    }
    
    /* get reply fields */
    switch (arg->reply_type) {
    case ETPAN_THREAD_REPLY:
      r = etpan_reply_to_get_fields(app, folder, fields, &new_fields);
      break;
      
    case ETPAN_THREAD_REPLY_ALL:
      r = etpan_reply_all_get_fields(app, folder, fields, &new_fields);
      break;
      
    case ETPAN_THREAD_REPLY_FOLLOWUP_TO:
      r = etpan_followup_to_get_fields(app, folder, fields, &new_fields);
      break;
      
    case ETPAN_THREAD_REPLY_FORWARD:
    case ETPAN_THREAD_REPLY_FORWARD_AS_ATTACHMENT:
      r = etpan_forward_get_fields(app, folder, fields,
          arg->account, arg->dest_addr, arg->post_folder, &new_fields);
      break;
      
    default:
      r = ERROR_INVAL;
      break;
    }
  
    if (r != NO_ERROR) {
      log_op_info(op->thread_data->manager->app, "failed",
          op->cmd, NULL, folder);
      mailimf_fields_free(fields);
      res = r;
      goto close_quoted;
    }
    
    mailimf_fields_free(fields);
  }
  
  r = etpan_build_simple_mime(app, new_fields, quoted_filename,
      &mime, &edit_part);
  if (r != NO_ERROR) {
    log_op_info(op->thread_data->manager->app, "failed",
        op->cmd, NULL, folder);
    mailimf_fields_free(new_fields);
    res = r;
    goto close_quoted;
  }
  
  fclose(quoted_f);
  
  switch (arg->reply_type) {
  case ETPAN_THREAD_REPLY_FORWARD:
    /* get all non-plain/text part and attach them */
    r = get_message_mime_non_text_plain(op, mime);
    if (r != NO_ERROR) {
      log_op_info(op->thread_data->manager->app, "failed",
          op->cmd, NULL, folder);
      etpan_message_mime_clear(mime);
      mailmime_free(mime);
      res = r;
      goto unref_msg_mime;
    }
    break;
    
  case ETPAN_THREAD_REPLY_FORWARD_AS_ATTACHMENT:
    /* copy the original message to a new mime and attach it */
    r = get_message_mime_copy(op, &attached_mime);
    if (r != NO_ERROR) {
      log_op_info(op->thread_data->manager->app, "failed",
          op->cmd, NULL, folder);
      etpan_message_mime_clear(mime);
      mailmime_free(mime);
      res = r;
      goto unref_msg_mime;
    }
    
    r = mailmime_smart_add_part(mime, attached_mime);
    if (r != MAILIMF_NO_ERROR) {
      log_op_info(op->thread_data->manager->app, "failed",
          op->cmd, NULL, folder);
      etpan_message_mime_clear(attached_mime);
      mailmime_free(attached_mime);
      etpan_message_mime_clear(mime);
      mailmime_free(mime);
      res = r;
      goto unref_msg_mime;
    }
    
    break;
  }
  
  result = malloc(sizeof(* result));
  if (result == NULL) {
    log_op_info(op->thread_data->manager->app, "not enough memory",
        op->cmd, NULL, folder);
    etpan_message_mime_clear(mime);
    mailmime_free(mime);
    res = ERROR_MEMORY;
    goto unref_msg_mime;
  }
  
  result->mime = mime;
  result->edit_part = edit_part;
  result->edit_part_line = line;
  
  op->result = result;
  
#if 0
  if (op->mime == NULL)
    etpan_unref_msg_mime(op->thread_data, folder, msg);
#endif
  libetpan_message_mime_unref(engine, msg);
  
  return NO_ERROR;
  
 close_quoted:
  fclose(quoted_f);
  unlink(quoted_filename);
 unref_msg_mime:
  libetpan_message_mime_unref(engine, msg);
#if 0
  if (op->mime == NULL)
    etpan_unref_msg_mime(op->thread_data, folder, msg);
#endif
 err:
  return res;
}



static void folder_append_message_clean_arg(struct etpan_thread_op * op)
{
  if (op->arg != NULL) {
    char * filename;
    struct etpan_message_append_arg * arg;
    
    arg = op->arg;
    filename = arg->filename;
    unlink(filename);
    free(filename);
    
    if (arg->flags != NULL)
      mail_flags_free(arg->flags);
    
    free(arg);
  }
  op->arg = NULL;
}

static int folder_append_message(struct etpan_thread_op * op)
{
  char * filename;
  struct mailfolder * folder;
  int fd;
  int res;
  char * data;
  size_t size;
  struct stat stat_info;
  int r;
  struct etpan_message_append_arg * arg;
  struct mail_flags * flags;
  
  folder = op->data.mailaccess.folder;
  arg = op->arg;
  filename = arg->filename;
  flags = arg->flags;

#if 0  
  if (!folder_is_valid(op->thread_data, folder)) {
    res = ERROR_INVAL;
    goto err;
  }
#endif
  
  fd = open(filename, O_RDONLY);
  if (fd < 0) {
    log_op_info(op->thread_data->manager->app, "failed",
        op->cmd, NULL, folder);
    res = ERROR_FILE;
    goto err;
  }
  
  r = fstat(fd, &stat_info);
  if (r < 0) {
    log_op_info(op->thread_data->manager->app, "failed",
        op->cmd, NULL, folder);
    res = ERROR_FILE;
    goto close;
  }
  
  size = stat_info.st_size;
  
  data = mmap(NULL, size, PROT_READ, MAP_PRIVATE, fd, 0);
  if (data == MAP_FAILED) {
    log_op_info(op->thread_data->manager->app, "failed",
        op->cmd, NULL, folder);
    res = ERROR_FILE;
    goto close;
  }
  
#if 0
  r = mailsession_append_message(folder->fld_session, data, size);
#endif
  r = mailfolder_append_message_flags(folder, data, size, flags);
  if (r == MAIL_ERROR_STREAM) {
    log_op_info(op->thread_data->manager->app, "stream error",
        op->cmd, NULL, folder);
    
    res = ERROR_STREAM;
    goto unmap;
  }
  else if (r != MAIL_NO_ERROR) {
    log_op_info(op->thread_data->manager->app, "failed",
        op->cmd, NULL, folder);
    res = ERROR_APPEND;
    goto unmap;
  }

  munmap(data, size);
  
  return NO_ERROR;
  
 unmap:
  munmap(data, size);
 close:
  close(fd);
 err:
  return res;
}


#define MAX_BUF 1024

#if 0
static void message_send_clean_arg(struct etpan_thread_op * op)
{
  if (op->arg != NULL) {
    char * filename;
    
    filename = op->arg;
    unlink(filename);
    free(filename);
  }
  op->arg = NULL;
}

static int message_send(struct etpan_thread_op * op)
{
  char * filename;
  char quoted_filename[PATH_MAX];
  char cmd[PATH_MAX];
  char log_line[MAX_BUF];
  int r;
  FILE * p;
  int res;
  char * sendmail_path;
  struct etpan_app_config * config;
  char dot_stripped_filename[PATH_MAX];
  FILE * dot_stripped_f;
  char msg_line[MAX_BUF];
  FILE * f;

  filename = op->arg;
  
  dot_stripped_f = etpan_get_tmp_file(dot_stripped_filename,
      sizeof(dot_stripped_filename));
  if (dot_stripped_f == NULL) {
    log_op_info(op->thread_data->manager->app, "not enough memory",
        op->cmd, NULL, NULL);
    res = ERROR_MEMORY;
    goto err;
  }
  
  f = fopen(filename, "r");
  if (f == NULL) {
    fclose(dot_stripped_f);
    unlink(dot_stripped_filename);
    res = ERROR_FILE;
    goto err;
  }
  
  /* quote lines that starts with '.' */
  while (fgets(msg_line, sizeof(msg_line), f)) {
    if (msg_line[0] == '.')
      putc('.', dot_stripped_f);
    
    r = fputs(msg_line, dot_stripped_f);
    if (r < 0) {
      fclose(dot_stripped_f);
      unlink(dot_stripped_filename);
      res = ERROR_FILE;
      goto err;
    }
  }
  
  fclose(f);
  
  fclose(dot_stripped_f);
  
  r = etpan_quote_filename(quoted_filename, sizeof(quoted_filename),
      dot_stripped_filename);
  if (r < 0) {
    log_op_info(op->thread_data->manager->app, "not enough memory",
        op->cmd, NULL, NULL);
    res = ERROR_MEMORY;
    goto unlink_dot_stripped;
  }
  
  config = &op->thread_data->manager->app->config;
  sendmail_path = config->global_config->sendmail_path;
  snprintf(cmd, sizeof(cmd), "%s -t < %s 2>&1",
      sendmail_path, quoted_filename);
  
  log_op_info(op->thread_data->manager->app, "sending mail",
      op->cmd, NULL, NULL);
  ETPAN_APP_DEBUG((op->thread_data->manager->app, "sending mail %s", cmd));
  
  p = popen(cmd, "r");
  if (p == NULL) {
    log_op_info(op->thread_data->manager->app, "failed",
        op->cmd, NULL, NULL);
    res = ERROR_COMMAND;
    goto unlink_dot_stripped;
  }
  
  while (fgets(log_line, sizeof(log_line), p))
    ETPAN_APP_LOG((op->thread_data->manager->app, "sending mail - %s", log_line));
  
  r = pclose(p);
  if (WEXITSTATUS(r) != 0) {
    log_op_info(op->thread_data->manager->app, "failed",
        op->cmd, NULL, NULL);
    res = ERROR_COMMAND;
    goto unlink_dot_stripped;
  }
  
  unlink(dot_stripped_filename);
  
  log_op_info(op->thread_data->manager->app, "mail sent",
      op->cmd, NULL, NULL);
  
  return NO_ERROR;
  
 unlink_dot_stripped:
  unlink(dot_stripped_filename);
 err:
  return res;
}
#endif

#if 0
static int message_free(struct etpan_thread_op * op)
{
  mailmessage * msg;
  struct mailfolder * folder;
  struct mailprivacy * privacy;
  
  ETPAN_APP_DEBUG((op->thread_data->manager->app,
                      "BUG detected MESSAGE FREE was called"));
  
  folder = op->folder;
  msg = op->msg;

  if (folder != NULL) {
    ETPAN_APP_DEBUG((op->thread_data->manager->app,
                        "BUG detected free on folder message"));
    return ERROR_INVAL;
  }
  
  privacy = libetpan_engine_get_privacy(op->thread_data->manager->engine);
  
#if 0  
  etpan_msg_flush(op->thread_data->manager->app, folder, msg);
#endif
  mailprivacy_msg_flush(privacy, msg);
  mailmessage_free(msg);
  
  return NO_ERROR;
}
#endif


static void connect_folder(struct etpan_thread_data * thread_data,
    struct mailfolder * folder);


static void msglist_copy_clean_arg(struct etpan_thread_op * op)
{
  struct etpan_msglist_copy_arg * arg;
  
  arg = op->arg;
  
  carray_free(arg->msglist);
  free(arg);
  op->arg = NULL;
}

static struct etpan_thread_data *
get_storage_thread_data(struct etpan_thread_manager * thread_manager,
    struct mailstorage * storage, int do_create);

static int msglist_do_copy(struct etpan_thread_op * op,
    carray * msglist, struct mailfolder * dest_folder, int remove_source)
{
  struct mailfolder * folder;
  unsigned int i;
  int r;
  struct etpan_thread_data * dest_thread_data;
  pthread_mutex_t * lock;
  struct mailprivacy * privacy;
  
  privacy = libetpan_engine_get_privacy(op->thread_data->manager->engine);
  
  folder = op->data.mailaccess.folder;
  
  if (dest_folder == NULL) {
    debug_op_info(op->thread_data->manager->app, "BUG detected - invalid",
        op->cmd, NULL, folder);
    return ERROR_INVAL;
  }

  if (dest_folder->fld_storage == NULL) {
    debug_op_info(op->thread_data->manager->app, "BUG detected - invalid",
        op->cmd, NULL, folder);
    return ERROR_INVAL;
  }

  /* ** WARNING ** - we could be working in an other storage */
  dest_thread_data = get_storage_thread_data(op->thread_data->manager,
      dest_folder->fld_storage, 1);
  if (dest_thread_data == NULL) {
    log_op_info(op->thread_data->manager->app, "not enough memory",
        op->cmd, NULL, folder);
    
    return ERROR_MEMORY;
  }
  
  lock = NULL;
  if (op->thread_data != dest_thread_data)
    lock = &dest_thread_data->run_lock;
  
  for(i = 0 ; i < carray_count(msglist) ; i ++) {
    mailmessage * msg;
    char * content;
    size_t content_len;
    
    msg = carray_get(msglist, i);

#if 0    
    if (!message_is_valid(op->thread_data, folder, msg)) {
      return ERROR_INVAL;
    }
#endif

    connect_folder(op->thread_data, folder);

#if 0    
    r = etpan_msg_fetch(op->thread_data->manager->app,
        folder, msg, &content, &content_len);
#endif
    r = mailprivacy_msg_fetch(privacy, msg, &content, &content_len);
    if (r == MAIL_ERROR_STREAM) {
      log_op_info(op->thread_data->manager->app, "stream error",
          op->cmd, NULL, folder);
      
      return ERROR_STREAM;
    }
    else if (r != MAIL_NO_ERROR) {
      log_op_info(op->thread_data->manager->app, "failed",
          op->cmd, NULL, folder);
      return ERROR_FETCH;
    }

    if (lock != NULL) {
      /* lock only if this is an other storage */
      pthread_mutex_lock(lock);
    }
    
    connect_folder(dest_thread_data, dest_folder);
    
    /* if not connected */
    if (dest_folder->fld_session == NULL) {
      log_op_info(op->thread_data->manager->app, "failed",
          op->cmd, NULL, folder);
      
      if (lock != NULL) {
        /* unlock only if this is an other storage */
        pthread_mutex_unlock(lock);
      }
#if 0
      etpan_msg_fetch_result_free(op->thread_data->manager->app,
          folder, msg, content);
#endif
      mailprivacy_msg_fetch_result_free(privacy, msg, content);
      return ERROR_CONNECT;
    }
    
#if 0
    r = mailsession_append_message_flags(dest_folder->fld_session,
        content, content_len, msg->msg_flags);
#endif
    r = mailfolder_append_message_flags(dest_folder,
        content, content_len, msg->msg_flags);
    
    if (lock != NULL) {
      /* unlock only if this is an other storage */
      pthread_mutex_unlock(lock);
    }
    
    if (r == MAIL_ERROR_STREAM) {
      log_op_info(op->thread_data->manager->app, "stream error",
          op->cmd, NULL, folder);
      mailprivacy_msg_fetch_result_free(privacy, msg, content);
      
      return ERROR_STREAM;
    }
    else if (r != MAIL_NO_ERROR) {
      log_op_info(op->thread_data->manager->app, "failed",
          op->cmd, NULL, folder);
#if 0
      etpan_msg_fetch_result_free(op->thread_data->manager->app,
          folder, msg, content);
#endif
      mailprivacy_msg_fetch_result_free(privacy, msg, content);
      return ERROR_APPEND;
    }
    
    if (remove_source) {
      if (msg->msg_flags != NULL) {
        uint32_t old_flags;
        
        old_flags = msg->msg_flags->fl_flags;
        msg->msg_flags->fl_flags &= ~MAIL_FLAG_NEW;
        msg->msg_flags->fl_flags |= MAIL_FLAG_SEEN;
        msg->msg_flags->fl_flags |= MAIL_FLAG_DELETED;
        if (old_flags != msg->msg_flags->fl_flags)
          mailmessage_check(msg);
      }
    }

#if 0    
    etpan_msg_fetch_result_free(op->thread_data->manager->app,
        folder, msg, content);
#endif
    mailprivacy_msg_fetch_result_free(privacy, msg, content);
  }
  
  return NO_ERROR;
}

static int msglist_copy(struct etpan_thread_op * op)
{
  struct etpan_msglist_copy_arg * arg;
  
  arg = op->arg;
  
  return msglist_do_copy(op, arg->msglist, arg->folder, 0);
}


static void msglist_move_clean_arg(struct etpan_thread_op * op)
{
  struct etpan_msglist_move_arg * arg;
  
  arg = op->arg;
  
  carray_free(arg->msglist);
  free(arg);
  op->arg = NULL;
}

static int msglist_move(struct etpan_thread_op * op)
{
  struct etpan_msglist_move_arg * arg;
  
  arg = op->arg;
  
  return msglist_do_copy(op, arg->msglist, arg->folder, 1);
}


static void msglist_spam_clean_arg(struct etpan_thread_op * op)
{
  struct etpan_msglist_spam_arg * arg;
  
  arg = op->arg;
  
  carray_free(arg->msglist);
  free(arg);
  op->arg = NULL;
}

static int msglist_spam(struct etpan_thread_op * op)
{
  struct etpan_msglist_spam_arg * arg;
  struct mailfolder * folder;
  unsigned int i;
  FILE * f;
  char filename[PATH_MAX];
  char command[PATH_MAX];
  int r;
  FILE * p;
  char log_line[MAX_BUF];
  int res;
  struct mailprivacy * privacy;
  struct etpan_app * app;
  
  app = op->thread_data->manager->app;
  privacy = libetpan_engine_get_privacy(op->thread_data->manager->engine);
  
  folder = op->data.mailaccess.folder;
  arg = op->arg;
  
  f = etpan_get_tmp_file(filename, sizeof(filename));
  if (f == NULL) {
    log_op_info(app, "failed", op->cmd, NULL, folder);
    res = ERROR_FILE;
    goto err;
  }
  
  for(i = 0 ; i < carray_count(arg->msglist) ; i ++) {
    mailmessage * msg;
    char * content;
    size_t content_len;
    size_t written;
    
    msg = carray_get(arg->msglist, i);

#if 0    
    if (!message_is_valid(op->thread_data, folder, msg)) {
      fclose(f);
      res = ERROR_INVAL;
      goto unlink;
    }
#endif
    
#if 0
    r = etpan_msg_fetch(op->thread_data->manager->app,
        folder, msg, &content, &content_len);
#endif
    r = mailprivacy_msg_fetch(privacy, msg, &content, &content_len);
    if (r == MAIL_ERROR_STREAM) {
      log_op_info(app, "stream error", op->cmd, NULL, folder);
      fclose(f);
      res = ERROR_STREAM;
      goto unlink;
    }
    else if (r != MAIL_NO_ERROR) {
      log_op_info(app, "failed", op->cmd, NULL, folder);
      fclose(f);
      res = ERROR_FETCH;
      goto unlink;
    }

    written = fwrite(content, 1, content_len, f);
    if (written != content_len) {
      log_op_info(app, "failed", op->cmd, NULL, folder);
#if 0
      etpan_msg_fetch_result_free(op->thread_data->manager->app,
          folder, msg, content);
#endif
      mailprivacy_msg_fetch_result_free(privacy, msg, content);
      fclose(f);
      res = ERROR_FILE;
      goto unlink;
    }

#if 0    
    etpan_msg_fetch_result_free(op->thread_data->manager->app,
        folder, msg, content);
#endif
    mailprivacy_msg_fetch_result_free(privacy, msg, content);
  }
  fclose(f);
  
  switch (arg->spam_type) {
  case ETPAN_THREAD_SPAM:
    snprintf(command, sizeof(command), "bogofilter -s < %s", filename);
    break;
  case ETPAN_THREAD_NON_SPAM:
    snprintf(command, sizeof(command), "bogofilter -n < %s", filename);
    break;
  case ETPAN_THREAD_FIX_SPAM:
    snprintf(command, sizeof(command), "bogofilter -Ns < %s", filename);
    break;
  case ETPAN_THREAD_FIX_NON_SPAM:
    snprintf(command, sizeof(command), "bogofilter -Sn < %s", filename);
    break;
  default:
    log_op_info(app, "BUG detected", op->cmd, NULL, folder);
    res = ERROR_INVAL;
    goto unlink;
  }
  
  log_op_info(app, "spam operations ...", op->cmd, NULL, NULL);
  ETPAN_APP_DEBUG((app, "spam operations %s", command));
  
  p = popen(command, "r");
  if (p == NULL) {
    log_op_info(app, "failed", op->cmd, NULL, NULL);
    res = ERROR_FILE;
    goto unlink;
  }
  
  while (fgets(log_line, sizeof(log_line), p))
    ETPAN_APP_LOG((app, "spam operations - %s", log_line));
  
  r = pclose(p);
  if (WEXITSTATUS(r) != 0) {
    log_op_info(app, "failed", op->cmd, NULL, NULL);
    res = ERROR_COMMAND;
    goto unlink;
  }
  
  log_op_info(app, "spam operations done", op->cmd, NULL, NULL);
  
  unlink(filename);
  
  return NO_ERROR;

 unlink:
  unlink(filename);
 err:
  return res;
}




static void message_mime_copy_clean_result(struct etpan_thread_op * op)
{
  struct etpan_message_mime_copy_result * result;
  
  result = op->result;

#if 0  
  etpan_message_mime_clear(result->mime);
#endif
  mailprivacy_mime_clear(result->mime);
  mailmime_free(result->mime);
  free(result);
  
  op->result = NULL;
}

static int message_mime_copy(struct etpan_thread_op * op)
{
  struct mailmime * mime;
  int res;
  struct etpan_message_mime_copy_result * result;
  struct mailfolder * folder;
  int r;
  
  folder = op->data.mailaccess.folder;
  
  r = get_message_mime_copy(op, &mime);
  if (r != NO_ERROR) {
    log_op_info(op->thread_data->manager->app, "failed",
        op->cmd, NULL, folder);
    res = r;
    goto err;
  }
  
  result = malloc(sizeof(* result));
  if (result == NULL) {
    log_op_info(op->thread_data->manager->app, "not enough memory",
        op->cmd, NULL, folder);
    res = r;
    goto free_mime;
  }
  result->mime = mime;
  
  op->result = result;

  return NO_ERROR;
  
 free_mime:
#if 0
  etpan_message_mime_clear(mime);
#endif
  mailprivacy_mime_clear(mime);
  mailmime_free(mime);
 err:
  return res;
}


static int ref_message(struct etpan_thread_op * op)
{
  libetpan_message_ref(op->thread_data->manager->engine,
      op->data.mailaccess.msg);
#if 0
  etpan_ref_message(op->thread_data, op->folder, op->msg);
#endif
  
  return NO_ERROR;
}

static int unref_message(struct etpan_thread_op * op)
{
  libetpan_message_unref(op->thread_data->manager->engine,
      op->data.mailaccess.msg);
#if 0
  etpan_unref_message(op->thread_data, op->folder, op->msg);
#endif

  return NO_ERROR;
}


static int nntp_list_groups(struct etpan_thread_op * op)
{
  struct mailstorage * storage;
  carray * info_array;
  int r;
  
  storage = op->data.mailaccess.storage;
  
  /* get NNTP low-level things */
  if (strcasecmp(storage->sto_driver->sto_name, "nntp") != 0)
    return ERROR_INVAL;
  
  r = etpan_nntp_list_groups(storage,
      storage->sto_session, op->arg, &info_array);
  if (r != NO_ERROR)
    return r;
  
  op->result = info_array;
  
  return NO_ERROR;
}

static void nntp_list_groups_clean_result(struct etpan_thread_op * op)
{
  carray * info_array;
  
  info_array = op->result;
  
  etpan_nntp_list_groups_free(info_array);
}

static void nntp_list_groups_clean_arg(struct etpan_thread_op * op)
{
  free(op->arg);
}


static int imap_list_mailboxes(struct etpan_thread_op * op)
{
  struct mailstorage * storage;
  carray * info_array;
  int r;
  
  storage = op->data.mailaccess.storage;
  
  /* get NNTP low-level things */
  if (strcasecmp(storage->sto_driver->sto_name, "imap") != 0)
    return ERROR_INVAL;
  
  r = etpan_imap_list_mailboxes(storage->sto_session, op->arg, &info_array);
  if (r != NO_ERROR)
    return r;
  
  op->result = info_array;
  
  return NO_ERROR;
}

static void imap_list_mailboxes_clean_result(struct etpan_thread_op * op)
{
  carray * info_array;
  
  info_array = op->result;
  
  etpan_imap_list_mailboxes_free(info_array);
}

static void imap_list_mailboxes_clean_arg(struct etpan_thread_op * op)
{
  free(op->arg);
}


static int imap_select(struct etpan_thread_op * op)
{
  struct mailstorage * storage;
  int r;
  
  storage = op->data.mailaccess.storage;
  
  /* get NNTP low-level things */
  if (strcasecmp(storage->sto_driver->sto_name, "imap") != 0)
    return ERROR_INVAL;
  
  r = etpan_imap_select(storage->sto_session, op->arg);
  if (r != NO_ERROR)
    return r;
  
  return NO_ERROR;
}

static void imap_select_clean_arg(struct etpan_thread_op * op)
{
  free(op->arg);
}


static int imap_create(struct etpan_thread_op * op)
{
  struct mailstorage * storage;
  int r;
  
  storage = op->data.mailaccess.storage;
  
  /* get NNTP low-level things */
  if (strcasecmp(storage->sto_driver->sto_name, "imap") != 0)
    return ERROR_INVAL;
  
  r = etpan_imap_create(storage->sto_session, op->arg);
  if (r != NO_ERROR)
    return r;
  
  return NO_ERROR;
}

static void imap_create_clean_arg(struct etpan_thread_op * op)
{
  free(op->arg);
}



static void done_config_clean_arg(struct etpan_thread_op * op)
{
  struct etpan_app_config * config;
  
  config = op->arg;
  if (config != NULL) {
    etpan_app_config_done(config);
    free(config);
    op->arg = NULL;
  }
}

static void storage_thread_stop(struct etpan_thread_manager * manager,
    struct mailstorage * storage);

static void storage_thread_free(struct etpan_thread_data * data);

static void
delete_storage_thread_data(struct etpan_thread_manager * thread_manager,
    struct mailstorage * storage);

static void sync_thread_stop(struct etpan_thread_data * thread_data);

static void
delete_thread_data_index(struct etpan_thread_manager * thread_manager,
    int thread_index);

static void
delete_abook_thread_data(struct etpan_thread_manager * thread_manager,
    struct etpan_abook * abook);

static void
delete_sender_thread_data(struct etpan_thread_manager * thread_manager,
    struct etpan_sender_item * sender);

static int is_local_storage(struct mailstorage * storage);

static int is_local_abook(struct etpan_abook * abook)
{
  return ((strcasecmp(abook->driver->name, "local") == 0) ||
      (strcasecmp(abook->driver->name, "vcard") == 0));
}

static int is_local_sender(struct etpan_sender_item * sender)
{
  return (sender->type == SENDER_TYPE_COMMAND);
}

static int done_config(struct etpan_thread_op * op)
{
  struct etpan_thread_manager * manager;
  unsigned int i;
  struct etpan_app_config * config;
  
  config = op->arg;
  
  manager = op->thread_data->manager;
  
  if (config->storage_config != NULL) {
    for(i = 0 ; i < carray_count(config->storage_config->storage_tab) ; i ++) {
      struct mailstorage * storage;
      struct etpan_thread_data * thread_data;

      storage = carray_get(config->storage_config->storage_tab, i);
      thread_data = get_storage_thread_data(manager, storage, 0);
      
      if (thread_data != NULL) {
        struct etpan_thread_op * op;
        
        op = etpan_thread_op_add(manager,
            ETPAN_THREAD_STORAGE_DISCONNECT,
            storage, NULL, NULL, NULL, NULL, 0);
        if (op != NULL) {
          etpan_thread_op_wait(op);
          etpan_thread_op_ack(op);
        }
        
        if (!is_local_storage(storage))
          delete_storage_thread_data(manager, storage);
      }
    }
  }
  
#if 0
  for(i = 0 ; i< ETPAN_LOCALSTORAGE_THREAD_COUNT ; i ++) {
    if (manager->thread_tab_local_mailaccess[i] != NULL) {
      sync_thread_stop(manager->thread_tab_local_mailaccess[i]);
    }
  }
#endif

#if 0
  if (config->storage_config != NULL) {
    for(i = 0 ; i < carray_count(config->storage_config->storage_tab) ; i ++) {
      struct mailstorage * storage;
      struct etpan_thread_data * thread_data;
      
      storage = carray_get(config->storage_config->storage_tab, i);
      
      thread_data = get_storage_thread_data(manager, storage, 0);
      if (thread_data != NULL) {
        if (is_local_storage(storage)) {
          delete_storage_thread_data(manager, storage);
        }
      }
    }
  }
  
  for(i = 0 ; i< ETPAN_LOCALSTORAGE_THREAD_COUNT ; i ++) {
    if (manager->thread_tab_local_mailaccess[i] != NULL) {
      delete_thread_data_index(manager,
          manager->thread_tab_local_mailaccess[i]->thread_index);
      storage_thread_free(manager->thread_tab_local_mailaccess[i]);
      manager->thread_tab_local_mailaccess[i] = NULL;
    }
  }
#endif
  
  if (config->abook_config != NULL) {
    for(i = 0 ; i < carray_count(config->abook_config->abook_tab) ; i ++) {
      struct etpan_abook * abook;
      int remote_abook;
      
      abook = carray_get(config->abook_config->abook_tab, i);
      
      remote_abook = 1;
      
      if (is_local_abook(abook))
        remote_abook = 0;
      
      if (abook->connected) {
        if (remote_abook) {
          struct etpan_thread_op * op;
          
          op = etpan_thread_abook_op_add(manager,
              ETPAN_THREAD_ABOOK_DISCONNECT,
              abook, NULL, 0);
          if (op != NULL) {
            etpan_thread_op_wait(op);
            etpan_thread_op_ack(op);
          }
          
          delete_abook_thread_data(manager, abook);
        }
        else {
          struct etpan_thread_op * op;
          
          op = etpan_thread_abook_op_add(manager,
              ETPAN_THREAD_ABOOK_DISCONNECT,
              abook, NULL, 1);
        }
      }
    }
  }
  
  if (config->sender_config != NULL) {
    chashiter * iter;
    
    for(iter = chash_begin(config->sender_config->sender_hash) ;
        iter != NULL ;
        iter = chash_next(config->sender_config->sender_hash, iter)) {
      struct etpan_sender_item * sender;
      int remote_sender;
      chashdatum value;
      
      chash_value(iter, &value);
      sender = value.data;
      
      remote_sender = 1;
      
      if (is_local_sender(sender))
        remote_sender = 0;
      
      /* release sender thread */
      delete_sender_thread_data(manager, sender);
    }
  }
  
  etpan_app_config_done(config);
  free(config);
  op->arg = NULL;
 
  return NO_ERROR;
}


static int stop_storage_thread(struct etpan_thread_op * op)
{
  struct etpan_thread_manager * manager;
  struct mailstorage * storage;
  struct etpan_thread_data * thread_data;
#if 0
  struct etpan_thread_data * thread_data;
  chashdatum key;
#endif
  
  manager = op->thread_data->manager;
  
  storage = op->arg;
  
  thread_data = get_storage_thread_data(manager, storage, 0);
  
  if (thread_data != NULL) {
    struct etpan_thread_op * op;
    
    ETPAN_APP_DEBUG((manager->app, "disconnect storage thread"));
    op = etpan_thread_op_add(manager,
        ETPAN_THREAD_STORAGE_DISCONNECT,
        storage, NULL, NULL, NULL, NULL, 0);
    if (op != NULL) {
      ETPAN_APP_DEBUG((manager->app, "wait disconnect"));
      etpan_thread_op_wait(op);
      ETPAN_APP_DEBUG((manager->app, "ok disconnect"));
      etpan_thread_op_ack(op);
    }
    ETPAN_APP_DEBUG((manager->app, "disconnected"));
    
    ETPAN_APP_DEBUG((manager->app, "delete thread data"));
    if (!is_local_storage(storage))
      delete_storage_thread_data(manager, storage);
  }
  else {
    ETPAN_APP_DEBUG((manager->app, "nothing to disconnected"));
  }
  
  
  
#if 0
  thread_data = get_thread_data(manager, storage, 0);
  if (thread_data == NULL)
    return NO_ERROR;
#endif

#if 0  
  storage_thread_stop(manager, storage);
#endif

#if 0  
  /* wait for thread to finish */
  
  debug_op_info(manager->app, "waiting end of thread",
      -1, storage, NULL);
  pthread_join(thread_data->th_id, NULL);
  storage_thread_free(thread_data);
  
  key.data = &storage;
  key.len = sizeof(storage);
  chash_delete(manager->thread_hash, &key, NULL);
#endif
  
  return NO_ERROR;
}


static int storage_disconnect(struct etpan_thread_op * op)
{
  struct mailstorage * storage;
  struct mailengine * engine;
  
  storage = op->data.mailaccess.storage;
  engine = op->thread_data->manager->engine;
  
  if (storage != NULL)
    libetpan_storage_disconnect(engine, storage);
  
#if 0
  libetpan_storage_remove(engine, storage);
#endif
#if 0
  delete_storage_thread_data(op->thread_data->manager,
      storage);
#endif

  if (is_local_storage(storage))
    delete_storage_thread_data(op->thread_data->manager, storage);
  
  return NO_ERROR;
}


static void abook_lookup_clean_result(struct etpan_thread_op * op)
{
  carray * entry_list;
  unsigned int i;
  
  entry_list = op->result;
  
  if (entry_list != NULL) {
    for(i = 0 ; i < carray_count(entry_list) ; i ++) {
      struct etpan_abook_entry * entry;
      
      entry = carray_get(entry_list, i);
      etpan_abook_entry_free(entry);
    }
    
    carray_free(entry_list);
  }
}

static void abook_lookup_clean_arg(struct etpan_thread_op * op)
{
  free(op->arg);
}


static int abook_lookup(struct etpan_thread_op * op)
{
  char * key;
  carray * entry_list;
  int r;

  ETPAN_APP_DEBUG((op->thread_data->manager->app, "** 1"));
  
  r = etpan_abook_connect(op->data.abook.abook);
  if (r != NO_ERROR)
    return r;

  ETPAN_APP_DEBUG((op->thread_data->manager->app, "** 2"));
  
  key = op->arg;

  r = etpan_abook_lookup(op->data.abook.abook, key, &entry_list);
  if (r != NO_ERROR)
    return r;

  ETPAN_APP_DEBUG((op->thread_data->manager->app, "** 3"));
  
  op->result = entry_list;
  
  return NO_ERROR;
}


static int abook_disconnect(struct etpan_thread_op * op)
{
  struct etpan_abook * abook;
  int remote_abook;
  
  abook = op->data.abook.abook;

#if 0
  remote_abook = 1;
  
  if (strcasecmp(abook->driver->name, "local") == 0)
    remote_abook = 0;
  
  if (remote_abook)
    delete_abook_thread_data(op->thread_data->manager, abook);
#endif
  
  return NO_ERROR;
}

static int discover_folder(struct etpan_thread_op * op)
{
  struct etpan_discovery_info * info;
  int r;
  
  info = op->arg;
  
  r = etpan_discovery_info_update(info);
  
  return r;
}

static void sendmail_file_clean_arg(struct etpan_thread_op * op)
{
  if (op->arg != NULL) {
    char * filename;
    
    filename = op->arg;
    unlink(filename);
    free(filename);
  }
  op->arg = NULL;
}

static int sendmail_file(struct etpan_thread_op * op)
{
  char * filename;
  char quoted_filename[PATH_MAX];
  char cmd[PATH_MAX];
  char log_line[MAX_BUF];
  int r;
  FILE * p;
  int res;
  char * sendmail_path;
  struct etpan_app_config * config;
  char dot_stripped_filename[PATH_MAX];
  FILE * dot_stripped_f;
  char msg_line[MAX_BUF];
  FILE * f;
  struct etpan_sender_item * sender;
  
  sender = op->data.sender.sender;
  
  filename = op->arg;
  
  dot_stripped_f = etpan_get_tmp_file(dot_stripped_filename,
      sizeof(dot_stripped_filename));
  if (dot_stripped_f == NULL) {
    log_op_info(op->thread_data->manager->app, "not enough memory",
        op->cmd, NULL, NULL);
    res = ERROR_MEMORY;
    goto err;
  }
  
  f = fopen(filename, "r");
  if (f == NULL) {
    fclose(dot_stripped_f);
    unlink(dot_stripped_filename);
    res = ERROR_FILE;
    goto err;
  }
  
  /* quote lines that starts with '.' */
  while (fgets(msg_line, sizeof(msg_line), f)) {
    if (msg_line[0] == '.')
      putc('.', dot_stripped_f);
    
    r = fputs(msg_line, dot_stripped_f);
    if (r < 0) {
      fclose(dot_stripped_f);
      unlink(dot_stripped_filename);
      res = ERROR_FILE;
      goto err;
    }
  }
  
  fclose(f);
  
  fclose(dot_stripped_f);
  
  r = etpan_send_message_file(sender, dot_stripped_filename);
  if (r != NO_ERROR) {
    res = r;
    goto unlink_dot_stripped;
  }
  
  unlink(dot_stripped_filename);
  
  log_op_info(op->thread_data->manager->app, "mail sent",
      op->cmd, NULL, NULL);
  
  return NO_ERROR;
  
 unlink_dot_stripped:
  unlink(dot_stripped_filename);
 err:
  return res;
}



/* end of thread operation */
/* ******************************************************************* */

#define MAX_TRY_CONNECT 1

static void connect_folder(struct etpan_thread_data * thread_data,
    struct mailfolder * folder)
{
  int r;
  
  r = libetpan_folder_connect(thread_data->manager->engine, folder);
}

static void connect_storage(struct etpan_thread_data * thread_data,
    struct mailstorage * storage)
{
  int r;
  
  r = libetpan_storage_connect(thread_data->manager->engine, storage);
}

static const char * op_to_str(int cmd)
{
  struct thread_info * info;
  char * name;
  
  info = get_thread_info(cmd);
  
  name = NULL;
  if (info != NULL) {
    if (info->name != NULL) {
      name = info->name;
    }
  }
  
  if (name == NULL)
    name = "[unknown]";
  
  return name;
}

static void log_op_info(struct etpan_app * app,
    char * prefix, int cmd,
    struct mailstorage * storage,
    struct mailfolder * folder)
{
  const char * cmd_name;
  char * folder_name;
  char * storage_name;
  char * path;
  char output[MAX_OUTPUT];
  
  cmd_name = op_to_str(cmd);
  
  if ((storage == NULL) && (folder != NULL))
    storage = folder->fld_storage;
  
  if (storage == NULL)
    storage_name = "[no storage]";
  else if (storage->sto_id == NULL)
    storage_name = "[no name]";
  else
    storage_name = storage->sto_id;
  
  folder_name = NULL;
  if (folder != NULL) {
    folder_name =  etpan_folder_get_virtual_path(folder);
    if (folder_name != NULL)
      path = folder_name;
    else
      path = "[no path]";
  }
  else
    path = "[no folder]";
  
  if (cmd == -1) {
    if (folder == NULL) {
      snprintf(output, sizeof(output), "%s - %s",
          prefix, storage_name);
    }
    else {
      snprintf(output, sizeof(output), "%s - %s - %s",
          prefix, storage_name, path);
    }
  }
  else {
    if (folder == NULL) {
      snprintf(output, sizeof(output), "%s - %s - %s",
          prefix, cmd_name, storage_name);
    }
    else {
      snprintf(output, sizeof(output), "%s - %s - %s - %s",
          prefix, cmd_name, storage_name, path);
    }
  }
  if (folder_name != NULL)
    free(folder_name);
  ETPAN_APP_LOG((app, "%s", output));
}

static void debug_op_info(struct etpan_app * app,
    char * prefix, int cmd,
    struct mailstorage * storage,
    struct mailfolder * folder)
{
  const char * cmd_name;
  char * folder_name;
  char * storage_name;
  char * path;
  char output[MAX_OUTPUT];
  
  cmd_name = op_to_str(cmd);

  if ((storage == NULL) && (folder != NULL))
    storage = folder->fld_storage;
  
  if (storage == NULL)
    storage_name = "[no storage]";
  else if (storage->sto_id == NULL)
    storage_name = "[no name]";
  else
    storage_name = storage->sto_id;
  
  folder_name = NULL;
  if (folder != NULL) {
    folder_name =  etpan_folder_get_virtual_path(folder);
    if (folder_name != NULL)
      path = folder_name;
    else
      path = "[no path]";
  }
  else
    path = "[no folder]";
  
  if (cmd == -1) {
    if (folder == NULL) {
      snprintf(output, sizeof(output), "%s - %s",
          prefix, storage_name);
    }
    else {
      snprintf(output, sizeof(output), "%s - %s - %s",
          prefix, storage_name, path);
    }
  }
  else {
    if (folder == NULL) {
      snprintf(output, sizeof(output), "%s - %s - %s",
          prefix, cmd_name, storage_name);
    }
    else {
      snprintf(output, sizeof(output), "%s - %s - %s - %s",
          prefix, cmd_name, storage_name, path);
    }
  }
  if (folder_name != NULL)
    free(folder_name);
  ETPAN_APP_DEBUG((app, "%s", output));
}

static int storage_execute_op(struct etpan_thread_op * op)
{
  struct thread_info * info;
  struct etpan_thread_data * thread_data;
  int res;
  int do_run;

  info = get_thread_info(op->cmd);
  thread_data = op->thread_data;
  
  if (info == NULL) {
    ETPAN_APP_DEBUG((thread_data->manager->app,
                        "thread manager - unknown operation - %i", op->cmd));
    return ERROR_INVAL;
  }
  
  if (info->run == NULL) {
    ETPAN_APP_DEBUG((thread_data->manager->app,
                        "thread manager - no action for - %s", info->name));
    return ERROR_INVAL;
  }
  
  debug_op_info(thread_data->manager->app, "execute op",
      op->cmd, NULL, NULL);
  
  pthread_mutex_lock(&thread_data->run_lock);
  if (info->need_connect == CONNECT_MAILACCESS) {
    if (op->data.mailaccess.storage != NULL) {
      if (op->data.mailaccess.folder != NULL) {
        connect_folder(thread_data, op->data.mailaccess.folder);
        if (op->data.mailaccess.folder->fld_session == NULL) {
          goto unlock;
        }
      }
      else {
        connect_storage(thread_data, op->data.mailaccess.storage);
        if (op->data.mailaccess.storage->sto_session == NULL) {
          goto unlock;
        }
      }
    }
  }
  
  do_run = 1;
  if (info->need_connect == CONNECT_MAILACCESS) {
    if ((op->data.mailaccess.msg != NULL) && (op->data.mailaccess.folder != NULL)) {
      if (op->data.mailaccess.msg->msg_session !=
          op->data.mailaccess.folder->fld_session) {
        ETPAN_APP_DEBUG((thread_data->manager->app,
                            "BUG detected, disconnection did not restore properly "));
        do_run = 0;
      }
    }
  }

  debug_op_info(thread_data->manager->app, "do run",
      op->cmd, NULL, NULL);
  
  if (!do_run)
    res = ERROR_INVAL;
  else
    res = info->run(op);
  
  debug_op_info(thread_data->manager->app, "do run ok",
      op->cmd, NULL, NULL);
  
  pthread_mutex_unlock(&thread_data->run_lock);
  
  debug_op_info(thread_data->manager->app, "completed op",
      op->cmd, NULL, NULL);
  
#if 0
  if (op->folder != NULL) {
    struct etpan_msg_params * params;
    
    params = get_msg_params(thread_data, op->folder);
    ETPAN_APP_DEBUG((thread_data->manager->app,
                        "remaining messages in folder : %i", 
                        chash_count(params->hash)));
  }
#endif
  
  return res;
  
 unlock:
  pthread_mutex_unlock(&thread_data->run_lock);
  debug_op_info(thread_data->manager->app, "action could not be run",
      op->cmd, NULL, NULL);
  /* return a connect error */
  return ERROR_CONNECT;
}


static struct etpan_thread_op *
etpan_thread_op_new(struct etpan_thread_data * thread_data,
    int cmd,
    struct mailstorage * storage,
    struct mailfolder * folder,
    mailmessage * msg,
    struct mailmime * mime,
    void * arg, int detached)
{
  struct etpan_thread_op * op;
  int r;
  
  op = malloc(sizeof(* op));
  if (op == NULL)
    goto err;
  
  op->data.mailaccess.storage = storage;
  op->data.mailaccess.folder = folder;
  op->data.mailaccess.msg = msg;
  op->data.mailaccess.mime = mime;
  op->cmd = cmd;
  op->arg = arg;
  op->result = NULL;
  op->err = NO_ERROR;
  if (detached)
    op->state = THREAD_STATE_DETACHED;
  else
    op->state = THREAD_STATE_NOT_STARTED;
  
  r = pthread_mutex_init(&op->state_lock, NULL);
  if (r != 0)
    goto free;
  
  op->thread_data = thread_data;
  
  return op;

 free:
  free(op);
 err:
  return NULL;
}

static void etpan_thread_op_free(struct etpan_thread_op * op)
{
  pthread_mutex_destroy(&op->state_lock);
  free(op);
}


enum {
  SEM_TYPE_SUPPORT_NORMAL,
  /* support for Darwin where sem_init does not exist */
  SEM_TYPE_SUPPORT_BROKEN,
};

static int etpan_sem_init(struct etpan_app * app,
    char * prefix, void * data, int value,
    sem_t ** result_sem, int * result_sem_type)
{
  sem_t * sem;
  int sem_type;
  int r;
  
  sem = malloc(sizeof(* sem));
  if (sem == NULL)
    return ERROR_MEMORY;
  
  r = sem_init(sem, 0, value);
  if (r != -1) {
    sem_type = SEM_TYPE_SUPPORT_NORMAL;
  }
  else {
    char sem_name[256];
    
    free(sem);
    snprintf(sem_name, sizeof(sem_name), "%s-%i-%p",
        prefix, app->pid, data);
    sem = sem_open(sem_name, O_CREAT, 0700, value);
    if (sem == (sem_t *) SEM_FAILED)
      return ERROR_NOT_SUPPORTED;
    sem_type = SEM_TYPE_SUPPORT_BROKEN;
  }
  
  * result_sem = sem;
  * result_sem_type = sem_type;

  return NO_ERROR;
}

static void etpan_sem_destroy(struct etpan_app * app,
    char * prefix, void * data, sem_t * sem, int sem_type)
{
  if (sem_type == SEM_TYPE_SUPPORT_NORMAL) {
    sem_destroy(sem);
    free(sem);
  }
  else {
    char sem_name[256];
    
    snprintf(sem_name, sizeof(sem_name), "%s-%i-%p",
        prefix, app->pid, data);
    
    sem_close(sem);
    sem_unlink(sem_name);
  }
}

static struct etpan_thread_data *
storage_thread_new(struct etpan_thread_manager * manager,
    int thread_type)
{
  struct etpan_thread_data * data;
  int r;
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    goto err;
  
  pthread_mutex_lock(&manager->last_thread_index_lock);
  manager->last_thread_index ++;
  data->thread_index = manager->last_thread_index;
  pthread_mutex_unlock(&manager->last_thread_index_lock);
  
  data->type = thread_type;
  data->manager = manager;
#if 0
  data->data = storage;
#endif
  r = pthread_mutex_init(&data->run_lock, NULL);
  if (r != 0)
    goto free;
  
  r = pthread_mutex_init(&data->lock, NULL);
  if (r != 0)
    goto destroy_storage_mutex;
  
  /*
    we use a socketpair instead of two pipes to save file descriptors.
    
    write to data->fd[1] is for thread end notification.
    read from data->fd[0] is to check if thread has ended.
    
    write to data->fd[0] is to acknowledge a thread end notification.
    read from data->fd[1] is to check if the thread end notification
        was acknowledged.
  */
  r = socketpair(PF_UNIX, SOCK_STREAM, 0, data->fd);
  if (r < 0)
    goto destroy_mutex;

#if 0  
  data->sem = malloc(sizeof(* data->sem));
  if (data->sem == NULL)
    goto destroy_pipe_ack;
  
  r = sem_init(data->sem, 0, 0);
  if (r != -1) {
    data->sem_type = SEM_TYPE_SUPPORT_NORMAL;
  }
  else {
    free(data->sem);
    snprintf(sem_name, sizeof(sem_name), "%i-%p",
        manager->app->pid, data);
    data->sem = sem_open(sem_name, O_CREAT, 0700);
    if (data->sem == SEM_FAILED)
      goto destroy_pipe_ack;
    data->sem_type = SEM_TYPE_SUPPORT_BROKEN;
  }
#endif
  
  r = etpan_sem_init(manager->app, "thread", data, 0,
      &data->sem, &data->sem_type);
  if (r != NO_ERROR)
    goto destroy_pipe_ack;
  
  data->op_list = carray_new(16);
  if (data->op_list == NULL)
    goto destroy_sem;
  
  return data;

 destroy_sem:
  etpan_sem_destroy(manager->app, "thread", data,
      data->sem, data->sem_type);
#if 0
  if (data->sem_type == SEM_TYPE_SUPPORT_NORMAL) {
    sem_destroy(data->sem);
    free(data->sem);
  }
  else {
    sem_close(data->sem);
    sem_unlink(sem_name);
  }
#endif
 destroy_pipe_ack:
  close(data->fd[1]);
  close(data->fd[0]);
 destroy_mutex:
  pthread_mutex_destroy(&data->lock);
 destroy_storage_mutex:
  pthread_mutex_destroy(&data->run_lock);
 free:
  free(data);
 err:
  return NULL;
}

static void storage_thread_free(struct etpan_thread_data * data)
{
  char sem_name[20];
  
  ETPAN_APP_DEBUG((data->manager->app, "free thread data %p %i", data, data->type));
  
  snprintf(sem_name, sizeof(sem_name), "%i-%p",
      data->manager->app->pid, data);

  carray_free(data->op_list);
  
  etpan_sem_destroy(data->manager->app, "thread", data,
      data->sem, data->sem_type);
#if 0
  if (data->sem_type == SEM_TYPE_SUPPORT_NORMAL) {
    sem_destroy(data->sem);
  }
  else {
    sem_close(data->sem);
    sem_unlink(sem_name);
  }
#endif
  
  close(data->fd[1]);
  close(data->fd[0]);
  pthread_mutex_destroy(&data->lock);
  pthread_mutex_destroy(&data->run_lock);
  free(data);
}

static void * do_thread_loop(void * data);

static void thread_start(struct etpan_thread_data * data)
{
  int r;
  pthread_attr_t attr;

  switch (data->type) {
  case ETPAN_THREAD_TYPE_REMOTE_ABOOK:
    debug_op_info(data->manager->app, "starting thread remote abook", -1,
        NULL, NULL);
    break;
    
  case ETPAN_THREAD_TYPE_MISC:
    debug_op_info(data->manager->app, "starting thread misc", -1,
        NULL, NULL);
    break;
    
  case ETPAN_THREAD_TYPE_LOCAL_MAILACCESS:
    debug_op_info(data->manager->app, "starting thread local mail", -1,
        NULL, NULL);
    break;

  case ETPAN_THREAD_TYPE_REMOTE_MAILACCESS:
    debug_op_info(data->manager->app, "starting thread remote mail", -1,
        NULL, NULL);
    break;

  case ETPAN_THREAD_TYPE_SENDER:
    debug_op_info(data->manager->app, "starting thread sender", -1,
        NULL, NULL);
    break;
  }
  
  r = pthread_attr_init(&attr);
  if (r != 0)
    goto err;
  
#ifdef THREAD_STACK_SIZE
  r = pthread_attr_setstacksize(&attr, THREAD_STACK_SIZE);
  if (r != 0) {
    debug_op_info(data->manager->app, "could not set thread stack size", -1,
        NULL, NULL);
    /* ignore error */
  }
#endif
  
  r = pthread_create(&data->th_id, &attr, do_thread_loop, data);
  if (r < 0)
    goto destroy_attr;
  
  debug_op_info(data->manager->app, "thread started", -1,
      NULL, NULL);

  pthread_attr_destroy(&attr);
  return;
  
 destroy_attr:
  pthread_attr_destroy(&attr);
 err:
  debug_op_info(data->manager->app, "could not start thread", -1,
      NULL, NULL);
}

static void thread_op_run(struct etpan_thread_op * op);

static void
delete_local_storage_thread_data(struct etpan_thread_manager * thread_manager,
    struct mailstorage * storage);

static void storage_thread_run(struct etpan_thread_data * thread_data)
{
  int stop;

  ETPAN_APP_DEBUG((thread_data->manager->app, "start thread"));
  stop = 0;
  while (!stop) {
    struct etpan_thread_op * op;
    
    sem_wait(thread_data->sem);
    
    if (carray_count(thread_data->op_list) == 0) {
      debug_op_info(thread_data->manager->app, "BUG ! op list empty", -1,
          NULL, NULL);
    }
    else {
      int do_run;
      
      op = carray_get(thread_data->op_list, 0);
      
      do_run = 1;
      if (op->cmd == ETPAN_THREAD_END) {
        do_run = 0;
        stop = 1;
      }
      
      ETPAN_APP_DEBUG((thread_data->manager->app, "thread data x %p",
                          thread_data));
      if (do_run)
        thread_op_run(op);
      
      if ((thread_data->type == ETPAN_THREAD_TYPE_LOCAL_MAILACCESS) &&
          (op->data.mailaccess.storage != NULL)) {
        chashdatum key;
        chashdatum value;
        struct etpan_local_mailaccess_info * info;
        int r;
        
        key.data = &op->data.mailaccess.storage;
        key.len = sizeof(op->data.mailaccess.storage);
        lock(thread_data->manager);
        r = chash_get(thread_data->manager->thread_local_mailaccess,
            &key, &value);
        unlock(thread_data->manager);
        if (r == 0) {
          info = value.data;
          info->op_count --;
        }
      }
      
      etpan_thread_op_free(op);
      pthread_mutex_lock(&thread_data->lock);
      carray_delete_slow(thread_data->op_list, 0);
      pthread_mutex_unlock(&thread_data->lock);
    }
  }
}

static void thread_stop(struct etpan_thread_data * thread_data)
{
  thread_op_add(thread_data, ETPAN_THREAD_END,
      NULL, NULL, NULL, NULL, NULL, 1);
}

static void sync_thread_stop(struct etpan_thread_data * thread_data)
{
  ETPAN_APP_DEBUG((thread_data->manager->app, "send stop %p %i", thread_data,
        thread_data->type));
  thread_stop(thread_data);
  ETPAN_APP_DEBUG((thread_data->manager->app, "wait stop %p", thread_data));
  pthread_join(thread_data->th_id, NULL);
  ETPAN_APP_DEBUG((thread_data->manager->app, "stopped %p", thread_data));
}

static void storage_thread_stop(struct etpan_thread_manager * manager,
    struct mailstorage * storage)
{
  struct etpan_thread_data * thread_data;
  
  thread_data = get_storage_thread_data(manager, storage, 0);
  if (thread_data == NULL)
    return;
  
  switch (thread_data->type) {
  case ETPAN_THREAD_TYPE_REMOTE_MAILACCESS:
#if 0
    thread_op_add(thread_data, ETPAN_THREAD_STORAGE_DISCONNECT,
        storage, NULL, NULL, NULL, NULL, 1);
#endif

#if 0    
    debug_op_info(manager->app, "waiting end of thread",
        -1, storage, NULL);
    sync_thread_stop(thread_data);
    debug_op_info(manager->app, "thread ended",
        -1, storage, NULL);
#endif

    break;
    
  case ETPAN_THREAD_TYPE_LOCAL_MAILACCESS:
    /* do nothing */
    break;
  }
  
  delete_storage_thread_data(manager, storage);
}

static void * do_thread_loop(void * data)
{
  storage_thread_run(data);
  
  return NULL;
}

struct etpan_thread_manager * etpan_thread_manager_new(struct etpan_app * app)
{
  struct etpan_thread_manager * thread_manager;
  int r;
  int i;

  thread_manager = malloc(sizeof(* thread_manager));
  if (thread_manager == NULL)
    goto err;
  
  thread_manager->app = app;
  
  r = pthread_mutex_init(&thread_manager->thread_hash_lock, NULL);
  if (r != 0)
    goto free;

  thread_manager->last_thread_index = 0;
  r = pthread_mutex_init(&thread_manager->last_thread_index_lock, NULL);
  if (r != 0)
    goto destroy_thread_hash_lock;
  
  thread_manager->thread_hash = chash_new(CHASH_DEFAULTSIZE,
      CHASH_COPYKEY);
  if (thread_manager->thread_hash == NULL)
    goto destroy_last_thread_index_lock;
  
  thread_manager->thread_misc = NULL;

  thread_manager->thread_local_mailaccess = chash_new(CHASH_DEFAULTSIZE,
      CHASH_COPYKEY);
  if (thread_manager->thread_local_mailaccess == NULL)
    goto free_thread_hash;

  for(i = 0 ; i < ETPAN_LOCALSTORAGE_THREAD_COUNT ; i ++) {
    thread_manager->thread_tab_local_mailaccess[i] = NULL;
    thread_manager->thread_local_mailaccess_count[i] = 0;
  }
  
  thread_manager->thread_remote_mailaccess = chash_new(CHASH_DEFAULTSIZE,
      CHASH_COPYKEY);
  if (thread_manager->thread_remote_mailaccess == NULL)
    goto free_local_thread;

  thread_manager->thread_remote_abook = chash_new(CHASH_DEFAULTSIZE,
      CHASH_COPYKEY);
  if (thread_manager->thread_remote_abook == NULL)
    goto free_remote_thread;
  
  thread_manager->thread_sender = chash_new(CHASH_DEFAULTSIZE,
      CHASH_COPYKEY);
  if (thread_manager->thread_sender == NULL)
    goto free_remote_thread;
  
  thread_manager->engine = libetpan_engine_new(app->privacy);
  if (thread_manager->engine == NULL)
    goto free_remote_abook;
  
  return thread_manager;
  
 free_remote_abook:
  chash_free(thread_manager->thread_remote_abook);
 free_remote_thread:
  chash_free(thread_manager->thread_remote_mailaccess);
 free_local_thread:
  chash_free(thread_manager->thread_local_mailaccess);
 free_thread_hash:
  chash_free(thread_manager->thread_hash);
 destroy_last_thread_index_lock:
  pthread_mutex_destroy(&thread_manager->last_thread_index_lock);
 destroy_thread_hash_lock:
  pthread_mutex_destroy(&thread_manager->thread_hash_lock);
 free:
  free(thread_manager);
 err:
  return NULL;
}

void etpan_thread_manager_free(struct etpan_thread_manager * thread_manager)
{
  
  if (chash_count(thread_manager->thread_hash) != 0) {
    ETPAN_APP_DEBUG((thread_manager->app, "remaining running thread : %i",
                        chash_count(thread_manager->thread_hash)));
  }
  
  libetpan_engine_free(thread_manager->engine);
  chash_free(thread_manager->thread_sender);
  chash_free(thread_manager->thread_remote_abook);
  chash_free(thread_manager->thread_remote_mailaccess);
  chash_free(thread_manager->thread_local_mailaccess);
  chash_free(thread_manager->thread_hash);
  pthread_mutex_destroy(&thread_manager->last_thread_index_lock);
  pthread_mutex_destroy(&thread_manager->thread_hash_lock);
  free(thread_manager);
}


static void delete_thread_data(struct etpan_thread_manager * thread_manager,
    struct etpan_thread_data * thread_data);

static struct etpan_thread_data *
get_misc_thread_data(struct etpan_thread_manager * thread_manager,
    int do_create);

int etpan_thread_manager_stop(struct etpan_thread_manager * thread_manager)
{
  chashiter * cur;
  struct etpan_thread_op * op;
  struct etpan_app_config * config;
  int i;
  struct etpan_thread_data * misc_thread;
  
  config = malloc(sizeof(* config));
  if (config == NULL) {
    ETPAN_APP_LOG((thread_manager->app,
                      "load new config - not enough memory"));
    return ERROR_MEMORY;
  }
  
  memcpy(config, &thread_manager->app->config, sizeof(* config));
  
  op = etpan_thread_op_add(thread_manager,
      ETPAN_THREAD_DONE_CONFIG,
      NULL, NULL, NULL, NULL, config, 0);
  if (op == NULL) {
    /* ignore error */
  }
  
  etpan_thread_op_wait(op);
  etpan_thread_op_ack(op);
#if 0
  for(cur = chash_begin(thread_manager->thread_hash) ; cur != NULL ;
      cur = chash_next(thread_manager->thread_hash, cur)) {
    struct etpan_thread_data * data;
    chashdatum value;
    
    chash_value(cur, &value);
    data = value.data;
    
    debug_op_info(thread_manager->app, "waiting end of thread",
        -1, NULL, NULL);
    sync_thread_stop(data);
    
    delete_storage_thread_data(manager, storage);
  }
  chash_clear(thread_manager->thread_hash);
#endif

#if 0
  if (config->storage_config != NULL) {
    for(i = 0 ; i < carray_count(config->storage_config->storage_tab) ; i ++) {
      struct mailstorage * storage;
      struct etpan_thread_data * thread_data;
      
      storage = carray_get(config->storage_config->storage_tab, i);
      
      thread_data = get_storage_thread_data(manager, storage, 0);
      if (thread_data != NULL) {
        if (is_local_storage(storage)) {
          delete_storage_thread_data(manager, storage);
        }
      }
    }
  }
#endif
  
  /* stop misc so that we are sure that done config are all done */
  misc_thread = get_misc_thread_data(thread_manager, 0);
  if (misc_thread != NULL)
    sync_thread_stop(misc_thread);
  
  delete_misc_thread_data(thread_manager);
 
  /* terminate local threads */
  for(i = 0 ; i< ETPAN_LOCALSTORAGE_THREAD_COUNT ; i ++) {
    if (thread_manager->thread_tab_local_mailaccess[i] != NULL) {
      sync_thread_stop(thread_manager->thread_tab_local_mailaccess[i]);
      delete_thread_data_index(thread_manager,
          thread_manager->thread_tab_local_mailaccess[i]->thread_index);
      storage_thread_free(thread_manager->thread_tab_local_mailaccess[i]);
      thread_manager->thread_tab_local_mailaccess[i] = NULL;
    }
  }
  
  
#if 0
  /* unuseful */
  while ((cur = chash_begin(thread_manager->thread_hash)) != NULL) {
    struct etpan_thread_data * thread_data;
    chashdatum value;
    chashdatum key;
    struct mailstorage * storage;
    
    chash_value(cur, &value);
    thread_data = value.data;
    chash_key(cur, &key);
    memcpy(&storage, key.data, sizeof(storage));
    
    fprintf(stderr, "storage %p", storage);
#if 0
    debug_op_info(thread_manager->app, "waiting end of thread",
        -1, NULL, NULL);
    sync_thread_stop(thread_data);
    
    delete_thread_data(thread_manager, thread_data);
#endif
  }
#endif
  
  return NO_ERROR;
}

void etpan_thread_op_cancel(struct etpan_thread_op * op)
{
  static struct thread_info * info;
  
  info = get_thread_info(op->cmd);
  
  if (!info->can_cancel) {
    pthread_mutex_lock(&op->state_lock);
    switch (op->state) {
      case THREAD_STATE_DETACHED:
      case THREAD_STATE_DETACHED_EXECUTING:
      case THREAD_STATE_DETACHED_FINISHED:
        break;
        
      case THREAD_STATE_NOT_STARTED:
        op->state = THREAD_STATE_DETACHED;
        break;

      case THREAD_STATE_EXECUTING:
        op->state = THREAD_STATE_DETACHED_EXECUTING;
        break;
        
      case THREAD_STATE_WAIT:
        etpan_thread_op_wait(op);
        etpan_thread_op_ack(op);
        op->state = THREAD_STATE_DETACHED_FINISHED;
        break;
        
      case THREAD_STATE_NOTIFIED:
        /* could not detach */
        break;
    }
    pthread_mutex_unlock(&op->state_lock);
    return;
  }
  
  debug_op_info(op->thread_data->manager->app, "cancel",
      op->cmd, NULL, NULL);
  
  pthread_mutex_lock(&op->state_lock);
  switch (op->state) {
  case THREAD_STATE_DETACHED:
  case THREAD_STATE_DETACHED_EXECUTING:
  case THREAD_STATE_DETACHED_FINISHED:
    debug_op_info(op->thread_data->manager->app,
        "BUG detected could not cancel DETACHED thread operation",
        op->cmd, NULL, NULL);
    break;
  case THREAD_STATE_NOT_STARTED:
    op->state = THREAD_STATE_NOT_STARTED_CANCELLED;
    break;
  case THREAD_STATE_NOT_STARTED_CANCELLED:
    debug_op_info(op->thread_data->manager->app,
        "BUG detected thread operation already cancelled",
        op->cmd, NULL, NULL);
    break;
  case THREAD_STATE_EXECUTING:
    op->state = THREAD_STATE_EXECUTING_CANCELLED;
    break;
  case THREAD_STATE_EXECUTING_CANCELLED:
    debug_op_info(op->thread_data->manager->app,
        "BUG detected thread operation already cancelled",
        op->cmd, NULL, NULL);
    break;
  case THREAD_STATE_WAIT:
    op->state = THREAD_STATE_WAIT_CANCELLED;
    etpan_thread_op_wait(op);
    etpan_thread_op_ack(op);
    break;
  case THREAD_STATE_WAIT_CANCELLED:
    debug_op_info(op->thread_data->manager->app,
        "thread operation already cancelled",
        op->cmd, NULL, NULL);
    break;
  case THREAD_STATE_NOTIFIED:
    debug_op_info(op->thread_data->manager->app,
        "BUG detected could not cancel NOTIFIED thread operation",
        op->cmd, NULL, NULL);
    break;
  }
  pthread_mutex_unlock(&op->state_lock);
}

/*
  next_state()
  
  go to next state.
  returns -1 if the thread was cancelled.
*/

static int next_state(struct etpan_thread_op * op)
{
  int cancelled;
  int detached;
  
  cancelled = 0;
  detached = 0;
  pthread_mutex_lock(&op->state_lock);
  switch (op->state) {
  case THREAD_STATE_DETACHED:
    op->state = THREAD_STATE_DETACHED_EXECUTING;
    detached = 1;
    break;
  case THREAD_STATE_DETACHED_EXECUTING:
    op->state = THREAD_STATE_DETACHED_FINISHED;
    detached = 1;
    break;
  case THREAD_STATE_DETACHED_FINISHED:
    debug_op_info(op->thread_data->manager->app,
        "BUG detected no next state for DETACHED FINISHED",
        op->cmd, NULL, NULL);
    detached = 1;
    break;
  case THREAD_STATE_NOT_STARTED:
    op->state = THREAD_STATE_EXECUTING;
    break;
  case THREAD_STATE_NOT_STARTED_CANCELLED:
    cancelled = 1;
    break;
  case THREAD_STATE_EXECUTING:
    op->state = THREAD_STATE_WAIT;
    break;
  case THREAD_STATE_EXECUTING_CANCELLED:
    cancelled = 1;
    break;
  case THREAD_STATE_WAIT:
    op->state = THREAD_STATE_NOTIFIED;
    break;
  case THREAD_STATE_WAIT_CANCELLED:
    cancelled = 1;
    break;
  case THREAD_STATE_NOTIFIED:
    debug_op_info(op->thread_data->manager->app,
        "BUG detected no next state for NOTIFIED",
        op->cmd, NULL, NULL);
    break;
  }
  pthread_mutex_unlock(&op->state_lock);
  
  if (cancelled) {
    debug_op_info(op->thread_data->manager->app,
        "operation cancelled",
        op->cmd, NULL, NULL);
    return -1;
  }
  
  if (detached)
    return 0;
  else
    return 1;
}

static int is_cancelled(struct etpan_thread_op * op)
{
  int cancelled;
  
  cancelled = 0;
  pthread_mutex_lock(&op->state_lock);
  switch (op->state) {
  case THREAD_STATE_NOT_STARTED_CANCELLED:
  case THREAD_STATE_EXECUTING_CANCELLED:
  case THREAD_STATE_WAIT_CANCELLED:
    cancelled = 1;
    break;
  }
  pthread_mutex_unlock(&op->state_lock);
  
  return cancelled;
}


static struct etpan_thread_op *
thread_op_add(struct etpan_thread_data * data,
    int cmd,
    struct mailstorage * storage,
    struct mailfolder * folder,
    mailmessage * msg,
    struct mailmime * mime,
    void * arg, int detached)
{
  struct etpan_thread_op * op;
  int r;
  
  ETPAN_APP_DEBUG((data->manager->app, "thread data %p", data));
  debug_op_info(data->manager->app, "enqueue op",
      cmd, storage, folder);
  
  op = etpan_thread_op_new(data, cmd,
      storage, folder, msg, mime, arg, detached);
  if (op == NULL) {
    goto err;
  }
  
  pthread_mutex_lock(&data->lock);
  r = carray_add(data->op_list, op, NULL);
  pthread_mutex_unlock(&data->lock);
  if (r < 0) {
    goto free_op;
  }
  
  debug_op_info(data->manager->app, "op enqueued",
      cmd, storage, folder);
  
  sem_post(data->sem);
  
  return op;

 free_op:
  etpan_thread_op_free(op);
 err:
  debug_op_info(data->manager->app, "could not enqueue op",
      cmd, storage, folder);
  return NULL;
}

int etpan_storage_op_count(struct etpan_thread_manager * thread_manager,
    struct mailstorage * storage)
{
  chashdatum key;
  chashdatum result;
  struct etpan_thread_data * data;
  int r;
  
  key.data = &storage;
  key.len = sizeof(storage);
  lock(thread_manager);
  r = chash_get(thread_manager->thread_hash, &key, &result);
  unlock(thread_manager);
  if (r < 0)
    data = NULL;
  else 
    data = result.data;
  
  return carray_count(data->op_list);
}


struct etpan_thread_op *
etpan_thread_op_add(struct etpan_thread_manager * thread_manager,
    int cmd,
    struct mailstorage * storage,
    struct mailfolder * folder,
    mailmessage * msg,
    struct mailmime * mime,
    void * arg, int detached)
{
  struct etpan_thread_data * data;
  struct etpan_thread_op * op;
  
  if (storage == NULL) {
    data = get_misc_thread_data(thread_manager, 1);
  }
  else {
    data = get_storage_thread_data(thread_manager, storage, 1);
  }
  if (data == NULL)
    return NULL;
  
  if (data->type == ETPAN_THREAD_TYPE_LOCAL_MAILACCESS) {
    chashdatum key;
    chashdatum value;
    struct etpan_local_mailaccess_info * info;
    
    key.data = &storage;
    key.len = sizeof(storage);
    
    lock(thread_manager);
    chash_get(thread_manager->thread_local_mailaccess, &key, &value);
    unlock(thread_manager);
    info = value.data;
    info->op_count ++;
  }
  
  op = thread_op_add(data, cmd, storage, folder, msg, mime, arg, detached);
  
  return op;
}

void etpan_thread_op_wait(struct etpan_thread_op * op)
{
  int ack;
  
  read(op->thread_data->fd[0], &ack, sizeof(ack));
}

void etpan_thread_op_ack(struct etpan_thread_op * op)
{
  int ack;
  
  ack = 0;
  write(op->thread_data->fd[0], &ack, sizeof(ack));
}

static inline void etpan_thread_op_end_notify(struct etpan_thread_op * op)
{
  int ack;
  
  ack = 0;
  write(op->thread_data->fd[1], &ack, sizeof(ack));
}

static inline void etpan_thread_op_ack_wait(struct etpan_thread_op * op)
{
  int ack;
  
  read(op->thread_data->fd[1], &ack, sizeof(ack));
}

int etpan_thread_op_is_finished(struct etpan_thread_op * op)
{
  int result;
  
  pthread_mutex_lock(&op->state_lock);
  switch (op->state) {
  case THREAD_STATE_WAIT:
  case THREAD_STATE_NOTIFIED:
  case THREAD_STATE_DETACHED_FINISHED:
    result = 1;
    break;
  default:
    result = 0;
    break;
  }
  pthread_mutex_unlock(&op->state_lock);

  return result;
}

void etpan_thread_op_set_fds(struct etpan_thread_op * op, fd_set * fds,
    int * maxfd)
{
  FD_SET(op->thread_data->fd[0], fds);
  if (op->thread_data->fd[0] > * maxfd)
    * maxfd = op->thread_data->fd[0];
}



static void clean_arg(struct etpan_thread_op * op)
{
  struct thread_info * info;
  
  info = get_thread_info(op->cmd);
  
  if (info == NULL)
    return;
  
  if (info->clean_arg == NULL)
    return;
  
  if (op->arg == NULL)
    return;
  
  info->clean_arg(op);
}

static void clean_result(struct etpan_thread_op * op)
{
  struct thread_info * info;
  
  info = get_thread_info(op->cmd);
  
  if (info == NULL)
    return;
  
  if (info->clean_result == NULL)
    return;
  
  if (op->result == NULL)
    return;
  
  info->clean_result(op);
}


static inline void clean_before_start(struct etpan_thread_op * op)
{
  clean_arg(op);
}

static inline void clean_after_start(struct etpan_thread_op * op)
{
  clean_result(op);
  clean_arg(op);
}

static void thread_op_run(struct etpan_thread_op * op)
{
  int r;
  
  /* go to EXECUTING state */
  if (next_state(op) < 0) {
    clean_before_start(op);
    return;
  }
  
  r = storage_execute_op(op);
  op->err = r;
  
  /* go to WAIT state */
  r = next_state(op);
  if (r < 0) {
    clean_after_start(op);
    return;
  }
  
  if (r == 0) {
    debug_op_info(op->thread_data->manager->app, "detached end",
        op->cmd, NULL, NULL);
    clean_after_start(op);
    /* detached, do not notify */
    return;
  }
  
  debug_op_info(op->thread_data->manager->app, "send notification",
      op->cmd, NULL, NULL);
  etpan_thread_op_end_notify(op);
  debug_op_info(op->thread_data->manager->app,
      "wait for acknowledge of notification",
      op->cmd, NULL, NULL);
  etpan_thread_op_ack_wait(op);
  debug_op_info(op->thread_data->manager->app,
      "acknowledged",
      op->cmd, NULL, NULL);
  
  /* go to NOTIFIED state */
  if (next_state(op) < 0) {
    clean_after_start(op);
    return;
  }
}


/*
  etpan_stop_storage_thread
  
  * WARNING *
  this must not be called threaded !
*/

#if 0
void etpan_stop_storage_thread(struct etpan_thread_manager * thread_manager,
    struct mailstorage * storage)
{
  struct etpan_thread_data * thread_data;
  chashdatum key;
  
  thread_data = get_thread_data(thread_manager, storage, 0);
  if (thread_data == NULL)
    return;
  
  thread_op_add(thread_data, ETPAN_THREAD_END,
      NULL, NULL, NULL, NULL, NULL, 0);
  
  debug_op_info(thread_manager->app, "waiting end of thread",
      -1, NULL, NULL);
  
  pthread_join(thread_data->th_id, NULL);
  
  storage_thread_free(thread_data);
  
  key.data = &storage;
  key.len = sizeof(storage);
  chash_delete(thread_manager->thread_hash, &key, NULL);
  
  return;
}
#endif

/* helper functions */

/* message/folder/storage */

struct etpan_thread_op *
etpan_thread_msg_op_add(struct etpan_thread_manager * thread_manager,
    int cmd,
    mailmessage * msg,
    struct mailmime * mime,
    void * arg, int detached)
{
  struct mailfolder * folder;
#if 0
  chashdatum key;
  chashdatum value;
  int r;
#endif

#if 0  
  key.data = (char *) &msg;
  key.len = sizeof(msg);
  
  r = chash_get(thread_manager->message_hash, &key, &value);
  if (r == 0)
    folder = (struct mailfolder *) value.data;
  else
    folder = NULL;
#endif
  folder = libetpan_message_get_folder(thread_manager->engine, msg);

  return etpan_thread_folder_op_add(thread_manager,
      cmd, folder, msg, mime, arg, detached);
}


struct etpan_thread_op *
etpan_thread_folder_op_add(struct etpan_thread_manager * thread_manager,
    int cmd,
    struct mailfolder * folder,
    mailmessage * msg,
    struct mailmime * mime,
    void * arg, int detached)
{
  struct mailstorage * storage;
  
  if (folder != NULL)
    storage = folder->fld_storage;
  else
    storage = NULL;
  
  return etpan_thread_op_add(thread_manager, cmd,
    storage, folder, msg, mime, arg, detached);
}


static struct etpan_thread_data *
lookup_thread_data_index(struct etpan_thread_manager * thread_manager,
    int thread_index)
{
  chashdatum key;
  chashdatum value;
  int r;
  struct etpan_thread_data * thread_data;

  key.data = &thread_index;
  key.len = sizeof(thread_index);
  lock(thread_manager);
  r = chash_get(thread_manager->thread_hash, &key, &value);
  unlock(thread_manager);
  if (r < 0)
    return NULL;
  
  thread_data = value.data;
  
  return thread_data;
}

static int
set_thread_data_index(struct etpan_thread_manager * thread_manager,
    int thread_index, struct etpan_thread_data * thread_data)
{
  chashdatum key;
  chashdatum value;
  int r;

  key.data = &thread_index;
  key.len = sizeof(thread_index);
  value.data = thread_data;
  value.len = 0;
  lock(thread_manager);
  r = chash_set(thread_manager->thread_hash, &key, &value, NULL);
  unlock(thread_manager);
  if (r < 0)
    return ERROR_MEMORY;
  
  return NO_ERROR;
}

static void
delete_thread_data_index(struct etpan_thread_manager * thread_manager,
    int thread_index)
{
  chashdatum key;
  
  key.data = &thread_index;
  key.len = sizeof(thread_index);
  lock(thread_manager);
  chash_delete(thread_manager->thread_hash, &key, NULL);
  unlock(thread_manager);
}

static struct etpan_thread_data *
lookup_remote_storage_thread_data(struct etpan_thread_manager * thread_manager,
    struct mailstorage * storage)
{
  chashdatum key;
  chashdatum value;
  int r;
  struct etpan_thread_data * thread_data;
  
  key.data = &storage;
  key.len = sizeof(storage);
  lock(thread_manager);
  r = chash_get(thread_manager->thread_remote_mailaccess, &key, &value);
  unlock(thread_manager);
  if (r < 0)
    return NULL;

#if 0  
  pthread_index = value.data;
  
  return lookup_thread_data_index(thread_manager, * pthread_index);
#endif

  thread_data = value.data;

  return thread_data;
}

static void
delete_remote_storage_thread_data(struct etpan_thread_manager * thread_manager,
    struct mailstorage * storage)
{
  struct etpan_thread_data * thread_data;
  chashdatum key;

  thread_data = lookup_remote_storage_thread_data(thread_manager, storage);
  sync_thread_stop(thread_data);
  
  key.data = &storage;
  key.len = sizeof(storage);
  lock(thread_manager);
  chash_delete(thread_manager->thread_remote_mailaccess, &key, NULL);
  unlock(thread_manager);
  delete_thread_data_index(thread_manager, thread_data->thread_index);

  libetpan_storage_remove(thread_manager->engine, storage);
  
  storage_thread_free(thread_data);
}

static int
set_remote_storage_thread_data(struct etpan_thread_manager * thread_manager,
    struct mailstorage * storage, struct etpan_thread_data * thread_data)
{
  chashdatum key;
  chashdatum value;
  int r;
  int res;
  
  r = libetpan_storage_add(thread_manager->engine, storage);
  if (r != MAIL_NO_ERROR) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  r = set_thread_data_index(thread_manager, thread_data->thread_index,
      thread_data);
  if (r != NO_ERROR) {
    res = r;
    goto remove_storage;
  }
  
  key.data = &storage;
  key.len = sizeof(storage);
  value.data = thread_data;
  value.len = 0;
  lock(thread_manager);
  r = chash_set(thread_manager->thread_remote_mailaccess, &key, &value, NULL);
  unlock(thread_manager);
  if (r < 0) {
    res = ERROR_MEMORY;
    goto remove_thread;
  }
  
  return NO_ERROR;

 remove_thread:
  delete_thread_data_index(thread_manager, thread_data->thread_index);
 remove_storage:
  libetpan_storage_remove(thread_manager->engine, storage);
 err:
  return res;
}

static struct etpan_thread_data *
lookup_local_storage_thread_data(struct etpan_thread_manager * thread_manager,
    struct mailstorage * storage)
{
  chashdatum key;
  chashdatum value;
  int r;
  struct etpan_local_mailaccess_info * info;
  
  key.data = &storage;
  key.len = sizeof(storage);
  lock(thread_manager);
  r = chash_get(thread_manager->thread_local_mailaccess, &key, &value);
  unlock(thread_manager);
  if (r < 0)
    return NULL;

#if 0  
  pthread_index = value.data;
  
  return lookup_thread_data_index(thread_manager, * pthread_index);
#endif
  
  info = value.data;
  
  return info->thread_data;
}

static int
set_local_storage_thread_data(struct etpan_thread_manager * thread_manager,
    struct mailstorage * storage, struct etpan_thread_data * thread_data)
{
  chashdatum key;
  chashdatum value;
  int r;
  int res;
  struct etpan_local_mailaccess_info * info;
  
  info = malloc(sizeof(* info));
  if (info == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  info->storage = storage;
  info->thread_data = thread_data;
  info->op_count = 0;
  
  r = libetpan_storage_add(thread_manager->engine, storage);
  if (r != MAIL_NO_ERROR) {
    res = ERROR_MEMORY;
    goto free_info;
  }
  
  r = set_thread_data_index(thread_manager, thread_data->thread_index,
      thread_data);
  if (r != NO_ERROR) {
    res = r;
    goto remove_storage;
  }
  
  ETPAN_APP_DEBUG((thread_manager->app, "add storage %s %p",
                      storage->sto_id, thread_data));
  
  key.data = &storage;
  key.len = sizeof(storage);
  value.data = info;
  value.len = 0;
  lock(thread_manager);
  r = chash_set(thread_manager->thread_local_mailaccess, &key, &value, NULL);
  unlock(thread_manager);
  if (r < 0) {
    res = ERROR_MEMORY;
    goto remove_thread;
  }
  
  return NO_ERROR;
  
 remove_thread:
  delete_thread_data_index(thread_manager, thread_data->thread_index);
 remove_storage:
  libetpan_storage_remove(thread_manager->engine, storage);
 free_info:
  free(info);
 err:
  return res;
}

static void
delete_local_storage_thread_data(struct etpan_thread_manager * thread_manager,
    struct mailstorage * storage)
{
  struct etpan_thread_data * thread_data;
  chashdatum key;
  chashdatum value;
  int i;
  struct etpan_local_mailaccess_info * info;
  int r;
  
  thread_data = lookup_local_storage_thread_data(thread_manager, storage);

  key.data = &storage;
  key.len = sizeof(storage);
  lock(thread_manager);
  r = chash_get(thread_manager->thread_local_mailaccess, &key, &value);
  unlock(thread_manager);
  if (r < 0)
    return;

  info = value.data;
  
  if (info == NULL)
    return;
  
  lock(thread_manager);
  chash_delete(thread_manager->thread_local_mailaccess, &key, NULL);
  unlock(thread_manager);
  
  free(info);
  
  lock(thread_manager);
  for(i = 0 ; i < ETPAN_LOCALSTORAGE_THREAD_COUNT ; i ++)
    if (thread_data == thread_manager->thread_tab_local_mailaccess[i])
      thread_manager->thread_local_mailaccess_count[i] --;
  unlock(thread_manager);
  
  libetpan_storage_remove(thread_manager->engine, storage);
}

static int is_local_storage(struct mailstorage * storage)
{
  int type;
  int remote_storage;
  
  type = etpan_cfg_storage_get_type(storage->sto_driver->sto_name);
  
  remote_storage = 0;
  
  switch (type) {
  case STORAGE_TYPE_IMAP:
  case STORAGE_TYPE_POP3:
  case STORAGE_TYPE_NNTP:
    remote_storage = 1;
    break;
    
  case STORAGE_TYPE_MH:
  case STORAGE_TYPE_MBOX:
  case STORAGE_TYPE_MAILDIR:
  case STORAGE_TYPE_DB:
    break;
  }
  
  return !remote_storage;
}

static void
delete_storage_thread_data(struct etpan_thread_manager * thread_manager,
    struct mailstorage * storage)
{
  if (!is_local_storage(storage))
    delete_remote_storage_thread_data(thread_manager, storage);
  else
    delete_local_storage_thread_data(thread_manager, storage);
}

static struct etpan_thread_data *
get_storage_thread_data(struct etpan_thread_manager * thread_manager,
    struct mailstorage * storage, int do_create)
{
  struct etpan_thread_data * thread_data;
  int r;
  int type;
  int remote_storage;
  
  type = etpan_cfg_storage_get_type(storage->sto_driver->sto_name);
  
  remote_storage = 0;
  
  switch (type) {
  case STORAGE_TYPE_IMAP:
  case STORAGE_TYPE_POP3:
  case STORAGE_TYPE_NNTP:
    remote_storage = 1;
    break;
    
  case STORAGE_TYPE_MH:
  case STORAGE_TYPE_MBOX:
  case STORAGE_TYPE_MAILDIR:
  case STORAGE_TYPE_DB:
    break;
  }
  
  if (remote_storage)
    thread_data = lookup_remote_storage_thread_data(thread_manager, storage);
  else
    thread_data = lookup_local_storage_thread_data(thread_manager, storage);
  
  if (do_create && (thread_data == NULL)) {
    int created;
    
    /* add new thread */
    
    created = 0;
    
    if (remote_storage) {
      thread_data = storage_thread_new(thread_manager,
          ETPAN_THREAD_TYPE_REMOTE_MAILACCESS);
      created = 1;
    }
    else {
      int count_min;
      int i;
      int i_min;

      lock(thread_manager);
      i_min = 0;
      count_min = thread_manager->thread_local_mailaccess_count[0];
      ETPAN_APP_DEBUG((thread_manager->app, "count %i %i",
                          0,
                          thread_manager->thread_local_mailaccess_count[0]));
      for(i = 1 ; i < ETPAN_LOCALSTORAGE_THREAD_COUNT ; i ++) {
        int count;
        
        ETPAN_APP_DEBUG((thread_manager->app, "count %i %i",
                            i,
                            thread_manager->thread_local_mailaccess_count[i]));
        count = thread_manager->thread_local_mailaccess_count[i];
        if (count < count_min) {
          count_min = count;
          i_min = i;
        }
      }
      unlock(thread_manager);
      
      lock(thread_manager);
      thread_data = thread_manager->thread_tab_local_mailaccess[i_min];
      if (thread_data == NULL) {
        thread_manager->thread_tab_local_mailaccess[i_min] =
          storage_thread_new(thread_manager,
              ETPAN_THREAD_TYPE_LOCAL_MAILACCESS);
        thread_data = thread_manager->thread_tab_local_mailaccess[i_min];
        created = 1;
      }
      thread_manager->thread_local_mailaccess_count[i_min] ++;
      unlock(thread_manager);
    }
    
    if (thread_data == NULL)
      goto err;
    
    if (remote_storage)
      r = set_remote_storage_thread_data(thread_manager, storage, thread_data);
    else
      r = set_local_storage_thread_data(thread_manager, storage, thread_data);
    if (r != NO_ERROR)
      goto free_thread_data;
    
    if (created)
      thread_start(thread_data);
  }
  
  return thread_data;
  
 free_thread_data:
  if (remote_storage)
    storage_thread_free(thread_data);
 err:
  return NULL;
}


/* misc */

static struct etpan_thread_data *
lookup_misc_thread_data(struct etpan_thread_manager * thread_manager)
{
  return thread_manager->thread_misc;
}

static void
delete_misc_thread_data(struct etpan_thread_manager * thread_manager)
{
  delete_thread_data_index(thread_manager,
      thread_manager->thread_misc->thread_index);
  libetpan_storage_remove(thread_manager->engine, NULL);
  storage_thread_free(thread_manager->thread_misc);
  thread_manager->thread_misc = NULL;
}

static int
set_misc_thread_data(struct etpan_thread_manager * thread_manager,
    struct etpan_thread_data * thread_data)
{
  int r;
  int res;
  
  r = libetpan_storage_add(thread_manager->engine, NULL);
  if (r != MAIL_NO_ERROR) {
    res = r;
    goto err;
  }
  
  r = set_thread_data_index(thread_manager, thread_data->thread_index,
      thread_data);
  if (r != NO_ERROR) {
    res = r;
    goto remove_storage;
  }
  
  thread_manager->thread_misc = thread_data;
  
  return NO_ERROR;

 remove_storage:
  libetpan_storage_remove(thread_manager->engine, NULL);
 err:
  return res;
}

static struct etpan_thread_data *
get_misc_thread_data(struct etpan_thread_manager * thread_manager,
    int do_create)
{
  struct etpan_thread_data * thread_data;
  int r;
  
  thread_data = lookup_misc_thread_data(thread_manager);
  
  if (do_create && (thread_data == NULL)) {
    /* add new thread */
    
    thread_data = storage_thread_new(thread_manager, ETPAN_THREAD_TYPE_MISC);
    if (thread_data == NULL)
      goto err;
    
    r = set_misc_thread_data(thread_manager, thread_data);
    if (r != NO_ERROR)
      goto free_thread_data;
    
    thread_start(thread_data);
  }
  
  return thread_data;
  
 free_thread_data:
  storage_thread_free(thread_data);
 err:
  return NULL;
}

static void delete_thread_data(struct etpan_thread_manager * thread_manager,
  struct etpan_thread_data * thread_data)
{
  switch (thread_data->type) {
  case ETPAN_THREAD_TYPE_MISC:
    delete_misc_thread_data(thread_manager);
    break;
    
  case ETPAN_THREAD_TYPE_LOCAL_MAILACCESS:
    break;
    
  case ETPAN_THREAD_TYPE_REMOTE_MAILACCESS:
    ETPAN_APP_DEBUG((thread_manager->app,
                        "BUG detected - remaining mail access thread"));
    break;
    
  case ETPAN_THREAD_TYPE_REMOTE_ABOOK:
    ETPAN_APP_DEBUG((thread_manager->app,
                        "BUG detected - remaining abook thread"));
    break;

  case ETPAN_THREAD_TYPE_SENDER:
    ETPAN_APP_DEBUG((thread_manager->app,
                        "BUG detected - remaining sender thread"));
    break;
  }
}

int etpan_thread_manager_start(struct etpan_thread_manager * manager)
{
  struct etpan_thread_data * thread_data;
  
  thread_data = get_misc_thread_data(manager, 1);
  if (thread_data == NULL)
    goto err;
  
  return NO_ERROR;
  
 err:
  return ERROR_MEMORY;
}


/* abook */

static struct etpan_thread_op *
etpan_thread_abook_op_new(struct etpan_thread_data * thread_data,
    int cmd,
    struct etpan_abook * abook,
    void * arg, int detached)
{
  struct etpan_thread_op * op;
  int r;
  
  op = malloc(sizeof(* op));
  if (op == NULL)
    goto err;
  
  op->data.abook.abook = abook;
  op->cmd = cmd;
  op->arg = arg;
  op->result = NULL;
  op->err = NO_ERROR;
  if (detached)
    op->state = THREAD_STATE_DETACHED;
  else
    op->state = THREAD_STATE_NOT_STARTED;
  
  r = pthread_mutex_init(&op->state_lock, NULL);
  if (r != 0)
    goto free;
  
  op->thread_data = thread_data;
  
  return op;

 free:
  free(op);
 err:
  return NULL;
}

static struct etpan_thread_data *
get_abook_thread_data(struct etpan_thread_manager * thread_manager,
    struct etpan_abook * abook, int do_create);

static struct etpan_thread_op *
thread_abook_op_add(struct etpan_thread_data * data,
    int cmd,
    struct etpan_abook * abook,
    void * arg, int detached)
{
  struct etpan_thread_op * op;
  int r;
  
  ETPAN_APP_DEBUG((data->manager->app, "enqueue op abook"));
  
  op = etpan_thread_abook_op_new(data, cmd,
      abook, arg, detached);
  if (op == NULL) {
    goto err;
  }
  
  pthread_mutex_lock(&data->lock);
  r = carray_add(data->op_list, op, NULL);
  pthread_mutex_unlock(&data->lock);
  if (r < 0) {
    goto free_op;
  }
  
  debug_op_info(data->manager->app, "op enqueued abook",
      cmd, NULL, NULL);
  
  sem_post(data->sem);
  
  return op;

 free_op:
  etpan_thread_op_free(op);
 err:
  debug_op_info(data->manager->app, "could not enqueue op",
      cmd, NULL, NULL);
  return NULL;
}


struct etpan_thread_op *
etpan_thread_abook_op_add(struct etpan_thread_manager * thread_manager,
    int cmd,
    struct etpan_abook * abook,
    void * arg, int detached)
{
  struct etpan_thread_data * data;
  struct etpan_thread_op * op;
  
  data = get_abook_thread_data(thread_manager, abook, 1);
  if (data == NULL)
    return NULL;
  
  op = thread_abook_op_add(data, cmd, abook, arg, detached);
  
  return op;
}

static struct etpan_thread_data *
lookup_remote_abook_thread_data(struct etpan_thread_manager * thread_manager,
    struct etpan_abook * abook)
{
  chashdatum key;
  chashdatum value;
  int r;
  struct etpan_thread_data * thread_data;
  
  key.data = &abook;
  key.len = sizeof(abook);
  lock(thread_manager);
  r = chash_get(thread_manager->thread_remote_abook, &key, &value);
  unlock(thread_manager);
  if (r < 0)
    return NULL;

#if 0  
  pthread_index = value.data;
  
  return lookup_thread_data_index(thread_manager, * pthread_index);
#endif
  
  thread_data = value.data;
  
  return thread_data;
}

static void
delete_remote_abook_thread_data(struct etpan_thread_manager * thread_manager,
    struct etpan_abook * abook)
{
  struct etpan_thread_data * thread_data;
  chashdatum key;

  thread_data = lookup_remote_abook_thread_data(thread_manager, abook);
  sync_thread_stop(thread_data);
  
  key.data = &abook;
  key.len = sizeof(abook);
  lock(thread_manager);
  chash_delete(thread_manager->thread_remote_abook, &key, NULL);
  unlock(thread_manager);
  delete_thread_data_index(thread_manager, thread_data->thread_index);
  
  storage_thread_free(thread_data);
}

static int
set_remote_abook_thread_data(struct etpan_thread_manager * thread_manager,
    struct etpan_abook * abook, struct etpan_thread_data * thread_data)
{
  chashdatum key;
  chashdatum value;
  int r;
  int res;
  
  r = set_thread_data_index(thread_manager, thread_data->thread_index,
      thread_data);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  key.data = &abook;
  key.len = sizeof(abook);
  value.data = thread_data;
  value.len = 0;
  lock(thread_manager);
  r = chash_set(thread_manager->thread_remote_abook, &key, &value, NULL);
  unlock(thread_manager);
  if (r < 0) {
    res = ERROR_MEMORY;
    goto remove_thread;
  }
  
  return NO_ERROR;

 remove_thread:
  delete_thread_data_index(thread_manager, thread_data->thread_index);
 err:
  return res;
}

static struct etpan_thread_data *
get_abook_thread_data(struct etpan_thread_manager * thread_manager,
    struct etpan_abook * abook, int do_create)
{
  struct etpan_thread_data * thread_data;
  int r;
  int remote_abook;
  
  remote_abook = 1;
  
  if (is_local_abook(abook))
    remote_abook = 0;
  
  if (!remote_abook)
    return get_misc_thread_data(thread_manager, do_create);
  
  thread_data = lookup_remote_abook_thread_data(thread_manager, abook);
  
  if (do_create && (thread_data == NULL)) {
    int created;
    
    /* add new thread */
    
    created = 0;
    
    thread_data = storage_thread_new(thread_manager,
        ETPAN_THREAD_TYPE_REMOTE_ABOOK);
    if (thread_data == NULL)
      goto err;
    
    r = set_remote_abook_thread_data(thread_manager, abook, thread_data);
    if (r != NO_ERROR)
      goto free_thread_data;
    
    thread_start(thread_data);
  }
  
  return thread_data;
  
 free_thread_data:
  if (remote_abook)
    storage_thread_free(thread_data);
 err:
  return NULL;
}

static void
delete_abook_thread_data(struct etpan_thread_manager * thread_manager,
    struct etpan_abook * abook)
{
  int remote_abook;
  
  remote_abook = 1;
  
  if (is_local_abook(abook))
    remote_abook = 0;
  
  if (remote_abook)
    delete_remote_abook_thread_data(thread_manager, abook);
}


/* sender */

static struct etpan_thread_op *
etpan_thread_sender_op_new(struct etpan_thread_data * thread_data,
    int cmd,
    struct etpan_sender_item * sender,
    void * arg, int detached)
{
  struct etpan_thread_op * op;
  int r;
  
  op = malloc(sizeof(* op));
  if (op == NULL)
    goto err;
  
  op->data.sender.sender = sender;
  op->cmd = cmd;
  op->arg = arg;
  op->result = NULL;
  op->err = NO_ERROR;
  if (detached)
    op->state = THREAD_STATE_DETACHED;
  else
    op->state = THREAD_STATE_NOT_STARTED;
  
  r = pthread_mutex_init(&op->state_lock, NULL);
  if (r != 0)
    goto free;
  
  op->thread_data = thread_data;
  
  return op;

 free:
  free(op);
 err:
  return NULL;
}

static struct etpan_thread_data *
get_sender_thread_data(struct etpan_thread_manager * thread_manager,
    struct etpan_sender_item * sender, int do_create);

static struct etpan_thread_op *
thread_sender_op_add(struct etpan_thread_data * data,
    int cmd,
    struct etpan_sender_item * sender,
    void * arg, int detached)
{
  struct etpan_thread_op * op;
  int r;
  
  ETPAN_APP_DEBUG((data->manager->app, "enqueue op sender"));
  
  op = etpan_thread_sender_op_new(data, cmd,
      sender, arg, detached);
  if (op == NULL) {
    goto err;
  }
  
  pthread_mutex_lock(&data->lock);
  r = carray_add(data->op_list, op, NULL);
  pthread_mutex_unlock(&data->lock);
  if (r < 0) {
    goto free_op;
  }
  
  debug_op_info(data->manager->app, "op enqueued sender",
      cmd, NULL, NULL);
  
  sem_post(data->sem);
  
  return op;

 free_op:
  etpan_thread_op_free(op);
 err:
  debug_op_info(data->manager->app, "could not enqueue op",
      cmd, NULL, NULL);
  return NULL;
}


struct etpan_thread_op *
etpan_thread_sender_op_add(struct etpan_thread_manager * thread_manager,
    int cmd,
    struct etpan_sender_item * sender,
    void * arg, int detached)
{
  struct etpan_thread_data * data;
  struct etpan_thread_op * op;
  
  data = get_sender_thread_data(thread_manager, sender, 1);
  if (data == NULL)
    return NULL;
  
  op = thread_sender_op_add(data, cmd, sender, arg, detached);
  
  return op;
}

static struct etpan_thread_data *
lookup_remote_sender_thread_data(struct etpan_thread_manager * thread_manager,
    struct etpan_sender_item * sender)
{
  chashdatum key;
  chashdatum value;
  int r;
  struct etpan_thread_data * thread_data;
  
  key.data = &sender;
  key.len = sizeof(sender);
  lock(thread_manager);
  r = chash_get(thread_manager->thread_sender, &key, &value);
  unlock(thread_manager);
  if (r < 0)
    return NULL;

#if 0  
  pthread_index = value.data;
  
  return lookup_thread_data_index(thread_manager, * pthread_index);
#endif
  
  thread_data = value.data;
  
  return thread_data;
}

static void
delete_remote_sender_thread_data(struct etpan_thread_manager * thread_manager,
    struct etpan_sender_item * sender)
{
  struct etpan_thread_data * thread_data;
  chashdatum key;

  thread_data = lookup_remote_sender_thread_data(thread_manager, sender);
  if (thread_data == NULL)
    return;
  
  sync_thread_stop(thread_data);
  
  key.data = &sender;
  key.len = sizeof(sender);
  lock(thread_manager);
  chash_delete(thread_manager->thread_sender, &key, NULL);
  unlock(thread_manager);
  delete_thread_data_index(thread_manager, thread_data->thread_index);
  
  storage_thread_free(thread_data);
}

static int
set_remote_sender_thread_data(struct etpan_thread_manager * thread_manager,
    struct etpan_sender_item * sender, struct etpan_thread_data * thread_data)
{
  chashdatum key;
  chashdatum value;
  int r;
  int res;
  
  r = set_thread_data_index(thread_manager, thread_data->thread_index,
      thread_data);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  key.data = &sender;
  key.len = sizeof(sender);
  value.data = thread_data;
  value.len = 0;
  lock(thread_manager);
  r = chash_set(thread_manager->thread_sender, &key, &value, NULL);
  unlock(thread_manager);
  if (r < 0) {
    res = ERROR_MEMORY;
    goto remove_thread;
  }
  
  return NO_ERROR;

 remove_thread:
  delete_thread_data_index(thread_manager, thread_data->thread_index);
 err:
  return res;
}

static struct etpan_thread_data *
get_sender_thread_data(struct etpan_thread_manager * thread_manager,
    struct etpan_sender_item * sender, int do_create)
{
  struct etpan_thread_data * thread_data;
  int r;
  int remote_sender;
  
  remote_sender = 1;
  
  if (sender->type == SENDER_TYPE_COMMAND)
    remote_sender = 0;
  
  if (!remote_sender)
    return get_misc_thread_data(thread_manager, do_create);
  
  thread_data = lookup_remote_sender_thread_data(thread_manager, sender);
  
  if (do_create && (thread_data == NULL)) {
    int created;
    
    /* add new thread */
    
    created = 0;
    
    thread_data = storage_thread_new(thread_manager,
        ETPAN_THREAD_TYPE_SENDER);
    if (thread_data == NULL)
      goto err;
    
    r = set_remote_sender_thread_data(thread_manager, sender, thread_data);
    if (r != NO_ERROR)
      goto free_thread_data;
    
    thread_start(thread_data);
  }
  
  return thread_data;
  
 free_thread_data:
  if (remote_sender)
    storage_thread_free(thread_data);
 err:
  return NULL;
}

static void
delete_sender_thread_data(struct etpan_thread_manager * thread_manager,
    struct etpan_sender_item * sender)
{
  int remote_sender;
  
  remote_sender = 1;
  
  if (sender->type == SENDER_TYPE_COMMAND)
    remote_sender = 0;
  
  if (remote_sender)
    delete_remote_sender_thread_data(thread_manager, sender);
}

static void lock(struct etpan_thread_manager * manager)
{
  pthread_mutex_lock(&manager->thread_hash_lock);
}

static void unlock(struct etpan_thread_manager * manager)
{
  pthread_mutex_unlock(&manager->thread_hash_lock);
}

