/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-cfg-edit-common.c,v 1.5 2004/12/17 10:20:51 hoa Exp $
 */

#include "etpan-cfg-edit-common.h"

#include "etpan-app.h"
#include "etpan-subapp.h"
#include "etpan-app-subapp.h"
#include "etpan-search-input.h"
#include "etpan-filename-input.h"
#include "etpan-address-input.h"
#include "etpan-errors.h"
#include <stdlib.h>
#include <string.h>
#include "etpan-imf-helper.h"
#include "etpan-mime-tools.h"
#include "etpan-nntp-input.h"
#include "etpan-imap-input.h"

enum {
  EDIT_STRING,
  EDIT_FILENAME,
  EDIT_NEWSGROUP,
  EDIT_IMAP_MAILBOX,
  EDIT_INT,
  EDIT_ADDRESS,
  EDIT_TIME,
};

struct string_edit_data {
  int type;
  int * dest_int;
  char * dest_str;
  time_t * dest_time;
  struct mailimf_address_list ** dest_addr;
  size_t size;
  struct etpan_subapp * app;
};


static time_t parse_time(char * str)
{
  struct tm time_date;
  time_t now;
  int mday;
  int month;
  int year;
  int hour;
  int min;
  int sec;
  char * time_part;
  int r;
  
  now = time(NULL);
  
  localtime_r(&now, &time_date);
  time_date.tm_isdst = -1;
  
  time_part = strchr(str, ' ');

  r = sscanf(str, "%i/%i/%i", &mday, &month, &year);
  if (r >= 2) {
    time_date.tm_mday = mday;
    time_date.tm_mon = month - 1;
    if (r == 3)
      time_date.tm_year = year - 1900;
  }
  
  if (time_part != NULL) {
    r = sscanf(time_part, "%i:%i:%i", &hour, &min, &sec);
    if (r >= 2) {
      time_date.tm_hour = hour;
      time_date.tm_min = min;
      if (r == 3)
        time_date.tm_sec = sec;
      else
        time_date.tm_sec = 0;
    }
  }

  return mktime(&time_date);
}

static void edit_upcall(struct etpan_subapp * input_app, int valid,
    void * data)
{
  char * val;
  int * dest_int;
  time_t * dest_time;
  char * dest_str;
  struct mailimf_address_list ** dest_addr;
  size_t size;
  int type;
  struct etpan_subapp * app;
  struct string_edit_data * edit_data;
  size_t cur_token;
  struct mailimf_address_list * addr_list;
  int r;
  
  edit_data = data;
  type = edit_data->type;
  dest_str = edit_data->dest_str;
  dest_int = edit_data->dest_int;
  dest_time = edit_data->dest_time;
  dest_addr = edit_data->dest_addr;
  size = edit_data->size;
  app = edit_data->app;
  free(edit_data);

  if (valid == ETPAN_INPUT_COMMON_CANCEL) {
    ETPAN_APP_LOG((app->app, "cancel edition"));
    goto err;
  }
  
  switch (type) {
  case EDIT_FILENAME:
    val = etpan_filename_input_get_value(input_app);
    break;
  case EDIT_STRING:
  case EDIT_INT:
  case EDIT_TIME:
    val = etpan_search_input_get_value(input_app);
    break;
  case EDIT_ADDRESS:
    val = etpan_address_input_get_value(input_app);
    break;
  case EDIT_NEWSGROUP:
    val = etpan_nntp_input_get_value(input_app);
    break;
  case EDIT_IMAP_MAILBOX:
    val = etpan_imap_input_get_value(input_app);
    break;
  default:
    ETPAN_APP_DEBUG((app->app, "BUG detected - invalid type of edition"));
    goto err;
  }
  
  switch (type) {
  case EDIT_NEWSGROUP:
  case EDIT_IMAP_MAILBOX:
  case EDIT_FILENAME:
  case EDIT_STRING:
    strncpy(dest_str, val, size);
    dest_str[size - 1] = '\0';
    break;
  case EDIT_INT:
    * dest_int = strtoul(val, NULL, 10);
    break;
  case EDIT_TIME:
    * dest_time = parse_time(val);
    break;
  case EDIT_ADDRESS:
    cur_token = 0;
    r = mailimf_address_list_parse(val, strlen(val), &cur_token,
        &addr_list);
    if (r != MAILIMF_NO_ERROR) {
      ETPAN_APP_LOG((app->app, "header editor - error parsing address list"));
      goto err_no_quit;
    }
    
    etpan_addr_list_encode(app->app, addr_list);
    
    * dest_addr = addr_list;
    break;
  }
  
  etpan_app_quit_subapp(input_app);
  
  return;
  
 err:
  etpan_app_quit_subapp(input_app);
 err_no_quit:
  return;
}


int etpan_cfg_edit_string(struct etpan_subapp * app, char * prompt,
    char * string, size_t size, int flags)
{
  struct etpan_subapp * input_app;
  struct string_edit_data * edit_data;
  int r;

  input_app = etpan_app_find_subapp(app->app, "search-input",
    0, NULL, NULL);
  if (input_app == NULL) {
    input_app = etpan_search_input_new(app->app);
    if (input_app == NULL)
      return ERROR_MEMORY;
  }
  
  etpan_subapp_set_parent(input_app, app);
  
  edit_data = malloc(sizeof(* edit_data));
  if (edit_data == NULL)
    return ERROR_MEMORY;

  edit_data->type = EDIT_STRING;
  edit_data->dest_int = NULL;
  edit_data->dest_time = NULL;
  edit_data->dest_str = string;
  edit_data->dest_addr = NULL;
  edit_data->size = size;
  edit_data->app = app;
  
  r = etpan_search_input_set(input_app, prompt, size - 1, string,
      flags, edit_upcall, edit_data);
  if (r != NO_ERROR) {
    free(edit_data);
    return r;
  }
  
  etpan_app_switch_subapp(input_app, 0);
  
  return NO_ERROR;
}

int etpan_cfg_edit_filename_with_type(struct etpan_subapp * app,
    char * prompt, char * string, size_t size, int type)
{
  struct etpan_subapp * input_app;
  struct string_edit_data * edit_data;
  int r;

  input_app = etpan_app_find_subapp(app->app, "filename-input",
    0, NULL, NULL);
  if (input_app == NULL) {
    input_app = etpan_filename_input_new(app->app);
    if (input_app == NULL)
      return ERROR_MEMORY;
  }
  
  etpan_subapp_set_parent(input_app, app);
  
  edit_data = malloc(sizeof(* edit_data));
  if (edit_data == NULL)
    return ERROR_MEMORY;

  edit_data->type = EDIT_FILENAME;
  edit_data->dest_int = NULL;
  edit_data->dest_time = NULL;
  edit_data->dest_str = string;
  edit_data->dest_addr = NULL;
  edit_data->size = size;
  edit_data->app = app;
  
  if (string[0] == '\0') {
    char default_name[PATH_MAX];
    
    snprintf(default_name, sizeof(default_name), "%s", getenv("HOME"));
    r = etpan_filename_input_set(input_app, prompt, size - 1, default_name,
        edit_upcall, edit_data);
  }
  else {
    r = etpan_filename_input_set(input_app, prompt, size - 1, string,
        edit_upcall, edit_data);
  }

  if (r != NO_ERROR) {
    free(edit_data);
    return r;
  }

  etpan_filename_input_set_file_type(input_app, type);
  
  etpan_app_switch_subapp(input_app, 0);
  
  return NO_ERROR;
}

int etpan_cfg_edit_filename(struct etpan_subapp * app, char * prompt,
    char * string, size_t size)
{
  return etpan_cfg_edit_filename_with_type(app, prompt, string, size,
      FILENAME_TYPE_REGULAR);
}

int etpan_cfg_edit_newsgroup(struct etpan_subapp * app,
    struct mailstorage * storage,
    char * prompt,
    char * string, size_t size)
{
  struct etpan_subapp * input_app;
  struct string_edit_data * edit_data;
  int r;

  input_app = etpan_app_find_subapp(app->app, "nntp-input",
    0, NULL, NULL);
  if (input_app == NULL) {
    input_app = etpan_nntp_input_new(app->app);
    if (input_app == NULL)
      return ERROR_MEMORY;
  }
  
  etpan_subapp_set_parent(input_app, app);
  
  edit_data = malloc(sizeof(* edit_data));
  if (edit_data == NULL)
    return ERROR_MEMORY;

  edit_data->type = EDIT_NEWSGROUP;
  edit_data->dest_int = NULL;
  edit_data->dest_time = NULL;
  edit_data->dest_str = string;
  edit_data->dest_addr = NULL;
  edit_data->size = size;
  edit_data->app = app;
  
  r = etpan_nntp_input_set(input_app, storage,
      prompt, size - 1, string,
      edit_upcall, edit_data);
  if (r != NO_ERROR) {
    free(edit_data);
    return r;
  }
  
  etpan_app_switch_subapp(input_app, 0);
  
  return NO_ERROR;
}

int etpan_cfg_edit_imap_mailbox(struct etpan_subapp * app,
    struct mailstorage * storage,
    char * prompt,
    char * string, size_t size)
{
  struct etpan_subapp * input_app;
  struct string_edit_data * edit_data;
  int r;

  input_app = etpan_app_find_subapp(app->app, "imap-input",
    0, NULL, NULL);
  if (input_app == NULL) {
    input_app = etpan_imap_input_new(app->app);
    if (input_app == NULL)
      return ERROR_MEMORY;
  }
  
  etpan_subapp_set_parent(input_app, app);
  
  edit_data = malloc(sizeof(* edit_data));
  if (edit_data == NULL)
    return ERROR_MEMORY;

  edit_data->type = EDIT_IMAP_MAILBOX;
  edit_data->dest_int = NULL;
  edit_data->dest_time = NULL;
  edit_data->dest_str = string;
  edit_data->dest_addr = NULL;
  edit_data->size = size;
  edit_data->app = app;
  
  r = etpan_imap_input_set(input_app, storage,
      prompt, size - 1, string,
      edit_upcall, edit_data);
  if (r != NO_ERROR) {
    free(edit_data);
    return r;
  }
  
  etpan_app_switch_subapp(input_app, 0);
  
  return NO_ERROR;
}

int etpan_cfg_edit_int(struct etpan_subapp * app, char * prompt,
    int * value)
{
  struct etpan_subapp * input_app;
  struct string_edit_data * edit_data;
  int r;
  char value_str[20];

  input_app = etpan_app_find_subapp(app->app, "search-input",
    0, NULL, NULL);
  if (input_app == NULL) {
    input_app = etpan_search_input_new(app->app);
    if (input_app == NULL)
      return ERROR_MEMORY;
  }
  
  etpan_subapp_set_parent(input_app, app);
  
  edit_data = malloc(sizeof(* edit_data));
  if (edit_data == NULL)
    return ERROR_MEMORY;

  edit_data->type = EDIT_INT;
  edit_data->dest_int = value;
  edit_data->dest_time = NULL;
  edit_data->dest_str = NULL;
  edit_data->dest_addr = NULL;
  edit_data->size = 0;
  edit_data->app = app;
  
  snprintf(value_str, sizeof(value_str), "%i", * value);
  
  r = etpan_search_input_set(input_app, prompt, 20, value_str,
      0, edit_upcall, edit_data);
  if (r != NO_ERROR) {
    free(edit_data);
    return r;
  }
  
  etpan_app_switch_subapp(input_app, 0);
  
  return NO_ERROR;
}

int etpan_cfg_edit_addr(struct etpan_subapp * app, char * prompt,
    struct mailimf_address_list ** value)
{
  struct etpan_subapp * input_app;
  struct string_edit_data * edit_data;
  int r;
  char * value_str;
  
  input_app = etpan_app_find_subapp(app->app, "address-input",
    0, NULL, NULL);
  if (input_app == NULL) {
    input_app = etpan_address_input_new(app->app);
    if (input_app == NULL)
      return ERROR_MEMORY;
  }
  
  etpan_subapp_set_parent(input_app, app);
  
  edit_data = malloc(sizeof(* edit_data));
  if (edit_data == NULL)
    return ERROR_MEMORY;

  edit_data->type = EDIT_ADDRESS;
  edit_data->dest_int = NULL;
  edit_data->dest_time = NULL;
  edit_data->dest_str = NULL;
  edit_data->dest_addr = value;
  edit_data->size = 0;
  edit_data->app = app;
  
  if (* value != NULL) {
    struct mailimf_address_list * dup_addr_list;
    
    dup_addr_list = etpan_dup_address_list(* value);
    if (dup_addr_list == NULL) {
      free(edit_data);
      return ERROR_MEMORY;
    }
    
    etpan_addr_list_decode(app->app, dup_addr_list);
    value_str = mailimf_address_list_to_string(dup_addr_list);
    mailimf_address_list_free(dup_addr_list);
    if (value_str == NULL) {
      free(edit_data);
      return ERROR_MEMORY;
    }
  }
  else
    value_str = NULL;
  
  r = etpan_address_input_set(input_app, prompt, 1024, value_str,
      edit_upcall, edit_data);
  if (value_str != NULL)
    free(value_str);
  if (r != NO_ERROR) {
    free(edit_data);
    return r;
  }
  
  etpan_app_switch_subapp(input_app, 0);
  
  return NO_ERROR;
}


int etpan_cfg_edit_time(struct etpan_subapp * app, char * prompt,
    time_t * value)
{
  struct etpan_subapp * input_app;
  struct string_edit_data * edit_data;
  int r;
  char value_str[40];
  struct tm time_date;
  
  input_app = etpan_app_find_subapp(app->app, "search-input",
    0, NULL, NULL);
  if (input_app == NULL) {
    input_app = etpan_search_input_new(app->app);
    if (input_app == NULL)
      return ERROR_MEMORY;
  }
  
  etpan_subapp_set_parent(input_app, app);
  
  edit_data = malloc(sizeof(* edit_data));
  if (edit_data == NULL)
    return ERROR_MEMORY;

  edit_data->type = EDIT_TIME;
  edit_data->dest_int = NULL;
  edit_data->dest_time = value;
  edit_data->dest_str = NULL;
  edit_data->dest_addr = NULL;
  edit_data->size = 0;
  edit_data->app = app;
  
  localtime_r(value, &time_date);
  snprintf(value_str, sizeof(value_str), "%02i/%02i/%i %02i:%02i:%02i",
      time_date.tm_mday, time_date.tm_mon + 1, time_date.tm_year + 1900,
      time_date.tm_hour, time_date.tm_min, time_date.tm_sec);
  
  r = etpan_search_input_set(input_app, prompt, 20, value_str,
      0, edit_upcall, edit_data);
  if (r != NO_ERROR) {
    free(edit_data);
    return r;
  }
  
  etpan_app_switch_subapp(input_app, 0);
  
  return NO_ERROR;
}
