#include "etpan-abook-ldap.h"

#include "etpan-abook-driver.h"
#include "etpan-errors.h"
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "etpan-ldap.h"
#include <stdio.h>
#include <libetpan/libetpan.h>

static int lookup(struct etpan_abook * abook,
    const char * key, carray ** result);

static void uninitialize(struct etpan_abook * abook);

static int connect(struct etpan_abook * abook);

static struct etpan_abook_driver ldap_abook_driver = {
  .name = "ldap",
  .connect = connect,
  .lookup = lookup,
  .uninitialize = uninitialize,
};

#ifdef USE_LDAP
struct ldap_data {
  struct etpan_ldap_config * ldap_config;
  struct etpan_ldap_session * ldap_session;
  struct etpan_global_config * global_config;
};
#endif

static int connect(struct etpan_abook * abook)
{
#ifdef USE_LDAP
  struct ldap_data * data;
  int r;
  
  data = abook->data;
  
  r = etpan_ldap_connect(data->ldap_session);
  switch (r) {
  case ETPAN_LDAP_NO_ERROR:
    return NO_ERROR;
    
  default:
    return ERROR_CONNECT;
  }
#else
  return ERROR_CONNECT;
#endif
}

struct etpan_abook *
etpan_abook_ldap_new(struct etpan_global_config * global_config,
    struct etpan_ldap_config * ldap_config)
{
  struct etpan_abook * abook;
#ifdef USE_LDAP
  struct etpan_ldap_session * ldap_session;
  struct ldap_data * data;
#endif
  
#ifdef USE_LDAP
  abook = malloc(sizeof(* abook));
  if (abook == NULL)
    goto err;

  ldap_session = etpan_ldap_session_new(ldap_config);
  if (ldap_session == NULL)
    goto free;
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    goto free_session;
  
  data->ldap_session = ldap_session;
  data->ldap_config = ldap_config;
  data->global_config = global_config;
  
  abook->data = data;
  abook->driver = &ldap_abook_driver;
  abook->connected = 0;
#else
  abook = NULL;
#endif
  
  return abook;
  
#ifdef USE_LDAP
 free_session:
  etpan_ldap_session_free(ldap_session);
 free:
  free(abook);
 err:
  return NULL;
#endif
}

static int match_abook(char * pattern, int pattern_len, char * value)
{
  char * dup_value;
  int r;
  char * p;

  r = 0;

  if (value == NULL)
    goto err;

  dup_value = strdup(value);
  if (dup_value == NULL)
    goto err;

  for(p = dup_value ; * p != 0 ; p ++)
    * p = (char) toupper((unsigned char) * p);

  if (strncasecmp(pattern, dup_value, pattern_len) == 0)
    r = 1;
  
  free(dup_value);
  
  return r;
  
 err:
  return r;
}

static struct etpan_abook_entry *
get_entry_from_ldap_entry(struct etpan_ldap_abook_entry * local_entry)
{
  struct etpan_abook_entry * entry;
  char * name;
  char * addr;
  
  name = strdup(local_entry->name);
  if (name == NULL)
    goto err;
    
  addr = strdup(local_entry->addr);
  if (addr == NULL)
    goto free_name;

  entry = etpan_abook_entry_new(name, addr, NULL);
  if (entry == NULL)
    goto free_addr;
  
  return entry;
  
 free_addr:
  free(addr);
 free_name:
  free(name);
 err:
  return NULL;
}

static int lookup(struct etpan_abook * abook,
    const char * key, carray ** result)
{
#ifdef USE_LDAP
  struct etpan_ldap_session * ldap_session;
  struct ldap_data * data;
  char * dup_key;
  size_t key_len;
  carray * entry_list;
  char * p;
  int r;
  unsigned int i;
  carray * ldap_result;
  int res;
  char * utf8_key;
  
  data = abook->data;
  ldap_session = data->ldap_session;
  
  r = charconv("utf-8", data->global_config->display_charset,
      key, strlen(key), &utf8_key);
  if (r != MAIL_CHARCONV_NO_ERROR) {
    utf8_key = strdup(key);
  }
  
  r = etpan_ldap_search(ldap_session, utf8_key, &ldap_result);
  free(utf8_key);
  
  if ((r != ETPAN_LDAP_NO_ERROR) && (r != ETPAN_LDAP_ERROR_PARTIAL)) {
    /* etpan_ldap_error(ldap_session) */
    res = ERROR_LDAP_SEARCH;
    goto err;
  }
  
  dup_key = strdup(key);
  if (dup_key == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  key_len = strlen(dup_key);
  
  for(p = dup_key ; * p != 0 ; p ++)
    * p = (char) toupper((unsigned char) * p);
  
  entry_list = carray_new(16);
  if (entry_list == NULL) {
    res = ERROR_MEMORY;
    goto free_key;
  }

  for(i = 0 ; i < carray_count(ldap_result) ; i ++) {
    struct etpan_ldap_abook_entry * ldap_entry;
    struct etpan_abook_entry * entry;
    
    ldap_entry = carray_get(ldap_result, i);
    
    if (ldap_entry->name != NULL) {
      char * name;
      
      r = charconv(data->global_config->display_charset,
          "utf-8", ldap_entry->name, strlen(ldap_entry->name), &name);
      if (r == MAIL_CHARCONV_NO_ERROR) {
        free(ldap_entry->name);
        
        ldap_entry->name = name;
      }
    }
    
    if (match_abook(dup_key, key_len, ldap_entry->name) || 
        match_abook(dup_key, key_len, ldap_entry->addr)) {
      /* item is the abook response from lookup */
      /* entry is the abook entry of local abook configuration */
      entry = get_entry_from_ldap_entry(ldap_entry);
      if (entry == NULL) {
        res = ERROR_MEMORY;
        goto free_entry_list;
      }
      
      free(ldap_entry->addr);
      free(ldap_entry->name);
      free(ldap_entry);
      
      r = carray_add(entry_list, entry, NULL);
      if (r != NO_ERROR) {
        etpan_abook_entry_free(entry);
        res = ERROR_MEMORY;
        goto free_entry_list;
      }
    }
  }
  free(ldap_result);
  
  * result = entry_list;

  return NO_ERROR;
  
 free_entry_list:
  for(i = 0 ; i < carray_count(entry_list) ; i ++) {
    struct etpan_abook_entry * entry;
    
    entry = carray_get(entry_list, i);
    etpan_abook_entry_free(entry);
  }
 free_key:
  free(dup_key);
 err:
  return res;
#else
  return ERROR_LDAP_SEARCH;
#endif
}

static void uninitialize(struct etpan_abook * abook)
{
#ifdef USE_LDAP
  struct ldap_data * data;

  data = abook->data;
  
  etpan_ldap_session_free(data->ldap_session);
  
  free(data->ldap_config->attrs.sn);
  free(data->ldap_config->attrs.givenname);
  free(data->ldap_config->attrs.cn);
  free(data->ldap_config->attrs.mail);
  free(data->ldap_config->filter);
  free(data->ldap_config->bindpw);
  free(data->ldap_config->binddn);
  free(data->ldap_config->base);
  free(data->ldap_config->hostname);
  free(data->ldap_config);
  free(data);
#endif
}
