/////////////////////////////////////////////////////////////
//                                                         //
// Copyright (c) 2003-2014 by The University of Queensland //
// Centre for Geoscience Computing                         //
// http://earth.uq.edu.au/centre-geoscience-computing      //
//                                                         //
// Primary Business: Brisbane, Queensland, Australia       //
// Licensed under the Open Software License version 3.0    //
// http://www.opensource.org/licenses/osl-3.0.php          //
//                                                         //
/////////////////////////////////////////////////////////////

#ifndef _CONSOLE_H_
#define _CONSOLE_H_

//--project includes--
#include "Foundation/Matrix3.h"
#include "Foundation/vec3.h"

//--STL includes--
#include <string>
#include <sstream>
#include <fstream>

using std::string;
using std::ostringstream;

class CCounter;
class CCounterList;

/*!
  \class BasicCon
  \brief Handle message ouput on the console 
  
  \author David Place, Steffen Abe
  $Revision$
  $Date$
*/
class BasicCon 
{
public:
  enum VerboseLevel { all=0, dbg, info, msg, warning, err, critical, silent } ;
    
protected:
  static int VLevel ;           //!< current verbose level
  static bool mute,quiet ;      //!< flag to mute console (loats output) or quiet console
  static bool timing;
    
  char *CurBuffer ;             //!< buffer for output & process
  ostream *os ;                 //!< Output stream (cout usually)

  virtual void process(ostringstream *str_os) ;
  virtual void show(char *level,bool h) ;
    
public:
  BasicCon();
  BasicCon(ostream *Ios);
  virtual ~BasicCon() ;
    
  inline void SetQuiet(bool Q) { quiet = Q ; };  
  inline void SetTiming(bool T) { timing = T ; };  
  static void SetVerbose(int vl=all) ; //!< set verbose level
  inline static int GetVerbose() { return VLevel; } ;

  virtual void flush() ;
  
  virtual void SetOStream(ostream *Ios);
            
  BasicCon & Message(bool h=true) ;  //!< set verbose level of next message to "msg"
  BasicCon & Error(bool h=true) ;    //!< set verbose level of next message to "err"
  BasicCon & Warning(bool h=true) ;  //!< set verbose level of next message to "wrn"
  BasicCon & Critical(bool h=true) ; //!< set verbose level of next message to "crt"
  BasicCon & Info(bool h=true) ;     //!< set verbose level of next message to "inf"
  BasicCon & Debug(bool h=true) ;    //!< set verbose level of next message to "dbg"
  BasicCon & XDebug(bool h=true) ;   //!< set verbose level of next message to "xdg"
  BasicCon & Timing(bool h=true) ;   //!< set verbose level of next message to "tme"
    
  BasicCon &  operator<<(const char* s);
  BasicCon &  operator<<(char s);
  BasicCon &  operator<<(short s) ;
  BasicCon &  operator<<(int s) ;
  BasicCon &  operator<<(long s) ;
  BasicCon &  operator<<(float s) ;
  BasicCon &  operator<<(double s) ;
  BasicCon &  operator<<(unsigned char s) ;
  BasicCon &  operator<<(unsigned short s);
  BasicCon &  operator<<(unsigned int s);
  BasicCon &  operator<<(unsigned long s);
  BasicCon &  operator<<(void* s) ;
  BasicCon &  operator<<(Vec3 s);
  BasicCon &  operator<<(const Matrix3&);
  BasicCon &  operator<<(CCounter &s);
  BasicCon &  operator<<(CCounterList &s);
  BasicCon &  operator<<(const string&);
} ;

/*!
  \class ConsoleWindow
  \brief Console with history, and handle message output on a console window
*/
class ConsoleWindow {
protected:
  static ConsoleWindow* Window ;
  virtual void UpdateCon() = 0 ;
public:
  virtual ~ConsoleWindow() {}
  static bool immediate ;
  static void Update() ;
  static bool IsOpen() { return Window!=NULL ; } ;
} ;

/*!
  \class Con
  \brief console with window buffer
*/
class Con : public BasicCon {
private:
  int CurCol ;
  char Buffer[4096] ;
protected:
  virtual void process(ostringstream *str_os) ;
  Con();

public:
  Con(ostream *Ios);
  virtual ~Con() ;
  virtual char *GetLast(char *buff) ;
} ;

/*!
  \class FCon
  \brief console writing into per-process file
*/
class FCon : public Con
{
  private:
  std::ofstream m_debugfile;

 public:
  FCon();
  virtual ~FCon() ;
};



#ifdef _ENABLE_DEBUG_FILE
extern FCon console ;
#else
extern Con console ;
#endif // _ENABLE_DEBUG_FILE
extern BasicCon tconsole ;


#endif
