#include <glib.h>
#include "entity.h"

/* The treewalk function is a utility function for walking through the tree
 * in a relatively fast and easy manner.  This is a utility function for the
 * enode api, and is not to be used directly unless you know what you are
 * doing. */

ENodeTreeWalk *
enode_treewalk_new (ENode * topnode)
{
    static ENodeTreeWalk *treewalk = NULL;

    if (!treewalk)
	treewalk = g_new (ENodeTreeWalk, 1);

    memset (treewalk, 0, sizeof (ENodeTreeWalk));
    treewalk->top_node = topnode;

    return (treewalk);
}

/* It's a NOP, but please use it!  In the future this may change to your
 * usual allocation/free thing, and I don't want to have to go fix all the
 * old crap.. :) */
void
enode_treewalk_free (ENodeTreeWalk * treewalk)
{
    return;
}

void
enode_treewalk (ENodeTreeWalk * treewalk,
		ENodeTreeWalkCallback child_callback,
		ENodeTreeWalkCallback parent_callback)
{
    ENode *curnode;
    ENode *parentnode;
    GSList *child;
    GQueue *q;
    GQueue *cq;
    gint keep_going = TRUE;

    ECHECK_RET (treewalk != NULL);
    ECHECK_RET (child_callback != NULL);
    ECHECK_RET (treewalk->top_node != NULL);

    q = g_queue_create ();
    cq = g_queue_create ();

    g_queue_push_tail (q, NULL);

    curnode = treewalk->top_node;
    parentnode = treewalk->top_node;
    child = curnode->children;
    g_queue_push_tail (cq, child);

    while (TRUE) {
	while (child) {
	    /* save the current parent */
	    g_queue_push_tail (q, curnode);
	    /* save the current child */
	    g_queue_push_tail (cq, child);
	    /* delve into next child */
	    curnode = child->data;

	    treewalk->curnode = curnode;
	    treewalk->parentnode = NULL;
	    keep_going = child_callback (treewalk);
	    if (!keep_going)
		goto cleanup;

	    /* find first child and descend */
	    child = curnode->children;
	}

	/* when we end up here, it means we've descended all the way down a
	 * fork. Now we climb back out and set it to the next child if it's
	 * found */
	parentnode = g_queue_pop_tail (q);

	/* if we've returned back beyond the top, we've completed the run. */
	if (parentnode == NULL)
	    goto cleanup;

	treewalk->curnode = curnode;
	treewalk->parentnode = parentnode;
	if (parent_callback) {
	    keep_going = parent_callback (treewalk);
	    if (!keep_going)
		goto cleanup;
	}

	/* find the next child in the parents list */
	child = g_queue_pop_tail (cq);

	if (child)
	    child = child->next;

	curnode = parentnode;
    }
  cleanup:
    g_queue_free (q);
    g_queue_free (cq);
    return;
}

static gint
make_enode_list_forwards (ENodeTreeWalk * treewalk)
{
    GSList *list = treewalk->user_data1;
    GSList *tail = treewalk->user_data2;

    list = g_slist_append_tail (list, treewalk->curnode, &tail);

    treewalk->user_data1 = list;
    treewalk->user_data2 = tail;
    return (TRUE);
}

static gint
make_enode_list_backwards (ENodeTreeWalk * treewalk)
{
    GSList *list = treewalk->user_data1;

    list = g_slist_prepend (list, treewalk->curnode);

    treewalk->user_data1 = list;
    return (TRUE);
}


/* forward is boolean.. switches between forward or reverse list. */
GSList *
enode_child_list (ENode * topnode, gint forward)
{
    GSList *list = NULL;
    ENodeTreeWalk *walker;

    walker = enode_treewalk_new (topnode);

    ECHECK_RETVAL (topnode != NULL, NULL);
    if (forward)
	enode_treewalk (walker, make_enode_list_forwards, NULL);
    else
	enode_treewalk (walker, make_enode_list_backwards, NULL);

    list = walker->user_data1;

    enode_treewalk_free (walker);

    return (list);
}


