/* 
   elmo - ELectronic Mail Operator

   Copyright (C) 2003, 2004 rzyjontko

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; version 2.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

   ----------------------------------------------------------------------
   
   Functions related to asking for a password when none defined.
   
*/
/****************************************************************************
 *    IMPLEMENTATION HEADERS
 ****************************************************************************/

#include <string.h>
#include <stdio.h>

#include "ask.h"
#include "choose.h"
#include "read.h"
#include "xmalloc.h"
#include "cmd.h"
#include "confhold.h"
#include "error.h"
#include "gettext.h"

/****************************************************************************
 *    IMPLEMENTATION PRIVATE DEFINITIONS / ENUMERATIONS / SIMPLE TYPEDEFS
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE CLASS PROTOTYPES / EXTERNAL CLASS REFERENCES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE STRUCTURES / UTILITY CLASSES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION REQUIRED EXTERNAL REFERENCES (AVOID)
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE DATA
 ****************************************************************************/
/****************************************************************************
 *    INTERFACE DATA
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE FUNCTION PROTOTYPES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE FUNCTIONS
 ****************************************************************************/

static char *
make_prompt (const char *p, const char *r)
{
        int   len    = strlen (p) + (r ? strlen (r) : 0) + 1;
        char *result = xmalloc (len + 3);

        if (r)
                sprintf (result, "%s:%s: ", p, r);
        else
                sprintf (result, "%s: ", p);
        return result;
}


/****************************************************************************
 *    INTERFACE FUNCTIONS
 ****************************************************************************/


ask_t *
ask_select (const char *name)
{
        int        index;
        ask_t     *result = xmalloc (sizeof (ask_t));
        rstring_t *items;

        result->selected = -1;
        result->cd       = confhold_open (name);
        if (result->cd == -1){
                xfree (result);
                return NULL;
        }

        items = confhold_get_fields (result->cd, "name");
        if (items == NULL){
                confhold_close (result->cd);
                xfree (result);
                return NULL;
        }

        if (items->count == 0){
                confhold_close (result->cd);
                rstring_delete (items);
                xfree (result);
                return NULL;
        }

        result->items = items;

        if (items->count == 1){
                result->selected = 0;
                confhold_select (result->cd, 0);
                return result;
        }

        choose_open (items->count, items->array);
        cmd_choose ();

        index = choose_index ();

        if (index == -1){
                confhold_close (result->cd);
                rstring_delete (items);
                xfree (result);
                return NULL;
        }

        result->selected = index;
        confhold_select (result->cd, index);
  
        return result;
}


ask_t *
ask_select_default (const char *name)
{
        ask_t *result = xmalloc (sizeof (ask_t));

        result->cd = confhold_open (name);
        if (result->cd == -1){
                confhold_close (result->cd);
                xfree (result);
                return NULL;
        }

        result->items    = confhold_get_fields (result->cd, "name");
        result->selected = 0;
        confhold_select (result->cd, 0);

        return result;
}



ask_t *
ask_select_next (ask_t *ask)
{
        if (ask->selected >= ask->items->count - 1){
                ask_destroy (ask);
                return NULL;
        }

        ask->selected++;
        confhold_select (ask->cd, ask->selected);
        return ask;
}



void
ask_select_different (ask_t *ask)
{
        int index;
  
        if (ask->items->count < 2){
                error_ (0, _("no other definition available"));
        }

        choose_open (ask->items->count, ask->items->array);
        cmd_choose ();

        index = choose_index ();

        if (index == -1){
                return;
        }

        ask->selected = index;
        confhold_select (ask->cd, index);
}



char *
ask_get_field (ask_t *ask, const char *field)
{
        if (ask == NULL)
                return NULL;

        return confhold_field_from_selected (ask->cd, field);
}


int
ask_get_field_int_default (ask_t *ask, const char *field, int def)
{
        char *ret;

        if (ask == NULL)
                return def;
        
        ret = confhold_field_from_selected (ask->cd, field);

        if (ret == NULL)
                return def;

        if (strcasecmp (ret, "yes") == 0)
                return 1;
        if (strcasecmp (ret, "true") == 0)
                return 1;
        return atoi (ret);
}


int
ask_get_field_int (ask_t *ask, const char *field)
{
        return ask_get_field_int_default (ask, field, 0);
}


void
ask_change_where (ask_t *ask, const char *field, const char *desired)
{
        confhold_select_where (ask->cd, field, desired);
}



void
ask_destroy (ask_t *ask)
{
        if (ask == NULL)
                return;
        
        confhold_close (ask->cd);
        rstring_delete (ask->items);
        xfree (ask);
}



char *
ask_for_simple (const char *name)
{
        int        index;
        char      *txt;
        char      *prompt;
        rstring_t *items;

        items = confhold_get_all (name);

        if (items == NULL || items->count == 0){
                prompt = make_prompt (name, NULL);
                txt    = read_argument (prompt, NULL, COMPLETE_NONE, HIDE_NO);
                xfree (prompt);
                if (*txt == '\0')
                        txt = NULL;
                return txt;
        }

        if (items->count == 1){
                txt = items->array[0];
                rstring_delete (items);
                return txt;
        }

        choose_open (items->count, items->array);
        cmd_choose ();

        index = choose_index ();

        if (index == -1){
                rstring_delete (items);
                return NULL;
        }

        txt = items->array[index];
        rstring_delete (items);
        return txt;
}



char *
ask_for_default (const char *name, const char *field)
{
        char *txt;

        txt = confhold_get_first (name, field);
        return txt;
}



int
ask_for_default_int (const char *name, const char *field, int def)
{
        char *txt;

        txt = confhold_get_first (name, field);

        if (txt == NULL)
                return def;
        if (strcasecmp (txt, "yes") == 0)
                return 1;
        if (strcasecmp (txt, "true") == 0)
                return 1;
        return atoi (txt);
}


int
ask_if_sure (char *question)
{
        char *answer = read_argument (question, NULL, COMPLETE_NONE, HIDE_NO);

        if (answer == NULL)
                return -1;
  
        if (*answer == 'y' || *answer == 'Y')
                return 1;
        return 0;
}


/****************************************************************************
 *    INTERFACE CLASS BODIES
 ****************************************************************************/
/****************************************************************************
 *
 *    END MODULE ask.c
 *
 ****************************************************************************/
