/* Copyright (C) 2006 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the src/utils sub-directory);
   if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA, 02111-1307, USA.

However, it is not intended that the object code of a program whose
source code instantiates a template from this file should by reason
only of that instantiation be subject to the restrictions of use in
the GNU Lesser General Public License.  With that in mind, the words
"and instantiations of templates (of any length)" shall be treated as
inserted in the fourth paragraph of section 5 of that licence after
the words "and small inline functions (ten lines or less in length)".
This does not affect any other reason why object code may be subject
to the restrictions in that licence (nor for the avoidance of doubt
does it affect the application of section 2 of that licence to
modifications of the source code in this file).

*/

/*
  This is a class which provides some of the functionality of a std::queue
  object (but note that the AsyncQueue::pop(value_type& obj) method provides
  the pop()ed element - see the comments below for the reason), except that
  it has mutex locking of the data container so as to permit push()ing and
  pop()ing from different threads.  It is therefore useful for passing
  data between threads, perhaps in response to a signal being emitted from
  a Notifier object.  Passing the data by means of a SharedLockPtr object
  or an IntrusivePtr object referencing data derived from
  IntrusiveLockCounter would be ideal.

  By default the queue uses a std::list object as its container because in
  the kind of use mentioned above it is unlikely to hold many objects but
  it can be changed to, say, a std::deque object by specifying it as the
  second template parameter
*/

#ifndef ASYNC_QUEUE_H
#define ASYNC_QUEUE_H

#include <queue>
#include <list>
#include <exception>
#include "utils/mutex.h"

struct AsyncQueuePopError: public std::exception {
  virtual const char* what() const throw() {return "Popping from empty AsyncQueue object";}
};


template <class T, class Container = std::list<T> > class AsyncQueue {
public:
  typedef typename Container::value_type value_type;
  typedef typename Container::size_type size_type;
  typedef Container container_type;
private:
  std::queue<T, Container> q;
  mutable Thread::Mutex mutex;

  // AsyncQueue objects cannot be copied - they are mainly
  // intended to pass data between two known threads
  AsyncQueue(const AsyncQueue&);
  AsyncQueue& operator=(const AsyncQueue&);
public:
  void push(const value_type& obj) {
    Thread::Mutex::Lock lock(mutex);
    q.push(obj);
  }

  // in order to complete pop() operations atomically under a single lock,
  // we have to both copy and extract the object at the front of the queue
  // in the pop() method.  To retain strong exception safety, the object
  // into which the pop()ed data is to be placed is passed as an
  // argument by reference (this avoids a copy from a temporary object
  // after the data has been extracted from the queue).  If the queue is
  // empty when a pop is attempted, this method will throw AsyncQueuePopError.
  void pop(value_type& obj) {
    Thread::Mutex::Lock lock(mutex);
    if (q.empty()) throw AsyncQueuePopError();
    obj = q.front();
    q.pop();
  }

  // this method is for use if all that is wanted is to discard the
  // element at the front of the queue.  If the queue is empty when
  // a pop is attempted, this method will throw AsyncQueuePopError.
  void pop(void) {
    Thread::Mutex::Lock lock(mutex);
    if (q.empty()) throw AsyncQueuePopError();
    q.pop();
  }

  bool empty(void) const {
    Thread::Mutex::Lock lock(mutex);
    return q.empty();
  }

  size_type size(void) const {
    Thread::Mutex::Lock lock(mutex);
    return q.size();
  }

  AsyncQueue(void) {}
};


#endif
