/* Copyright (C) 2001 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/


#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>

#include <ctime>
#include <cstdlib>
#include <cstring>
#include <iostream>

#include <gtk--/main.h>
#include <gtk--/menu.h>
#include <gtk--/scrollbar.h>
#include <gtk--/adjustment.h>
#include <gtk--/style.h>
#include <gdk--/color.h>

#include "mainwindow.h"
#include "dialogs.h"
#include "addressbook.h"
#include "settings.h"

#define TIMER_INTERVAL 200              // number of milliseconds between timer events for the main timer
#define LOGFILE_TIMER_INTERVAL 60000    // number of milliseconds between flushing of logfile

extern "C" void childexit_signalhandler(int);
extern "C" void close_signalhandler(int);
static volatile sig_atomic_t close_flag = false;


Pipe_fifo MainWindow::error_pipe(Pipe_fifo::non_block);
bool MainWindow::connected_to_stderr = false;

MainWindow::MainWindow(const string& messages, const char* filename): // filename has a default value of 0
                              Gtk::Window(GTK_WINDOW_TOPLEVEL), standard_size(24),
                              table(5, 3, false), file_button("File to fax:"),
			      number_button("Tel number: "), send_button("Send fax"),
			      receive_answer_button("Answer call"), receive_takeover_button("Take over call"),
			      receive_standby_button("Standby"), stop_button("Stop"),
                              status_line(standard_size) {

  // catch any relevant Unix signals for an orderly closedown
  // catch SIGQUIT, SIGTERM SIGINT SIGHUP and SIGPIPE
  // it is safe to use signal() for these
  signal(SIGQUIT, close_signalhandler);
  signal(SIGTERM, close_signalhandler);
  signal(SIGINT, close_signalhandler);
  signal(SIGHUP, close_signalhandler);
  signal(SIGPIPE, close_signalhandler);

  // set up child signal handler
  struct sigaction sig_act;
  sig_act.sa_handler = childexit_signalhandler;
  // we don't need to mask off any signals
  sigemptyset(&sig_act.sa_mask);
  // the default flag set will be ok here
  sig_act.sa_flags = 0;

  sigaction(SIGCHLD, &sig_act, 0);

  file_button.set_usize(standard_size * 4, standard_size);
  number_button.set_usize(standard_size * 4, standard_size);

  file_entry.set_usize(standard_size * 7, standard_size);
  number_entry.set_usize(standard_size * 7, standard_size);

  file_box.pack_start(file_button, false, false, standard_size/2);
  file_box.pack_start(file_entry, true, true, 0);

  number_box.pack_start(number_button, false, false, standard_size/2);
  number_box.pack_start(number_entry, true, true, 0);

  text_window.set_usize(standard_size * 13, standard_size * 5);

  table.attach(file_box, 0, 3, 0, 1, GTK_EXPAND | GTK_FILL,
         0, standard_size/3, standard_size/3);
  table.attach(number_box, 0, 3, 1, 2, GTK_EXPAND | GTK_FILL,
         0, standard_size/3, standard_size/3);
  table.attach(text_window, 0, 3, 2, 3, GTK_EXPAND | GTK_FILL,
         GTK_EXPAND | GTK_FILL, standard_size/3, standard_size/3);

  table.attach(send_button, 0, 1, 3, 4, GTK_EXPAND,
	 0, standard_size/3, standard_size/3);
  table.attach(receive_answer_button, 1, 2, 3, 4, GTK_EXPAND,
	 0, standard_size/3, standard_size/3);
  table.attach(receive_takeover_button, 2, 3, 3, 4, GTK_EXPAND,
	 0, standard_size/3, standard_size/3);

  table.attach(receive_standby_button, 0, 1, 4, 5, GTK_EXPAND,
	 0, standard_size/3, standard_size/3);
  table.attach(stop_button, 2, 3, 4, 5, GTK_EXPAND,
	 0, standard_size/3, standard_size/3);

  file_button.clicked.connect(SigC::slot(this, &MainWindow::get_file_slot));
  number_button.clicked.connect(SigC::slot(this, &MainWindow::addressbook_slot));
  send_button.clicked.connect(SigC::slot(this, &MainWindow::sendfax_slot));
  receive_answer_button.clicked.connect(SigC::bind(SigC::slot(this, &MainWindow::receive_slot),
						   EfaxController::receive_answer));
  receive_takeover_button.clicked.connect(SigC::bind(SigC::slot(this, &MainWindow::receive_slot),
						     EfaxController::receive_takeover));
  receive_standby_button.clicked.connect(SigC::bind(SigC::slot(this, &MainWindow::receive_slot),
						    EfaxController::receive_standby));
  stop_button.clicked.connect(SigC::slot(&efax_controller, &EfaxController::stop_slot));

  destroy.connect(SigC::slot(this, &MainWindow::destroy_slot));

  send_button.set_usize(standard_size * 4, standard_size);
  receive_answer_button.set_usize(standard_size * 4, standard_size);
  receive_takeover_button.set_usize(standard_size * 4, standard_size);
  receive_standby_button.set_usize(standard_size * 4, standard_size);
  stop_button.set_usize(standard_size * 4, standard_size);

  // set up the menu bar
  {
    using namespace Gtk::Menu_Helpers;
    // Create the file menu
    Gtk::Menu* file_menu_p = manage(new Gtk::Menu);
    // now fill the file menu
    MenuList& file_list = file_menu_p->items();
    
    file_list.push_back(MenuElem("List _received faxes",
				 SigC::bind(SigC::slot(this, &MainWindow::fax_list_slot), 
					    FaxListDialog::received)));
    file_list.push_back(MenuElem("_List sent faxes",
				 SigC::bind(SigC::slot(this, &MainWindow::fax_list_slot), 
					    FaxListDialog::sent)));
    file_list.push_back(SeparatorElem());
    file_list.push_back(MenuElem("_Find file",
				 SigC::slot(this, &MainWindow::get_file_slot)));
    file_list.push_back(MenuElem("_Address book",
				 SigC::slot(this, &MainWindow::addressbook_slot)));
    file_list.push_back(SeparatorElem());
    file_list.push_back(MenuElem("_Settings",
				 SigC::slot(this, &MainWindow::settings_slot)));
    file_list.push_back(SeparatorElem());
    file_list.push_back(MenuElem("_Quit", destroy.slot()));

    // Create the about menu
    Gtk::Menu* help_menu_p = manage(new Gtk::Menu);
    // now fill the about menu
    MenuList& help_list = help_menu_p->items();
    
    help_list.push_back(MenuElem("About efax-_gtk",
				 SigC::bind(SigC::slot(this, &MainWindow::about_slot), true)));
    help_list.push_back(MenuElem("About _efax",
				 SigC::bind(SigC::slot(this, &MainWindow::about_slot), false)));
    help_list.push_back(SeparatorElem());
    help_list.push_back(MenuElem("_Helpfile",
				 SigC::slot(this, &MainWindow::helpfile_slot)));

    // now fill the menu bar
    MenuList& bar_list = menu_bar.items();
    bar_list.push_front(MenuElem("_Help", "<mod1>h", *help_menu_p));
    bar_list.push_front(MenuElem("_File", "<mod1>f", *file_menu_p));
  }

  window_box.pack_start(menu_bar, false, false);
  window_box.pack_start(table, true, true);
  window_box.pack_start(status_line, false, false);

  add(window_box);

  table.set_border_width(standard_size/3);

  if (filename) {
    file_entry.set_text(filename);
    number_entry.grab_focus();
  }
  else file_button.grab_focus();

  Gtk::Main::timeout.connect(SigC::slot(this, &MainWindow::timer_event_handler), TIMER_INTERVAL);

  Gtk::Main::input.connect(SigC::slot(this, &MainWindow::read_error_slot),
			   error_pipe.get_read_fd(), GDK_INPUT_READ);

  efax_controller.stdout_message.connect(SigC::slot(&text_window, &MessageText::write_black_slot));
  efax_controller.write_state.connect(SigC::slot(&status_line, &StatusLine::write_status));

  // we want to minimise the effect on efax, so make writing to the error pipe non-blocking
  error_pipe.make_write_non_block();

  show_all();
  if (!messages.empty()) {
    text_window.write_red_slot(messages.c_str());
    text_window.write_red_slot("\n\n");
  }

  // make sure that we have the faxin, faxout, faxsent, faxin/oldfax  and faxsent/oldfax directories
  chdir(prog_config.homedir.c_str());
  mkdir("faxin", S_IRUSR | S_IWUSR | S_IXUSR);
  mkdir("faxout", S_IRUSR | S_IWUSR | S_IXUSR);
  mkdir("faxsent", S_IRUSR | S_IWUSR | S_IXUSR);
  mkdir("faxin/oldfax", S_IRUSR | S_IWUSR | S_IXUSR);
  mkdir("faxsent/oldfax", S_IRUSR | S_IWUSR | S_IXUSR);

  // set the working directory for the parent process
  chdir("faxin");

  // if there is no config file installed, then bring up the settings dialog
  if (!prog_config.found_rcfile) {
    // we don't want to use MainWindow::settings_slot(), or the absence of
    // a configuration file will be reported twice -- not a big deal, but ...
    // so pass true as the last parameter to skip detection of settings file
    SettingsDialog dialog(standard_size, *this, true);
    dialog.accepted.connect(SigC::slot(this, &MainWindow::settings_changed_slot));
    dialog.run();
  }
}

gint MainWindow::key_press_event_impl(GdkEventKey* event_p) {
  if (event_p->keyval == GDK_F1) helpfile_slot();
  else Gtk::Window::key_press_event_impl(event_p);
  return true; // processing stops here
}

void MainWindow::read_error_slot(gint fd, GdkInputCondition) {

  char pipe_buffer[PIPE_BUF + 1];
  ssize_t result;

  if (fd == error_pipe.get_read_fd()) {
    while ((result = error_pipe.read(pipe_buffer, PIPE_BUF)) > 0) {
      pipe_buffer[result] = 0;
      text_window.write_red_slot(pipe_buffer);
    }
  }
}

void MainWindow::destroy_slot(void) {
  // there are a number of ways to close the program
  // one is to call Gtk::Main::quit() and then to follow it with
  // an explicit emission of destroy()
  // or delete_event_impl() can call Gtk::Main::quit() and return false
  // (which in fact just emits destroy() and so has the same
  // effect as the first case)
  // another (which we do here) is to call Gtk::Main::quit() within a method
  // connected to the destroy signal, and quit the program by emitting
  // the destroy signal

  // first, make sure we close down efax() if it is active
  efax_controller.efax_closedown();

  // now quit
  Gtk::Main::quit();
}

void MainWindow::sendfax_slot(void) {

  string file(file_entry.get_text());
  string number(number_entry.get_text());

  if (!prog_config.found_rcfile) {
    text_window.write_red_slot("Can't send fax -- no efax-gtkrc configuration file found\n\n");
    beep();
  }
  
  else if (prog_config.lock_file.empty()) {
    text_window.write_red_slot("Can't send fax -- no valid serial device specified\n\n");
    beep();
  }

  else if (file.empty() || number.empty()) beep();

  else efax_controller.sendfax(file, number);
}


void MainWindow::receive_slot(int mode) {

  if (!prog_config.found_rcfile) {
    text_window.write_red_slot("Can't receive fax -- no efax-gtkrc configuration file found\n\n");
    beep();
  }
  
  else if (prog_config.lock_file.empty()) {
    text_window.write_red_slot("Can't receive fax -- no valid serial device specified\n\n");
    beep();
  }

  else efax_controller.receive(mode);
}

gint MainWindow::timer_event_handler(void) {

  if (close_flag) destroy(); // we must have picked up an external kill signal
                             // so we need an orderly close down
  else efax_controller.timer_event();
  return true; // we want a multi-shot timer
}

void MainWindow::get_file_slot(void) {
  FileReadSelectDialog file_dialog(standard_size, *this);
  string result(file_dialog.run());
  if (!result.empty()) {
    file_entry.set_text(result);
    if (result[result.size() - 1] != '/') number_entry.grab_focus();
  }
}

void MainWindow::fax_list_slot(FaxListDialog::Mode mode) {

  if ((mode == FaxListDialog::received &&
       !FaxListDialog::get_is_fax_received_list())
      || (mode == FaxListDialog::sent &&
       !FaxListDialog::get_is_fax_sent_list())) {
    FaxListDialog* dialog_p = new FaxListDialog(mode, standard_size);
    if (!dialog_p) {
      cerr << "Memory allocation error in MainWindow::fax_list_slot()" << endl;
      exit(MEM_ERROR);
    }
    // there is no memory leak -- FaxListDialog is modeless and will delete its own memory
    // when it is closed
  }
}

void MainWindow::addressbook_slot(void) {

  if (!AddressBook::get_is_address_list()) {
    AddressBook* dialog_p = new AddressBook(standard_size, *this);
    if (!dialog_p) {
      cerr << "Memory allocation error in MainWindow::addressbook_slot()" << endl;
      exit(MEM_ERROR);
    }
    dialog_p->accepted.connect(SigC::slot(this, &MainWindow::set_number));
    // there is no memory leak -- AddressBook will delete its own memory
    // when it is closed
  }
}

void MainWindow::settings_slot(void) {

  if (efax_controller.get_state() != EfaxController::inactive) {
    InfoDialog* dialog_p;
    const char message[] = "Can't change settings unless\n"
                           "the program is inactive\n\n"
                           "Press the Stop button to make it inactive\n";
    dialog_p = new InfoDialog(message, "Change settings", standard_size, InfoDialog::warning, *this);
    if (!dialog_p) {
      cerr << "Memory allocation error in MainWindow::settings_slot()" << endl;
      exit(MEM_ERROR);
    }
    // there is no memory leak -- run() was not called so InfoDialog
    // will delete its own memory when it is closed
  }

  else {
    SettingsDialog* dialog_p = new SettingsDialog(standard_size, *this);
    if (!dialog_p) {
      cerr << "Memory allocation error in MainWindow::settings_slot()" << endl;
      exit(MEM_ERROR);
    }
    dialog_p->accepted.connect(SigC::slot(this, &MainWindow::settings_changed_slot));
    // there is no memory leak -- SettingsDialog will delete its own memory
    // when it is closed
  }
}

void MainWindow::settings_changed_slot(const string& messages) {
  text_window.reset_logfile();
  if (!messages.empty()) {
    text_window.write_red_slot(messages.c_str());
    text_window.write_red_slot("\n\n");
  }
}

void MainWindow::helpfile_slot(void) {

  HelpDialog* dialog_p = new HelpDialog(standard_size);
  if (!dialog_p) {
    cerr << "Memory allocation error in MainWindow::helpfile_slot()" << endl;
    exit(MEM_ERROR);
  }
  // there is no memory leak -- HelpDialog is modeless and will delete its own memory
  // when it is closed
}

void MainWindow::set_number(const string& number) {

  if (!number.empty()) {
    number_entry.set_text(number);
    if (file_entry.get_text_length()) send_button.grab_focus();
  }
}

void MainWindow::about_slot(bool efax_gtk) {
  InfoDialog* dialog_p;

  if (efax_gtk) {
    const char message[] = "efax-gtk-1.0.1a\n"
                           "Copyright (C) 2001 Chris Vine\n\n"
                           "This program is released under the\n"
                           "GNU General Public License, version 2\n";
    dialog_p = new InfoDialog(message, "About efax-gtk", standard_size, InfoDialog::information, *this);
  }
  else {
    const char message[] = "This program is a front end for efax\n\n"
                           "efax is a program released under the\n"
                           "GNU General Public License, version 2 by Ed Casas\n\n"
                           "The copyright to efax is held by Ed Casas\n"
                           "(the copyright to this program is held by Chris Vine)\n";
    dialog_p = new InfoDialog(message, "About efax", standard_size, InfoDialog::information, *this);
  }

  if (!dialog_p) {
    cerr << "Memory allocation error in MainWindow::about_slot()" << endl;
    exit(MEM_ERROR);
  }
  // there is no memory leak -- run() was not called so InfoDialog
  // will delete its own memory when it is closed
}

MessageText::MessageText(void) : Gtk::Table(1, 2, false) {

  attach(text, 0, 1, 0, 1, GTK_FILL | GTK_EXPAND,
	 GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 0);
  // put a VScrollbar in the upper right
  Gtk::Scrollbar* scrollbar_p = manage(new Gtk::VScrollbar(*(text.get_vadjustment())));
  attach (*scrollbar_p, 1, 2, 0, 1, GTK_FILL, GTK_EXPAND | GTK_FILL | GTK_SHRINK, 0, 0);

  text.set_word_wrap(true);
  text.set_line_wrap(true);
  text.set_editable(false);
  text.unset_flags(GTK_CAN_FOCUS);

  Gdk_Color red;
  red.set_rgb_p(0.8, 0, 0);
  red_context.set_foreground(red);

  // save the current working directory before MainWindow::MainWindow can get
  // at it, in case it is needed in future by reset_logfile

#ifndef PATH_MAX
#define PATH_MAX 1023
#endif
  char* path_buffer;
  int count;
  void* success;
  for (count = 1, success = 0; !success && count < 10; count++) {
    path_buffer = new char[(PATH_MAX * count) + 1];
    success = getcwd(path_buffer, PATH_MAX * count);
    if (success) {
      starting_dirname = path_buffer;
      starting_dirname += '/';
    }
    delete[] path_buffer;
  }

  // now open the log file if required
  if (!prog_config.logfile_name.empty()) {

    string temp(prog_config.logfile_name);
    if (temp[0] != '/') temp.insert(0, starting_dirname); // provide an absolute path name
    logfile.open(temp.c_str(), ios::app | ios::out);
    if (!logfile) {
      prog_config.logfile_name = "";
      string message("Can't open logfile ");
      message += temp;
      message += '\n';
      write_red_slot(message.c_str());
    }
    else {
      struct tm* time_p;
      time_t time_count;
      time(&time_count);
      time_p = localtime(&time_count);
      
      const char date_description_format[] = "%H%M %Z %d %b %Y";
      const int max_description_datesize = 126;
      char date_description[max_description_datesize];
      strftime(date_description, max_description_datesize, date_description_format, time_p);

      logfile << "\n\n***********************\n"
                 "Beginning fax log at " << date_description << '\n' << endl;
      // set a timeout to flush every one minute
      timer_slot_connection = Gtk::Main::timeout.connect(SigC::slot(this,
                                         &MessageText::flush_logfile_timer_slot), LOGFILE_TIMER_INTERVAL);
    }
    // save the logfile name in the local version of logfile_name 
    // in case reset_logfile() is called later so we can test whether it has changed
    logfile_name = prog_config.logfile_name;
  }
}

MessageText::~MessageText(void) {
  struct tm* time_p;
  time_t time_count;
  time(&time_count);
  time_p = localtime(&time_count);
  
  const char date_description_format[] = "%H%M %Z %d %b %Y";
  const int max_description_datesize = 126;
  char date_description[max_description_datesize];
  strftime(date_description, max_description_datesize, date_description_format, time_p);
  
  logfile << "\n\nEnding fax log at " << date_description << '\n'
	  << "***********************\n" << endl;
  logfile.close();
}

void MessageText::reset_logfile(void) {

  // check pre-conditions
  if (logfile_name == prog_config.logfile_name) return; // no change!

  // proceed
  // first close the log file if required
  if (!logfile_name.empty()) {

    struct tm* time_p;
    time_t time_count;
    time(&time_count);
    time_p = localtime(&time_count);
  
    const char date_description_format[] = "%H%M %Z %d %b %Y";
    const int max_description_datesize = 126;
    char date_description[max_description_datesize];
    strftime(date_description, max_description_datesize, date_description_format, time_p);
  
    logfile << "\n\nEnding fax log at " << date_description << '\n'
	    << "***********************\n" << endl;
    logfile.close();
    // and now disconnect the old timer connection
    timer_slot_connection.disconnect();
  }

  // now open the new log file if required
  if (!prog_config.logfile_name.empty()) {

    string temp(prog_config.logfile_name);
    if (temp[0] != '/') temp.insert(0, starting_dirname); // provide an absolute path name

    logfile.open(temp.c_str(), ios::app | ios::out);
    if (!logfile) {
      prog_config.logfile_name = "";
      string message("Can't open logfile ");
      message += temp;
      message += '\n';
      write_red_slot(message.c_str());
    }
    else {
      struct tm* time_p;
      time_t time_count;
      time(&time_count);
      time_p = localtime(&time_count);
      
      const char date_description_format[] = "%H%M %Z %d %b %Y";
      const int max_description_datesize = 126;
      char date_description[max_description_datesize];
      strftime(date_description, max_description_datesize, date_description_format, time_p);

      logfile << "\n\n***********************\n"
                 "Beginning fax log at " << date_description << '\n' << endl;
      // set a timeout to flush every one minute
      timer_slot_connection = Gtk::Main::timeout.connect(SigC::slot(this,
					 &MessageText::flush_logfile_timer_slot), LOGFILE_TIMER_INTERVAL);
    }
  }
  // save the logfile name in the local version of logfile_name 
  // in case reset_logfile() is called again later
  logfile_name = prog_config.logfile_name;
}

void MessageText::cleanify(string& message) {

  string::size_type chunk_start = 0;
  string::size_type chunk_end;
  while (chunk_start != string::npos) {
    string::size_type chunk_size;
    chunk_end = message.find('\n', chunk_start);

    // include in the search chunk any '\n' found
    if (chunk_end + 1 >= message.size()) chunk_end = string::npos;
    else if (chunk_end != string::npos) chunk_end++;

    // now search the relevant string/substring and erase from message
    if (chunk_end == string::npos) chunk_size = message.size() - chunk_start;
    else chunk_size = chunk_end - chunk_start;
    if (message.substr(chunk_start, chunk_size).find(" Copyright ") != string::npos
	|| message.substr(chunk_start, chunk_size).find(" compiled ") != string::npos
	|| message.substr(chunk_start, chunk_size).find("Error: terminating on signal 15") != string::npos) {
      message.erase(chunk_start, chunk_size);
      // if we have erased this chunk then we don't want to reset chunk_start
      // unless we have finished examining message
      if (chunk_end == string::npos) chunk_start = string::npos;
    }
    // if we didn't erase the last chunk, then we need to reset
    // chunk_start to beginning of next substring (or to string::npos)
    else chunk_start = chunk_end;
  }
}

void MessageText::write_black_slot(const char* message) {
  string temp(message);
  cleanify(temp);
  text.insert(temp);
  // now relay to standard output
  write(1, temp.c_str(), temp.size());
  // now log to file if required
  if (!prog_config.logfile_name.empty()) logfile << temp;
}

void MessageText::write_red_slot(const char* message) {
  string temp(message);
  cleanify(temp);
  text.insert(red_context, temp);
  // now relay to standard error
  write(2, temp.c_str(), temp.size());
  // now log to file if required
  if (!prog_config.logfile_name.empty()) logfile << temp;
}

gint MessageText::flush_logfile_timer_slot(void) {
  logfile.flush();
  return true; // we want a multi-shot timer
}

StatusLine::StatusLine(const int standard_size): 
                                       Gtk::HBox(false, 0), status_label("Inactive"),
                                       flexi_label("Press F1 for help") {

  set_border_width(2);
  status_label.set_usize(standard_size * 7, standard_size - 4);
  status_frame.set_shadow_type(GTK_SHADOW_IN);
  status_frame.add(status_label);
  pack_start(status_frame, false, false, 0);
  
  flexi_label.set_usize(standard_size * 6, standard_size - 4);
  flexi_frame.set_shadow_type(GTK_SHADOW_IN);
  flexi_frame.add(flexi_label);
  pack_end(flexi_frame, true, true, 0);

  // make the status label display in red
  // Note: we need to take a copy of the existing style, as the existing
  // style is shared by all the program's widgets.  If we operate
  // directly on the Gtk::Style object returned (by pointer) by
  // Gtk::Widget::get_style(), say with Gtk::Style::set_fg() or
  // Gtk::Style::set_font(), we will alter it for everything!
  Gtk::Style* new_style_p = status_label.get_style()->copy();
  Gdk_Color red;
  red.set_rgb_p(0.9, 0, 0);
  new_style_p->set_fg(GTK_STATE_NORMAL, red);
  status_label.set_style(*new_style_p);

  show_all();
}

int connect_to_stderr(void) {
  int result = MainWindow::error_pipe.connect_to_stderr();
  if (!result) MainWindow::connected_to_stderr = true;
  return result;
} 

ssize_t write_error(const char* message) {
  if (!MainWindow::connected_to_stderr) return MainWindow::error_pipe.write(message);
  else return ::write(2, message, strlen(message));
}

void close_signalhandler(int) {
  close_flag = true;
}

void childexit_signalhandler(int) {
  int exit_code = -1;
  int stat_val;
  pid_t result = waitpid(-1, &stat_val, WNOHANG);
  if (result > 0 && WIFEXITED(stat_val)) exit_code = WEXITSTATUS(stat_val);
  //if (!efax_controller_childexit_handler(result, exit_code))  // for debugging
    //cout << "Not efax_controller child" << endl;
    //cout << "Child pid is " << result << endl;
    //cout << "Child exit code is " << exit_code << endl;

  efax_controller_childexit_handler(result, exit_code);
}
