/* eblook.c - interactive EB interface command
 *
 * Copyright (C) 1997,1998,1999 Keisuke Nishida <knishida@ring.gr.jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <errno.h>

#if STDC_HEADERS
# include <stdlib.h>
# include <string.h>
#else
# ifndef HAVE_STRCHR
#  define strchr index
#  define strrchr rindex
# else
char *strchr (), *strrchr ();
# endif
#endif

#include "getopt.h"
#include "codeconv.h"

#ifdef EBCONF_ENABLE_PTHREAD
#define ENABLE_PTHREAD
#endif

#include "eb/eb.h"
#include "eb/text.h"
#include "eb/font.h"
#include "eb/appendix.h"
#include "eb/error.h"

#ifndef BUFSIZ
#define BUFSIZ	1024
#endif

/*
 *	maximum path length, same as in eb source files.
 */
#ifndef PATH_MAX
#ifdef MAXPATHLEN
#define PATH_MAX        MAXPATHLEN
#else /* not MAXPATHLEN */
#define PATH_MAX        1024
#endif /* not MAXPATHLEN */
#endif /* not PATH_MAX */

#define MAX_HIT_SIZE	256
#define MAX_TEXT_SIZE	8192
#define MAX_DUMP_SIZE	2048

#define USER_INIT_FILE  "~/.eblookrc"

/*
 * String A-list
 */
typedef struct _StringAlist {
  char                 *key;
  char                 *value;
  struct _StringAlist  *next;
} StringAlist;

/*
 * Internal functions
 */
char *read_command EB_P ((char *, int, FILE *));
int excute_command EB_P ((char *));
int parse_command_line EB_P ((char *, char *[]));

void command_book EB_P ((int, char *[]));
void command_info EB_P ((int, char *[]));
void command_list EB_P ((int, char *[]));
void command_select EB_P ((int, char *[]));
void command_subinfo EB_P ((int, char *[]));
void command_copyright EB_P ((int, char *[]));
void command_menu EB_P ((int, char *[]));
void command_search EB_P ((int, char *[]));
void command_content EB_P ((int, char *[]));
void command_dump EB_P ((int, char *[]));
void command_font EB_P ((int, char *[]));
void command_show EB_P ((int, char *[]));
void command_set EB_P ((int, char *[]));
void command_unset EB_P ((int, char *[]));
void command_help EB_P ((int, char *[]));

void command_candidate EB_P ((int, char *[]));
void command_label EB_P ((int, char *[]));

void show_entry_candidate EB_P(( EB_Book *, int, int ));
void show_label EB_P((EB_Book *, int));

int check_book EB_P (());
int check_subbook EB_P (());

int internal_set_font EB_P ((EB_Book *, char *));
int parse_dict_id EB_P ((char *, EB_Book *));
int parse_entry_id EB_P ((char *, EB_Position *));

int search_pattern EB_P ((EB_Book *, EB_Appendix *, char *, int, int));
int insert_content EB_P ((EB_Book *, EB_Appendix *, EB_Position *, int, int));
int insert_dump EB_P ((EB_Book *, EB_Appendix *, EB_Position *, int));
int insert_font EB_P ((EB_Book *, const char *));
int insert_font_list EB_P ((EB_Book *));

EB_Error_Code hook_font EB_P ((EB_Book *, EB_Appendix *, void *, EB_Hook_Code, int, const unsigned int *));
EB_Error_Code hook_stopcode EB_P ((EB_Book *, EB_Appendix *, void *, EB_Hook_Code, int, const unsigned int *));
EB_Error_Code hook_tags EB_P ((EB_Book *, EB_Appendix *, void *, EB_Hook_Code, int, const unsigned int *));

EB_Error_Code can_menu_narrow_char EB_P ((EB_Book *, EB_Appendix *, void *, EB_Hook_Code, int, const unsigned int *));
EB_Error_Code can_menu_wide_char EB_P ((EB_Book *, EB_Appendix *, void *, EB_Hook_Code, int, const unsigned int *));
EB_Error_Code can_menu_gaiji EB_P ((EB_Book *, EB_Appendix *, void *, EB_Hook_Code, int, const unsigned int *));

void show_version EB_P ((void));
void show_help EB_P ((void));
void show_try_help EB_P ((void));
void set_error_message EB_P ((EB_Error_Code));
void unset_error_message EB_P (());

StringAlist *salist_set EB_P ((StringAlist *, const char *, const char *));
char *salist_ref EB_P ((StringAlist *, const char *));

EB_Error_Code eblook_search_keyword EB_P ((EB_Book *, const char *));
EB_Error_Code eblook_search_multi EB_P ((EB_Book *, const char *));
static void output_multi_information EB_P ((EB_Book *));

#define variable_set(key, value) \
     variable_alist = salist_set (variable_alist, key, value)
#define variable_ref(key) \
     salist_ref (variable_alist, key)

#ifdef WIN32 /* in win32.c */
size_t euc_to_sjis(const char *, char *, size_t);
size_t sjis_to_euc(const char *, char *, size_t);
void dos_fix_path(char *, int);
#endif       /* WIN32 */

/*
 * Constants
 */
const char *program_name = PACKAGE;
const char *program_version = VERSION;
const char *default_prompt = "eblook> ";
const char *default_method = "glob";

/*
 * Internal variables
 */
const char *invoked_name;

EB_Book current_book;
EB_Appendix current_appendix;

EB_Hookset text_hookset;
StringAlist *variable_alist = NULL;
int last_search_begin = 0;
int last_search_length = 0;
int (*last_search_function) EB_P ((EB_Book *, const char *)) = NULL;

/*
 * Interactive command table
 */
struct {
  const char *name;
  const char *option_string;
#if defined(__STDC__) || defined(WIN32)
  void (*func) (int, char *[]);
#else /* not __STDC__ */
  void (*func) ();
#endif /* not __STDC__ */
  const char *help;
} command_table[] = {
  {"book", "[directory [appendix]]", command_book, "Set a book directory.\n"},
  {"info", "", command_info, "Show information of the selected book.\n"},
  {"list", "", command_list, "List all dictionaries in the selected book.\n"},
  {"select", "subbook", command_select, "Select a subbook.\n"},
  {"subinfo", "", command_subinfo, "Show information of the selected subbook.\n"},
  {"copyright", "", command_copyright, "Show copyright of the selected subbook.\n"},
  {"menu", "", command_menu, "Show the menu of the selected subbook.\n"},
  {"search", "pattern [offset]", command_search, "Search for a word\n"},
  {"content", "entry [offset]", command_content, "Display contents of entry.\n"},
  {"dump", "entry [offset]", command_dump, "Display dumps of entry.\n"},
  {"font", "[id]", command_font, "Display the bitmap of gaiji.\n"},
  {"show", "[variable]", command_show, "Show the value of variables.\n"},
  {"set", "variable value", command_set, "Set a variable to the value.\n"},
  {"unset", "variable...", command_unset, "Unset variables.\n"},
  {"candidate", "", command_candidate, "Show candidates for multi search\n"},
  {"label", "[id]", command_label, "Show label for multi search\n"},
  {"help", "", command_help, "Show this message.\n"},
  {"quit", "", NULL, "Quit program.\n"},
  {NULL, NULL, NULL, NULL}
};

/*
 * Text hooks
 */
EB_Hook text_hooks[] = {
  {EB_HOOK_NARROW_JISX0208, eb_hook_euc_to_ascii},
  {EB_HOOK_NARROW_FONT,     hook_font},
  {EB_HOOK_WIDE_FONT,	    hook_font},
  {EB_HOOK_NEWLINE,         eb_hook_newline},
  {EB_HOOK_STOP_CODE,       hook_stopcode},
/*   {EB_HOOK_BEGIN_PICTURE,   hook_tags}, */
/*   {EB_HOOK_END_PICTURE,     hook_tags}, */
/*   {EB_HOOK_BEGIN_SOUND,     hook_tags}, */
/*   {EB_HOOK_END_SOUND,       hook_tags}, */
  {EB_HOOK_BEGIN_REFERENCE, hook_tags},
  {EB_HOOK_END_REFERENCE,   hook_tags},
  {EB_HOOK_BEGIN_CANDIDATE, hook_tags},
  {EB_HOOK_END_CANDIDATE_GROUP, hook_tags},
  {EB_HOOK_NULL, NULL},
};

static const char *short_options = "e:hqv";
static struct option long_options[] = {
  {"encoding",     required_argument, NULL, 'e'},
  {"help",         no_argument,       NULL, 'h'},
  {"no-init-file", no_argument,       NULL, 'q'},
  {"version",      no_argument,       NULL, 'v'},
  {NULL,           no_argument,       NULL, 0}
};


int
main (argc, argv)
     int argc;
     char *const *argv;
{
  int optch;
  int no_init = 0;
  char buff[BUFSIZ];
  const char *book, *appendix, *s;
  FILE *fp;
  EB_Character_Code charcode;
  EB_Error_Code error_code = EB_SUCCESS;

  invoked_name = argv[0];
#ifdef _WIN32
  locale_init("SJIS");
#else
  locale_init(NULL);
#endif

  /* parse command line options */
  while ((optch = getopt_long(argc, argv, short_options, long_options, NULL))
	 != EOF) {
    switch (optch) {
    case 'e':
      locale_init(optarg);
      break;
    case 'h':
      show_help ();
      exit (0);
    case 'v':
      show_version ();
      exit (0);
    case 'q':
      no_init = 1;
      break;
    default:
      show_try_help ();
      exit (1);
    }
  }

  /* check the rest arguments */
  book = appendix = NULL;
  switch (argc - optind) {
  case 2:
    appendix = argv[optind + 1];
  case 1:
    book = argv[optind];
  case 0:
    break;

  default:
    xfprintf (stderr, "%s: too many arguments\n", invoked_name);
    show_try_help ();
    exit (1);
  }

  /* initialize variables */
  eb_initialize_library ();
  eb_initialize_book (&current_book);
  eb_initialize_appendix (&current_appendix);
  eb_initialize_hookset (&text_hookset);
  eb_set_hooks (&text_hookset, text_hooks);

  variable_set ("prompt", default_prompt);
  variable_set ("search-method", default_method);

  sprintf (buff, "%d", MAX_HIT_SIZE);
  variable_set ("max-hits", buff);

  sprintf (buff, "%d", MAX_TEXT_SIZE);
  variable_set ("max-text", buff);

  sprintf (buff, "%d", MAX_DUMP_SIZE);
  variable_set ("max-dump", buff);

  sprintf (buff, "%s %s (with EB %d.%d)", program_name, program_version,
	   EB_VERSION_MAJOR, EB_VERSION_MINOR);
  variable_set ("version", buff);

  variable_set ("multi-search-id", "1");

  /* load init file */
  if (!no_init) {
    if (!strncmp (USER_INIT_FILE, "~/", 2)) {
#ifdef	WIN32
      char *homedir = getenv ("HOME");
      if (homedir) {
        strcpy (buff, homedir);
        strcat (buff, USER_INIT_FILE + 1);
      } else {
        strcpy (buff, USER_INIT_FILE);
      }
#else	/* !WIN32 */
      strcpy (buff, getenv ("HOME"));
      strcat (buff, USER_INIT_FILE + 1);
#endif	/* WIN32 */
    } else {
      strcpy (buff, USER_INIT_FILE);
    }
    if ((fp = fopen (buff, "r")) != NULL)
      while (read_command (buff, BUFSIZ, fp) != NULL)
	if (!excute_command (buff))
	  break;
  }

  /* set book and appendix */
  if (book) {
#ifndef WIN32
    error_code = eb_bind (&current_book, book);
    if (EB_SUCCESS != error_code) {
      xprintf ("Warning: invalid book directory: %s\n", book);
      set_error_message (error_code);
    }
#else	/* WIN32 */
    strncpy(buff, book, sizeof(buff));
    dos_fix_path(buff, 0);
    error_code = eb_bind (&current_book, buff);
    if (EB_SUCCESS != error_code) {
      sjis_to_euc(book, buff, sizeof(buff));
      xprintf ("Warning: invalid book directory: %s\n", buff);
    }
#endif	/* WIN32 */
  }
  if (appendix) {
#ifndef WIN32
    error_code = eb_bind_appendix (&current_appendix, appendix);
    if (EB_SUCCESS != error_code) {
      xprintf ("Warning: invalid appendix directory: %s\n", appendix);
      set_error_message (error_code);
    }
#else	/* WIN32 */
    strncpy(buff, appendix, sizeof(appendix));
    dos_fix_path(buff, 0);
    if (EB_SUCCESS != error_code) {
      xprintf ("Warning: invalid appendix directory: %s\n", buff);
      set_error_message (error_code);
    }
#endif	/* !WIN32 */
  }

  /* check the book directory */
  if (!eb_is_bound (&current_book))
    xputs ("Warning: you should specify a book directory first\n");

  /* enter command loop */
  while (1) {
    /* kanji code */
    error_code = eb_character_code (&current_book, &charcode);
    if (EB_CHARCODE_JISX0208 != error_code) {
      if ((s = variable_ref ("kanji-code")) != NULL) {
	if (strcasecmp (s, "JIS") == 0)
	  locale_init ("JIS");
	else if (strcasecmp (s, "SJIS") == 0)
	  locale_init ("SJIS");
	else if (strcasecmp (s, "EUC") == 0)
	  locale_init ("EUC");
	else if (strcasecmp (s, "UTF8") == 0)
	  locale_init ("UTF8");
	else if (strcasecmp (s, "AUTO") == 0)
	  locale_init ("NULL");
	else {
	  xprintf ("Invalid kanji code: %s\n", s);
	  variable_set ("kanji-code", NULL);
	}
      }
    }

    /* prompt */
    if ((s = variable_ref ("prompt")) == NULL)
      s = default_prompt;
    xfputs (s, stdout);
    fflush (stdout);

    /* read and excute */
    unset_error_message ();
    if (read_command (buff, BUFSIZ, stdin) == NULL)
      break;
    if (!excute_command (buff))
      break;
  }

  eb_finalize_library ();
  return 0;
}

char *
read_command (command_line, size, stream)
     char *command_line;
     int size;
     FILE *stream;
{
  char *p;

  /* read string */
  if (xfgets (command_line, size, stream) == NULL)
    return NULL;

  /* delete '\n' */
  if ((p = strchr (command_line, '\n')) != NULL) {
    *p = '\0';
  } else {
    xputs ("Input is too long");
    while (xfgets (command_line, BUFSIZ, stdin) != NULL &&
	   strchr (command_line, '\n') == NULL);
    command_line[0] = '\0';
  }
  return command_line;
}

int
excute_command (command_line)
     char *command_line;
{
  int i, argc;
  char *argv[BUFSIZ / 2];			/* xxx: no good? */

  argc = parse_command_line (command_line, argv);

  /* if input is empty, do nothing */
  if (argc == 0)
    return 1;

  /* if input is "quit", we should quit */
  if (strcmp (argv[0], "quit") == 0)
    return 0;

  /* otherwise, search command and execute */
  for (i = 0; command_table[i].name != NULL; i++) {
    if (strcmp (argv[0], command_table[i].name) == 0) {
      command_table[i].func (argc, argv);
      return 1;
    }
  }
  if (command_table[i].name == NULL)
    xprintf ("Unkown command: %s\n", argv[0]);
  return 1;
}

int
parse_command_line (command_line, argv)
     char *command_line;
     char *argv[];
{
  int num;
  int reserved, in_quote;
  char *p;

  /* devide string into tokens by white spaces */
  num = reserved = in_quote = 0;
  for (p = command_line; *p != '\0'; p++) {
    switch (*p) {
    case '"':
      if (!reserved) {
	argv[num++] = p;
	reserved = 1;
      }
      strcpy (p, p + 1);
      p--;
      in_quote = !in_quote;
      break;

    case ' ':
    case '\t':
      if (!in_quote) {
	*p = '\0';
	reserved = 0;
      }
      break;

    case '\\':
      strcpy (p, p + 1);
    default:
      if (!reserved) {
	argv[num++] = p;
	reserved = 1;
      }
    }
  }

  return num;
}

void
command_book (argc, argv)
     int argc;
     char *argv[];
{
  EB_Error_Code error_code = EB_SUCCESS;

  switch (argc) {
  case 3:
#ifdef WIN32
    dos_fix_path(argv[2], 1);
#endif	/* WIN32 */
    error_code = eb_bind_appendix (&current_appendix, argv[2]);
    if (EB_SUCCESS != error_code) {
      xprintf ("Invalid appendix directory: %s\n", argv[2]);
      set_error_message (error_code);
    }
  case 2:
#ifdef WIN32
    dos_fix_path(argv[1], 1);
#endif	/* WIN32 */
    error_code = eb_bind (&current_book, argv[1]);
    if (EB_SUCCESS != error_code) {
      xprintf ("Invalid book directory: %s\n", argv[1]);
      set_error_message (error_code);
    }
    break;

  case 1:
    if (eb_is_bound (&current_book)) {
#ifndef WIN32
      char temp[BUFSIZ];
      eb_path (&current_book, temp);
      xprintf ("book\t%s\n", temp);
      if (eb_is_appendix_bound (&current_appendix)) {
	eb_appendix_path (&current_appendix, temp);
	xprintf ("appendix\t%s\n", temp);
      }
#else	/* WIN32 */
      char temp[PATH_MAX], temp_euc[PATH_MAX];
      eb_path (&current_book, temp);
      sjis_to_euc(temp, temp_euc, sizeof(temp_euc));
      xprintf ("book\t%s\n", temp_euc);
      if (eb_is_appendix_bound (&current_appendix)) {
        eb_appendix_path (&current_appendix, temp);
        sjis_to_euc(temp, temp_euc, sizeof(temp_euc));
        xprintf ("appendix\t%s\n", temp_euc);
      }
#endif	/* !WIN32 */
    } else {
      xputs ("No book is specified");
    }
    break;

  default:
    xprintf ("%s: too many arguments\n", argv[0]);
  }
}

void
command_info (argc, argv)
     int argc;
     char *argv[];
{
  EB_Subbook_Code code[EB_MAX_SUBBOOKS];
  if (argc > 1) {
    xprintf ("%s: too many arguments\n", argv[0]);
    return;
  }

  if (check_book ()) {
    int subcount;
    EB_Disc_Code disccode;
    EB_Character_Code charcode;

    /* disc type */
    eb_disc_type (&current_book, &disccode);
    if (disccode >= 0) {
      xfputs (" disc type: ", stdout);
      xputs ((disccode == EB_DISC_EB) ? "EB/EBG/EBXA" : "EPWING");
    }

    /* character code */
    eb_character_code (&current_book, &charcode);
    if (charcode >= 0) {
      xfputs (" character code: ", stdout);
      xputs ((charcode == EB_CHARCODE_JISX0208) ? "JIS X 0208" : "ISO 8859-1");
    }

    /* the number of dictionarys */
    if (EB_SUCCESS == eb_subbook_list (&current_book, code, &subcount)
        && subcount >= 0)
      xprintf (" the number of dictionries: %d\n", subcount);
  }
}

void
command_list (argc, argv)
     int argc;
     char *argv[];
{
  EB_Error_Code error_code = EB_SUCCESS;

  if (argc > 1) {
    xprintf ("%s: too many arguments\n", argv[0]);
    return;
  }

  if (check_book ()) {
    int i, num;
    char buff[PATH_MAX + 1];
    EB_Subbook_Code list[EB_MAX_SUBBOOKS];

    error_code = eb_subbook_list (&current_book, list, &num);
    if (EB_SUCCESS != error_code)
      goto error;

    for (i = 0; i < num; i++) {
      xprintf ("%2d. ", i + 1);

      error_code = eb_subbook_directory2 (&current_book, list[i], buff);
      if (EB_SUCCESS != error_code)
	goto error;

      xprintf ("%s\t", buff);
      error_code = eb_subbook_title2 (&current_book, list[i], buff);
      if (EB_SUCCESS != error_code)
	goto error;

      xfputs (buff, stdout);
      fputc ('\n', stdout);
    }

    return;

  error:
    xprintf ("An error occured in command_list: %s\n",
	    eb_error_message (error_code));
    set_error_message (error_code);
    return;
  }
}

void
command_select (argc, argv)
     int argc;
     char *argv[];
{
  switch (argc) {
  case 1:
    eb_unset_subbook (&current_book);
    eb_unset_appendix_subbook (&current_appendix);
    return;

  case 2:
    if (check_book ()) {
      if (parse_dict_id (argv[1], &current_book)) {
	if (eb_is_appendix_bound (&current_appendix))
	  {
	    EB_Subbook_Code code;

	    eb_subbook (&current_book, &code);
	    eb_set_appendix_subbook (&current_appendix, code);
	  }
      }
    }
    return;

  default:
    xprintf ("%s: too many arguments\n", argv[0]);
  }
}

void
command_subinfo (argc, argv)
     int argc;
     char *argv[];
{
  if (argc > 1) {
    xprintf ("%s: too many arguments\n", argv[0]);
    return;
  }

  if (check_subbook ()) {
    int i, num;
    char buff[PATH_MAX + 1];
    EB_Font_Code list[EB_MAX_FONTS];

    /* title */
	if (EB_SUCCESS == eb_subbook_title (&current_book, buff)) {
      xfputs (" title: ", stdout);
      xfputs (buff, stdout);
      fputc ('\n', stdout);
	}

    /* directory */
    if (EB_SUCCESS == eb_subbook_directory (&current_book, buff))
      xprintf (" directory: %s\n", buff);

    /* search methods */
    xfputs (" search methods:", stdout);
    if (eb_have_word_search (&current_book))
      xfputs (" word", stdout);
    if (eb_have_endword_search (&current_book))
      xfputs (" endword", stdout);
    if (eb_have_exactword_search (&current_book))
      xfputs (" exactword", stdout);
    if (eb_have_keyword_search (&current_book))
      xfputs (" keyword", stdout);
    if (eb_have_multi_search (&current_book))
      xfputs (" multi", stdout);
    if (eb_have_menu (&current_book))
      xfputs (" menu", stdout);
/*     if (eb_have_graphic_search (&current_book)) */
/*       xfputs (" graphic", stdout); */
    fputc ('\n', stdout);

    /* font size */
    xfputs (" font sizes:", stdout);

    eb_font_list (&current_book, list, &num);
    for (i = 0; i < num; i++) {
      switch (list[i]) {
      case EB_FONT_16: xprintf (" 16"); break;
      case EB_FONT_24: xprintf (" 24"); break;
      case EB_FONT_30: xprintf (" 30"); break;
      case EB_FONT_48: xprintf (" 48"); break;
      default: xprintf (" 0"); break;
      }
    }

    fputc ('\n', stdout);
  }
}

void
command_copyright (argc, argv)
     int argc;
     char *argv[];
{
  if (argc > 1) {
    xprintf ("%s: too many arguments\n", argv[0]);
    return;
  }

  if (check_subbook ()) {
    EB_Position pos;
    if (EB_SUCCESS != eb_copyright (&current_book, &pos))
      xputs ("Current dictionary has no copyright information.");
    else
      insert_content (&current_book, &current_appendix, &pos, 0, 0);
  }
}

void
command_menu (argc, argv)
     int argc;
     char *argv[];
{
  if (argc > 1) {
    xprintf ("%s: too many arguments\n", argv[0]);
    return;
  }

  if (check_subbook ()) {
    EB_Position pos;
    if (EB_SUCCESS != eb_menu (&current_book, &pos))
      xputs ("Current dictionary has no menu.");
    else
      insert_content (&current_book, &current_appendix, &pos, 0, 0);
  }
}

void
command_search (argc, argv)
     int argc;
     char *argv[];
{
  int begin, length;
  char *pattern;

  begin = 1;
  pattern = variable_ref ("max-hits");
  length = pattern ? atoi (pattern) : 0;
  pattern = NULL;

  switch (argc) {
  case 3:
    begin = atoi (argv[2]);
  case 2:
    pattern = argv[1];
  case 1:
    if (check_subbook ())
      search_pattern (&current_book, &current_appendix, pattern, begin, length);
    return;

  default:
    xprintf ("%s: too many arguments\n", argv[0]);
  }
}

void
command_content (argc, argv)
     int argc;
     char *argv[];
{
  int begin, length;
  char *s;
  EB_Position pos;

  begin = 1;
  s = variable_ref ("max-text");
  length = s ? (atoi (s) / EB_SIZE_PAGE) : 0;

  switch (argc) {
  case 1:
    xprintf ("%s: too few arguments\n", argv[0]);
    return;

  case 4:
    length = atoi (argv[3]);
  case 3:
    begin = atoi (argv[2]);
  case 2:
    if (check_subbook ()) {
      if (parse_entry_id (argv[1], &pos))
	insert_content (&current_book, &current_appendix, &pos, begin, length);
    }
    return;

  default:
    xprintf ("%s: too many arguments\n", argv[0]);
  }
}

void
command_dump (argc, argv)
     int argc;
     char *argv[];
{
  int begin, length;
  char *s;
  EB_Position pos;

  begin = 1;
  s = variable_ref ("max-dump");
  length = s ? (atoi (s) / EB_SIZE_PAGE) : (MAX_DUMP_SIZE / EB_SIZE_PAGE);
  if (length == 0)
    length = 1;

  switch (argc) {
  case 1:
    xprintf ("%s: too few arguments\n", argv[0]);
    return;

  case 4:
    length = atoi (argv[3]);
  case 3:
    begin = atoi (argv[2]);
  case 2:
    if (check_subbook ()) {
      if (parse_entry_id (argv[1], &pos))
	insert_dump (&current_book, &current_appendix, &pos, length);
    }
    return;

  default:
    xprintf ("%s: too many arguments\n", argv[0]);
  }
}

void
command_font (argc, argv)
     int argc;
     char *argv[];
{
  if (argc > 2) {
    xprintf ("%s: too many arguments\n", argv[0]);
    return;
  }

  if (check_subbook () && internal_set_font (&current_book, NULL)) {
    if (argc == 1)
      insert_font_list (&current_book);
    else
      insert_font (&current_book, argv[1]);
  }
}

void
command_show (argc, argv)
     int argc;
     char *argv[];
{
  char *s;
  StringAlist *var;
  switch (argc) {
  case 1:
    /*
     * Show all variables and their values
     */
    for (var = variable_alist; var != NULL; var = var->next)
      if (var->key[0] != '_')
	xprintf ("%s\t%s\n", var->key, var->value);
    return;

  case 2:
    /*
     * Show value of variable
     */
    if ((s = variable_ref (argv[1])) != NULL)
      xputs (s);
    else
      xprintf ("Unbounded variable: %s\n", argv[1]);
    return;

  default:
    xprintf ("%s: too many arguments\n", argv[0]);
  }
}

void
command_set (argc, argv)
     int argc;
     char *argv[];
{
  switch (argc) {
  case 1:
    xprintf ("%s: too few arguments\n", argv[0]);
    return;

  case 2:
    argv[2] = "";
  case 3:
    variable_set (argv[1], argv[2]);
    return;

  default:
    xprintf ("%s: too many arguments\n", argv[0]);
  }
}

void
command_unset (argc, argv)
     int argc;
     char *argv[];
{
  int i;
  if (argc == 1) {
    xprintf ("%s: too few arguments\n", argv[0]);
  } else {
    for (i = 1; i < argc; i++)
      variable_set (argv[i], NULL);
    return;
  }
}

void
command_help (argc, argv)
     int argc;
     char *argv[];
{
  int i;
  char buff[256];
  const char *p;
  switch (argc) {
  case 1:
    /*
     * List up all command helps
     */
    for (i = 0; command_table[i].name != NULL; i++) {
      sprintf (buff, "%s %s",
	      command_table[i].name, command_table[i].option_string);
      xprintf (" %-22s - ", buff);
      for (p = command_table[i].help;
	   *p != '\0' && *p != '.' && *p != '\n';
	   p++)
	putchar (*p);
      putchar ('\n');
    }
    return;

  case 2:
    /*
     * Show command help
     */
    for (i = 0; command_table[i].name != NULL; i++) {
      if (strcmp (command_table[i].name, argv[1]) == 0)
	break;
    }
    if (command_table[i].name == NULL) {
      xprintf ("No such command: %s\n", argv[1]);
    } else {
      xprintf ("Usage: %s %s\n\n%s",
	      command_table[i].name, command_table[i].option_string,
	      command_table[i].help);
    }
    return;

  default:
    xprintf ("%s: too many arguments\n", argv[0]);
  }
}

int
check_book ()
{
  if (eb_is_bound (&current_book)) {
    return 1;
  } else {
    xputs ("You should specify a book directory first");
    return 0;
  }
}

int
check_subbook ()
{
  if (check_book ()) {
    EB_Subbook_Code code;
    if (EB_SUCCESS == eb_subbook (&current_book, &code))
      return 1;
    else
      xputs ("You should select a subbook first");
  }
  return 0;
}

int
internal_set_font (book, height)
     EB_Book *book;
     char *height;
{
  EB_Font_Code font;
  EB_Error_Code error_code = EB_SUCCESS;

  if (height == NULL)
    if ((height = variable_ref ("font")) == NULL)
      height = "16";

  font = atoi (height);
  switch (font) {
  case 16: font = EB_FONT_16; break;
  case 24: font = EB_FONT_24; break;
  case 30: font = EB_FONT_30; break;
  case 48: font = EB_FONT_48; break;
  default:
    xprintf ("Illegal font height: %s\n", height);
    return 0;
  }

  if (!eb_have_font (book, font) ) {
    xprintf ("Invalid font for %s: %s\n",
	    book->subbook_current->directory_name,
	    height);
    /* set_error_message (); */
    return 0;
  }

  error_code = eb_set_font (book, font);
  if (EB_SUCCESS != error_code) {
    xprintf ("An error occurred in internal_set_font: %s\n",
	   eb_error_message (error_code));
    set_error_message (error_code);
    return 0;
  }
  return 1;
}

int
parse_dict_id (name, book)
     char *name;
     EB_Book *book;
{
  int i, num;
  EB_Subbook_Code sublist[EB_MAX_SUBBOOKS];
  EB_Error_Code error_code = EB_SUCCESS;

#ifdef	_DEBUG
  /* to avoid strange behavior in VC++ 6.0 debug mode. */
  memset (sublist, -1, sizeof (sublist));
#endif	/* !DEBUG */
  error_code = eb_subbook_list (book, sublist, &num);
  if (EB_SUCCESS != error_code)
    goto error;

  if ((i = atoi (name)) > 0) {
    /*
     * Numbered dictionary
     */
    if (--i < num) {
      error_code = eb_set_subbook (book, sublist[i]);
      if (EB_SUCCESS != error_code)
	goto error;
      return 1;
    } else {
      xprintf ("No such numberd dictionary : %s\n", name);
      return 0;
    }
  } else {
    /*
     * Named dictionary
     */
    char dir[PATH_MAX + 1];

    for (i = 0; i < num; i++) {
      error_code = eb_subbook_directory2 (book, sublist[i], dir);
      if (EB_SUCCESS != error_code)
	goto error;

      if (strcmp (name, dir) == 0) {
	error_code = eb_set_subbook (book, sublist[i]);
        if (EB_SUCCESS != error_code)
	  goto error;
	return 1;
      }
    }
    xprintf ("No such dictionary: %s\n", name);
    return 0;
  }

 error:
  xprintf ("An error occurred in parse_dict_id: %s\n",
	  eb_error_message (error_code));
  set_error_message (error_code);
  return 0;
}

int
parse_entry_id (code, pos)
     char *code;
     EB_Position *pos;
{
  EB_Error_Code error_code = EB_SUCCESS;

  if (strchr (code, ':') != NULL) {
    /*
     * Encoded position
     */
    char *endp;
    pos->page = strtol (code, &endp, 16);
    if (*endp != ':')
      goto illegal;

    pos->offset = strtol (endp + 1, &endp, 16);
    if (*endp != '\0')
      goto illegal;

    return 1;

  illegal:
    xprintf ("Illegal position: %s\n", code);
    return 0;

  } else {
    /*
     * Numbered entry
     */
    int num, count;
    const char *pattern = variable_ref ("_last_search_pattern");
    EB_Hit list[MAX_HIT_SIZE];

    if (!pattern) {
      xputs ("No search has been executed yet.");
      return 0;
    }
    if ((count = atoi (code) - 1) < 0) {
      xprintf ("Invalid entry number: %s\n", code);
      return 0;
    }
    if (check_subbook ()) {
      error_code = last_search_function (&current_book, pattern);
      if (EB_SUCCESS != error_code) {
	xprintf ("An error occured in parse_entry_id: %s\n",
		eb_error_message (error_code));
	set_error_message (error_code);
	return 0;
      }
      while (EB_SUCCESS == eb_hit_list (&current_book, MAX_HIT_SIZE, list,
					&num) && 0 < num) {
	if (count < num) {
	  pos->page = list[count].text.page;
	  pos->offset = list[count].text.offset;
	  return 1;
	}
	count -= num;
	pattern = NULL;
      }
      if (num == 0)
	xprintf ("Too big: %s\n", code);
    }
    return 0;
  }
}


int
search_pattern (book, appendix, pattern, begin, length)
     EB_Book *book;
     EB_Appendix *appendix;
     char *pattern;
     int begin;
     int length;
{
  int i, num, point;
  char headbuf1[BUFSIZ];
  char headbuf2[BUFSIZ];
  char *head;
  const char *s;
  EB_Error_Code (*search) EB_P ((EB_Book *, const char *));
  EB_Hit hitlist[MAX_HIT_SIZE];
  EB_Error_Code error_code = EB_SUCCESS;

  char* prevhead;
  int prevpage;
  int prevoffset;
  ssize_t heading_len;

  if (pattern == NULL) {
    /* check last search */
    begin = last_search_begin;
    length = last_search_length;
    search = last_search_function;
    pattern = variable_ref ("_last_search_pattern");
    if (pattern == NULL) {
      xputs ("No search has been executed yet.");
      return 0;
    }
    if (last_search_begin == 0) {
      xprintf ("Last search had finished\n");
      return 0;
    }
  } else {
    /* get search method */
    if ((s = variable_ref ("search-method")) == NULL)
      s = default_method;

    if (strchr(pattern, '=') && eb_have_keyword_search(book))
      search = eblook_search_keyword;
    else if (strchr(pattern, ':') && eb_have_multi_search(book))
      search = eblook_search_multi;
    else if (strcmp (s, "exact") == 0)
      search = eb_search_exactword;
    else if (strcmp (s, "word") == 0)
      search = eb_search_word;
    else if (strcmp (s, "endword") == 0)
      search = eb_search_endword;
    else if (strcmp (s, "glob") == 0) {
      search = eb_search_exactword;

      /* check for word search */
      i = strlen (pattern) - 1;
      if (pattern[i] == '*') {
	pattern[i] = '\0';
	search = eb_search_word;
      }

      /* check for endword search */
      if (pattern[0] == '*') {
	pattern++;
	search = eb_search_endword;
      }
    } else {
      xprintf ("Invalid search method: %s\n", s);
      return 0;
    }
  }

  /* reserve search information */
  /* use EB_Book structure directly here so as not to use more buffer. */
  variable_set ("_last_search_book", book->path);
  variable_set ("_last_search_dict", book->subbook_current->directory_name);
  variable_set ("_last_search_pattern", pattern);
  last_search_begin = 0;
  last_search_length = length;
  last_search_function = search;

  /* search */
  point = 0;
  error_code = search (book, pattern);
  if (EB_SUCCESS != error_code) {
    xprintf ("An error occured in search_pattern: %s\n",
	    eb_error_message (error_code));
    set_error_message (error_code);
    return 0;
  }

  head = headbuf1;
  prevhead = headbuf2;
  *prevhead = '\0';
  prevpage = 0;
  prevoffset = 0;

  while (EB_SUCCESS == eb_hit_list (book, MAX_HIT_SIZE, hitlist, &num)
	 && 0 < num) {
    for (i = 0; i < num; i++) {
      point++;
      if (point >= begin + length && length > 0) {
	xprintf ("<more point=%d>\n", point);
	last_search_begin = point;
	goto exit;
      }

      if (point >= begin) {
  	error_code = eb_seek_text (book, &hitlist[i].heading);
        if (error_code != EB_SUCCESS)
	  continue;
	error_code = eb_read_heading (book, appendix, &text_hookset, NULL,
				      BUFSIZ - 1, head, &heading_len);
        if (error_code != EB_SUCCESS || heading_len == 0)
	  continue;
        *(head + heading_len) = '\0';

	if (prevpage == hitlist[i].text.page &&
	    prevoffset == hitlist[i].text.offset &&
	    strcmp (head, prevhead) == 0)
	  continue;

	xprintf ("%2d. %x:%x\t", point,
	       hitlist[i].text.page, hitlist[i].text.offset);
	xfputs (head, stdout);
	fputc ('\n', stdout);
      }

      if (head == headbuf1) {
	head = headbuf2;
	prevhead = headbuf1;
      } else {
	head = headbuf1;
	prevhead = headbuf2;
      }
      prevpage = hitlist[i].text.page;
      prevoffset = hitlist[i].text.offset;
    }
  }

 exit:
  return 1;
}

int
insert_content (book, appendix, pos, begin, length)
     EB_Book *book;
     EB_Appendix *appendix;
     EB_Position *pos;
     int begin;
     int length;
{
  int point;
  ssize_t len;
  char last = '\n';
  char buff[EB_SIZE_PAGE];
  EB_Error_Code error_code = EB_SUCCESS;

  /* insert */
  point = 0;
  error_code = eb_seek_text(book, pos);
  if (error_code != EB_SUCCESS) {
    xprintf("An error occured in seek_position: %s\n",
	   eb_error_message(error_code));
    set_error_message (error_code);
    return 0;
  }
  while (EB_SUCCESS == eb_read_text (book, appendix, &text_hookset, NULL,
				     EB_SIZE_PAGE - 1, buff, &len) &&
	 0 < len) {
    *(buff + len) = '\0';
    /* count up */
    point++;
    if (point >= begin + length && length > 0) {
      xprintf ("<more point=%d>\n", point);
      goto exit;
    }

    /* insert */
    if (point >= begin) {
      xfputs (buff, stdout);
      last = buff[len - 1];
    }
  }

  /* insert a newline securely */
  if (last != '\n')
    putchar ('\n');

 exit:
  return 1;
}

int
insert_dump (book, appendix, pos, length)
     EB_Book *book;
     EB_Appendix *appendix;
     EB_Position *pos;
     int length;
{
  int page;
  ssize_t len;
  unsigned char buff[EB_SIZE_PAGE];
  EB_Error_Code error_code = EB_SUCCESS;
  int i, count;
  long position;

  /* insert */
  for (page = 0; page < length; page++) {
    error_code = eb_seek_text(book, pos);
    if (error_code != EB_SUCCESS) {
      xprintf("An error occured in seek_position: %s\n",
	      eb_error_message(error_code));
      set_error_message (error_code);
      return 0;
    }

    position = (pos->page - 1) * 2048 + pos->offset;

    error_code = eb_read_rawtext (book, EB_SIZE_PAGE - 1, buff, &len);
    if ((error_code != EB_SUCCESS) || len <= 0) {
      xprintf ("An error occured in command_dump: %s\n",
	       eb_error_message (error_code));
      set_error_message (error_code);
      return 0;
    }
    /* insert */
    count = 0;
    while (count < EB_SIZE_PAGE) {
      xprintf("%04x:%03x  ", position / 2048 + 1, position % 2048);
      for (i = 0; i < 16; i+= 2) {
	xprintf("%02x%02x", buff[count+i], buff[count+i+1]);
      }
      xprintf("  ");
      for (i = 0; i < 16; i+= 2) {
	if (0x21 <= buff[count+i] && buff[count+i] <=0x7e &&
	    0x21 <= buff[count+i+1] && buff[count+i+1] <= 0x7e) {
	  xprintf("[%c%c]", buff[count+i] | 0x80, buff[count+i+1] | 0x80);
	} else {
	  xprintf("%02x%02x", buff[count+i], buff[count+i+1]);
	}
      }
      xprintf("\n");
      position += 16;
      count += 16;
    }
    pos->page ++;
  }

  return 1;
}

int
insert_font (book, id)
     EB_Book *book;
     const char *id;
{
  int ch, width, height, start, end;
  size_t size;
  char bitmap[EB_SIZE_WIDE_FONT_48];
  char xbm[EB_SIZE_WIDE_FONT_48_XBM];
  EB_Error_Code error_code = EB_SUCCESS;

  switch (*id) {
  case 'h':
    ch = strtol (id + 1, NULL, 16);
    eb_narrow_font_start (book, &start);
    eb_narrow_font_end (book, &end);
    if (start <= ch && ch <= end) {
      eb_narrow_font_width (book, &width);
      error_code = eb_narrow_font_character_bitmap (book, ch, bitmap);
      if (EB_SUCCESS != error_code)
	goto error;
    } else {
      xprintf ("No such character font: %s\n", id);
      return 0;
    }
    break;

  case 'z':
    ch = strtol (id + 1, NULL, 16);
    eb_wide_font_start (book, &start);
    eb_wide_font_end (book, &end);
    if (start <= ch && ch <= end) {
      eb_wide_font_width (book, &width);
      error_code = eb_wide_font_character_bitmap (book, ch, bitmap);
      if (EB_SUCCESS != error_code)
	goto error;
    } else {
      xprintf ("No such character font: %s\n", id);
      return 0;
    }
    break;

  default:
    xprintf ("Invalid font id: %s\n", id);
    return 0;
  }

  eb_font_height (book, &height);
  eb_bitmap_to_xbm (bitmap, width, height, xbm, &size);
  xbm[size] = '\0';
  xfputs (xbm, stdout);
  return 1;

 error:
  xprintf ("An error occured in insert_font: %s\n",
	  eb_error_message (error_code));
  set_error_message (error_code);
  return 0;
}

int
insert_font_list (book)
     EB_Book *book;
{
  int ch, width, height, start, end;
  size_t size;
  char bitmap[EB_SIZE_WIDE_FONT_48];
  char xbm[EB_SIZE_WIDE_FONT_48_XBM];

  eb_font_height (book, &height);

  eb_narrow_font_width (book, &width);
  eb_narrow_font_start (book, &start);
  eb_narrow_font_end (book, &end);

  for (ch = start; ch < end; ch++)
    if (EB_SUCCESS == eb_narrow_font_character_bitmap (book, ch, bitmap)) {
      eb_bitmap_to_xbm (bitmap, width, height, xbm, &size);
      xbm[size] = '\0';
      xfprintf (stdout, "\nid = h%04x\n", ch);
      xfputs (xbm, stdout);
    }

  eb_wide_font_width (book, &width);
  eb_wide_font_start (book, &start);
  eb_wide_font_end (book, &end);
  for (ch = start; ch < end; ch++)
    if (EB_SUCCESS == eb_wide_font_character_bitmap (book, ch, bitmap)) {
      eb_bitmap_to_xbm (bitmap, width, height, xbm, &size);
      xbm[size] = '\0';
      xfprintf (stdout, "\nid = z%04x\n", ch);
      xfputs (xbm, stdout);
    }
  return 1;
}

void
command_candidate(argc, argv)
     int argc;
     char *argv[];
{
    switch (argc) {
    case 3:
	show_entry_candidate(&current_book, atoi(argv[1])-1, atoi(argv[2])-1);
	break;
    default:
	output_multi_information(&current_book);
	break;
    }
}

void
command_label(argc, argv)
    int argc;
    char *argv[];
{
    if (argc == 1) {
	show_label(&current_book, -1);
    } else {
	show_label(&current_book, atoi(argv[1]) - 1);
    }

}



EB_Error_Code
hook_font (book, appendix, container, code, argc, argv)
     EB_Book *book;
     EB_Appendix *appendix;
     void *container;
     EB_Hook_Code code;
     int argc;
     const unsigned int *argv;
{
  char buff[EB_MAX_ALTERNATION_TEXT_LENGTH + 1];
  switch (code) {
  case EB_HOOK_NARROW_FONT:
    if (EB_SUCCESS != eb_narrow_alt_character_text (appendix, argv[0], buff))
      sprintf (buff, "<gaiji=h%04x>", argv[0]);
    eb_write_text_string(book, buff);
    break;

  case EB_HOOK_WIDE_FONT:
    if (EB_SUCCESS != eb_wide_alt_character_text (appendix, argv[0], buff))
      sprintf (buff, "<gaiji=z%04x>", argv[0]);
    eb_write_text_string(book, buff);
    break;
  }
  return 0;
}

EB_Error_Code
hook_stopcode (book, appendix, container, code, argc, argv)
     EB_Book *book;
     EB_Appendix *appendix;
     void *container;
     EB_Hook_Code code;
     int argc;
     const unsigned int *argv;
{
  const char *stop = variable_ref ("stop-code");

  if (stop) {
    unsigned int c;
    if (strncmp (stop, "0x", 2) == 0)
      c = strtol (stop + 2, NULL, 16);
    else
      c = atoi (stop);

    if (c == (argv[0] << 16) + argv[1])
      return EB_ERR_STOP_CODE;
    else
      return EB_SUCCESS;
  }

  return (eb_hook_stop_code (book, appendix, container, code, argc, argv));
}

EB_Error_Code
hook_tags (book, appendix, container, code, argc, argv)
     EB_Book *book;
     EB_Appendix *appendix;
     void *container;
     EB_Hook_Code code;
     int argc;
     const unsigned int *argv;
{
  char buff[EB_MAX_ALTERNATION_TEXT_LENGTH + 1];
  switch (code) {
/*   case EB_HOOK_BEGIN_PICTURE: */
/*     strcpy (buff, "<picture>"); break; */
/*   case EB_HOOK_END_PICTURE: */
/*     sprintf (buff, "</picture=%x:%x>", argv[1], argv[2]); break; */

/*   case EB_HOOK_BEGIN_SOUND: */
/*     strcpy (buff, "<sound>"); break; */
/*   case EB_HOOK_END_SOUND: */
    /*
     *	prevent access to the invalid address, but here argc may be always
     *	1 for some subbook(s)... how should it be handled?
     */
/*     sprintf(buff, "</sound=%x:%x>", argv[1], (2 < argc) ? argv[2] : 0); */
/*     break; */

  case EB_HOOK_BEGIN_REFERENCE:
  case EB_HOOK_BEGIN_CANDIDATE:
    strcpy (buff, "<reference>");
    eb_write_text_string(book, buff);
    break;
  case EB_HOOK_END_REFERENCE:
  case EB_HOOK_END_CANDIDATE_GROUP:
    sprintf (buff, "</reference=%x:%x>", argv[1], argv[2]);
    eb_write_text_string(book, buff);
    break;
  }
  return 0;
}


void
show_version ()
{
  xprintf ("%s %s (with EB %d.%d)\n", program_name, program_version,
	  EB_VERSION_MAJOR, EB_VERSION_MINOR);
  xputs ("Copyright (C) 1997,1998,1999,2000 NISHIDA Keisuke");
  xputs ("Copyright (C) 2000,2001 Satomi");
  xputs ("Copyright (C) 2000,2001 Kazuhiko");
  xputs ("Copyright (C) 2000,2001 NEMOTO Takashi");
  xputs ("eblook may be distributed under the terms of the GNU General Public Licence;");
  xputs ("certain other uses are permitted as well.  For details, see the file");
  xputs ("`COPYING'.");
  xputs ("There is no warranty, to the extent permitted by law.");
}

void
show_help ()
{
  xfprintf (stderr, "Usage: %s [option...] [book-directory [appendix-directory]]\n", program_name);
  xfprintf (stderr, "Options:\n");
  xfprintf (stderr, "  -e, --encoding=NAME   specify input/output encoding\n");
  xfprintf (stderr, "  -q, --no-init-file    ignore user init file\n");
  xfprintf (stderr, "  -h, --help            show this message\n");
  xfprintf (stderr, "  -v, --version         show version number\n");
  fflush (stderr);
}

void
show_try_help ()
{
  xfprintf (stderr, "Try `%s --help' for more information.\n", invoked_name);
  fflush (stderr);
}

void
set_error_message (error_code)
     EB_Error_Code error_code;
{
  variable_set ("_error", strerror (errno));
  variable_set ("_eb_error", eb_error_message (error_code));
}

void
unset_error_message ()
{
  variable_set ("_error", NULL);
  variable_set ("_eb_error", NULL);
}

EB_Error_Code
eblook_search_keyword(book, pattern)
     EB_Book *book;
     const char *pattern;
{
  char *keyword[EB_MAX_KEYWORDS+2];
  EB_Error_Code error_code;
  char *p = strdup(pattern);
  int i;

  keyword[0] = strtok(p, "=");

  for (i=1; i<EB_MAX_KEYWORDS+2; i++)
    keyword[i] = strtok(0, "=");

  error_code = eb_search_keyword(book, (const char * const *)keyword);

  free(p);
  return error_code;
}

EB_Error_Code
eblook_search_multi(book, pattern)
     EB_Book *book;
     const char *pattern;
{
  char *candidate[EB_MAX_MULTI_ENTRIES+2];
  EB_Error_Code error_code;
  char *p = strdup(pattern);
  int i;
  int multi_id = atoi(variable_ref("multi-search-id")) - 1;

#ifdef EBLOOK_SEARCH_MULTI_DEBUG
  xprintf("multi_id = %d\n", multi_id);
#endif

  candidate[0] = strtok(p, ":");

  for (i=1; i<EB_MAX_MULTI_ENTRIES+2; i++) {
    candidate[i] = strtok(0, ":");
  }

  for (i=0; i<EB_MAX_MULTI_ENTRIES+2; i++) {
    if (candidate[i] && 0 == strcmp(candidate[i], "*")) {
      candidate[i] = "";
    }
#ifdef EBLOOK_SEARCH_MULTI_DEBUG
    xprintf("candidate[%d] = %s\n", i, candidate[i]);
#endif
  }

  error_code = eb_search_multi(book, multi_id,
    (const char * const *)candidate);

#ifdef EBLOOK_SEARCH_MULTI_DEBUG
  xprintf("return = %d(%s)\n", error_code, eb_error_message(error_code));
#endif

  free(p);
  return error_code;
}


StringAlist *
salist_set (alist, key, value)
     StringAlist *alist;
     const char *key;
     const char *value;
{
  StringAlist *var;

  if (value) {
    /*
     * Set KEY to VALUE
     */
    StringAlist *prev = NULL;
    for (var = alist; var != NULL; var = var->next) {
      if (strcmp (key, var->key) == 0) {
	/* update original value */
	char *p = strdup (value);
	if (p != NULL) {
	  free (var->value);
	  var->value = p;
	} else {
	  xputs ("memory full");
	  if (prev)
	    prev->next = var->next;
	  else
	    alist = var->next;

	  free (var->key);
	  free (var->value);
	  free (var);
	}
	break;
      }
      prev = var;
    }
    if (var == NULL) {
      /* add new element */
      if ((var = malloc (sizeof (StringAlist))) == NULL) {
	xputs ("memory full");
      } else if ((var->key = strdup (key)) == NULL ||
	       (var->value = strdup (value)) == NULL) {
	xputs ("memory full");
	free (var->key);
	free (var);
      } else {
	var->next = alist;
	alist = var;
      }
    }
  } else {
    /*
     * Delete element
     */
    StringAlist *prev = NULL;
    for (var = alist; var != NULL; var = var->next) {
      if (strcmp (key, var->key) == 0) {
	/* delete from alist */
	if (prev)
	  prev->next = var->next;
	else
	  alist = var->next;

	/* free */
	free (var->key);
	free (var->value);
	free (var);
	break;
      }
      prev = var;
    }
  }

  return alist;
}

char *
salist_ref (alist, key)
     StringAlist *alist;
     const char *key;
{
  StringAlist *var;
  for (var = alist; var != NULL; var = var->next)
    if (strcmp (key, var->key) == 0)
      return var->value;

  return NULL;
}

#include "eb/text.h"

const EB_Position zero_pos = { 0, 0 };

struct multi_can {
    char text[256];
    struct multi_can *child;
    EB_Position child_pos;
    struct multi_can *next;
    int terminated;
};

struct multi_can *head = 0, *tail = 0;

char can_word[256];

EB_Error_Code
can_menu_begin(book, appendix, workbuf, hook_code, argc, argv)
    EB_Book *book;
    EB_Appendix *appendix;
    char *workbuf;
    EB_Hook_Code hook_code;
    int argc;
    const int *argv;
{
    memset(can_word, 0, sizeof(can_word));
    return EB_SUCCESS;
}

EB_Error_Code
can_menu_end(book, appendix, workbuf, hook_code, argc, argv)
    EB_Book *book;
    EB_Appendix *appendix;
    char *workbuf;
    EB_Hook_Code hook_code;
    int argc;
    const int *argv;
{
#if MULTI_DEBUG
    xprintf(">> can_word = %s\n", can_word);
#endif

    if (head == 0) {
	head = malloc(sizeof(struct multi_can));
	tail = head;
    } else {
#if MULTI_DEBUG
	xprintf(">> current tail %s:%x->next %s:%x\n",
		tail->text, tail,
		tail->next->text, tail->next);
#endif
	tail->next = malloc(sizeof(struct multi_can));
#if MULTI_DEBUG
	xprintf(">> %s:%x->next = %s:%x\n", tail->text, tail, can_word, tail->next);
#endif
	tail = tail->next;
    }

    memset(tail, 0, sizeof(struct multi_can));
    strcpy(tail->text, can_word);
    memset(can_word, 0, sizeof(can_word));

#if MULTI_DEBUG
    xprintf(">> %s:%x\n", tail->text, tail);
#endif

    if (argv[1] || argv[2]) {
	tail->child_pos.page = argv[1];
	tail->child_pos.offset = argv[2];
    } else {
#if MULTI_DEBUG
	xprintf(">> %s\n", tail->text);
#endif
    }
    return EB_SUCCESS;
}

/*
 * EUC JP to ASCII conversion table.
 */
static const unsigned char euc_a1_to_ascii_table[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x00 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x08 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x10 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x18 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x20 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x28 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x30 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x38 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x40 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x48 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x50 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x58 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x60 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x68 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x70 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x78 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x80 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x88 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x90 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x98 */
    0x00, 0x20, 0x00, 0x00, 0x2c, 0x2e, 0x00, 0x3a,     /* 0xa0 */
    0x3b, 0x3f, 0x21, 0x00, 0x00, 0x00, 0x60, 0x00,     /* 0xa8 */
    0x5e, 0x7e, 0x5f, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0xb0 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x2f,     /* 0xb8 */
    0x5c, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x27,     /* 0xc0 */
    0x00, 0x22, 0x28, 0x29, 0x00, 0x00, 0x5b, 0x5d,     /* 0xc8 */
    0x7b, 0x7d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0xd0 */
    0x00, 0x00, 0x00, 0x00, 0x2b, 0x2d, 0x00, 0x00,     /* 0xd8 */
    0x00, 0x3d, 0x00, 0x3c, 0x3e, 0x00, 0x00, 0x00,     /* 0xe0 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5c,     /* 0xe8 */
    0x24, 0x00, 0x00, 0x25, 0x23, 0x26, 0x2a, 0x40,     /* 0xf0 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0xf8 */
};

static const unsigned char euc_a3_to_ascii_table[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x00 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x08 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x10 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x18 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x20 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x28 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x30 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x38 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x40 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x48 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x50 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x58 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x60 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x68 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x70 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x78 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x80 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x88 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x90 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0x98 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0xa0 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0xa8 */
    0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,     /* 0xb0 */
    0x38, 0x39, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0xb8 */
    0x00, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,     /* 0xc0 */
    0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,     /* 0xc8 */
    0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,     /* 0xd0 */
    0x58, 0x59, 0x5a, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0xd8 */
    0x00, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,     /* 0xe0 */
    0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,     /* 0xe8 */
    0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,     /* 0xf0 */
    0x78, 0x79, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x00,     /* 0xf8 */
};

/*
 * Convert the `workbuf' string from EUC to ASCII.
 */
EB_Error_Code
can_menu_narrow_char(book, appendix, container, hook_code, argc, argv)
    EB_Book *book;
    EB_Appendix *appendix;
    void *container;
    EB_Hook_Code hook_code;
    int argc;
    const unsigned int *argv;
{
    int in_code1, in_code2;
    int out_code = 0;
    unsigned char outchars[3];

    in_code1 = argv[0] >> 8;
    in_code2 = argv[0] & 0xFF;

    if (in_code1 == 0xa1)
	out_code = euc_a1_to_ascii_table[in_code2];
    else if (in_code1 == 0xa3)
	out_code = euc_a3_to_ascii_table[in_code2];

    if (out_code == 0) {
	outchars[0] = in_code1;
	outchars[1] = in_code2;
	outchars[2] = 0;
    } else {
	outchars[0] = out_code;
	outchars[1] = 0;
    }

    strcat(can_word, outchars);

    return EB_SUCCESS;
}

EB_Error_Code
can_menu_wide_char(book, appendix, container, hook_code, argc, argv)
    EB_Book *book;
    EB_Appendix *appendix;
    void *container;
    EB_Hook_Code hook_code;
    int argc;
    const unsigned int *argv;
{
    unsigned char outchars[3];

    outchars[0] = argv[0] >> 8;
    outchars[1] = argv[0] & 0xFF;
    outchars[2] = 0;

    strcat(can_word, outchars);

    return EB_SUCCESS;
}

EB_Error_Code
can_menu_gaiji(book, appendix, container, hook_code, argc, argv)
    EB_Book *book;
    EB_Appendix *appendix;
    void *container;
    EB_Hook_Code hook_code;
    int argc;
    const unsigned int *argv;
{
    char workbuf[EB_MAX_ALTERNATION_TEXT_LENGTH + 1];
    char c;

    switch (hook_code) {
    case EB_HOOK_NARROW_FONT:
	c = 'h';
	break;
    case EB_HOOK_WIDE_FONT:
	c = 'z';
	break;
    default:
	c = '?';
    }
    sprintf(workbuf, "<gaiji=%c%04x>", c, argv[0]);
    strcat(can_word, workbuf);
    return EB_SUCCESS;
}

EB_Hookset multi_candidate_hookset;
EB_Hook
multi_candidate_hooks [] = {
  {EB_HOOK_NARROW_FONT,	    can_menu_gaiji},
  {EB_HOOK_WIDE_FONT,	    can_menu_gaiji},
  {EB_HOOK_NARROW_JISX0208, can_menu_narrow_char},
  {EB_HOOK_WIDE_JISX0208,   can_menu_wide_char},
  {EB_HOOK_BEGIN_CANDIDATE,      can_menu_begin},
  {EB_HOOK_END_CANDIDATE_LEAF,   can_menu_end},
  {EB_HOOK_END_CANDIDATE_GROUP,  can_menu_end},
  {EB_HOOK_NULL,            NULL},
};

struct multi_can *
find_child(can)
    struct multi_can * can;
{
    while (can) {
	if (can->child_pos.page == 0 && can->child_pos.offset == 0) {
#if MULTI_DEBUG
	    xprintf(">> %s noref skip\n", can->text);
#endif
	    can = can->next;
	} else if (can->child != 0) {
#if MULTI_DEBUG
	    xprintf(">> %s has processed child skip\n", can->text);
#endif
	    can = can->next;
	} else {
#if MULTI_DEBUG
	    xprintf(">> unprocessed child found %x\n", can);
	    xprintf(">> %s[\n", can->text);
#endif
	    return can;
	}
    }
    return 0;
}

int
process_child(book, can)
    EB_Book *book;
    struct multi_can * can;
{
    EB_Error_Code error_code;
    char buf[2048];
    int buflen;

#if MULTI_DEBUG
    xprintf(">> seeking %x:%x\n", can->child_pos.page, can->child_pos.offset);
#endif
    error_code = eb_seek_text(book, &can->child_pos);
#if MULTI_DEBUG
    xprintf(">> eb_seek_text %s\n", eb_error_message(error_code));
#endif
    error_code = eb_read_text(book, 0, &multi_candidate_hookset, NULL, 2047, buf, &buflen);
#if MULTI_DEBUG
    xprintf(">> eb_read_text %s\n", eb_error_message(error_code));
    xprintf(">> buflen = %d\n", buflen);
    xprintf(">> ]\n");
#endif

    tail->terminated = 1;
    return(0);
}

show_candidates_level(can, level)
    struct multi_can * can;
    int level;
{
    char * indent;

    indent = malloc(level+1);
    if (!indent) {
      xputs("memory full");
      return;
    }
    memset(indent, '\t', level);
    *(indent + level) = '\0';

    while (1) {
	xprintf("%s%c%s\n", indent, can->child ? ' ': '*', can->text);
	if (can->child) {
	    show_candidates_level(can->child, level+1);
	}
	if (can->terminated)
	    break;
	can = can->next;
    }

    free(indent);
}

void
free_candidates_tree(can)
    struct multi_can * can;
{
    struct multi_can * next;

    do {
	next = can->next;
#if MULTI_DEBUG
	xfprintf(stderr, ">> freeing %s:%x\n", can->text, can);
	xfprintf(stderr, ">> next is %s:%x\n", next->text, next);
#endif
	free(can);
	can = next;
    } while (next);
}

int
show_candidate(book, pos0)
    EB_Book *book;
    EB_Position pos0;
{
    char buf[2048];
    int buflen;
    struct multi_can *child, *ptail;
    EB_Error_Code error_code = EB_SUCCESS;

    eb_initialize_hookset (&multi_candidate_hookset);
    eb_set_hooks (&multi_candidate_hookset, multi_candidate_hooks);

    error_code = eb_seek_text(book, &pos0);
    if (error_code != EB_SUCCESS) {
        xprintf("An error occured in seek_position: %s\n",
	       eb_error_message(error_code));
	set_error_message (error_code);
	return 0;
    }
    error_code = eb_read_text(book, 0, &multi_candidate_hookset, NULL,
			      2047, buf, &buflen);
    if (error_code != EB_SUCCESS) {
        xprintf("An error occured in read_text: %s\n",
	       eb_error_message(error_code));
	set_error_message (error_code);
	return 0;
    }
#if MULTI_DEBUG
    xprintf(">> buflen = %d\n", buflen);
#endif

    tail->terminated = 1;

    while ((child = find_child(head)) != 0) {
	ptail = tail;
#if MULTI_DEBUG
	xprintf(">> current tail %s:%x->next %s:%x\n",
		tail->text, tail,
		tail->next->text, tail->next);
#endif
	process_child(book, child);
	child->child = ptail->next;
    }

    show_candidates_level(head, 0);

    free_candidates_tree(head);
    head = 0;
    tail = 0;

    return 1;
}


/*
 * Output information about multi searches.
 */
static void
output_multi_information(book)
    EB_Book *book;
{
    EB_Error_Code error_code;
    EB_Multi_Search_Code multi_list[EB_MAX_MULTI_SEARCHES];
    EB_Multi_Entry_Code entry_list[EB_MAX_MULTI_ENTRIES];
    int multi_count;
    int entry_count;
    char entry_label[EB_MAX_MULTI_LABEL_LENGTH + 1];
    int i, j;

    error_code = eb_multi_search_list(book, multi_list, &multi_count);
    if (error_code != EB_SUCCESS) {
	xprintf("eb_multi_search_list %s\n", eb_error_message(error_code));
	return;
    }
    for (i = 0; i < multi_count; i++) {
	xprintf("  multi search %d:\n", i + 1);
	error_code = eb_multi_entry_list(book, multi_list[i], entry_list,
	    &entry_count);
	if (error_code != EB_SUCCESS) {
	    xprintf("eb_multi_entry_list %s\n", eb_error_message(error_code));
	    continue;
	}
	for (j = 0; j < entry_count; j++) {
	    error_code = eb_multi_entry_label(book, multi_list[i],
		entry_list[j], entry_label);
	    if (error_code != EB_SUCCESS) {
		xprintf("eb_multi_entry_label %s\n",
		eb_error_message(error_code));
		continue;
	    }

	    xprintf("    label %d: %s\n", j + 1, entry_label);
	    xfputs("      candidates: ", stdout);
	    if (eb_multi_entry_have_candidates(book, multi_list[i],
		entry_list[j])) {
		    EB_Position pos;

		    xfputs("exist\n", stdout);

		    eb_multi_entry_candidates(book,
				multi_list[i], entry_list[j], &pos);
#if MULTI_DEBUG
		    xprintf(">> candidate = %x:%x\n", pos.page, pos.offset);
#endif

		    show_candidate(book, pos);

		}
	    else
		xfputs("not-exist\n", stdout);
	}
    }
    fflush(stdout);
}



void
show_entry_candidate(book, search_id, entry_id)
    EB_Book *book;
    int search_id;
    int entry_id;
{
    EB_Error_Code error_code;
    EB_Multi_Search_Code multi_list[EB_MAX_MULTI_SEARCHES];
    EB_Multi_Entry_Code entry_list[EB_MAX_MULTI_ENTRIES];
    EB_Position candidate_pos;
    char entry_label[EB_MAX_MULTI_LABEL_LENGTH + 1];
    int multi_count;
    int entry_count;
    int i, j;

    if (!eb_have_multi_search(book))
	return;

    error_code = eb_multi_search_list(book, multi_list, &multi_count);
    if (error_code != EB_SUCCESS) {
	xprintf("eb_multi_search_list: %s\n", eb_error_message(error_code));
	return;
    }

    if (search_id >= multi_count || search_id < 0)
	return;
    error_code = eb_multi_entry_list(book, multi_list[search_id],
		entry_list, &entry_count);
    if (error_code != EB_SUCCESS) {
	xprintf("eb_multi_entry_list %s\n", eb_error_message(error_code));
	return;
    }

    if (entry_id >= entry_count || entry_count < 0)
	return;
    if (!eb_multi_entry_have_candidates(book, multi_list[search_id], entry_list[entry_id])) {
	xprintf(" no-candidate\n");
	return;
    }


    if (error_code = eb_multi_entry_candidates(book, multi_list[search_id],
		entry_list[entry_id], &candidate_pos)) {
	xprintf("eb_multi_entry_candidates %s\n", eb_error_message(error_code));
	return;
    }

    show_candidate(book, candidate_pos);
}

void
show_label(book, id)
    EB_Book *book;
    int id;
{
    EB_Error_Code error_code;
    EB_Multi_Search_Code multi_list[EB_MAX_MULTI_SEARCHES];
    EB_Multi_Entry_Code entry_list[EB_MAX_MULTI_ENTRIES];
    char entry_label[EB_MAX_MULTI_LABEL_LENGTH + 1];
    int multi_count;
    int entry_count;
    int i, j;


    if (!eb_have_multi_search(book))
	return;

    error_code = eb_multi_search_list(book, multi_list, &multi_count);
    if (error_code != EB_SUCCESS) {
	xprintf("eb_multi_search_list: %s\n", eb_error_message(error_code));
	return;
    }

    for (i=0; i<multi_count; i++) {
	if (id != -1 && id != i)
	    continue;

	xprintf("%2d. ", i+1);
	error_code = eb_multi_entry_list(book, multi_list[i], entry_list,
	    &entry_count);
	if (error_code != EB_SUCCESS) {
	    xprintf("eb_multi_entry_list %s\n", eb_error_message(error_code));
	    continue;
	}

	for (j=0; j<entry_count; j++) {

	    error_code = eb_multi_entry_label(book, multi_list[i],
		entry_list[j], entry_label);
	    if (error_code != EB_SUCCESS) {
		xprintf("eb_multi_entry_label %s\n", eb_error_message(error_code));
		continue;
	    }

	    xprintf("%s:", entry_label);
	}
	xprintf("\n");
    }

}
