/* et_core.h - 2001/10/21 */
/*
 *  EasyTAG - Tag editor for MP3 and OGG files
 *  Copyright (C) 2000  Jerome Couderc <j.couderc@ifrance.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifndef __ET_CORE_H__
#define __ET_CORE_H__


#include <glib.h>


/***************
 * Declaration *
 ***************/

#ifndef MAX
#    define MAX(a,b) ((a) > (b) ? (a) : (b))
#endif

#ifndef MIN
#    define MIN(a,b) ((a) < (b) ? (a) : (b))
#endif


/*
 * The main list of files (contains ET_File data)
 */
GList *ETFileList;               /* List of files to tag */
guint  ETFileList_Index;         /* Contains the position of the current item in the whole list */
guint  ETFileList_Length;        /* Contains the length of the list */
gfloat ETFileList_TotalSize;     /* Total of the size of all files in list */
gulong ETFileList_TotalDuration; /* Total of duration of all files in list */
GList *ETHistoryFileList;        /* History list of files changes for undo/redo actions */


/*
 * Types of sorting.
 */
typedef enum
{
    SORTING_BY_ASCENDING_FILENAME = 0,
    SORTING_BY_DESCENDING_FILENAME,
    SORTING_BY_ASCENDING_TRACK_NUMBER,
    SORTING_BY_DESCENDING_TRACK_NUMBER,
    SORTING_BY_ASCENDING_CREATION_DATE,
    SORTING_BY_DESCENDING_CREATION_DATE,
    SORTING_UNKNOWN
} ET_Sorting_Type;


/*
 * Types of files
 */
typedef enum
{
    MP2_FILE = 0,
    MP3_FILE,
    OGG_FILE,
    FLAC_FILE,
    UNKNOWN_FILE
} ET_File_Type;


/*
 * Types of tags
 */
typedef enum
{
    ID3_TAG = 0,
    OGG_TAG,
    UNKNOWN_TAG
} ET_Tag_Type;


/*
 * Description of each item of the FileNameList list
 */
typedef struct _File_Name File_Name;
struct _File_Name
{
    gulong key;
    gboolean saved;        /* Set to TRUE if this filename had been saved */
    gchar *value;          /* The filename containing the full path and the extension of the file */
};


/*
 * Description of each item of the TagList list
 */
typedef struct _File_Tag File_Tag;
struct _File_Tag
{
    gulong key;            /* Incremented value */
    gboolean saved;        /* Set to TRUE if this tag had been saved */
    gchar *title;          /* Title of track */
    gchar *artist;         /* Artist name */
    gchar *album;          /* Album name */
    gchar *year;           /* Year of track */
    gchar *track;          /* Position of track in the album */
    gchar *track_total;    /* The number of tracks for the album (ex: 12/20) */
    gchar *genre;          /* Genre of song */
    gchar *comment;        /* Comment */
};


/*
 * Structure containing informations of the header of file
 * Nota: This struct was copied from an "MP3 structure", and will change later.
 */
typedef struct _ET_File_Info ET_File_Info;
struct _ET_File_Info
{
    gint version;     /* Version of bitstream (mpeg version for mp3, encoder version for ogg) */
    gint mpeg25;      /* Version is MPEG 2.5? */
    gint layer;       /* "MP3 data" */
    gint bitrate;     /* Bitrate (kb/s) */
    gint variable_bitrate; /* Is a VBR file? */
    gint samplerate;  /* Samplerate (Hz) */
    gint mode;        /* Stereo, ... or channels for ogg */
    gint size;        /* The size of file (in bytes) */
    gint duration;    /* The duration of file (in seconds) */
};


/*
 * Structure for descripting supported files
 */
typedef struct _ET_File_Description ET_File_Description;
struct _ET_File_Description
{
    ET_File_Type FileType;    /* Type of file (ex: MP3) */
    gchar       *Extension;   /* Extension (ex: ".mp3") */
    ET_Tag_Type  TagType;     /* Type of tag (ex: ID3) */
};

/*
 * Description of supported files
 */
static ET_File_Description ETFileDescription[] = 
{
    {MP3_FILE,     ".mp3",  ID3_TAG},
#ifdef ENABLE_OGG
    {OGG_FILE,     ".ogg",  OGG_TAG},
#endif
    {MP2_FILE,     ".mp2",  ID3_TAG},
    {FLAC_FILE,    ".flac", ID3_TAG},
    {UNKNOWN_FILE, "",      UNKNOWN_TAG}    /* This item must be placed to the end! */
};
/* The last index of the previous tab */
#define ET_FILE_DESCRIPTION_SIZE ( sizeof(ETFileDescription)/sizeof(ETFileDescription[0]) - 1 )


/*
 * Description of each item of the ETFileList list
 */
typedef struct _ET_File ET_File;
struct _ET_File
{
    gulong ETFileKey;         /* Primary key to identify each item of the list */

    ET_File_Description *ETFileDescription;
    ET_File_Info        *ETFileInfo; /* Header infos: bitrate, duration, ... */

    GList *FileNameCur;       /* Points to item of FileNameList that represents the current value of filename state (i.e. file on hard disk) */
    GList *FileNameNew;       /* Points to item of FileNameList that represents the new value of filename state */
    GList *FileNameList;      /* Contains the history of changes about the file name */
    
    GList *FileTag;           /* Points to the current item of FileTagList */
    GList *FileTagList;       /* Contains the history of changes about file tag data */
    
};


/*
 * Description of each item of the ETHistoryFileList list
 */
typedef struct _ET_History_File ET_History_File;
struct _ET_History_File
{
    GList   *FileNameLast; /* Pointer to item of FileNameList before changes */
    GList   *FileNameNew;  /* Pointer to item of FileNameList after changes */
    GList   *FileTagLast;  /* Pointer to item of FileTagList before changes */
    GList   *FileTagNew;   /* Pointer to item of FileTagList after changes */
    ET_File *ETFile;       /* Pointer to item of ETFileList changed */
};



/**************
 * Prototypes *
 **************/

gboolean ET_File_Is_Supported            (gchar *filename);
gboolean ET_Check_If_File_Is_Saved       (ET_File *ETFile);
gboolean ET_Check_If_All_Files_Are_Saved (void);
GList   *ET_Add_File_To_File_List        (gchar *filename);
gboolean ET_Free_File_List               (void);
gboolean ET_Free_History_File_List       (void);

GList   *ET_File_List_First      (void);
GList   *ET_File_List_Previous   (void);
GList   *ET_File_List_Next       (void);
GList   *ET_File_List_Last       (void);
guint    ET_File_List_Get_Length (void);
GList   *ET_File_List_Nth_By_Key (gulong key);

void     ET_Display_File_Data_To_UI (ET_File *ETFile);
void     ET_Save_File_Data_From_UI  (ET_File *ETFile);
gboolean ET_Save_File_Tag_To_HD     (ET_File *ETFile);

gboolean ET_Undo_File_Data          (ET_File *ETFile);
gboolean ET_Redo_File_Data          (ET_File *ETFile);
gboolean ET_File_Data_Has_Undo_Data (ET_File *ETFile);
gboolean ET_File_Data_Has_Redo_Data (ET_File *ETFile);

gboolean ET_Undo_History_File_Data          (void);
gboolean ET_Redo_History_File_Data          (void);
gboolean ET_History_File_List_Has_Undo_Data (void);
gboolean ET_History_File_List_Has_Redo_Data (void);

void     ET_Mark_File_Name_As_Saved (ET_File *ETFile);
void     ET_Update_Directory_Name_Into_File_List(gchar* last_path, gchar *new_path);
gboolean ET_Delete_File (ET_File *ETFile);

gint ET_Comp_Func_Sort_File_By_Ascending_Filename       (ET_File *ETFile1, ET_File *ETFile2);
gint ET_Comp_Func_Sort_File_By_Descending_Filename      (ET_File *ETFile1, ET_File *ETFile2);
gint ET_Comp_Func_Sort_File_By_Ascending_Creation_Date  (ET_File *ETFile1, ET_File *ETFile2);
gint ET_Comp_Func_Sort_File_By_Descending_Creation_Date (ET_File *ETFile1, ET_File *ETFile2);
gint ET_Comp_Func_Sort_File_By_Ascending_Track_Number   (ET_File *ETFile1, ET_File *ETFile2);
gint ET_Comp_Func_Sort_File_By_Descending_Track_Number  (ET_File *ETFile1, ET_File *ETFile2);


/* Functions for debugging */
void ET_Debug_Print_File_List (gchar *file ,gint line);


#endif /* __ET_CORE_H__ */
