#include "DVB.hh"

static void dvb2txt(char *out, char *in, int len)
{
        //char *chartable=0;
        unsigned char *txt=(unsigned char *) in;
        if (!len)
	        return;
	switch (*txt) {
	case 0x01 ... 0x0f:
	        txt++;
	        len--;
	        break;
	case 0x10:
	        txt+=3;
                len-=3;
	        break;
	}
	while (len) {
	        switch(*txt) {
		case 0x01 ... 0x1f:
		case 0x7f ... 0xa0:
		        len--;
			txt++;
			break;
		case 0x00:
		        len=1;
		default:
		        *(out++)=(char) *(txt++);
			len--;
			break;
		}
	}
}

DVB::~DVB() {
        delete [] lnbs;
	delete [] diseqcs;
	delete [] rotors;
	delete [] tps;
	delete [] chans;
	delete [] bouqs;
	delete [] sats;
	delete [] swis;
	delete [] ntws;
	//close(fd_video);
	//close(fd_audio);
	close(fd_frontend);
	close(fd_sec);
	close(fd_demuxa);
	close(fd_demuxv);
	close(fd_demuxtt);
	close(fdvb);
}


void DVB::init(char *dvbname="/dev/video0", char *siname="/dev/vbi0",int minor = 0) {
	FrontendInfo feinfo;
	int failed =0;

	for (int i=0; i<NK; i++)
	        num[i]=0;

	lnbs   =new Lnb[maxs[LNB]];
	diseqcs=new DiSEqC[maxs[DIS]];
	rotors =new Rotor[maxs[ROTOR]];
	tps    =new Transponder[maxs[TRANS]];
	chans  =new Channel[maxs[CHAN]];
	bouqs  =new Bouquet[maxs[BOU]];
	sats   =new Sat[maxs[SAT]];
	swis   =new Switch[maxs[SWI]];
	ntws   =new Network[maxs[NTW]];
	if (minor < 0) return;

        if (fd_sec > 0) close(fd_sec);
        if (fd_frontend > 0) close(fd_frontend);
        if (fd_demuxa > 0) close(fd_demuxa);
        if (fd_demuxv > 0) close(fd_demuxv);
        if (fd_demuxtt > 0) close(fd_demuxtt);
	char devname[80];


	dvr_enabled = 0;

	sprintf(devname,"%s%d",FRONT_DEV,minor);
	fd_frontend=open(devname, O_RDWR);
	if (fd_frontend < 0) {
		cerr << "Could not open " << devname << endl;
		front_type=-1;
		perror(devname);
		fd_frontend = -1;
		failed = 1;
	}
	ioctl(fd_frontend, FE_GET_INFO, &feinfo);
	front_type=feinfo.type;

	if (front_type==FE_QPSK) {
	        sprintf(devname,"%s%d",SEC_DEV,minor);
		fd_sec=open(devname, O_RDWR);
		if (fd_sec < 0) {
		        cerr << "Could not open " << devname << endl;
			perror(devname);
			fd_sec = -1;
			failed = 1;
		}
	}

	sprintf(devname,"%s%d",DEMUX_DEV,minor);
	fd_demuxtt=open(devname, O_RDWR);
	if (fd_demuxtt < 0) {
		cerr << "Could not open " << devname << endl;
		perror(devname);
		fd_demuxtt = -1;
		failed = 1;
	}

	fd_demuxa=open(devname, O_RDWR);
	if (fd_demuxa < 0) {
		cerr << "Could not open " << devname << endl;
		perror(devname);
		fd_demuxa = -1;
		failed = 1;
	}

	fd_demuxv=open(devname, O_RDWR);
	if (fd_demuxv < 0) {
		cerr << "Could not open " << devname << endl;
		perror(devname);
		fd_demuxv = -1;
		failed = 1;
	}
}

int DVB::SetTP(unsigned int tpid, unsigned int satid)
{
        int i;
	Transponder *tp=0;
	Sat *sat=0;
	Lnb *lnb=0;

	for (i=0; i<num[TRANS]; i++) {
	        if (tps[i].id==tpid &&
		    // accept NOID for backwards compatibility 
		    (tps[i].satid==satid || satid==NOID )) {
		        tp=&tps[i];
			break;
		}
	}
	if (!tp)
	        return -1;

	for (i=0; i<num[SAT]; i++) {
		if (sats[i].id==tp->satid) {
		        sat=&sats[i];
			break;
		}
	}

	if (!sat)
	        return -1;

	for (i=0; i<num[LNB]; i++)  
               if (lnbs[i].id==sat->lnbid) {
		        lnb=&lnbs[i];
			break;
	       }
	if (!lnb)
	        return -1;

	static const uint8_t rfectab[9] = {1,2,3,0,4,0,5,0,0};

	switch (front_type) {
	case FE_QPSK:
	        if (tp->freq < lnb->slof) {
		        front_param.Frequency = (tp->freq - lnb->lof1);
			scmds.continuousTone = SEC_TONE_OFF;
		} else {
		        front_param.Frequency = (tp->freq - lnb->lof2);
			scmds.continuousTone = SEC_TONE_ON;
		}
		if (tp->pol) scmds.voltage = SEC_VOLTAGE_18;
		else scmds.voltage = SEC_VOLTAGE_13;
		set_diseqc_nb(lnb->diseqcnr);
		front_param.u.qpsk.SymbolRate = tp->srate;
		front_param.u.qpsk.FEC_inner = (CodeRate)rfectab[tp->fec];
		front_param.Inversion = INVERSION_AUTO;
		break;
	case FE_QAM:
//		printf("%d \n", tp->freq);
	        front_param.Frequency = tp->freq;
		front_param.Inversion = INVERSION_AUTO;
		front_param.u.qam.SymbolRate = tp->srate;
		front_param.u.qam.FEC_inner = (CodeRate)rfectab[tp->fec];
		front_param.u.qam.QAM=(Modulation) (tp->qam+1);
	        break;

	case FE_OFDM:
	        front_param.Frequency = tp->freq;
		front_param.Inversion = INVERSION_AUTO;
		front_param.u.ofdm.bandWidth = (BandWidth)tp->band;
		front_param.u.ofdm.HP_CodeRate = (CodeRate)tp->hp_rate;
		front_param.u.ofdm.LP_CodeRate = (CodeRate)tp->lp_rate;
		front_param.u.ofdm.Constellation = (Modulation)tp->mod;
		front_param.u.ofdm.TransmissionMode = 
			(TransmitMode)tp->transmode;
		front_param.u.ofdm.guardInterval = (GuardInterval)tp->guard;
		front_param.u.ofdm.HierarchyInformation = 
			(Hierarchy)tp->hierarchy;
	}
        return 0;
}

static uint16_t get_pid(uint8_t *pid)
{
        uint16_t pp;

        pp = (pid[0] & 0x1f) << 8;
        pp |= pid[1] &0xff;
        return pp;
}



void DVB::check_all_pids()
{
	for (int i = 0; i < num[CHAN]; i++){
		cerr << "checking " << chans[i].name << endl;
		SetChannel(i); 
	}
}

int DVB::check_pids(Channel *chan)
{
	uint8_t *prog;
	int slen, i, nprog, ilen, c;
	uint16_t prog_pid = 0, pnr = 0;
	int found = 0;
	int es_length;
	int e;
	int oldnum;

	unsigned char buf[MAXSECSIZE], msec=0, sec=0;
	
	oldnum=chan->apidnum;
	time_t count = time(0)+2;
	while (sec<=msec && !found && count > time(0)) {
		if (GetSection(buf, 0, 0, sec, msec)>0 && !buf[0]){
			sec++;
			slen = (((buf[1]&0x0f) << 8) | buf[2]) +3;
			nprog = (slen - 12) / 4;
			prog = buf+ 8;
			for (i = 0; i < nprog; i++){
				pnr = (prog[0] << 8)|prog[1];
			  if ( chan->pnr == pnr ){
					prog_pid = get_pid(prog+2);
					found = 1;
					break;
				} 
				prog += 4;
			} 
		}
	}
	if (!found) return -1;
	chan->apidnum = 0;
	sec = 0;
	msec = 0;
	
	while (sec<=msec && count > time(0)) {
		if (GetSection(buf,prog_pid, 2, sec, msec)>0){
			sec++;
			slen = (((buf[1]&0x0f) << 8) | buf[2]) +3;
			ilen = ((buf[10]&0x03)  <<8) | buf[11];
			chan->pcrpid =  get_pid(buf+8);
			c = 12+ilen;
			while (c < slen){
				if(count < time(0)) break;
				switch (buf[c]) {
				case 1: 
				case 2: 
					if (chan->vpid == NOPID)
						chan->vpid = get_pid(buf+c+1);
					break;
				case 3: 
				case 4: 
				{
					int afound = 0;
					uint16_t apid = get_pid(buf+c+1); 
					if (chan->apidnum>=MAXAPIDS) {
						cerr << "Need more apids\n";
						break;
					}
					
					for (int i = 0; 
					     i < chan->apidnum; i++){
						if ( apid ==
						     chan->apids[i]){
							afound = 1;
							break;
						}
					}
					if (! afound){
						chan->apids[chan->apidnum++]=
							apid;
					
						es_length = (buf[c+3]&0x0F) 
							<< 8;
						es_length |= buf[c+4];

						e = 0;
						while (buf[c+5+e] != 0x0a && 
							c+5+e < es_length) e++;
						if (buf[c+6+e] == 0x04){
							memcpy(chan->apids_name
							       +(chan->apidnum
								 -1)*4,
							       buf+c+7+e, 3);
							chan->apids_name[
								(chan->apidnum-1)*4+3] =
								'\0';
						}
					}
					break; 
				}
				case 6: 
					chan->ttpid = get_pid(buf+c+1);
					break;
				default:
					break;
				}
				uint16_t silen = (((buf[c+3] & 0x0f) << 8) | 
						  buf[c+4]);
				c += 5 + silen;
			}
		}
	}
	if (!chan->apidnum)
		chan->apidnum=oldnum;
	chan->checked = 1;
	return 0;
}

int DVB::get_all_progs(uint16_t *progbuf, uint16_t *pnrbuf, int length)
{
	int nfound = 0, oldone = 0, i, j;
	unsigned char buf[MAXSECSIZE], msec=0, sec=0;
	uint16_t prog_pid = 0, pnr = 0;
	int slen, nprog;
	uint8_t *prog;

	time_t count = time(0)+3;
	while (sec<=msec && count > time(0)) {
		if (GetSection(buf, 0, 0, sec, msec)>0 && !buf[0]){
			sec++;
			slen = (((buf[1]&0x0f) << 8) | buf[2]) +3;
			nprog = (slen - 12) / 4;
			prog = buf+ 8;
			for (i = 0; i < nprog; i++){
				pnr = (prog[0] << 8)|prog[1];
				prog_pid = get_pid(prog+2);
				
				oldone = 0;
				for(j=0; j<nfound; j++)
					if (pnr == pnrbuf[j]) oldone = 1;
					
				if ( !oldone && nfound < length ){
					pnrbuf[nfound]=pnr;
					progbuf[nfound]=prog_pid;
					nfound++;
				} else return -1;
				prog += 4;
			} 
		}
	}
	return nfound;
}

int DVB::get_pids(uint16_t prog_pid, uint16_t *vpid, uint16_t *apids)
{
	int slen, ilen, c;
	int found = 0;
	unsigned char buf[MAXSECSIZE], msec=0, sec=0;
	int apidnum = 0;
	
	time_t count = time(0)+2;

	sec = 0;
	msec = 0;
	*vpid = NOPID;
	while (sec<=msec && count > time(0)) {
		if (GetSection(buf,prog_pid, 2, sec, msec)>0){
			sec++;
			slen = (((buf[1]&0x0f) << 8) | buf[2]) +3;
			ilen = ((buf[10]&0x03)  <<8) | buf[11];
			c = 12+ilen;
			while (c < slen){
				if(count < time(0)) break;
				switch (buf[c]) {
				case 1: 
				case 2: 
					*vpid = get_pid(buf+c+1);
					found = 1;
					return found;
					break;
				case 3: 
				case 4: 
				{
					int afound = 0;
					uint16_t apid = get_pid(buf+c+1); 
					if (apidnum >= MAXAPIDS) {
						cerr << "Need more apids\n";
						break;
					}
					
					for (int i = 0; i < apidnum; i++){
						if ( apid == apids[i]){
							afound = 1;
							break;
						}
					}
					if (! afound){
						apids[apidnum++] = apid;
					}
					break; 
				}
				case 6: 
					break;
				default:
					break;
				}
				uint16_t silen = (((buf[c+3] & 0x0f) << 8) | 
						  buf[c+4]);
				c += 5 + silen;
			}
		}
	}

	return found+apidnum;
}

uint16_t DVB::find_pnr(uint16_t svpid, uint16_t sapid)
{
	int nfound = 0;
	int pfound = 0;
	uint16_t progs[100];
	uint16_t pnrs[100];
	uint16_t vpid;
	uint16_t apids[MAXAPIDS];

	nfound = get_all_progs(progs, pnrs, 100);
	for(int i=0; i < nfound; i++){
		if ( (pfound = get_pids(progs[i], &vpid, apids)) ){
			if(svpid != NOPID && vpid == svpid ) return pnrs[i];
			else if ( svpid == NOPID ){
				for (int j=0; j<pfound; j++){
					if (apids[j] == sapid) return pnrs[i];
				}
			}
		}
	}
	return 0;
}



void DVB::parse_sdt(uint8_t *buf, int count, Channel *chan)
{
	int c = 0;
	uint8_t dtag;
	uint8_t dlen;
	uint8_t slen=0;
	int i;
	char name[MAXNAM];

	while ( c < count ){
		dtag = buf[c++];
		dlen = buf[c++];
		while ( dlen ){
			switch ( dtag ){
			case 0x09: /* ca desc */
				chan->type = 1;
				dlen = 0;
				c+= dlen;
				break;

			case 0x48: /* service desc */
				i = c;
				i++; /* service type */
				i += buf[i]+1;  /* provider name */
				slen = buf[i++];
				memcpy(name,(char*)(&buf[i]), int(slen));
				name[slen]=0;
				memset(chan->name, 0, MAXNAM);
				dvb2txt(chan->name,name, int(slen));
				i+= slen;
				c+= dlen;
				dlen = 0;
				break;
			default:
				c+= dlen;
				dlen = 0;
			}
		}

	}

}

void DVB::scan_sdt(Channel *chan)
{
	int slen, ilen, c;
	unsigned char buf[MAXSECSIZE], msec=0, sec=0;
	uint16_t pnr;
	uint8_t ca;

	time_t count = time(0)+5;
	while (sec<=msec && count > time(0)) {
		if (GetSection(buf, 0x11, 0x42, sec, msec)>0){
			sec++;
			slen = (((buf[1]&0x0f) << 8) | buf[2]) +3;
			c = 11;
			while (c < slen-4){
				pnr = (buf[c]<<8)|buf[c+1];
				c+=3;
				ca=(buf[c]&0x10)>>4;
				ilen=((buf[c]&0x0f)<<8)|buf[c+1];
				c+=2;
				if ( chan->pnr == pnr ){
					parse_sdt(&buf[c], ilen, chan);
				}
				c+=ilen;
			}
		}
	}
}

int DVB::SetChannel(int chnr, char* apref=NULL, uint16_t *apidp=NULL, 
		    uint16_t *vpidp=NULL) 
{
	Channel *chan = 0;
	int i;
	int scan = 0;
	uint16_t apid=0, vpid=0;

	//cerr << "setchannel" << endl;
	if (chnr>=num[CHAN])
	        return -1;
	chan=&chans[chnr];

	get_front();
	SetTP(chan->tpid, chan->satid);
	set_front();
       
	if (chan->pnr == NOPID && (chan->vpid != NOPID || 
				   chan->apids[0] != NOPID))
		chan->pnr = find_pnr(chan->vpid, chan->apids[0]);
	if (chan->vpid == NOPID && chan->apidnum == 0 ) 
		if (!chan->checked){ 
			check_pids(chan);
			scan = 1;
		}
	
	if (chan->vpid != NOPID) set_vpid(chan->vpid);
	vpid = chan->vpid;
	if (chan->apidnum){
		if (apref){
			int found = 0;
			for (i=0; i < chan->apidnum; i++){
				if (!strncasecmp(chan->apids_name+4*i,
						 apref,3)){
					found=1;
					break;
				}
			}
			if (found) apid = chan->apids[i];
			else apid = chan->apids[0];
		} else apid = chan->apids[0];
	}
		
	set_apid(apid);

	if (chan->ttpid != NOPID) set_ttpid(chan->ttpid);
	
	if (scan)
		scan_sdt(chan);

	if (fdvb >= 0){
		struct tm *lt;
		time_t t;
		
		
		t = time(0);
		lt = localtime(&t);
		
		ostrstream fstr;
		osd.Clear();
		
		fstr << setw(2) << setfill(int('0')) << lt->tm_hour 
		     << ":" << setw(2) << setfill(int('0')) 
		     << lt->tm_min << "  "
		     << chan->name <<  ends;
		
		osd.Text(0, 0, 0, 1, fstr.str());
		osd.Show();
	}
	if (vpidp) *vpidp = vpid;
	if (apidp) *apidp = apid;
        return 0;

}

int DVB::SetChannel(uint16_t sid, uint16_t onid, 
		    uint16_t tpid, uint16_t satid)
{
        int chnr, i;
	Transponder *tp=0;
	Sat *sat=0;
	Lnb *lnb=0;
	Channel *chan=0;

	for (chnr=0; chnr<num[CHAN]; chnr++)  
	        if (chans[chnr].pnr==sid   && 
		    (onid==NOID  || chans[chnr].onid==onid) && 
		    (satid==NOID || chans[chnr].satid==satid)  && 
		    (tpid==NOPID  || chans[chnr].tpid==tpid)
		    ) {
		        chan=&chans[chnr];
			break;
		}
	if (!chan)
	        return -1;

	if (tpid==NOID)
 	        tpid=chan->tpid;
	if (onid==NOID)
 	        onid=chan->onid;
	if (satid==NOID)
 	        satid=chan->satid;

	for (i=0; i<num[TRANS]; i++)  
	        if (tps[i].id==tpid && tps[i].satid==satid) {
		        tp=&tps[i];
			break;
		}
	if (!tp)
	        return -1;

	for (i=0; i<num[SAT]; i++) {
		if (sats[i].id==tp->satid) {
		        sat=&sats[i];
			break;
		}
	}
	if (!sat)
	        return -1;

	for (i=0; i<num[LNB]; i++)  
               if (lnbs[i].id==sat->lnbid) {
		        lnb=&lnbs[i];
			break;
	       }
	if (!lnb)
	        return -1;

	get_front();

	static const uint8_t rfectab[9] = {1,2,3,0,4,0,5,0,0};
	                
	switch (front_type) {
	case FE_QPSK:
	        if (tp->freq < lnb->slof) {
		        front_param.Frequency = (tp->freq - lnb->lof1);
			scmds.continuousTone = SEC_TONE_OFF;
		} else {
		        front_param.Frequency = (tp->freq - lnb->lof2);
			scmds.continuousTone = SEC_TONE_ON;
		}
		if (tp->pol) scmds.voltage = SEC_VOLTAGE_18;
		else scmds.voltage = SEC_VOLTAGE_13;
		set_diseqc_nb(lnb->diseqcnr);
		front_param.u.qpsk.SymbolRate = tp->srate;
		front_param.u.qpsk.FEC_inner = (CodeRate)rfectab[tp->fec];
		front_param.Inversion = INVERSION_AUTO;
		break;

	case FE_QAM:
	        front_param.Frequency = tp->freq;
		front_param.u.qam.FEC_inner = (CodeRate)rfectab[tp->fec];
		front_param.u.qam.QAM = (Modulation)(tp->qam+1);
		front_param.Inversion = INVERSION_AUTO;
	        break;

	case FE_OFDM:
	        front_param.Frequency = tp->freq;
		front_param.Inversion = INVERSION_AUTO;
		front_param.u.ofdm.bandWidth = (BandWidth)tp->band;
		front_param.u.ofdm.HP_CodeRate = (CodeRate)tp->hp_rate;
		front_param.u.ofdm.LP_CodeRate = (CodeRate)tp->lp_rate;
		front_param.u.ofdm.Constellation = (Modulation)tp->mod;
		front_param.u.ofdm.TransmissionMode = 
			(TransmitMode)tp->transmode;
		front_param.u.ofdm.guardInterval = (GuardInterval)tp->guard;
		front_param.u.ofdm.HierarchyInformation = 
			(Hierarchy)tp->hierarchy;
		break;
	}

	set_front();
        set_vpid(chan->vpid);
        set_apid(chan->apids[0]);
	set_ttpid(chan->ttpid);

	return chnr;
}

bool DVB::Tune(int &newProgNr, int &oldProgNr, uint16_t ONID,
	       uint16_t TSID, int SID)
{
        if ((newProgNr=SetChannel(SID, TSID, 0xffff, ONID))<0)
	        return false;
	else
	        return true;
}



int DVB::GetChannel(int chnr, struct channel *) 
{
        int i;
	Channel *chan=0;
	Transponder *tp=0;
	Sat *sat=0;
	Lnb *lnb=0;

	if (chnr>=num[CHAN])
	        return -1;
	chan=&chans[chnr];
	
	for (i=0; i<num[TRANS]; i++)  
               if (tps[i].id==chan->tpid) {
		        tp=&tps[i];
			break;
	       }
	if (!tp)
	        return -1;

	for (i=0; i<num[SAT]; i++)  
               if (sats[i].id==tp->satid) {
		        sat=&sats[i];
			break;
	       }
	if (!sat)
	        return -1;

	for (i=0; i<num[LNB]; i++)  
               if (lnbs[i].id==sat->lnbid) {
		        lnb=&lnbs[i];
			break;
	       }
	if (!lnb)
	        return -1;
	//cerr << "ok\n";

	set_front();

	/*
	osd.Open(80, 500, 640, 540, 2, 0, 2);
	osd.SetColor(0, 0, 0, 0, 255);
	osd.SetColor(1, 240, 240, 240, 255);
	
	{
	  ostrstream fstr;
	  fstr << dec << chan->id << ": " << chan->name << ends;
	  osd.Text(0, 0, 3, 1, fstr.str());
	}
	*/
	/*
	VIDEOSTATE vstate;

	ioctl(fdvb, VIDIOCGVIDEOSTATE, &vstate);
	
	cout << dec << "HSize=" << vstate.HSize << endl;
	cout << dec << "VSize=" << vstate.VSize << endl;
	cout << dec << "BitRate=" << vstate.BitRate << endl;
	cout << dec << "BytesDecoded=" << vstate.BytesDecoded << endl;
	cout << dec << "InitDone=" << vstate.InitDone << endl;
	*/
        return 0;

}

/*
static dvb_sdt_service_t *find_service(dvb_pat_prog_t *prog, dvb_sdt_t *sdt) {
        dvb_sdt_service_t *service;
	  
	if (!sdt || !prog ) 
	        return 0;
	service = sdt->service;
	while (service) {
	        if (prog->prog_nr == service->service_id)
		        return service;
		service=service->next;
	}
	return 0;
}


int DVB::AddChannels(Transponder &tp, dvb_pat_t &pat, dvb_sdt_t &sdt)
{
        dvb_pmt_stream_t *stream;
        dvb_pat_prog_t *prog;
	dvb_sdt_service_t *service;
	Channel chan;

	prog=pat.prog;
	while (prog) {
		if (prog->prog_nr) {
		        chan.pnr=prog->prog_nr;
			cerr << "prog nr: " << hex << chan.pnr << endl;
		        if ((service=find_service(prog, &sdt))) {
			        chan.type|=service->free_ca_mode;
			        if (service->descr && service->descr->service
				    && service->descr->service->service_name) {
				        dvb2txt(chan.name,
						service->descr->service->service_name,
						maxname);
					cerr << "Found: " << chan.name << endl;
				}
			}
			
			if (prog->prog.pmt) {
			        chan.pcrpid=prog->prog.pmt->pcr_pid;
				cerr << "pcr pid: " << hex << chan.pcrpid 
				     << endl;
				stream = prog->prog.pmt->stream;
				while (stream) {
				        switch (stream->stream_type) {
					case 1: 
					case 2: 
					        chan.vpid=stream->pid;
						break;
					case 3: 
					case 4: 
					        if (chan.apidnum>=MAXAPIDS) {
						        cerr << "Need more apids\n";
							break;
						}
						chan.apids[chan.apidnum++]=stream->pid;
						break; 
					case 6: 
					        chan.ttpid=stream->pid;
						break;
					default:
					  //cerr << "Unknown: " <<  HEX(4) << (int) stream->stream_type << "  PID: " << HEX(2) << (int) stream->pid << endl;
					  break;
					}
					stream = stream->next;
				}
			}
			cerr << chan << endl;
			chan.tpid=tp.id;
			chan.satid = tp.satid;
			AddChannel(chan);
			chan.clearall();
		}
	        prog=prog->next;
	}  

        return 0;
}


int DVB::scan_tp(struct Transponder &tp) 
{
        dvb_sdt_t sdt;
        dvb_pat_t pat;
	dvb_pat_prog_t *prog;
	int pmtnum=0;

	memset(&pat, 0, sizeof(pat));

        if (GetPAT(&pat)<0)
	       return -1;

	//if(fdvb >=  0) osd.Text(0, 120, 2, 1, "PAT");
	cerr << "PAT" << endl;

	prog=pat.prog;
	while (prog) {
		if (!prog->prog_nr) {
		        dvb_nit_t *nit = (dvb_nit_t *)calloc (1, sizeof (dvb_nit_t));
			if (GetNIT(nit) >= 0) {
			        if (prog->prog.nit)
				        free(prog->prog.nit);
				prog->prog.nit = nit;
			} else 
			        free(nit);

			//if (fdvb >= 0) osd.Text(64, 120, 2, 1, "NIT");

			cerr << "NIT\n";

			// FIXME: fix NIT parsing 
			//if (front.type!=FE_QAM)
			        tp.freq=(uint32_t) (1000000*nit->freq+0.5);
		} else {
		  fprintf(stderr,"PMT: %04x:%04x\n", prog->prog_nr, prog->pid);
		        dvb_pmt_t *pmt = (dvb_pmt_t *)calloc (1, sizeof (dvb_pmt_t));
			if (GetPMT(pmt, prog->pid) >= 0) {
			        pmtnum++;
			        if (prog->prog.pmt)
				        free (prog->prog.pmt);
				prog->prog.pmt = pmt;
			} else 
			        free (pmt);
		}
	        prog=prog->next;
	}

	//if(fdvb >= 0) osd.Text(128, 120, 2, 1, "PMT");

	cerr << "freq:" << dec << tp.freq << endl;
	// Ignore TPs with no PMT or SDT 
	if (!pmtnum) {
	        free_pat(&pat);
	        return -1;
	}
	memset(&sdt, 0, sizeof(dvb_sdt_t));
	if (GetSDT(&sdt)<0) {
	        free_pat(&pat);
	        return -1;
	}
	//if(fdvb >= 0) osd.Text(196, 120, 2, 1, "SDT");

	tp.id=(tp.satid<<16)|pat.ts_id;
	int tpn = AddTP(tp);
	AddChannels(tps[tpn], pat, sdt);
	
	free_sdt(&sdt);
	free_pat(&pat);
        return 0;
}

int DVB::scan_sat(Sat &sat) 
{
	struct Transponder tp;
        uint srates[2]= { 27500000, 22000000 }; 
	uint freq, i;
	int pol, res, ttk;
	uint srate=0, ofreq;
	Lnb &lnb=*sat.lnb;
	

	set_lnb(lnb.diseqcnr);
	ostrstream satname;
			
	satname << "on " << sat.name << ends;
	//if(fdvb >= 0) osd.Text(100, 40, 2, 1, satname.str());

	for (pol=0; pol<2; pol++)
	for (freq=sat.fmin; freq<=sat.fmax; freq+=1000) {
	        if (freq<lnb.slof) {
		        ofreq=(freq-lnb.lof1)*1000;
			ttk=0;
		} else {
		        ofreq=(freq-lnb.lof2)*1000;
			ttk=1;
		}
		
		{
		        ostrstream fstr;
			fstr << dec << (freq/1000) << " MHz " << ends;
			cerr << fstr.str();
			//if(fdvb >= 0) osd.Text(400, 40, 2, 1, fstr.str());
		}
		for (i=0, res=-1; i<2 && res<0; i++) {
		        srate=srates[i];
			res=set_tp(&ofreq, ttk, pol, srate);

			if (!res) {
			        ostrstream fstr;
				fstr << "Signal lock at " << dec << (freq/1000)
				     << " MHz" << ends;
				cerr << fstr.str() << endl;
				if(fdvb >= 0){
				  //osd.Text(0, 80, 2, 1, fstr.str());
				  //osd.FillBlock(0, 120, 300, 159, 0);
				}
				tp.name[0]=0;
				tp.id=NOID;
				tp.type=1;
				tp.freq=freq;
				tp.pol=pol;
				tp.qam=-1;
				tp.srate=srate;
				tp.fec=8;
				tp.satid=sat.id;
				scan_tp(tp);
				freq+=8000;
			}
		}
	}
	
	return 0;
}

int DVB::scan_cable(Sat &sat) 
{

        struct Transponder tp;
        uint srates[2]= { 6900000 }; 
	uint freq, i;
	int res, qam=2;
	uint srate=0, ofreq;
	
	for (freq=sat.fmin; freq<=sat.fmax; freq+=4000000) {
		{
		        ostrstream fstr;
			fstr << dec << (freq/1000) << " KHz" << ends;
			//if(fdvb >= 0) osd.Text(400, 40, 2, 1, fstr.str());
		}
		for (i=0, res=-1; i<1 && res<0; i++) {
		        ofreq=freq;
		        srate=srates[i];
			res=set_tp(&ofreq, 0, 0, srate);

			if (!res) {
			        ostrstream fstr;
				fstr << "Signal lock at " << dec << (freq/1000)
				     << " KHz" << ends;
				if(fdvb >= 0){
				        //osd.Text(0, 80, 2, 1, fstr.str());
				  //osd.FillBlock(0, 120, 300, 159, 0);
				}
				tp.name[0]=0;
				tp.id=NOID;
				tp.type=2;
				tp.pol=-1;
				tp.freq=freq;
				tp.qam=qam;
				tp.srate=srate;
				tp.fec=8;
				tp.satid=sat.id;
				scan_tp(tp);
			}
		}
	}
	
	return 0;
}

int DVB::scan_lnb(struct Lnb &lnb) {
        int i;

	for (i=0; i<num[SAT]; i++) {
	        if (sats[i].lnbid==lnb.id)
		        switch (front_type) {
			case FE_QPSK:
			        scan_sat(sats[i]);
				break;
			case FE_QAM:
			  //cerr << "scan_cable\n";
			        scan_cable(sats[i]);
				break;
			}
	}
	return 0;
}


int DVB::scan_all_tps() 
{
        int itp;
	

	get_front();
	set_vpid(0);
	set_apid(0);
	set_ttpid(0);

	for (itp=0; itp<num[TRANS]; itp++) {
	  cerr << "TP " << dec << itp << "  id=" << hex << tps[itp].id << endl;
	  if (SetTP(tps[itp].id, tps[itp].satid))
	    continue;
	  set_front();
	  scan_tp(tps[itp]);
	}
	return 0;
}

int DVB::scan() {
        int i;
	
	if (!num[LNB])
	        return -1;
	

	for (i=0; i<num[LNB]; i++) {
	        ostrstream lnbname;
		lnbname << "LNB" << HEX(1) << lnbs[i].id << ends;

		//if(fdvb >= 0) osd.Text(0, 40, 2, 1, lnbname.str());

		if (scan_lnb(lnbs[i])<0){
		  cerr << "done" << endl;
		  return -1;
		}
	}
	cerr << "done" << endl;
	return 0;
}
*/

int DVB::AddLNB(int id, int t, uint l1, uint l2, uint sl,
		int dnr, int dis, int sw)
{
        if (num[LNB]>=maxs[LNB])
	        return -1;
	for (int i=0; i< num[LNB]; i++){
		if (lnbs[i].id == id && lnbs[i].diseqcnr == dnr){
			cerr << "Warning: LNB already defined:" << endl;
			cerr << "ID: " << id << "  DISEQCNR: " << dnr << endl;
			return -1;
		}
	}

	lnbs[num[LNB]].init(t, l1, l2, sl, dnr, dis, sw);
	lnbs[num[LNB]].id=id;
	num[LNB]++;
	return 0;
}



int DVB::AddTP(Transponder &tp)
{
        if (num[TRANS]>=maxs[TRANS])
	        return -1;

	if (tp.id == NOID){
		max_tpid++;
	        tp.id = max_tpid;
	} else if (tp.id > max_tpid) max_tpid = tp.id;

	for (int i=0; i<num[TRANS]; i++)
		if (tps[i].id==tp.id && tps[i].satid == tp.satid){
			cerr << "Warning: TP already defined:" 
			     << endl;
			cerr << "ID: " << hex << tp.id ;
			cerr << "  SATID: "<< hex  << tp.satid;
			cerr << endl;
			return i;
		} 
			
	tps[num[TRANS]]=tp;
	num[TRANS]++;
	return num[TRANS]-1;
}

void DVB::find_satid(Channel &chan)
{
	for (int i=num[TRANS]; i >= 0; i--){
		if (chan.tpid == tps[i].id){
			chan.satid = tps[i].satid;
			return;
		}
	}
}

int DVB::AddChannel(Channel &chan)
{
        int i;

        if (num[CHAN]>=maxs[CHAN])
	        return -1;
	
	if ( chan.satid == NOID)
		find_satid(chan);

        for (i=0; i<num[CHAN]; i++) {
		if (chan.pnr != NOPID && chan.pnr == chans[i].pnr && 
		    chan.satid == chans[i].satid
			&& chan.tpid == chans[i].tpid) {
		        cerr << "Channel " << chan.name << " ("
			     << hex << chan.pnr << ") exists" << endl;
			return i;
		}
		if (chan.pnr == NOPID && chan.vpid == chans[i].vpid &&
		    chan.apids[0] == chans[i].apids[0] &&
		    chan.satid == chans[i].satid
		    && chan.tpid == chans[i].tpid) {
		        cerr << "Channel " << chan.name << " ("
			     << hex << chan.pnr << ") exists" << endl;
			return i;
		}
	}

	chan.id = num[CHAN];
	chans[num[CHAN]] = chan;
	num[CHAN]++;
	return chan.id;
}

int DVB::AddSat(Sat &sat)
{
        int i;

        if (num[SAT]>=maxs[SAT])
	        return -1;
	if (!sat.id)
	        sat.id=num[SAT];

        for (i=0; i<num[SAT]; i++) {
 	        if (sat.lnbid == sats[i].lnbid) {
		        cerr << "Sat exists\n";
			return i;
		}
	}
	sats[num[SAT]]=sat;
	num[SAT]++;
	return sat.id;
}

int DVB::AddSat(int id, unsigned int lnbid, char *name, uint fmin, uint fmax)
{
        int i,j;

	if (num[SAT]==maxs[SAT]) 
	        return -1;
	for (i=0; i<num[LNB]; i++) {
	  if (lnbs[i].id==lnbid) {
		  for (j=0; j<num[SAT]; j++) {
			  if (lnbid == sats[j].lnbid) {
				  cerr << "Sat exists\n";
				  return j;
			  }
		  }
		  sats[num[SAT]].id=id;
		  sats[num[SAT]].lnb=&lnbs[i];
		  sats[num[SAT]].lnbid=lnbs[i].id;
		  strncpy(sats[num[SAT]].name, name, maxname);
		  sats[num[SAT]].name[maxname]=0;
		  sats[num[SAT]].fmin=fmin;
		  sats[num[SAT]].fmax=fmax;
		  num[SAT]++;
		  return num[SAT]-1;
		}
	}
	return -1;

}


ostream &operator<<(ostream &stream, DVB &x) {
	int i,j,k,l;
	
	for (i=0; i<x.num[LNB]; i++) {
		stream << x.lnbs[i];
		
		for (j=0; j<x.num[SAT]; j++)
			if (x.sats[j].lnbid==x.lnbs[i].id) {
				stream << x.sats[j];
				
				for (k=0; k<x.num[TRANS]; k++) 
					if (x.tps[k].satid==x.sats[j].id) {
						stream << x.tps[k];
						
						for (l=0; l<x.num[CHAN]; l++) 
							if (x.chans[l].tpid==
							    x.tps[k].id && 
							    x.chans[l].satid == x.tps[k].satid )
								stream << x.chans[l];
					}
			}
	}
	return stream;
}



int DVB::check_input_format(istream &ins)
{
	streampos pos = ins.tellg();
	int found = 0;
	char *test_keys[]={
		"LNB","DISEQC","ROTOR","TRANSPONDER","CHANNEL",
		"BOUQUET","SAT","SWITCH","NETWORK","<?xml", ":SAT", NULL
	};
	enum { XML_START=9, NOKIA_START};

	int f = -1;
	while(!found && !ins.eof()){
		char keybuf[MAXNAM];
		ins >> keybuf;
		int n=findkey(keybuf, test_keys);
		
		switch (n){
		case LNB:
		case DIS:
		case ROTOR:
		case TRANS:
		case CHAN:
		case BOU:
		case SAT:
			found = 1;
			f = DVB_ORIG;
			break;
			
		case NOKIA_START:
			found = 1;
			f = DVB_NOKIA;
			break;
			
		case XML_START:
			found = 1;
			f = DVB_XML;
			break;
		default:
			cerr << "Error: " << keybuf 
			     << " is not a valid keyword at " 
			     << endl;
			exit(0);
			
		}
	} 
	ins.seekg(pos);
	return f;
}

void DVB::read_original(istream &ins)
{
	char *names[] = {
		"LNB","DISEQC","ROTOR","TRANSPONDER","CHANNEL",
		"BOUQUET","SAT","SWITCH","NETWORK", NULL
	};

	cerr << "Reading original format" << endl;
	while(!ins.eof()){
		char keybuf[MAXNAM];
		ins >> keybuf;
		int n=findkey(keybuf, names);
		if (n<0) {
			cerr << "Error: " << keybuf 
			     << " is not a valid keyword at " 
			     << endl;
			exit(0);
		} else {
			if (num[n]< maxs[n]){
				switch (n){
				case LNB:
				{
					Lnb lnb;
					lnb.name[0]='\0';
					ins >> lnb;
					AddLNB(lnb.id, lnb.type, lnb.lof1, 
						 lnb.lof2, lnb.slof, 
						 lnb.diseqcnr, lnb.diseqcid, 
						 lnb.swiid);
					break;
				}
				case DIS:
					num[n]++;
					ins >> diseqcs[num[n]-1];
					break;
				case ROTOR:
					num[n]++;
					ins >> rotors[num[n]-1];
					break;
				case TRANS:
				{
					Transponder tp;
					ins >> tp;
					AddTP(tp);
					break;
				}
				case CHAN:
				{
					Channel chan;
					ins >> chan;
					AddChannel(chan);
					break;
				}
				case BOU:
					num[n]++;
					ins >> bouqs[num[n]-1];
					break;
				case SAT:
				{
					Sat sat;
					ins >> sat;
					AddSat(sat);
					break;
				}
				}
			} else {
				cerr << "not enough channels" << endl;
				break;
			}
		}
	} 
	
}

istream &operator>>(istream &ins, DVB &x)
{
	int format = x.check_input_format(ins);

	switch(format){
	case DVB_ORIG:
		x.read_original(ins);
		break;

	case DVB_NOKIA:
	{
		nokiaconv cc(&x);

		cc.lnb_sat.n = 4;
		cc.lnb_sat.diseqc[0] = 0;
		cc.lnb_sat.diseqc[1] = 1;
		cc.lnb_sat.diseqc[2] = 2;
		cc.lnb_sat.diseqc[3] = 3;
		strcpy(cc.lnb_sat.sat_names[0],"Astra");
		cc.lnb_sat.satid[0]=0x0192;
		strcpy(cc.lnb_sat.sat_names[1],"HotBird");
		cc.lnb_sat.satid[1]=0x0130;
		strcpy(cc.lnb_sat.sat_names[2],"Sirius");
		cc.lnb_sat.satid[2]=0x0050;
		cerr << "Reading NOKIA format" << endl;
		
		ins >> cc;
		break;
	}

	case DVB_XML:
	{
		xmlconv cc(&x);

		cc.lnb_sat.n = 4;
		cc.lnb_sat.diseqc[0] = 0;
		cc.lnb_sat.diseqc[1] = 1;
		cc.lnb_sat.diseqc[2] = 2;
		cc.lnb_sat.diseqc[3] = 3;
		strcpy(cc.lnb_sat.sat_names[0],"Astra");
		cc.lnb_sat.satid[0]=0x0192;
		strcpy(cc.lnb_sat.sat_names[1],"HotBird");
		cc.lnb_sat.satid[1]=0x0130;
		strcpy(cc.lnb_sat.sat_names[2],"Sirius");
		cc.lnb_sat.satid[2]=0x0050;
		cerr << "Reading XML format" << endl;
		
		ins >> cc;
		break;
	}
	default:
		cerr << "Unknown format. Can't open dvbrc. Exiting" << endl;
		exit(1);

	}
	return ins;
}




void hdump(unsigned char *buf, int n) 
{
	int i;
	
	for (i=0; i<n; i++)
		cerr << HEX(2) << (int) buf[i] << " ";
	cerr << endl;
}


void DVB::bar2(int x, int y, int w, int h, int val, int col1, int col2) 
{
	int sep=(w*val)>>16;
	
	if (fdvb >= 0) {
		osd.FillBlock(x, y, x+w-1-sep, y+h-1, col1);
		osd.FillBlock(x+w-1-sep, y, 515, y+h-1, col2);
	}
}

uint16_t DVB::SetFilter(uint16_t pid, uint16_t section, uint16_t mode) 
{ 
	struct dmxSctFilterParams secFilterParams;
	int fd_section=open("/dev/ost/demux", O_RDWR);
	
	secFilterParams.pid=pid;
	memset(&secFilterParams.filter.filter, 0, DMX_FILTER_SIZE);
	memset(&secFilterParams.filter.mask, 0, DMX_FILTER_SIZE);
	secFilterParams.timeout = 0;
	secFilterParams.flags=DMX_IMMEDIATE_START;
	
	secFilterParams.filter.filter[0]=(section>>8)&0xff;
	secFilterParams.filter.mask[0]=section&0xff;
	
	if (ioctl(fd_section, DMX_SET_FILTER, &secFilterParams) < 0)  
		return 0xffff;
	return fd_section;
} 

int DVB::SetFilter(uint16_t pid, unsigned char *filter, 
		   unsigned char *mask,
		   uint32_t timeout, uint32_t flags) 
{
	
	struct dmxSctFilterParams secFilterParams;
	int fd_section=open("/dev/ost/demux", O_RDWR);
	
	secFilterParams.pid=pid;
	memset(&secFilterParams.filter.filter, 0, DMX_FILTER_SIZE);
	memset(&secFilterParams.filter.mask, 0, DMX_FILTER_SIZE);
	secFilterParams.timeout = timeout;
	secFilterParams.flags=DMX_IMMEDIATE_START;
	
	for (int i = 0; i < DMX_FILTER_SIZE; i++){
		secFilterParams.filter.filter[i]=filter[i];
		secFilterParams.filter.mask[i]=mask[i];
	}
	if (ioctl(fd_section, DMX_SET_FILTER, &secFilterParams) < 0)  
		return 0xffff;
	return fd_section;
}


int DVB::CloseFilter(int h) 
{
	close(h);
	return 0;
}


int DVB::GetSection(unsigned char *buf, ushort PID, unsigned char sec,
		    unsigned char secnum, unsigned char &msecnum) 
{
	int seclen=0;
	uint16_t handle, pid;
	unsigned char section, sectionnum=0xff, maxsec=0;
	struct pollfd pfd;
	int loopc = 0;

	if ((handle=SetFilter(PID, (sec<<8)|0x00ff, 0))==0xffff)
		return -1;
	do {
		seclen=0;
		pfd.fd = handle;
		pfd.events=POLLIN;
		if (poll(&pfd, 1, 2000)==0) {
//			cerr << "Timeout\n";
			break;
		}
		loopc++;
		
		pid = PID;
		read(handle, buf, 3);
		seclen = 0;
		seclen |= ((buf[1] & 0x0F) << 8); 
		seclen |= (buf[2] & 0xFF);
		
		read(handle, buf+3, seclen);
		seclen+=3;
		section=buf[0];
		sectionnum=buf[6];
		maxsec=buf[7];
	} while ( loopc < maxsec*2 && 
		 (section != sec || pid != PID || sectionnum != secnum));
	msecnum=maxsec;
	CloseFilter(handle);
	return seclen;
}

int DVB::GetSection(uint8_t *buf, 
		    uint16_t PID, uint8_t *filter, uint8_t *mask,
		    uint8_t secnum, uint8_t &msecnum) 
{
	int seclen=0;
	uint16_t handle, pid;
	uint8_t section, sectionnum=0xff, maxsec=0;
	struct pollfd pfd;
	int loopc = 0;

	if ((handle=SetFilter(PID, filter, mask, 0, 0))==0xffff)
		return -1;
	do {
		seclen=0;
		pfd.fd=handle;
		pfd.events=POLLIN;
		if (poll(&pfd, 1, 20000)==0)
			break;
		loopc++;
		// read header
		pid = PID;
		read(handle, buf, 3);
		seclen = 0;
		seclen |= ((buf[1] & 0x0F) << 8); 
		seclen |= (buf[2] & 0xFF);
		
		read(handle, buf+3, seclen);
		seclen+=3;

		section=buf[0];
		sectionnum=buf[6];
		maxsec=buf[7];
	} while (loopc < maxsec*2 && ((section&mask[0]!=filter[0]) ||
		 pid!=PID || sectionnum!=secnum));
	msecnum=maxsec;
	CloseFilter(handle);
	return seclen;
}

int DVB::GetSection(uint8_t *buf, 
		    uint16_t PID, uint8_t TID, uint16_t TIDExt, 
		    uint16_t FilterTIDExt, 
		    uint8_t secnum, uint8_t &msecnum) 
{
	uint8_t filter[16], mask[16];
	
	memset(filter, 0, 16);
	memset(mask, 0, 16);
	
	filter[0]=TID;
	mask[0]=0xff;
	if (TIDExt!=0xffff) {
		filter[1]=(TIDExt>>8);
		filter[2]=TIDExt&0xff;
		mask[1]=(FilterTIDExt>>8);
		mask[2]=FilterTIDExt&0xff;
		}
	
	return GetSection(buf, PID, filter, mask, secnum, msecnum);
}		

/*
int DVB::GetSDT(dvb_sdt_t *sdt) 
{
	unsigned char buf[MAXSECSIZE], msec=0, sec=0;
	int ret=-1;
	
	
	while (sec<=msec) {
		if (GetSection(buf, PID_SDT, TID_SDT_ACT, sec, msec)>0)
			ret&=parse_sdt(buf, sdt);
		sec++;
	}
	return ret;
}

int DVB::GetPAT(dvb_pat_t *pat) 
{
	unsigned char buf[MAXSECSIZE], msec=0, sec=0;
	int ret=-1;
	
	while (sec<=msec) {
		if (GetSection(buf, PID_PAT, TID_PAT, sec, msec)>0)
			ret&=parse_pat(buf, pat);
		sec++;
	}
	return ret;
}

int DVB::GetPMT(dvb_pmt_t *pmt, unsigned short pid) 
{
	unsigned char buf[MAXSECSIZE], sec=0, msec=0;
	int ret=-1;
	while (sec<=msec) {
		if (GetSection(buf, pid, TID_PMT, sec, msec)>0)
			ret&=parse_pmt(buf, pmt);
		sec++;
	}
	return ret;
}

int DVB::GetNIT(dvb_nit_t *nit) 
{
	unsigned char buf[MAXSECSIZE], sec=0, msec=0;
	int ret=-1;
	while (sec<=msec) {
		if (GetSection(buf, PID_NIT, TID_NIT_ACT, sec, msec)>0)
			ret&=parse_nit(buf, nit);
		//hdump(buf, 0x120);
		sec++;
	}
	return ret;
}
*/

void DVB::get_front(void) 
{
	set_vpid(0);
	set_apid(0);
	set_ttpid(0);
	scmds.voltage = SEC_VOLTAGE_13; 
	scmds.miniCommand = SEC_MINI_NONE;
	scmds.continuousTone = SEC_TONE_OFF;
	scmds.numCommands=1;
	scmds.commands=&scmd;
}	        
  
int DVB::tune_it(FrontendParameters *front_param)
{
	FrontendEvent event;
	struct pollfd pfd[1];

	if (ioctl(fd_frontend, FE_SET_FRONTEND, front_param) <0)
		perror("setfront front");

	pfd[0].fd=fd_frontend;
	pfd[0].events=POLLIN;
	if (poll(pfd,1,2000)) {
		if (pfd[0].revents & POLLIN){
			if (ioctl(fd_frontend, FE_GET_EVENT, &event)  

			     == -EBUFFEROVERFLOW){
				perror("fe get event");
				return -1;
			}
			switch(event.type){
			case FE_UNEXPECTED_EV:
				perror("unexpected event\n");
				return -1;
			case FE_FAILURE_EV:
				perror("failure event\n");
				return -1;
				
			case FE_COMPLETION_EV:
//				printf("completion event\n");
				return 0;
			}
		}
	}
	return -1;
}


void DVB::set_front(void) 
{
	scmds.miniCommand = SEC_MINI_NONE;
	scmds.numCommands=1;
	scmds.commands=&scmd;
	
	set_vpid(0);  
	set_apid(0);
	set_ttpid(0);

	if (front_type==FE_QPSK) {
		if (ioctl(fd_sec, SEC_SEND_SEQUENCE, &scmds) < 0)
		        perror("setfront sec");
		usleep(70000);
	}
	tune_it(&front_param);
}	        

void DVB::set_diseqc_nb(int nr) 
{
	scmd.type=0;
	scmd.u.diseqc.addr = 0x10;
	scmd.u.diseqc.cmd = 0x38;
	scmd.u.diseqc.numParams = 1;
	scmd.u.diseqc.params[0] = 0xF0 | ((nr * 4) & 0x0F) | 
	  (scmds.continuousTone == SEC_TONE_ON ? 1 : 0) |
	  (scmds.voltage==SEC_VOLTAGE_18 ? 2 : 0);
}

int DVB::set_lnb(int dis) 
{
	get_front();
	set_diseqc_nb(dis);		
	if(ioctl(fd_sec, SEC_SEND_SEQUENCE, &scmds) < 0)
		perror("set_lnb");
	return 0;
}	        
  
void DVB::set_ttpid(ushort ttpid) 
{  
	if (ttpid==0 || ttpid== NOPID || ttpid==0x1fff) {
		ioctl(fd_demuxtt, DMX_STOP, 0);
		return;
	}
	pesFilterParamsTT.pid     = ttpid;
	pesFilterParamsTT.input   = DMX_IN_FRONTEND; 
	pesFilterParamsTT.output  = DMX_OUT_DECODER; 
	pesFilterParamsTT.pesType = DMX_PES_TELETEXT; 
	pesFilterParamsTT.flags   = DMX_IMMEDIATE_START;
	if (ioctl(fd_demuxtt, DMX_SET_PES_FILTER, 
		  &pesFilterParamsTT) < 0) {
	  printf("PID=%04x\n", ttpid);
		perror("set_ttpid");
	}
}

void DVB::set_vpid(ushort vpid) 
{  
	if (vpid==0 || vpid==NOPID || vpid==0x1fff) {
		ioctl(fd_demuxv, DMX_STOP, 0);
		return;
	}
	pesFilterParamsV.pid     = vpid;
	pesFilterParamsV.input   = DMX_IN_FRONTEND; 
	if (dvr_enabled)
		pesFilterParamsV.output  = DMX_OUT_TS_TAP; 
	else 
		pesFilterParamsV.output  = DMX_OUT_DECODER; 

	pesFilterParamsV.pesType = DMX_PES_VIDEO; 
	pesFilterParamsV.flags   = DMX_IMMEDIATE_START;
	if (ioctl(fd_demuxv, DMX_SET_PES_FILTER, 
		  &pesFilterParamsV) < 0)
		perror("set_vpid");
}

void DVB::set_apid(ushort apid) 
{  
	if (apid==0 || apid==NOPID || apid==0x1fff) {
		ioctl(fd_demuxa, DMX_STOP, apid);
		return;
	}

	pesFilterParamsA.pid = apid;
	pesFilterParamsA.input = DMX_IN_FRONTEND; 
	if (dvr_enabled)
		pesFilterParamsA.output = DMX_OUT_TS_TAP; 
	else
		pesFilterParamsA.output = DMX_OUT_DECODER;
 
	pesFilterParamsA.pesType = DMX_PES_AUDIO; 
	pesFilterParamsA.flags = DMX_IMMEDIATE_START;
	if (ioctl(fd_demuxa, DMX_SET_PES_FILTER, 
		  &pesFilterParamsA) < 0)
		perror("set_apid");
}


int DVB::set_tp(uint *freq, int ttk, int pol, uint srate) 
{
        int stat=0;

        switch (front_type) {
	  
	case FE_QPSK:
	        if (ttk) scmds.continuousTone = SEC_TONE_ON;
		else scmds.continuousTone = SEC_TONE_OFF;
		front_param.Frequency = *freq/1000;
		front_param.u.qpsk.SymbolRate = srate;
		if (pol) scmds.voltage=SEC_VOLTAGE_18;
		else scmds.voltage=SEC_VOLTAGE_13;
		front_param.u.qpsk.FEC_inner=FEC_AUTO;
		if (ioctl(fd_sec, SEC_SEND_SEQUENCE, &scmds) < 0)
			perror("set_tp sec");
		break;

	case FE_QAM:
		front_param.Frequency=*freq;
		front_param.u.qam.SymbolRate=srate;
		front_param.u.qam.FEC_inner=FEC_AUTO;
		front_param.u.qam.QAM=QAM_64;
		break;

	case FE_OFDM:
	        front_param.Frequency = *freq;
		front_param.Inversion = INVERSION_AUTO;
		front_param.u.ofdm.bandWidth = BANDWIDTH_8_MHZ;
		front_param.u.ofdm.HP_CodeRate = FEC_2_3;
		front_param.u.ofdm.LP_CodeRate = FEC_NONE;
		front_param.u.ofdm.Constellation = QAM_16;
		front_param.u.ofdm.TransmissionMode = TRANSMISSION_MODE_8K;
		front_param.u.ofdm.guardInterval = GUARD_INTERVAL_1_8;
		front_param.u.ofdm.HierarchyInformation = HIERARCHY_NONE;
	}
	stat = tune_it(&front_param);

	if (!stat) return 0;
	return -1;
}



istream &operator>>(istream &ins, nokiaconv &x)
{
	int n=-1;
	char keybuf[MAXNAM];
	char sname[MAXNAM];
	char dummy;
	int current_sat = -1;
	int current_tp = -1;
	int dint;

	enum { NSAT=0, NNET, NTRP, NCHAN, NEND};
	static char *nokiakeys[]={
		":SAT", ":NET", ":TRP", ":CHN", ":END", NULL
	};

	while(!ins.eof()){
		streampos pos = ins.tellg();
		ins >> keybuf;
		n=findkey(keybuf, nokiakeys);
		if (n<0) {
			ins.seekg(pos);
			break;
		}
		switch(n){
		case NSAT:
		{
			double did;
			int id=0;
			int lnbid = 5; 
			int found = 0;

			getname(sname,ins);
			//cerr << "Satellite \"" << sname << "\"" << endl;
			for(int i=0; i < x.lnb_sat.n; i++){
				if (!strcmp(x.lnb_sat.sat_names[i],sname)){
					lnbid = x.lnb_sat.diseqc[i]; 
					id = x.lnb_sat.satid[i]; 

					found = 1;
					break;
				}
					
			}

			x.dvb->AddLNB(lnbid, 1, 
				      9750000, 10600000, 
				      11700000, 
				      lnbid, 
				      NOID, NOID);
			
			ins >> did;
			current_sat =
				x.dvb->AddSat( id, lnbid,
					       sname, 
					       10700000 , 
					       12700000);
			ins >> dummy;
			
			break;
		}

		case NNET:
			getname(sname,ins);
			//cerr << "  Network \"" << sname << "\""<< endl; 
			ins >> dint;
			break;

		case NTRP:
		{
			Transponder trans;

			ins >> dec >> trans.id;
			ins >> trans.freq;
			ins >> trans.srate;
			ins >> dint;
			ins >> dummy;
			if (dummy == 'H') trans.pol = 1;
			if (dummy == 'V') trans.pol = 0;
			ins >> dint;

			trans.satid = x.dvb->sats[current_sat].id;
			trans.type = 1;
			trans.freq *= 10;
			trans.srate *= 100;

			ins >> dint;
			ins >> dummy;
			ins >> dint;

			switch (dint){
			case 2:
				trans.fec = 0;
				break;
			case 3:
				trans.fec = 1;
				break;
			case 4:
				trans.fec = 2;
				break;
			case 6:
				trans.fec = 4;
				break;
			case 8:
				trans.fec = 6;
				break;
			}

			current_tp = x.dvb->AddTP(trans);
			//cerr << "    Transponder "<< trans.id  << endl;
			break;
		}

		case NCHAN:
		{
			Channel chan;
			int cnum;

			getname(sname,ins);
			strncpy(chan.name, sname, maxname);
			ins >> chan.pnr;
			ins >> dummy;
			if (dummy == 'T'){
				ins.ignore(20, ':');
				ins.seekg(ins.tellg()-1);

				chan.satid = x.dvb->sats[current_sat].id;
				chan.tpid = x.dvb->tps[current_tp].id;
				cnum = x.dvb->AddChannel(chan);

				//cerr << "      Channel "<< sname  
				//     << " (" << cnum << ")" << endl;
			} else 	{
				ins.seekg(pos);
				ins.ignore(80,0x0a);
			}
			break;
		}

		case NEND:
			//cerr << "ALL DONE" << endl;
			return ins;
		}		
	}
	return ins;
}

static int get_keylen(istream &ins, char *keybuf)
{
	streampos pos = ins.tellg();
	int klen = strlen(keybuf);
	if (klen>2 && keybuf[1]!= '/' &&
	    keybuf[0]=='<' && keybuf[klen-1]=='>'){
		keybuf[klen-2]='\0';
		klen--;
		ins.seekg(pos-2);
	}
	return klen;
}

static int find_xml_key(istream &ins, char *keybuf, char *keys[])
{
	char *s;
	int n;
	streampos pos = ins.tellg();
	ins >> keybuf;
	int klen = get_keylen(ins, keybuf);
	s=keybuf;
	while (s[0] != '=' && s != keybuf+klen)s++;
	s[0]=0;
	ins.seekg(pos + (s-keybuf) +1 ); // go past =
	n=findkey(keybuf, keys);
	if (n<0) {
		ins.seekg(pos);
		cerr << "Unknown tag: " << keybuf << endl;
	}
	return n;
}

int xmlconv::read_sat(istream &ins, int csat = -1)
{
	int n=-1;
	int lnbid=-1;
	int satid;
	char keybuf[MAXNAM];
	char sname[MAXNAM];
	enum { XSATN, XSLNB, XSATID, XTRANS, XSATEND, XEND, XNEND};
	static char *xsat[]={
		"name", "lnb", "id", "<transponder", "</satellite>",
		">", "/>", NULL
	};
	while(!ins.eof()){
		if ( (n = find_xml_key( ins, keybuf, xsat)) < 0) break;
		switch(n){
		case XSATN:
			getname(sname,ins);			
			break;

		case XSLNB:
			ins >> satid;
			break;

		case XTRANS:
			if (csat >= 0)
				read_trans(ins, csat);
			else
				return -1;
			break;

		case XSATID:
			ins >> satid;
			break;

		case XSATEND:
			return 0;
			break;

		case XEND:
			if (satid >=0 && lnbid >= 0)
				csat =	dvb->AddSat(satid, 
						    lnbid,
						    sname, 
						    10700000 , 
						    12700000);
			break;

		case XNEND:
			return 0;
			break;

		default:
			skip_tag(ins,keybuf);
			break;
		}
	}

	return 0;
}

int xmlconv::read_trans(istream &ins, int csat)
{
	int n=-1;
	int ctp=-1;
	char keybuf[MAXNAM];
	char sname[MAXNAM];
	enum { XTYPE=0, XFREQ, XSRATE, XPOL, XFEC, XSERV, 
	       XTRANSEND, XEND, XNEND };
	static char *xtrans[]={
		"type", "freq", "srate", "polarity", "fec", 
		"<service", "</transponder>", 
		">", "/>", NULL
	};
	Transponder trans;
	trans.satid = dvb->sats[csat].id;
	trans.fec = 8;
	trans.id = NOID;

	while(!ins.eof()){
		if ( (n = find_xml_key( ins, keybuf, xtrans)) < 0) break;
		switch(n){
		case XTYPE:
			getname(sname,ins);
			switch(sname[0]){
			case 'S':
				trans.type = SAT_TYPE;
				break;
			case 'T':
				trans.type = TERRA_TYPE;
				break;
			case 'C':
				trans.type = CABLE_TYPE;
				break;
			}
			break;

		case XFREQ:
			getname(sname,ins);
			trans.freq=atoi(sname);
			break;

		case XSRATE:
			getname(sname,ins);
			trans.srate=atoi(sname);
			break;

		case XPOL:
			getname(sname,ins);
			if (sname[0] == 'H') trans.pol = 1;
			if (sname[0] == 'V') trans.pol = 0;
			break;

		case XFEC:
			int dint;
			getname(sname,ins);
			dint = atoi(sname);

			switch (dint){
			case 2:
				trans.fec = 0;
				break;
			case 3:
				trans.fec = 1;
				break;
			case 4:
				trans.fec = 2;
				break;
			case 6:
				trans.fec = 4;
				break;
			case 8:
				trans.fec = 6;
				break;
			}
			break;


		case XSERV:
			if (ctp>=0)
				read_serv(ins,ctp,csat);
			break;

		case XTRANSEND:
			return 0;
			break;

		case XEND:
			ctp = dvb->AddTP(trans);
			break;

		case XNEND:
			return 0;
			break;
		default:
			skip_tag(ins,keybuf);
			break;

		}
	}

	return 0;
}

int xmlconv::read_serv(istream &ins, int ctp, int csat)
{
	int n=-1;
	char keybuf[MAXNAM];
	char sname[MAXNAM];
	enum { XSID=0, XCA, XDESC, XSTREAM, 
	       XSERVEND, XEND, XNEND };
	static char *xserv[]={
		"id", "ca", 
		"<description", 
		"<stream", "</service>",  		
		">", "/>", 
		"<ca_descriptor", "<descriptor",
		"<country_availability", "<ca_system_id", NULL
	};
	Channel chan;
	int nchan=-1;

	chan.satid = dvb->sats[csat].id;
	chan.tpid = dvb->tps[ctp].id;

	while(!ins.eof()){
		if ( (n = find_xml_key( ins, keybuf, xserv)) < 0) break;
		switch(n){
		case XSID:
			getname(sname,ins);
			chan.pnr = atoi(sname);
			nchan = dvb->AddChannel(chan);
			break;

		case XCA:
			getname(sname,ins);
			if (nchan >= 0)
				dvb->chans[nchan].type = atoi(sname);
			else
				chan.type = atoi(sname);
			break;

		case XDESC:
			if (nchan>=0)
				read_desc(ins, nchan);
			else
				return -1;
			break;

		case XSTREAM: 
			if (nchan>=0)
				read_stream(ins,nchan);
			else
				return -1;
			break;

		case XSERVEND: 
			return 0;
			break;

		case XEND: 
			break;

		case XNEND:
			return 0;
			break;
		default:
			skip_tag(ins,keybuf);
			break;
		}
	}

	return 0;
}

int xmlconv::read_desc(istream &ins, int nchan)
{
	int n=-1;
	char keybuf[MAXNAM];
	char sname[MAXNAM];
	enum { XTAG=0, XTYPE, XPROV, XSNAME, XDESCEND, 
	       XEND, XNEND};
	static char *xdesc[]={
		"tag", "type", "provider_name", "service_name", 
		"</description>", ">", "/>", NULL
	};

	while(!ins.eof()){
		if ( (n = find_xml_key( ins, keybuf, xdesc)) < 0) break;
		switch(n){
		case XTAG:
			getname(sname,ins);
			break;

		case XTYPE:
			getname(sname,ins);
			break;

		case XPROV:
			getname(sname,ins);
			break;

		case XSNAME:
			getname(sname,ins);
			dvb2txt(dvb->chans[nchan].name,sname,MAXNAM);
			break;

		case XDESCEND:
			return 0;
			break;

		case XEND:
			break;

		case XNEND:
			return 0;
			break;

		default:
			skip_tag(ins,keybuf);
			break;
		}
	}

	return 0;
}

int xmlconv::read_stream(istream &ins, int nchan)
{
	int n=-1;
	char keybuf[MAXNAM];
	char sname[MAXNAM];
	int type = -1;
	int apids = 0;
	uint16_t pid = NOPID;
	enum { XTYPE, XPID, XISO, XSTREAMEND, XEND, XNEND };
	static char *xstream[]={
		"type", "pid", "<iso_639",
		"</stream>", ">", "/>", 
		"<ca_descriptor","<descriptor","<teletext","<stream_id", 
		"<canal_radio", "<audio_info", "<description", NULL
	};

	while(!ins.eof()){
		if ( (n = find_xml_key( ins, keybuf, xstream)) < 0) break;
		switch(n){
		case XTYPE:
			getname(sname,ins);
			type = atoi(sname);
			break;

		case XPID:
			getname(sname,ins);
			pid = atoi(sname);
			switch(type){
			case 1:
			case 2:
				if (pid != NOPID)
					dvb->chans[nchan].vpid = pid;
				break;
			case 3:
			case 4:
				if (pid == NOPID) break;
				apids = dvb->chans[nchan].apidnum;
				if (apids >= MAXAPIDS) break;
				dvb->chans[nchan].apidnum++;
				dvb->chans[nchan].apids[apids]=pid;
				break;
			case 6:
				if (pid != NOPID)
					dvb->chans[nchan].ttpid = pid;
				break;				
			}
			break;

		case XSTREAMEND:
			return 0;
			break;

		case XEND:
			break;
			
		case XNEND:
			return 0;
			break;

		case XISO:
			read_iso639(ins, nchan, apids);
			break;

		default:
			skip_tag(ins,keybuf);
			break;
		}
	}

	return 0;
}

int xmlconv::read_iso639(istream &ins, int nchan, int apids)
{
	int n=-1;
	char keybuf[MAXNAM];
	char sname[MAXNAM];
	enum { XTYPE, XLAN,XISOEND, XEND, XNEND };
	static char *xiso[]={
		"type", "language", "</iso_639>",
		">", "/>", NULL
	};

	while(!ins.eof()){
		if ( (n = find_xml_key( ins, keybuf, xiso)) < 0) break;
		switch(n){
		case XTYPE:
			getname(sname,ins);
			break;

		case XLAN:
			getname(sname,ins);
			strncpy(dvb->chans[nchan].apids_name+apids*4, sname, 4);
			break;

		case XISOEND:
			return 0;
			break;

		case XEND:
			break;
			
		case XNEND:
			return 0;
			break;

		default:
			skip_tag(ins,keybuf);
			break;
		}
	}

	return 0;
}

int xmlconv::skip_tag(istream &ins, char *tag)
{
	char sname[MAXNAM*2];
	char endtag[MAXNAM];
	int found = 0;
	streampos pos = ins.tellg();
	
	ostrstream etag(endtag,MAXNAM);
	etag << "</" << tag+1 << ">" << ends;
	int tlen = strlen(endtag)-1;
	
//	cerr << "find: " << endtag << endl;
	ins >> sname;
	if (sname[0] == '>')
		while(!found){
			if (!strncmp(sname,endtag,tlen)) 
				found=1;
			else 
				ins >> sname;
		}
	else {
		ins.seekg(pos);
		ins.ignore(1000,'>');
		pos = ins.tellg();
		ins.seekg(pos-2);
		ins >> sname;
		if (sname[0] == '/')
			ins.seekg(pos);
		else 
			while(!found){
				if (!strncmp(sname,endtag,tlen)) 
					found=1;
				else
					ins >> sname;
			}

	}
	return 0;
}

istream &operator>>(istream &ins, xmlconv &x)
{
	int n=-1;
	char keybuf[MAXNAM];
	char sname[MAXNAM];
	int current_sat = -1;
	int nsat = 0;

	enum { XMLSTART=0, XSAT, XNSAT, XLNB, XEND, XNEND};
	static char *xmltags[]={
		"<?xml","<satellite", "<satellite>"
		"<lnb",
		">", "/>", NULL
	};

	while(!ins.eof()){
		streampos pos = ins.tellg();
		ins >> keybuf;
		n=findkey(keybuf, xmltags);
		if (n<0) {
			ins.seekg(pos);
			cerr << "Unknown tag: " << keybuf << endl;
			break;
		}
		switch(n){
		case XMLSTART:
			cerr << "xml start found" << endl;
			ins.ignore(80,'>');
			break;

		case XNSAT:
		{
			int clnb;
			int lnbid = 5; 
			int satid = -1;

			if (nsat > XML_MAX_SAT) break;
			strcpy(sname,x.lnb_sat.sat_names[nsat]);
			lnbid = x.lnb_sat.diseqc[nsat]; 
			clnb = x.dvb->AddLNB(lnbid, 1, 
				      9750000, 10600000, 
				      11700000, 
				      lnbid, 
				      NOID, NOID);
			
			satid = x.lnb_sat.satid[nsat];
			current_sat =
				x.dvb->AddSat(satid, 
					       lnbid,
					       sname, 
					       10700000 , 
					       12700000);
			nsat++;
			x.read_sat(ins, current_sat);
			break;
		}

		case XSAT:
		{
			x.read_sat(ins, -1);
			break;
		}

		case XLNB:

			break;

		default:
			x.skip_tag(ins,keybuf);
			break;
		}		
	}
	return ins;
}

int get_dvbrc(char *path, DVB &dv, int dev, int len)
{
        ifstream dvbin;

	dvbin.open(path);
	if (!dvbin){
		cerr << "Using default dvbrc." << endl;
		const char *home = getenv("HOME");
		const char *file = ".dvbrc";
		ostrstream str(path,len-1);

		str << home << "/" << file ;
		if (dev)
			str << "." << dev ;
		str << ends;

		dvbin.open(path);
	}
        if (dvbin) {
                dvbin >> dv;
                return 1;
        }

        return 0;
}



