/* Drip - a transcoder for Unix
 * Copyright (C) 2001-2003 Jarl van Katwijk
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "../src/drip.h"
#include "plugin-loader.hh"
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <string.h>


filter filters[MAX_FILTERS]; // list of filters
gint filters_index = 0; // index to filters array
static gdouble* values;

/* Save settings of a plugin */
gboolean save_plugin_settings(gint plugin) {
    GIOChannel *configfile_channel = NULL;
    guint bytessaved;
    gint saveerror;
    gint configfile_fd;
    gint len;
    gboolean result = TRUE;

    /* Fetch values */
    gboolean active = filters[plugin].active;
    gchar* name = filters[plugin].module_name->str;
    GString *HOME = g_string_new(getenv("HOME"));
    GString *configfile = g_string_new("");

    /* Create (and remove old) configuration file */
    (*filters[plugin].module_values_function_get)(values);
    g_string_sprintf(configfile,"%s/.drip/plugins/%s.cfg",HOME->str,name);
    unlink(configfile->str);                                                 // delete old
    configfile_fd = open(configfile->str,O_WRONLY|O_CREAT,S_IREAD|S_IWRITE); // create config file
    if (configfile_fd>-1) {
        configfile_channel = g_io_channel_unix_new(configfile_fd);
        len = strlen(name);
        saveerror = g_io_channel_write(configfile_channel,(gchar*)&len,sizeof(gint),&bytessaved);
        saveerror = g_io_channel_write(configfile_channel,name,len,&bytessaved);
        saveerror = g_io_channel_write(configfile_channel,(gchar*)&active,sizeof(gboolean),&bytessaved);
        for (gint i = 0;i<VALUES;i++) {
            saveerror = g_io_channel_write(configfile_channel,(gchar*)&values[i],sizeof(gdouble),&bytessaved);
        }
        g_io_channel_close(configfile_channel);
        close(configfile_fd);
    } else {
        result = FALSE;
    }
    /* Clean & exit */
    g_string_free(configfile,TRUE);
    g_string_free(HOME,TRUE);
    return result;
}

/* Load settings of plugins */
gboolean load_plugin_settings(gint plugin) {
    GIOChannel *configfile_channel = NULL;
    guint bytesloaded;
    gint loaderror;
    gint configfile_fd;
    gboolean result = TRUE;
    gboolean active;
    gchar* name = filters[plugin].module_name->str;
    gint len;
    gpointer buf = malloc(1024);
    GString *HOME = g_string_new(getenv("HOME"));
    GString *configfile = g_string_new("");

    memset(buf,0,1024);
    /* Open configuration file */
    g_string_sprintf(configfile,"%s/.drip/plugins/%s.cfg",HOME->str,name);
    configfile_fd = open(configfile->str,O_RDONLY); // create config file
    if (configfile_fd>-1) {
        configfile_channel = g_io_channel_unix_new(configfile_fd);
        loaderror = g_io_channel_read(configfile_channel,(gchar*)&len,sizeof(gint),&bytesloaded);
        loaderror = g_io_channel_read(configfile_channel,(gchar*)buf,len,&bytesloaded);
        loaderror = g_io_channel_read(configfile_channel,(gchar*)&active,sizeof(gboolean),&bytesloaded);
        for (gint i = 0;i<VALUES;i++) {
            loaderror = g_io_channel_read(configfile_channel,(gchar*)&values[i],sizeof(gdouble),&bytesloaded);
        }
        g_io_channel_close(configfile_channel);
        close(configfile_fd);
        result = TRUE;
    } else {
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: load plugin settings: config file not found for %s",name);
        active = FALSE;
        for (gint i=0;i<VALUES;i++) {
            values[i] = 0;
        }
        result = FALSE;
    }
    /* Set values into plugin */
    filters[plugin].active = active;
    (*filters[plugin].module_values_function_set)(values);
    /* Clean & exit */
    g_string_free(configfile,TRUE);
    g_string_free(HOME,TRUE);
    free(buf);
    return result;
}

/* Load a symbol out a module */
gboolean load_symbol(GModule *module,gchar *name,gpointer *symbol) {
    static gboolean result;
    result = g_module_symbol(module,name,symbol);
    if (result==FALSE) {
        g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Plugin: loading failed, cant find symbol %s",name);
        return FALSE;
    } else 
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: found symbol %s",name);
    return TRUE;
}

/* Loads a plugin, checks it and updates list-of-loaded plugins */
gboolean load_plugin(gchar* plugin,gchar* directory) {
    GString* symbol_name = g_string_new("");
    gboolean result;
    
    /* Construct main symbol name */
    filters[filters_index].module_name = g_string_new("");
    g_string_sprintf(filters[filters_index].module_name,"drip_%s",plugin);
    /* Load & Scan module */
    filters[filters_index].module_filename = g_module_build_path((const gchar*)directory,(const gchar*)filters[filters_index].module_name->str);
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: loading module %s",filters[filters_index].module_filename);
    filters[filters_index].module = g_module_open((const gchar*)filters[filters_index].module_filename,G_MODULE_BIND_MASK);

    /* Module loaded? */
    if (filters[filters_index].module==NULL) {
        g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Plugin: %s",g_module_error());
        return FALSE;
    }

    /* -- Load symbols -- */
    /* Init Symbol in module? */
    g_string_sprintf(symbol_name,"%s_init",plugin);
    result = load_symbol(filters[filters_index].module,symbol_name->str,(gpointer*)(&filters[filters_index].module_init_function));
    if (result==FALSE) return FALSE;
    /* Default Symbol in module? */
    g_string_sprintf(symbol_name,"%s_default",plugin);
    result = load_symbol(filters[filters_index].module,symbol_name->str,(gpointer*)(&filters[filters_index].module_default_function));
    if (result==FALSE) return FALSE;
    /* Apply Symbol in module? */
    g_string_sprintf(symbol_name,"%s_apply",plugin);
    result = load_symbol(filters[filters_index].module,symbol_name->str,(gpointer*)(&filters[filters_index].module_apply_function));
    if (result==FALSE) return FALSE;
    /* GUI Symbol in module? */
    g_string_sprintf(symbol_name,"%s_gui",plugin);
    result = load_symbol(filters[filters_index].module,symbol_name->str,(gpointer*)(&filters[filters_index].module_gui_function));
    if (result==FALSE) return FALSE;
    /* Apply Symbol in module? */
    g_string_sprintf(symbol_name,"%s_destruct",plugin);
    result = load_symbol(filters[filters_index].module,symbol_name->str,(gpointer*)(&filters[filters_index].module_destruct_function));
    if (result==FALSE) return FALSE;
    /* Type Symbol in module? */
    g_string_sprintf(symbol_name,"%s_type",plugin);
    result = load_symbol(filters[filters_index].module,symbol_name->str,(gpointer*)(&filters[filters_index].module_type_function));
    if (result==FALSE) return FALSE;
    /* Phase Symbol in module? */
    g_string_sprintf(symbol_name,"%s_phase",plugin);
    result = load_symbol(filters[filters_index].module,symbol_name->str,(gpointer*)(&filters[filters_index].module_phase_function));
    if (result==FALSE) return FALSE;
    /* Values Symbol in module? */
    g_string_sprintf(symbol_name,"%s_values",plugin);
    result = load_symbol(filters[filters_index].module,symbol_name->str,(gpointer*)(&filters[filters_index].module_values_function_get));   
    if (result==FALSE) return FALSE;
    /* Values_function Symbol in module? */
    g_string_sprintf(symbol_name,"%s_values_function",plugin);
    result = load_symbol(filters[filters_index].module,symbol_name->str,(gpointer*)(&filters[filters_index].module_values_function_set));
    if (result==FALSE) return FALSE;


    /* Fill in filter info */
    filters[filters_index].loaded = TRUE;
    filters[filters_index].active = FALSE; // TODO: Set to TRUE based on configuration\plugin dialogs window
    filters[filters_index].module_GUI = NULL;
    filters[filters_index].module_type = (*filters[filters_index].module_type_function)();
    filters[filters_index].module_phase = (*filters[filters_index].module_phase_function)();

    /* Update filters_index */
    filters_index++;

    /* Clean & Exit */
    g_string_free(symbol_name,TRUE);
    return result;
}


/* Close all plugins (save their config, and call module_destruct_function) */
#ifdef __cplusplus
extern "C" {
#endif
void /*gboolean*/ close_plugins(void) {
    gboolean result = TRUE;
    gint i;
    
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Plugin: closing plugins");
    /* Destruct plugins (NOT unload them) */
    for (i=0;i<filters_index;i++) {
        result = result & save_plugin_settings(i);
        (*filters[i].module_destruct_function)();
        result = result & save_plugin_settings(i);
        
    }
 
    if (result==TRUE) {
        g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Closed %i plugins succesfully",i);
    } else { 
        g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Closing plugins failed, read log for details");
    }
    
    free(values);

    return; // result;
}   
#ifdef __cplusplus
}   
#endif


/* Initialize and load plugin environment */
#ifdef __cplusplus
extern "C" {
#endif
gboolean init_plugins(void) {
    gboolean result,loaded;
    GString *location = g_string_new("");
    /* Can we load plugins at all? */
    result = g_module_supported();
    if (result == FALSE) {
        return FALSE;
    }

    values = (gdouble*)malloc(VALUES*sizeof(gdouble));

    /* --- Load the filters
       TODO: scan lib drip for plugins, dont use hardcoded table */
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: loading plugins");
    g_string_sprintf(location,"%s/lib",PREFIX);
    /* Video filters */
    result = result & load_plugin("colourfilter",location->str);
    result = result & load_plugin("logofilter",location->str);
    result = result & load_plugin("bwfilter",location->str);
    result = result & load_plugin("magickfilter",location->str);
    /* Audio filters */
    result = result & load_plugin("crystalityfilter",location->str);
    result = result & load_plugin("volumefilter",location->str);
    /* Various \ Testing */
    #ifdef DEBUG
    result = result & load_plugin("noisefilter",location->str);
    result = result & load_plugin("testfilter",location->str);
    result = result & load_plugin("cleanfilter",location->str);
    #endif


    /* Load settings */
    for (gint i=0;i<filters_index;i++) {
        loaded = load_plugin_settings(i);
        if (!loaded) {
            (*filters[i].module_default_function)();
        }
    }
    /* --- */
    if (result==TRUE) {
        g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Loaded %i Drip plugins",filters_index);
    } else {
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Loading plugins failed, read log for details");
    }
    /* --- */
    g_string_free(location,TRUE);
    return result;
}
#ifdef __cplusplus
}
#endif

