/* Drip - a transcoder for Unix
 * Copyright (C) 2001-2003 Jarl van Katwijk
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/* Mpeg decoding */
#include "demuxer.hh"
#include "encoder.hh"
#include "a52dec/a52.h"
/* SPU */
#include "../libspu/spu.h"
/* Other */
#include "plugin-loader.hh"
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <math.h>
#include <pthread.h>
#include "fast_memcpy.hh"
#include "../src/external.h"
#include <mad.h>




/* Defines */
#define DEMUX_BUFFER_SIZE 2*1048576       // 1Mb
#define AUDIO_BUFFER_SIZE 60  *44100*4  // 30 seconds buffer, 5Mb
#define PTSMAX 8192

// loggin
//#define DEBUG
#ifdef DEBUG
#warning         -------------------------------------
#warning
#warning
#warning
#warning
#warning
#warning         DEBUGGING WILL PRODUCE HUGE LOG FILES
#warning
#warning
#warning
#warning
#warning
#warning         -------------------------------------
#endif
/* writeout audio channel 0 to /tmp/writeout_audio.raw */
//#define WRITEOUT_AUDIO

/* Globals */
static mpeg2dec_t *mpeg2dec;
static const mpeg2_info_t *mpeg2info;
static mpeg_read_buffer_t read_buffer;
static gboolean mpeg2_reader_handler_active = FALSE;
static gint mpeg2_reader_handler_pid;
static FILE *in_file;
static gint width = -1,height=-1;
static a52_state_t *output_liba52[2];
static glong total_frames = 0;
gboolean demuxing = FALSE;
static gboolean process = FALSE;
static void(*processor)(guint8*[3],uint32_t pts,gint pts_audio);
static gint16 gint16_samples[AUDIO_BUFFER_SIZE];
static gint16 *audio_buffer[2];
static gint audio_buffer_index[2];
static gint audio_buffer_base[2];
static gint audio_buffer_total[2];
static gint audio_fetched_total[2];
static gint audio_sample_rate,audio_bit_rate,audio_channels;
static guint8 _decode_buffer[1024*768*3];
static guint8 *decode_buffer[3];
static pthread_t mpeg2_reader;
glong SCRstream, SCRvideo, PTSprivate;
glong SCRstream_prev = 0;
static glong IDnumber[2];
static gint PTSindex[2];
gint PTSvideo = -1;
gint PTSaudio = -1;
gdouble PTSframe; // Here PTS value of latest video frame is stored.. or should be, not 100% garanteed for now, couse libmpeg2 cant supply the correct value
gboolean Audio_Cut = FALSE;
static gint16 silence[AUDIO_BUFFER_SIZE];
static gint16 *audio_buffer_chunk;
static gdouble AUDIO_BYTES_PER_FRAME = -1;
static gdouble PTS_PER_AUDIO_FRAME = 480;
static gdouble PTS_PER_AUDIO_BYTE = PTS_PER_AUDIO_FRAME/1024;
static gboolean non_silence[2];
gchar **input_files;
gint input_files_index = 0;
gint spu_track;
gint audio_track;
glong avifile_audio_outstream_bytes = 0;
gdouble avifile_audio_outstream_bytes_calculated = 0;
/* A52 vari's */
static guint8 *a52_buffer[2];
static guint8 *bufptr[2];
static guint8 *bufpos[2];
static gdouble PTSa52[2];
static guint8* start[2];
static gint avifile_audio_total_offset[2];
static gboolean offset_corrected[2];
static gboolean decoding_audio = FALSE;
static glong PTSstored[2];
mpeg_read_buffer_t *mad_buffer;
#ifdef WRITEOUT_AUDIO
gint writeout_audio_fd;
#endif

/* ----------- Functions -- */


/* Return audio samplerate */
gint drip_audio_samplerate(void) {
    #ifdef DEBUG
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"drip_audio_samplerate = %i",audio_sample_rate);
    #endif
    switch (audio_sample_rate) {
        case 0:  return 48000;
                 break;
        default: return audio_sample_rate;
                 break;
    }
}


/* Return audio bitrate */
gint drip_audio_bitrate(void) {
    return audio_bit_rate;
}


/* Return audio bitrate */
gint drip_audio_channels(void) {
    return audio_channels;
}


/* Return width of current open mpeg */
gint drip_mpeg_width(void) {
    gint w = -1;
    if (mpeg2info)
        w = mpeg2info->sequence->width;
    return w;
}


/* Return height of current open mpeg */
gint drip_mpeg_height(void) {
    gint h = -1;
    if (mpeg2info)
        h = mpeg2info->sequence->height;
    return h;
}


/* Determine framerate out frame_rate_code */
gdouble drip_framerate() {
    static gdouble framerate = (gdouble)((gdouble)27000000 / (gdouble)mpeg2info->sequence->frame_period);
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Video framerate = %f",framerate);
    return framerate;
}


/* Return TimeStamp value per video frame */
gdouble TimeStamp_Frame(void) {
    static gdouble TF;
    TF = 90000/drip_framerate();
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"TimeStamp_Frame: TF=%f  (FR=%f)",TF,drip_framerate());
    return TF;
}


/* Return PTS correction for video frame position calculation. Audio is encoded in multiples of 4,
    so it's a little higher per frame. This correction value compensates (divide Video PTS with this) */
gdouble drip_PTS_correction(void) {
    gdouble PTSvideo_correction;
    PTSvideo_correction = ((ceil((gdouble)drip_audio_samplerate()/(gdouble)drip_framerate()) * 4)/4*drip_framerate())/(gdouble)drip_audio_samplerate();
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"PTSvideo_correction = %f",PTSvideo_correction);
    return PTSvideo_correction;
}


/* Seek to percentage in mpeg file */
gint drip_mpeg_seek_percentage(gdouble perc) {
    gint result,pos;
    /* Check perc */
    if (perc<0 || perc>1) {
        perc = 0;
    }
    /* Calc position in bytes from perc */
    result = fseek(in_file,0,SEEK_END);
    pos = (gint)(ftell(in_file) * perc);
    /* Seek to pos */
    result = fseek(in_file,pos,SEEK_SET);
    return result;
}


/* float_to_int */
static inline gint16 convert(int32_t i) {
    if (i>0x43c07fff)
        return 32767;
    else if (i<0x43bf8000)
        return -32768;
    else
        return i-0x43c00000;
}


/* Typecast floats to int16s */
static inline gint float_to_int_stereo(float *_f, gint16 *s16) {
    guint i;
    int32_t *f = (int32_t*)_f;       /* XXX assumes IEEE float format */
    for (i=0; i<256; i++) {
        s16[i*2] = convert(f[i]);
        s16[i*2+1] = convert(f[i+256]);
    }
    return 1024;//256*2*(sizeof(gint16));
}


/* Reset audio buffer */
inline void reset_audio_buffer(gint channel) {
    audio_buffer_index[channel] = 0;
    PTSindex[channel] = 0;
    non_silence[channel] = FALSE;
    return;
}


/* Debug audio buffer, display status */
inline void audio_buffer_status(gint channel) {
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Audio Buffer: index=%i    base=%i   fetched=%i    stored=%i    bufferd=%i",audio_buffer_index[channel],audio_buffer_base[channel],audio_fetched_total[channel],audio_buffer_total[channel],audio_buffer_total[channel]-audio_fetched_total[channel]);
}


/* Returns audio buffer and resets index */
gint16* fetch_from_audio_buffer(gint samples, gint SCR,gint channel) {
    gint chunk_size = samples; // TODO: 2->channels

    /* Anything in buffer? */
    if ((audio_buffer_index[channel]==0)) {
        #ifdef DEBUG 
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Audio Fetch buffer empty (%i audio frames)",chunk_size,PTSindex);
        #endif
        return NULL;
    }

    /* Enough in buffer? */
    if (audio_buffer_index[channel]<chunk_size) {
        #ifdef DEBUG
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Audio Fetch buffer holds %ibytes, could not release %ibytes",audio_buffer_index[channel],chunk_size);
        #endif
        return NULL;
    }
    /* Moronic chuck_size? */
    if (chunk_size<1) {
        g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Audio Fetch buffer request for %ib canceled",chunk_size);
        return NULL;
    }

    #ifndef STANDALONE
    drip_lock("fetch_from_audio_buffer1");
    #endif

    /* Keep track total bytes put here */
    audio_fetched_total[channel]+=chunk_size;

    /* Update buffer */
    audio_buffer_index[channel] -= chunk_size;
    if(audio_buffer_index[channel]<0) {
        audio_fetched_total[channel] += audio_buffer_index[channel];
        audio_buffer_index[channel]=0;
        #ifdef DEBUG
        g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Audio Fetch buffer underflow");
        #endif
    }
    #ifdef DEBUG
    audio_buffer_status(channel);
    #endif

    /* Copy audio data into return pointer */
    audio_buffer_chunk = (gint16*)((glong)audio_buffer[channel]+(glong)audio_buffer_base[channel]);
    /* Update buffer base (offset) */
    audio_buffer_base[channel] += chunk_size;

    #ifndef STANDALONE
    drip_unlock("fetch_from_audio_buffer1");
    #endif
    #ifdef DEBUG
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Audio Fetch buffer returned chunk");
    #endif
    return audio_buffer_chunk;
}


/* Builds audio buffer for encoder */
void add_to_audio_buffer(gint16* audio_chunk, gint audio_chunk_size, gint PTS, gint channel) {
    static gint16* audio_buffer_dest;
    static gint i;
    static gdouble PTSoff[2];
    static guint8** chunk = NULL;
    glong audio_chunk_extra;
    gint16* audio_chunk_pointer;

    #ifdef DEBUG
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Audio Add buffer new sample, channel %i, PTSstored = %li,   PTS = %li,   sample bytes = %i",channel,PTSstored[channel],PTS,audio_chunk_size);
    #endif

    /* alloc chunk */
    audio_chunk_pointer = audio_chunk;
    if (chunk == NULL) {
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Audio Add buffer new sample, allocated chunk");
        chunk = (guint8**)malloc(3*sizeof(guint8*));
        for (i=0;i<AMAX;i++) {
            PTSoff[i] = 0;
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Audio Add buffer, allocated",i);
        }
        audio_buffer_chunk = (gint16*)malloc(2048);
    }

    /* Process audio data by the filters */
    chunk[0] = (guint8*)audio_chunk;
    chunk[1] = (guint8*)NULL;
    chunk[2] = (guint8*)NULL;
    /* - Plugin Filters, type AUDIO PRE - */
    for (gint i=0;i<filters_index;i++) {
        if (filters[i].active==TRUE & filters[i].module_type==AUDIO && filters[i].module_phase==PRE) {
            chunk = (*filters[i].module_apply_function)(chunk,audio_chunk_size,(gulong)SCRframe,PTS);
        }
    }
    /* - Plugin Filters, type AUDIO NORMAL - */
    for (gint i=0;i<filters_index;i++) {
        if (filters[i].active==TRUE & filters[i].module_type==AUDIO && filters[i].module_phase==NORMAL) {
            chunk = (*filters[i].module_apply_function)(chunk,audio_chunk_size,(gulong)SCRframe,PTS);
        }
    }
    /* - Plugin Filters, type AUDIO POST - */
    for (gint i=0;i<filters_index;i++) {
        if (filters[i].active==TRUE & filters[i].module_type==AUDIO && filters[i].module_phase==POST) {
            chunk = (*filters[i].module_apply_function)(chunk,audio_chunk_size,(gulong)SCRframe,PTS);
        }
    }

    #ifndef STANDALONE
    drip_lock("add_to_audio_buffer1");
    #endif
    #ifdef DEBUG
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"add_to_audio_buffer1, %li bytes audio sample with PTS %li",audio_chunk_size,PTS);
    #endif

    /* Did a plugin allocated new memory? */
    if (audio_chunk != (gint16*)chunk[0]) {
        free(audio_chunk);
        audio_chunk = (gint16*)chunk[0];
    }

    /* Audio does not start behind video? */
    if ( !(PTSstored[channel] > PTSvideo) ) {
        Audio_Cut = TRUE;
    }

    /* Less stored as needed? */
    if ( PTSvideo!=-1 && ( PTS > PTSstored[channel] ||
                           ( PTSstored[channel] > PTSvideo
                             && Audio_Cut == FALSE
                           )
                         )
       ) {
        /* Add silence chunk */
        g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Audio Add buffer , PTS = %li, PTSstored[channel] = %li, PTSvideo = %li",PTS,PTSstored[channel],PTSvideo);
        /* Which gap are we filling: PTS > PTSstored[channel] of PTSstored[channel] > PTSvideo */
        if (PTS > PTSstored[channel]) {
            /* PTS > PTSstored[channel] */
            audio_chunk_extra = (glong)((PTS-PTSstored[channel])/PTS_PER_AUDIO_BYTE);
        } else {
            /* PTSstored[channel] > PTSvideo */
            audio_chunk_extra = (glong)((PTSstored[channel]-PTSvideo)/PTS_PER_AUDIO_BYTE);
        }
        if (audio_chunk_extra > AUDIO_BUFFER_SIZE) { audio_chunk_extra = AUDIO_BUFFER_SIZE; }
        audio_chunk_extra = ((glong)(audio_chunk_extra/4))*4; // round to 4 (sterio 16bit)
        if ((audio_chunk_size + audio_chunk_extra) > AUDIO_BUFFER_SIZE) {
            g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Audio Add buffer extra reduced in size due to buffer size limit");
            audio_chunk_extra = (((AUDIO_BUFFER_SIZE-audio_chunk_size)/4)*4);
        } else Audio_Cut = TRUE;
        if(audio_chunk_extra>0) {
                /*memmove*/fast_memcpy((gint16*)((glong)audio_chunk+audio_chunk_extra),audio_chunk,audio_chunk_size);
                memset(audio_chunk,0,audio_chunk_extra);
                audio_chunk_size += audio_chunk_extra;
                g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Audio Add buffer added %li extra bytes",audio_chunk_extra);
        } else if (audio_chunk_extra<0) {
                g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Audio Add buffer couldn't add silence, too large (%libytes)",audio_chunk_extra);
                PTSstored[channel] = PTS;
                g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Audio Add buffer resetted PTS to %li for channel %i",PTSstored[channel],channel);
        }
    } 

    /* Track stored audio */
    #ifdef DEBUG
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Audio Add buffer, PTSstored = %li, new PTSstored = %li",PTSstored[channel],PTSstored[channel]+PTS_PER_AUDIO_BYTE*audio_chunk_size);
    #endif
    PTSstored[channel] += (glong)(PTS_PER_AUDIO_BYTE*audio_chunk_size);

    /* Keep track total bytes put here */
    audio_buffer_total[channel]+=audio_chunk_size;
    /* 1st time here? */
    if (audio_buffer[channel]==NULL) {
        audio_buffer[channel] = (gint16*)malloc(AUDIO_BUFFER_SIZE);
    }
    /* Space left in buffer? */
    if (((glong)audio_buffer_index[channel]+(glong)audio_chunk_size) > (AUDIO_BUFFER_SIZE-1)) {
        #ifdef DEBUG
        g_log(DRIP_CB_LD,G_LOG_LEVEL_WARNING,"Audio Add buffer overflow, resetting audio buffer");
        #endif
        #ifndef STANDALONE
        drip_unlock("add_to_audio_buffer1");
        #endif
        reset_audio_buffer(channel);
        return;
    }

    /* Need to rotate buffer? */
    if (((glong)audio_buffer_index[channel]+(glong)audio_chunk_size+(glong)audio_buffer_base[channel]) > (AUDIO_BUFFER_SIZE-1)) {
        #ifdef DEBUG
        g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Audio Add buffer rotating buffer");
        #endif
        /*memmove*/fast_memcpy(audio_buffer[channel],(gint16*)((glong)audio_buffer[channel]+(glong)audio_buffer_base[channel]),(glong)audio_buffer_index[channel]);
        audio_buffer_base[channel] = 0;
    }

    /* Copy audio chunk into buffer and update index */
    audio_buffer_dest = (gint16*)((glong)audio_buffer[channel] + (glong)audio_buffer_base[channel] + (glong)audio_buffer_index[channel]);
    #ifdef DEBUG
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Audio Add buffer : adding new audio to buffer (dest=%p, source=%p, size=%li)",audio_buffer_dest,audio_chunk,audio_chunk_size);
    #endif
    fast_memcpy(audio_buffer_dest,audio_chunk,(size_t)audio_chunk_size);
    audio_buffer_index[channel] += audio_chunk_size;

    #ifdef DEBUG
    audio_buffer_status(channel);
    audio_chunk = audio_chunk_pointer;
    #endif
    #ifndef STANDALONE
    drip_unlock("add_to_audio_buffer1");
    #endif

    return;
}


/* Add some silence to audio fifo buffer, it will be in sync with given PTS after adding */
void add_silence_to_audio_buffer(gint PTS,gboolean frame) {
    gint frame_size;

    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG," add_silence_to_audio_buffer, PTS=%li, frame = %li\n",PTS,frame);
    /* Add silence to audio buffer(s) to fill gaps when the video buffer is full */

    /* Calc AUDIO_BYTES_PER_FRAME */
    if (AUDIO_BYTES_PER_FRAME==-1) {
        AUDIO_BYTES_PER_FRAME = ceil((gdouble)drip_audio_samplerate()/(gdouble)drip_framerate()) * 4;
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Output_audio(1): AUDIO_BYTES_PER_FRAME = %f  (drip_audio_samplerate=%i, drip_mpeg_framerate = %f, PTSvideo_correction=%f)",AUDIO_BYTES_PER_FRAME,drip_audio_samplerate(),drip_framerate(),drip_PTS_correction());
    }
    /* Add a frame of silence , or are we just syncing? */
    if (frame) {
        frame_size = (gint)AUDIO_BYTES_PER_FRAME;
    } else {
        frame_size = 0;
    }
    add_to_audio_buffer(silence, frame_size, PTS, 0);
    if (Config.anum == 2) { add_to_audio_buffer(silence, frame_size, PTS, 1); }

    return;
}


/* Add audio data to output: here an attempt to writeout to Aviencoder is done,
   available audio and video are checked before writing */
gboolean output_audio(glong frame) {
    static gdouble SCR = -1;
    gint16* audio_chunk;
    static gdouble result;
    gint audio_bytes_to_fetch;

    if (demuxing == FALSE) return FALSE;

    if (encode_frame(NULL,NULL,(gdouble)0,FALSE,TRUE,PTSaudio) == FALSE) {
        return FALSE;
    } else {
        #ifdef DEBUG
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Output_audio: Video buffer has data, continueing audio handling");
        #endif
    }

    /* Init SCR & calc AUDIO_BYTES_PER_FRAME */
    if (SCR==-1) {
        if (PTSaudio!=-1) {
            SCR = (gdouble)PTSaudio;
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Output_audio: SCR initialised to %f",SCR);
        } else {
            #ifdef DEBUG
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Output_audio: PTSaudio=-1, cancelling");
            #endif
            return FALSE;
        }
    }

    /* Calc AUDIO_BYTES_PER_FRAME */
    if (AUDIO_BYTES_PER_FRAME==-1) {
        AUDIO_BYTES_PER_FRAME = ceil((gdouble)drip_audio_samplerate()/(gdouble)drip_framerate()) * 4;
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Output_audio(1): AUDIO_BYTES_PER_FRAME = %f  (drip_audio_samplerate=%i, drip_mpeg_framerate = %f, PTSvideo_correction=%f)",AUDIO_BYTES_PER_FRAME,drip_audio_samplerate(),drip_framerate(),drip_PTS_correction());
    }

    /* Output audio till it's in sync with video */
    /* Amount of audio bytes = rounded UP */
    audio_buffer_filled = FALSE;
    avifile_audio_outstream_bytes_calculated += AUDIO_BYTES_PER_FRAME;
    audio_bytes_to_fetch = (gint)ceil(avifile_audio_outstream_bytes_calculated - (gdouble)avifile_audio_outstream_bytes);
    audio_bytes_to_fetch = (gint)ceil(((gdouble)audio_bytes_to_fetch)/2);
    audio_bytes_to_fetch = audio_bytes_to_fetch << 1;

    result = 0;
    audio_chunk = fetch_from_audio_buffer(audio_bytes_to_fetch,(gint)SCRframe,0);
    if (audio_chunk != NULL) {
        decoding_audio = TRUE;
        #ifdef DEBUG
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Output_audio: Encoding audio frame with audio_bytes_to_fetch = %li (chunk=%p)",audio_bytes_to_fetch,audio_chunk);
        #endif
        avifile_audio_outstream[0]->AddData(audio_chunk,(gint)audio_bytes_to_fetch/2);
        if (Config.anum == 2) { // 2nd encoded audio channel
            #ifdef DEBUG
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Output_audio: Fetching audio for 2nd channel");
            #endif
            audio_chunk = fetch_from_audio_buffer(audio_bytes_to_fetch,(gint)SCRframe,1);
            avifile_audio_outstream[1]->AddData(audio_chunk,(gint)audio_bytes_to_fetch/2);
        }
        avifile_audio_outstream_bytes += audio_bytes_to_fetch;
        audio_buffer_filled = TRUE;
        result = encode_frame(stream,im,PTSframe,TRUE,FALSE,PTSaudio);
        if (result != FALSE) {
            SCR += PTS_PER_AUDIO_BYTE*AUDIO_BYTES_PER_FRAME;
        } else {
            #ifdef DEBUG
            g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Output_audio: Encoding frame failed..");
            #endif
            return FALSE;
        }
        decoding_audio = FALSE;
    } else {
        /* Got a video frame, but NO audio chunk */
        avifile_audio_outstream_bytes_calculated -= AUDIO_BYTES_PER_FRAME;
        #ifdef DEBUG
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Output_audio: NO audio chunk available");
        #endif
        return FALSE;
    }

    /* Update, clean & exit */
    return TRUE;
}


/* Handles decoding of a52 audio blocks */
void a52_decode_data(guint8 *data, guint8 *end, glong PTS, glong PTSpos, gint channel) {
    gint flags;
    sample_t level; // Volume adjustment, TODO less for 5.1?
    sample_t bias;
    gint i;
    gint pcm_size;
    gint len,len_max;
    gint length;

    #ifdef DEBUG
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"a52_decode_data, channel = %i, PTS = %li, PTSpos = %li (%li)",channel,PTS,PTSpos,(glong)start[channel]-(glong)data);
    #endif

    start[channel] = data;
    while (1) { 
        len = end - start[channel];
        if (!len)  break; /* Exit decode loop */
        len_max = bufpos[channel] - bufptr[channel];
        if (len > len_max)  len = len_max;
        fast_memcpy(bufptr[channel],start[channel],len);
        bufptr[channel] += len;
        start[channel] += len;

        if ((PTS!=-1) && (((glong)start[channel]-(glong)data)>PTSpos)) {
            /* Only set the PTS once we've hit the magic pointer */
            #ifdef DEBUG
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"a52 PTSa52 set to %i  (PTSstored[0]=%li)",PTS,PTSstored[0]);
            #endif
            PTSa52[channel] = PTS;
            PTS = -1;
        }

        if (bufptr[channel] == bufpos[channel]) {
            if (bufpos[channel] == &a52_buffer[channel][7]) {
                length = a52_syncinfo(a52_buffer[channel],&flags,&audio_sample_rate,&audio_bit_rate);
                if (!length) {
                    for (bufptr[channel] = a52_buffer[channel];  bufptr[channel] < &a52_buffer[channel][6];  bufptr[channel]++) {
                        bufptr[channel][0] = bufptr[channel][1];
                    }
                    continue;
                }
                bufpos[channel] = &a52_buffer[channel][length];
            } else {
                flags = A52_ADJUST_LEVEL | A52_STEREO;
                level = 1.0; // Volume adjustment, TODO less for 5.1?
                bias = 384.0;
                if (a52_frame(output_liba52[channel],a52_buffer[channel],&flags,(sample_t*)&level,bias)) {
                    g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Audio decoder: Channel %i, Error in a52_frame",channel);
                    goto error;
                }
                flags &= A52_CHANNEL_MASK;
                audio_channels = 2; /* flags[channel] = A52_ADJUST_LEVEL | A52_STEREO */
                /* Decode audio */
                if (PTSa52[channel]>-1) {
                    for (i=0; i<6; i++) {
                        if (a52_block(output_liba52[channel])) {
                            g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Audio decoder: Channel %i, Error in a52_block",channel);
                            goto error;
                        }
                        pcm_size = float_to_int_stereo(a52_samples(output_liba52[channel]),gint16_samples);
                        #ifdef WRITEOUT_AUDIO
                        write(writeout_audio_fd,gint16_samples,pcm_size);
                        #endif
                        add_to_audio_buffer(gint16_samples,pcm_size,(gint)PTSa52[channel],channel);
                        PTSa52[channel] += PTS_PER_AUDIO_FRAME;
                    }
                } else {
                    #ifdef DEBUG
                    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"a52 block not added, PTS still set to -1");
                    #endif
                }
                bufptr[channel] = &a52_buffer[channel][0];
                bufpos[channel] = &a52_buffer[channel][7];
                continue;
            error:
                bufptr[channel] = &a52_buffer[channel][0];
                bufpos[channel] = &a52_buffer[channel][7];
            }
        }
    }
    return;
}


/* THREAD for reading mpeg2 data ; this way IO and encoding could be parallel if threads supported ?? */
gpointer mpeg2_reader_handler(gpointer data) {
    size_t rbuf,rbuf2;
    gint read_size,i;

    while (mpeg2_reader_handler_active == TRUE) { usleep(100);} //poll (...);
    mpeg2_reader_handler_active = TRUE;
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"mpeg2_reader_handler spawned");
    /* Reset buffer */
    read_buffer.lenght = 0;
    mpeg2_reader_handler_pid = getpid();
    do {
        #ifndef STANDALONE
        drip_lock("mpeg2_reader_handler1");
        #endif
        rbuf = fread(read_buffer.buffer, 1, DEMUX_BUFFER_SIZE, in_file);
        #ifndef STANDALONE
        drip_unlock("mpeg2_reader_handler1");
        #endif
        read_buffer.lenght = rbuf;
        
        if (rbuf != ((glong)DEMUX_BUFFER_SIZE) && input_files[input_files_index]!=NULL) {
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Could only read %lu bytes, opening next file |%s|",rbuf,input_files[input_files_index]);
            #ifndef STANDALONE
            drip_lock("mpeg2_reader_handler2");
            #endif
            fclose(in_file);
            in_file = fopen(input_files[input_files_index],"rb");
            #ifndef STANDALONE
            drip_unlock("mpeg2_reader_handler2");
            #endif
            if (!in_file) {
                g_log(DRIP_LD,G_LOG_LEVEL_ERROR,"%s - couldnt open file |%s|", strerror(errno),input_files[input_files_index]);
            } else {
                g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Opened %s for input",input_files[input_files_index]);
            }
            input_files_index++;
            read_size = ((glong)DEMUX_BUFFER_SIZE) - rbuf;
            #ifndef STANDALONE
            drip_lock("mpeg2_reader_handler3");
            #endif
            rbuf2 = fread(read_buffer.buffer+rbuf, 1, read_size, in_file);
            #ifndef STANDALONE
            drip_unlock("mpeg2_reader_handler3");
            #endif
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Read left %lu bytes (%u total in buffer)",rbuf2,rbuf+rbuf2);
            rbuf += rbuf2;
            read_buffer.lenght = rbuf;
        }
        /* flip buffer */ 
    } while ( (demuxing==TRUE) && (quit==FALSE) ); 
    /* Clean & Close */
    mpeg2_reader_handler_active = FALSE;
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"mpeg2_reader_handler exiting");
    pthread_exit(0);
    return NULL;
}


/* Parse mpeg2 data */
gboolean drip_parse_mpeg2(guint8* tmp1,guint8* tmp2) {
    gint state;
    guint8* mpeg2buf[3];
    gboolean done = FALSE;

    mpeg2_buffer(mpeg2dec, tmp1, tmp2);
    mpeg2info = mpeg2_info(mpeg2dec);

    while (!done) {
        state = mpeg2_parse(mpeg2dec);
        switch(state) {
        case -1:
            done = TRUE;
            break;
        case STATE_SEQUENCE:
            /* Mpeg2 sequence start */
            break;
        case STATE_PICTURE:
            /* Mpeg2 new frame start */
        case STATE_PICTURE_2ND:
            /* should not do anything */
            break;
        case STATE_SLICE:
        case STATE_END:
            /* Mpeg2 frame ready, feed to Drip frame processor */
            if (mpeg2info->display_fbuf) {
                mpeg2buf[0] = mpeg2info->display_fbuf->buf[0];
                mpeg2buf[1] = (guint8*)((gint)mpeg2info->display_fbuf->buf[0] + drip_mpeg_width()*drip_mpeg_height());
                mpeg2buf[2] = (guint8*)((gint)mpeg2info->display_fbuf->buf[0] + drip_mpeg_width()*drip_mpeg_height()*5/4);
                /* Call current processor (autoclipper, frame fifo, etc) */
                processor(mpeg2buf,(uint32_t)PTSframe,PTSaudio);
            }
            break;
        }
    }

    /* Clean & Exit */
    return !done;
}

static inline signed int scale(mad_fixed_t sample) {
    /* round */
    sample += (1L << (MAD_F_FRACBITS - 16));
    /* clip */
    if (sample >= MAD_F_ONE)
        sample = MAD_F_ONE - 1;
    else if (sample < -MAD_F_ONE)
        sample = -MAD_F_ONE;
    /* quantize */
    return sample >> (MAD_F_FRACBITS + 1 - 16);
}


/* MAD lib handler for input event */
enum mad_flow mad_input(void *data, struct mad_stream *stream) {
    mpeg_read_buffer_t *buffer = (mpeg_read_buffer_t*)data;

    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"MAD decoding requested new input data (buffer=%p, len=%li)",buffer,buffer->lenght);
    // TODO : fetch here from a special MP3 buffer, which should be filled up by the demuxer

    /* Stop decoding stream? */
    if (buffer->lenght == -1) {
        g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"MAD decoding STOPPING");
        return MAD_FLOW_STOP;
    }
    /* Wait for data? */
    while (buffer->buffer == NULL || buffer->lenght < 1) {
        // TODO replace by mutex locking..
        g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Waiting for mad mp3 data (length=%li,%p, start=%p)",buffer->lenght,&(buffer->lenght),buffer->buffer);
        usleep(10);
    }
    /* Feed mp3 data to decoder */
    #ifndef STANDALONE
    drip_lock("mad_flow1");
    #endif
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"MAD decoding MP3 data (%li bytes)",buffer->lenght);
    mad_stream_buffer(stream,(const unsigned char*)buffer->buffer,buffer->lenght);
    buffer->lenght = 0;
    #ifndef STANDALONE
    drip_unlock("mad_flow1");
    #endif

    /* Return continue */
    return MAD_FLOW_CONTINUE;
}

/* MAD lib handler for output event */
static enum mad_flow mad_output(void *data,struct mad_header const *header,struct mad_pcm *pcm) {
    guint nchannels, nsamples;
    mad_fixed_t const *left_ch, *right_ch;

    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"MAD decoding has new audio decoded");
    // TODO add decoded audio to drip-audio-fifo
    // TODO figure out format returned by MAD

    /* pcm->samplerate contains the sampling frequency */
    nchannels = pcm->channels;
    nsamples  = pcm->length;
    left_ch   = pcm->samples[0];
    right_ch  = pcm->samples[1];
    while (nsamples--) {
        signed int sample;
        /* output sample(s) in 16-bit signed little-endian PCM */
        sample = scale(*left_ch++);
        putchar((sample >> 0) & 0xff);
        putchar((sample >> 8) & 0xff);
        if (nchannels == 2) {
            sample = scale(*right_ch++);
            putchar((sample >> 0) & 0xff);
            putchar((sample >> 8) & 0xff);
        }
    }

    return MAD_FLOW_CONTINUE;
}

static enum mad_flow mad_error(void *data,struct mad_stream *stream,struct mad_frame *frame) {
    mpeg_read_buffer_t *buffer = (mpeg_read_buffer_t*)data;
    g_log(DRIP_CB_LD,G_LOG_LEVEL_WARNING,"MAD decoding error 0x%04x (%s) at byte offset %u\n",stream->error, mad_stream_errorstr(stream),stream->this_frame - (const unsigned char*)buffer->buffer);
    return MAD_FLOW_BREAK;
}



/* Demux mpeg stream */
gpointer drip_demux(gpointer data, gint** audio_channel) {
    size_t rbuf2;
    gint read_size,i;
    static gint mpeg1_skip_table[16] = {
             1, 0xffff,      5,     10, 0xffff, 0xffff, 0xffff, 0xffff,
        0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff
    };
    #define DEMUX_HEADER 0
    #define DEMUX_DATA 1
    #define DEMUX_SKIP 2
    guint8 *buf;
    size_t rbuf; 
    guint8 *end = NULL;
    guint8 *tmp1 = NULL;
    guint8 *tmp2;
    gint complain_loudly;
    gint len;
    guint packetlen;
    guint a1, a2;
    glong PTS, DTS, SCR;
    glong buf_previous = 0;
    gint audio_len;
    gint audio_bytes;
    gboolean result;
    struct mad_decoder _mad_decoder;
    mad_decoder *mad_decoder_ = &_mad_decoder;
    static gboolean mp3first = FALSE;


    #ifdef DEBUG
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Demuxer: Audio channel 0 = %i, 1 = %i,  Config.anum = %i",*audio_channel[0],*audio_channel[1],Config.anum);
    #endif
    complain_loudly = 1;
    buf = read_buffer.buffer;

    do {
        #ifndef STANDALONE
        drip_lock("mpeg2_reader_handler1");
        #endif
        rbuf = fread(read_buffer.buffer, 1, DEMUX_BUFFER_SIZE, in_file);
        #ifndef STANDALONE
        drip_unlock("mpeg2_reader_handler1");
        #endif
        read_buffer.lenght = rbuf;

        if (rbuf != ((glong)DEMUX_BUFFER_SIZE) && input_files[input_files_index]!=NULL) {
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Could only read %lu bytes, opening next file |%s|",rbuf,input_files[input_files_index]);
            #ifndef STANDALONE
            drip_lock("mpeg2_reader_handler2");
            #endif
            fclose(in_file);
            in_file = fopen(input_files[input_files_index],"rb");
            #ifndef STANDALONE
            drip_unlock("mpeg2_reader_handler2");
            #endif
            if (!in_file) {
                g_log(DRIP_LD,G_LOG_LEVEL_ERROR,"%s - couldnt open file |%s|", strerror(errno),input_files[input_files_index]);
            } else {
                g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Opened %s for input",input_files[input_files_index]);
            }
            input_files_index++;
            read_size = ((glong)DEMUX_BUFFER_SIZE) - rbuf;
            #ifndef STANDALONE
            drip_lock("mpeg2_reader_handler3");
            #endif
            rbuf2 = fread(read_buffer.buffer+rbuf, 1, read_size, in_file);
            #ifndef STANDALONE
            drip_unlock("mpeg2_reader_handler3");
            #endif
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Read left %lu bytes (%u total in buffer)",rbuf2,rbuf+rbuf2);
            rbuf += rbuf2;
            read_buffer.lenght = rbuf;
        }
        /* Mark status read */
        if (quit==TRUE) {
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Demuxer: Quitting from inner loop");
            demuxing = FALSE;
            break;
        }
        #ifndef STANDALONE
        drip_lock("drip_demux2");
        #endif
        rbuf = read_buffer.lenght;
        buf = read_buffer.buffer;
        #ifndef STANDALONE
        drip_unlock("drip_demux2");
        #endif
        end = buf + rbuf;
        while (( buf+4 <= end) && (quit == FALSE)) {
            /* check start code */
            buf_previous = (glong)buf;
            if (buf[0] || buf[1] || (buf[2] != 0x01)) {
                if (complain_loudly) {
                    if ((buf[0] == 0) && (buf[1] == 0) && (buf[2] == 0))
                    complain_loudly = 0;
                }
                buf++;
                continue;
            }
            /* Get SCR */
            a1 = buf[4] << 24 | buf[5] << 16 | buf[6]  << 8 | buf[7];
            a2 = buf[8] << 24 | buf[9] << 16 | buf[10] << 8 | buf[11];
            SCR  = (a1 & 0x38000000) << 3;    // shifting the top bit out ..  . ...
            SCR |= (a1 & 0x03fff800) << 4;
            SCR |= (a1 & 0x000003ff) << 5;
            SCR |= (a2 & 0xf8000000) >> 27; 

            /* Get PTS or DTS */
            if( buf[7] & 0x80 ) { /* PTS */
                PTS  = (buf[ 9] & 0x0E) << 29 ;
                PTS |=  buf[10]         << 22 ;
                PTS |= (buf[11] & 0xFE) << 14 ;
                PTS |=  buf[12]         <<  7 ;
                PTS |= (buf[13] & 0xFE) >>  1 ;
            } else {
                PTS = 0;
            }
            if( buf[7] & 0x40 ) { /* DTS */
                DTS  = (buf[14] & 0x0E) << 29 ;
                DTS |=  buf[15]         << 22 ;
                DTS |= (buf[16] & 0xFE) << 14 ;
                DTS |=  buf[17]         <<  7 ;
                DTS |= (buf[18] & 0xFE) >>  1 ;
            } else {
                DTS = 0;
            }

            /* Handle id */
            switch (buf[3]) {
            case 0xb9:  /* program end code */
                demuxing = FALSE;
                return NULL;
            case 0xbb:  /* pss header */
                tmp1 = buf + (buf[4] << 8) + buf[5] + 5;
                buf = tmp1;
                break;
            case 0xba:  /* pack header */
                /* skip */
                if ((buf[4] & 0xc0) == 0x40) {   /* mpeg2 */
                    SCRstream = SCR;
                    tmp1 = buf + 14 + (buf[13] & 7);
                } else if ((buf[4] & 0xf0) == 0x20) {      /* mpeg1 */
                    SCRstream = SCR;  // TODO: SCR mpeg1 diff from mpeg2??
                    tmp1 = buf + 12;
                } else {
                    g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Demuxer: Weird pack header");
                }
                buf = tmp1;
                break;
            case 0xe0:  /* video */
                tmp2 = buf + 6 + (buf[4] << 8) + buf[5];
                if ((buf[6] & 0xc0) == 0x80) {   /* mpeg2 */
                    tmp1 = buf + 9 + buf[8];
                    if (buf[7] & 0x80) {
                        PTSframe = (gdouble)PTS;
                        mpeg2_pts(mpeg2dec,(glong)PTSframe);
                    }
                } else {  /* mpeg1 */
                    for (tmp1 = buf + 6; *tmp1 == 0xff; tmp1++) {
                        if (tmp1 == buf + 6 + 16) {
                            buf = tmp2;
                            break;
                        }
                    }
                    if ((*tmp1 & 0xc0) == 0x40)
                        tmp1 += 2;
                    tmp1 += mpeg1_skip_table [*tmp1 >> 4];
                    //TODO: calc pts for mpeg1
                    PTSframe = (gdouble)PTS;
                    mpeg2_pts(mpeg2dec,(glong)PTSframe);
                }
                if (tmp1 < tmp2) {
                    gint num_frames;
                    if (PTSvideo == -1) {
                        PTSvideo = (glong)PTSframe; //Store PTS of 1st frame
                        Video_Synced = FALSE;
                        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Demuxer: 1st video frame has Presentation Time Stamp %i",PTSvideo);
                    }
                    result = drip_parse_mpeg2(tmp1,tmp2);
                }
                buf = tmp2;
                break;
            case 0xbd: /* Private streams (SPU, Audio, others) */
                PTSprivate = PTS; /* store presentation time */
                len = buf[8];
                audio_len = 10 + buf[8] + 3;
                audio_bytes = 6 + (buf[4] << 8) + buf[5] - audio_len;
                tmp1 = (guint8*)((glong)buf + 6 + (buf[4]<<8) + buf[5]);
                spu_track = buf[len+9] & 0x1F;
                audio_track = buf[len+9] & 0x0F;
                /* Subpictures */
                if ((buf[len+9] & 0xE0) == 0x20) {
                    #ifdef DEBUG
                    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Demuxer: Got a SPU package, pts = %i",PTS);
                    #endif
                    spu_render(buf-14,PTSprivate); /* Call SPU handler of libdripspu */
                } 
                /* Audio (PCM or A52) */
                if (((buf[len+9] & 0xF0) == 0x80) && (audio_track==*audio_channel[0] || audio_track==*audio_channel[1])) {
                    if (PTSaudio == -1) {
                        PTSaudio = PTSprivate; //Store PTS of 1st frame
                        PTSstored[0] = (glong)PTSaudio;//video; //Store PTS of 1st frame in audio channel 0
                        PTSstored[1] = (glong)PTSaudio;//PTSvideo; //Store PTS of 1st frame in audio channel 1
                        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Demuxer: 1st audio frame has PTS %i (set stored to video PTS %li)",PTSaudio,PTSvideo);
                    }
                    /* A52 audio */
                    #ifdef DEBUG
                    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Demuxer: Audio PTS reads %li",PTSprivate);
                    #endif
                    if(audio_channel>=0) {
                        if (audio_track==*audio_channel[0]) {
                            a52_decode_data((guint8*)((glong)buf+len+13),(guint8*)((glong)buf+len+13+audio_bytes),PTSprivate,buf[len+11]<<8|buf[len+12],0);
                        } else if (audio_track==*audio_channel[1]) {
                            a52_decode_data((guint8*)((glong)buf+len+13),(guint8*)((glong)buf+len+13+audio_bytes),PTSprivate,buf[len+11]<<8|buf[len+12],1);
                        }
                    }
                } else if (((buf[len+9] & 0xF0)==0xA0) || (audio_track==(*audio_channel[0]-16)) || (audio_track==(*audio_channel[1]-16))) {
                    /* PCM audio */
                    #ifdef DEBUG
                    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Demuxer: PCM audio");
                    #endif
                    guint pcm_offset;
                    guint offset = (buf[len+10]<<8) + buf[len+11];
                    const gdouble PTSsub_const = (90000.0/drip_audio_samplerate())/4;
                    gint PTSsub = (gint)((double)(offset)*PTSsub_const);
                    PTSprivate = PTS - PTSsub;
                    packetlen = (buf[4]<<8) + buf[5] - 3 - buf[8];
                    for (pcm_offset=0; ++pcm_offset<packetlen-1; ) {
                        if (buf[len+9+pcm_offset]==0x01 && buf[len+9+pcm_offset+1]==0x80 ) {
                            /* Start */
                            pcm_offset += 2;
                            break;
                        }
                    }
                    if (audio_channel>=0) {
                        guint8* audio_data = (guint8*)((glong)&buf[len+9+pcm_offset]);
                        guint8 tmp;
                        for (guint i=0;i<(packetlen-pcm_offset);i+=2) {
                           tmp = audio_data[i];
                           audio_data[i] = audio_data[i+1];
                           audio_data[i+1] = tmp;
                        }
                        if (PTS) {
                            #ifdef DEBUG
                            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"drip_demux: adding PCM audio to buffer");
                            #endif
                            fast_memcpy(gint16_samples,(gint16*)(audio_data),packetlen-pcm_offset);
                            if (audio_track==*audio_channel[0]) {
                                add_to_audio_buffer(gint16_samples,packetlen-pcm_offset,PTSprivate,0);
                            } else {
                                add_to_audio_buffer(gint16_samples,packetlen-pcm_offset,PTSprivate,1);
                            }
                        } else {
                            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Demuxer: no PTS on PCM audio frame");
                        }
                    } else {
                        #ifdef DEBUG
                        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"drip_demux: PCM audio channel %i not used",audio_channel);
                        #endif
                    }
                } 
                buf = tmp1;
                break;
            default:
                if (buf[3] < 0xb9) {
                    g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Demuxer: looks like a video stream, not system stream");
                } else if ( (buf[3]-0xc0) == *audio_channel[0] || (buf[3]-0xc0) == *audio_channel[1]) {// && buf[3]<=0xdf) {
                    /* MP3 mpeg audio (libmad) */
                    #ifdef DEBUG
                    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Demuxer: MP3 audio (channel %i)",buf[3]-0xc0);
                    #endif
                    if (!mp3first) {
                        g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Demuxer: Source material contains MP3 audio for which support is work-in-progress");
                        mp3first = TRUE;
                        mad_buffer = (mpeg_read_buffer_t*)malloc(sizeof(mpeg_read_buffer_t));
                        mad_buffer->buffer = NULL;
                        mad_buffer->lenght = 0;
                        mad_decoder_init(mad_decoder_,mad_buffer,mad_input,0/*header*/,0/*filter*/,mad_output,mad_error,0/*message*/);
                        /* start decoding in async mode, iow running in it's own thread */
                        result = mad_decoder_run(mad_decoder_,MAD_DECODER_MODE_ASYNC);
                    }
                    while (mad_buffer->lenght>0) {
                        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Demuxer: Waiting for MAD buffer to become ready (length=%li,%p)",mad_buffer->lenght,&(mad_buffer->lenght));
                        usleep(10);
                        //vreemd = (gint)random();
                    }
                    audio_bytes = 6 + (buf[4] << 8) + buf[5] - 13;
                    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Demuxer: Addind %li bytes to MAD buffer (len=%p, start=%p)",audio_bytes,&mad_buffer->lenght,(gpointer)&buf[13]);
                    #ifndef STANDALONE
                    drip_lock("drip_demux3");
                    #endif
                    mad_buffer->buffer = (guint8*)&buf[13];
                    mad_buffer->lenght = audio_bytes;
                    #ifndef STANDALONE
                    drip_unlock("drip_demux3");
                    #endif
g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Demuxer: done adding data to mp3 buffer");
                }

                /* skip */
                tmp1 = buf + 6 + (buf[4] << 8) + buf[5];
                buf = tmp1;
                break;
            }
        }
    } while ( (end == read_buffer.buffer+DEMUX_BUFFER_SIZE) && (quit==FALSE) );
    #ifdef DEBUG
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Demuxer: end = %p, read_buffer.buffer+DEMUX_BUFFER_SIZE = %p, quit = %i",end,read_buffer.buffer+DEMUX_BUFFER_SIZE,quit);
    #endif
    demuxing = FALSE;

    /* Release the MP3 decoder */
    mad_decoder_finish(mad_decoder_);
    #ifndef STANDALONE
    drip_lock("drip_demux4");
    #endif
    //mad_buffer->buffer = NULL;
    //mad_buffer->lenght = 0;
    #ifndef STANDALONE
    drip_unlock("drip_demux4");
    #endif
    mp3first = FALSE;

    /* Close libraries */
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Demuxer: Ended task, stopping");
    return NULL;
}


/* Start demuxing dvd stream */
void drip_mpeg_start_demuxing(gint** audio_channel) {
    gint i;
    /* Start (threadless?) demuxing */
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Starting demux thread");
    demuxing = TRUE;
    process = TRUE;
    quit = FALSE;
    PTSvideo = -1;
    PTSaudio = -1;
    Audio_Cut = FALSE;
    Video_Synced = FALSE;
    /* Default audio vari's */
    for (i=0;i<AMAX;i++) {
        audio_buffer[i] = NULL;
        audio_buffer_index[i] = 0;
        audio_buffer_base[i] = 0;
        audio_buffer_total[i] = 0;
        audio_fetched_total[i] = 0;
        IDnumber[i] = 0;
        PTSindex[i] = 0;
        non_silence[i] = FALSE;
        /* A52 vari's */
        bufptr[i] = &a52_buffer[i][0];
        bufpos[i] = &a52_buffer[i][7];
        PTSa52[i] = -1;
        start[i] = 0;
        avifile_audio_total_offset[i] = 0;
        offset_corrected[i] = FALSE;
    }

    /* All initialized, start the batch */
    drip_demux(NULL,audio_channel);

    /* Clean & Exit */
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Ended demux thread");
    return;
}


/* Opens libmpeg2 handlers */
gint drip_mpeg_open(gchar *input,void handler(guint8*[3],uint32_t,gint)) {
    gint i;
    uint32_t accel = 1; 
    // open input file
    demuxing = TRUE;
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Opening input mpeg from %s",input);
    #ifndef STANDALONE
    drip_lock("drip_mpeg_open1");
    #endif
    input_files = g_strsplit(input,":",256);
    input_files_index = 0;
    in_file = fopen(input_files[input_files_index],"rb");
    #ifndef STANDALONE
    drip_unlock("drip_mpeg_open1");
    #endif

    if (!in_file) 
        g_log(DRIP_LD,G_LOG_LEVEL_ERROR,"MPEG Open: %s - couldnt open file %s", strerror(errno),input_files[input_files_index]);
    else 
        g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Opened %s for input",input_files[input_files_index]);
    input_files_index++;

    // Publish handler
    processor = handler;

    // alloc read buffer
    if (read_buffer.buffer != NULL) free(read_buffer.buffer);
    read_buffer.buffer = (guint8*)malloc(DEMUX_BUFFER_SIZE);
    read_buffer.lenght = 0;

    //DISABLED // start reader thread; hope we run with kernel threads so the IO wont lock encoding
    //DISABLED mpeg2_reader_id = pthread_create(&mpeg2_reader,NULL,mpeg2_reader_handler,NULL);

    /* init libmpeg2 (video) */
    #ifndef STANDALONE
    drip_lock("drip_mpeg_open4");
    #endif
    mpeg2dec = mpeg2_init();
    mpeg2info = mpeg2_info(mpeg2dec); 
    #ifndef STANDALONE
    drip_unlock("drip_mpeg_open4");
    #endif

    /* init silence chunk */
    memset(silence,AUDIO_BUFFER_SIZE*sizeof(gint16),0);
    /* alloc a52 buffers */
    a52_buffer[0] = (guint8*)malloc(3840*sizeof(guint8));
    a52_buffer[1] = (guint8*)malloc(3840*sizeof(guint8));
    /* init liba52 (audio) */
    #ifdef WRITEOUT_AUDIO
    #ifndef STANDALONE
    drip_lock("drip_mpeg_open5");
    #endif
    writeout_audio_fd = open("/tmp/writeout_audio.raw",O_WRONLY|O_CREAT|O_TRUNC,S_IRWXU);
    #ifndef STANDALONE
    drip_unlock("drip_mpeg_open5");
    #endif
    #endif
    #ifndef STANDALONE
    drip_lock("drip_mpeg_open6");
    #endif
    output_liba52[0] = a52_init(accel);			output_liba52[1] = a52_init(accel);
    #ifndef STANDALONE
    drip_unlock("drip_mpeg_open6");
    #endif
    if (output_liba52==NULL) g_log(DRIP_LD,G_LOG_LEVEL_ERROR,"Failed to initialize audio");

    /* Clean & Exit */
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Initialized import of mpeg video and audio");
    return TRUE;
}


/* Closes libmpeg2 handlers */
gint drip_mpeg_close(gboolean lock) {
    gint result;
    const char* domain;

    /* Define logging domain (locked\unlocked) */
    if (lock) {
        domain = DRIP_LD;
    } else {
        domain = DRIP_CB_LD;
    }

    quit = TRUE;
    /* Cancel mpeg2 reader thread */
    result = pthread_join(mpeg2_reader,NULL);
    while (mpeg2_reader_handler_active) { usleep(20);}
    /* Close mpeg and a52 handlers */
    demuxing = FALSE;
    while (decoding_audio == TRUE) { usleep(20);}

    free(audio_buffer[0]);
    audio_buffer[0] = NULL;
    if (audio_buffer[1]) {
        free(audio_buffer[1]);
        audio_buffer[1] = NULL;
    }
    if (read_buffer.buffer!=NULL) {
        free(read_buffer.buffer);
        read_buffer.buffer = NULL;
    }
    mpeg2_close(mpeg2dec);

    /* Shutdown */
    #ifndef STANDALONE
    drip_lock("drip_mpeg_close1");
    #endif
    g_strfreev(input_files);
    reset_audio_buffer(0);				reset_audio_buffer(1);
    fclose(in_file);
    #ifndef STANDALONE
    drip_unlock("drip_mpeg_close1");
    #endif

    #ifdef WRITEOUT_AUDIO
    close(writeout_audio_fd);
    #endif
    /* reset */
    quit = FALSE;
    return TRUE;

}

