/*
 *  Dr Geo an interactive geometry software
 * (C) Copyright Free Software Foundation 2001
 * Auhtor: hilaire@ofset.org 
 * 
 *
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public Licences as by published
 * by the Free Software Foundation; either version 2; or (at your option)
 * any later version
 *
 * This program is distributed in the hope that it will entertaining,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILTY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Publis License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include "drgeo_polygon.h"
#include "drgeo_point.h"
#include "drgeo_repere.h"
#include "drgeo_drgeoStyle.h"
#include "drgeo_drawable.h"

extern int polygonColor;

polygon::
polygon (liste_elem& parents, polygonType type,
	 gboolean createdFromMacro, liste_elem *figureList):
	geometricObject (createdFromMacro, figureList)
{
	gpointer item;

	style.color = (drgeoColorType) polygonColor;
	this->type = (gint) type;
	category = POLYGON;
	switch (type)
	{
	case POLYGON_NPTS:
		/* the object parentList will contain
		   a list of POINT object */
		parents.init_lire ();
		while (item = (gpointer) parents.lire(0))
			parentList = g_list_append 
				(parentList, gpointer (item));
		break;
	}
	update ();
	initName ();
}

polygon::
polygon (xmlNodePtr tree, GHashTable *itemIdToAdress,
	 liste_elem *figureList):
	geometricObject (tree, figureList)
{
	gchar *attr;
	xmlNodePtr item;
	void *obj;

	attr = (gchar *) xmlGetProp (tree, BAD_CAST "type");
	if (!attr)
		/* FIXME: do something there,
		   mandatory attribute missing */
		printf("point::point missing mandatory attribute\n");
	else
	{
		category = POLYGON;
		if (!strcmp (attr,"npts"))
			type = POLYGON_NPTS;
		
		/* get the parent ref */
		item = xml_search_next (tree->childs, "parent");
		while (item)
		{
			if ((obj = xmlInsertParent (item, itemIdToAdress)) == NULL)
				exit (1); /* FIXME: implement a nicer way */
			parentList = g_list_append (parentList, obj); 
			item = xml_search_next (item->next, "parent");
		}
	}
	g_free (attr);
 	update ();
	initName ();
}

void polygon::
update ()
{
	exist = parentExist ();
	/* Nothing to update actualy */
}

void polygon::
move (drgeoVector& t)
{
	GList *parent;
	switch (type)
	{
	case POLYGON_NPTS:
		parent = g_list_first (parentList);
		while (parent != NULL)
		{
			((geometricObject *)parent->data)->move (t);
			parent = g_list_next (parent);
		}
		break;
	}
}

void polygon::
save (xmlNodePtr tree, liste_elem& figureList)
{
	xmlNodePtr item;
	GList *parent;

	parent = g_list_first (parentList);
	item = xmlNewChild (tree, NULL, BAD_CAST "polygon", NULL);	

	switch (type)
	{
	case POLYGON_NPTS:
		save_attribute (item, this, "npts");
		break;
	}
	while (parent)
	{
		xmlAddParent (item, (geometricObject *)parent->data);
		parent = g_list_next (parent);
	}	
}

void polygon::
updateDescription ()
{
	gint nb;
	geometricObject::updateDescription ();

	switch (type)
	{
	case POLYGON_NPTS:		
		description = new gchar * [g_list_length(parentList) + 1];
		description[0] = g_strdup_printf(_("Polygon:: %s"),name);
		for (nb = 0; nb < g_list_length (parentList); nb++)
			description[nb + 1] = g_strdup_printf(_("Point %s"),
							      getNthName(nb));
		break;
	}
}

void polygon::
draw (drgeoDrawable& area, char force)
{
	if ((style.mask == alway) || (style.mask == yes && force == FALSE)
	    || !exist)
		return;
	area.drawPolygon (style, parentList);
}

gboolean polygon::
overObject (drgeoPoint& mouse, gdouble range)
{
	drgeoVector u;
	GList *parent;
	point *old, *recent;
	drgeoPoint origin;
	
	parent = g_list_first (parentList);
	old = (point *)parent->data;
	while (parent = g_list_next (parent))
	{
		recent = (point *)parent->data;
		origin = old->getCoordinate ();
		u = recent->getCoordinate() - origin;
		if (ABS (u ^ (mouse - origin)) / u.norm () <= range
		    && ((mouse - origin) * u) >= 0
		    && ((mouse - origin) * u) <= u.squareNorm ())
			return TRUE;
		old = recent;
	}	
	// Check with the segment formed by the first and last point
	parent = g_list_first (parentList);
	old = (point *)parent->data;
	origin = old->getCoordinate ();
	u = recent->getCoordinate() - origin;
	if (ABS (u ^ (mouse - origin)) / u.norm () <= range
	    && ((mouse - origin) * u) >= 0
	    && ((mouse - origin) * u) <= u.squareNorm ())
		return TRUE;	
	return FALSE;
}

void polygon::
initName ()
{
	int l, nb;
	gboolean autoName = TRUE;


	if (style.mask == alway)
		return;
	if (typeName != NULL)
		g_free (typeName);
	l = strlen (_("this polygon %1"))  + 1;

	nb = g_list_length (parentList);
	while (nb-- && autoName)
		if (!getNthName(nb)) 
			autoName = FALSE;
		else if (strlen (getNthName(nb)) == 0)
			autoName = FALSE;	
	if (autoName)
	{
		gchar *tmp;
		g_free (name);
		name = g_strdup ("");
		for (nb = 0; nb < g_list_length (parentList); nb++)
		{
			tmp = g_strconcat (name, getNthName (nb), NULL);
			g_free (name);
			name = tmp;
		}
	}

	if (name)
		l += strlen (name);
	typeName = (gchar *) g_malloc (l);
	strcpy (typeName, _("this polygon %1"));
	strinsmsg (typeName, name, "%1");
}
