/*
 *  dlisp / a dancer's lisp parser library
 *  Copyright (C) 2002-2003 Junichi Uekawa
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * main parser.
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "dlispparser.h"

/**@name Internal static functions used inside parser reader
   This section describes function that are used only inside the
   parser system.
 */
/*@{*/


/**
   generate an error, and exit. 
   Noting the current character position
 */
static void
lisp_error (FILE*f /** input file stream*/, const char * sentence /** error message*/)
{
  fprintf (stderr, "Fatal error at position %ld: %s\n", 
	   ftell(f),
	   sentence);
  exit (1);
}

/**
   skip up to the next newline
 */
static void
skip_to_newline (FILE*f)
{
  int c;
  
  while ((c = getc (f)) != EOF)
    {
      switch (c) 
	{
	case '\r':
	  return;
	case '\n':
	  return;
	case EOF:
	  return;
	}
      
    }
}


/** peek one character, only for read_noncomment_char */
static int
peek_char_raw (FILE*f)
{
  int c = getc (f);
  if (c != EOF)
    ungetc (c,f);
  return c;
}

/**
   Multi-line comment procedure.
   skip to !# 
 */
static void
skip_to_comment_terminator(FILE*f)
{
  int c;
  while (( c = getc (f))!= EOF)
    {
      if (c == '!')
	{
	  if (peek_char_raw(f) == '#')
	    {
	      getc(f);
	      return;
	    }
	}
    }
}

/**
   read a new character, excluding comments.
 */
static int
read_noncomment_char (FILE *f )
{
  int c;
  
  while (( c = getc(f)) != EOF)
    {
      switch (c)
	{
	case ';':
	  skip_to_newline (f);
	  continue;
	case '\r':
	case '\n':
	  return ' '; 		/* return space instead of returns. */
	case '#':		/* comment #! -- !# */
	  if (peek_char_raw(f) == '!')
	    {
	      skip_to_comment_terminator(f);
	      continue;
	    }
	  else
	    return c;
	default:
	  return c;
	}  
    }
  return c;
}

/** 
   peek one character, taking comments in consideration.
*/
static int
peek_char (FILE*f)
{
  int c = read_noncomment_char (f);
  if (c != EOF)
    ungetc (c,f);
  return c;
}

/**
   Force-read char, and give error.
 */
static int 
force_read_char (FILE*f, int want_char, const char * accept_char /** can be null*/) 
{
  int c;
  
  while ((c=read_noncomment_char(f))!=EOF)
    {
      if ( want_char == c )
	return c;
      else
	{
	  if (accept_char && strchr (accept_char, c))
	    continue;
	  else
	    {
	      lisp_error (f, "Unacceptable character found");
	      return EOF;	/* this is error here! */
	    }
	}
      
    }
  return c;
}

/**
   skip space, ignoring comments.
 */
static void
skip_spaces(FILE*f)
{
  int c;
  while ((c = read_noncomment_char(f))!=EOF)
    {
      if (!isspace(c))
	break;
    }
  if (c != EOF)
    ungetc (c,f);
  return;
}


/**
  allocate lisp entry structure.
 */
static dlisp_lispentry *
alloc_lispentry(FILE*f)
{
  dlisp_lispentry* l = malloc (sizeof (dlisp_lispentry));
  if (!l)
    {
      lisp_error (f, "Out of memory");
    }
  
  l->next = l->child = NULL;
  l->type = dlisp_error;  
  l->name = NULL;
  return l;
}

/**
   read lisp symbol, delimited by terminatorstring.
   escape is done with "\"
 */
static char * 
read_lisp_name (FILE*f, const char * terminatorstring)
{
  int c;
  const int BUFBLOCK=512;
  int bufsize = BUFBLOCK;
  char * buf = malloc (bufsize);
  int currentpos = 0;		/** current buffer position */
  int prev = 0;			/** previous char, used for checking \ */

  while ((c=peek_char(f))!=EOF)
    {
      if (prev != '\\' && strchr(terminatorstring, c))
	{
	  /* end of sequence */
	  buf[currentpos]=0;
	  return buf;
	}
      else 
	{
	  if (currentpos > (bufsize - 2))
	    buf = realloc ( buf, bufsize += BUFBLOCK );
	  buf[currentpos ++ ] = read_noncomment_char(f);
	}
      prev = c;
    }

  /* end of file buffer while reading entity. */
  buf [currentpos]=0;  
  return buf ;
}



/**
   read a entity. 
   entity -> lisp-bracket | term 
   lisp-bracket -> (entity*) 
 */
static dlisp_lispentry *
read_lisp_entity (FILE*f)
{
  int c;
  dlisp_lispentry * l = alloc_lispentry (f);
  c = peek_char(f);

  switch (c)
    {
    case EOF:
      lisp_error(f, "unexpected EOF!!");
      return NULL;
    case '(':
      l->child = dlisp_read_lisp_bracket(f);
      l->type = dlisp_bracket;
      return l;
    case '\'':
      /* this is a "quote" */
      force_read_char(f, '\'', NULL);
      l->child = read_lisp_entity(f);
      l->type = dlisp_quotedtext;
      return l;
    case '#':
      /* possible start of constant vector */
      force_read_char(f, '#', NULL);
      if (peek_char(f) != '(')
	{
	  char * tmp = read_lisp_name (f, " \n\r\t()");
	  if (!tmp)
	    lisp_error(f, "Internal error, NULL from reading lisp name");
	  if ((l->name = malloc (strlen (tmp) + 2 ))==NULL)
	    lisp_error(f, "Out of memory");
	  
	  l->name[0]='#';
	  strcpy (l->name + 1, tmp);
	  l->type = dlisp_cdata;
	  free (tmp);
	}
      else
	{
	  l->child = read_lisp_entity(f);
	  l->type = dlisp_sharp_vector_constant;
	}
      return l;
    case '`':
      /* quasiquote */
      force_read_char(f, '`', NULL);
      l->child = read_lisp_entity(f);
      l->type = dlisp_quasiquote;
      return l;
    case ',':
      /* in-quasiquote */
      force_read_char(f, ',', NULL);
      if (peek_char(f) == '@')
	{
	  force_read_char(f, '@', NULL);
	  l->child = read_lisp_entity(f);
	  l->type = dlisp_commaat;
	}
      else
	{
	  l->child = read_lisp_entity(f);
	  l->type = dlisp_comma;
	}
      return l;
    case '"':			/* start of string literal */
      force_read_char(f, '\"', NULL);
      l->name = read_lisp_name (f, "\"");
      l->type = dlisp_stringconstant;
      force_read_char(f, '\"', NULL);
      return l;
    default:
      l->name = read_lisp_name (f, " \n\r\t()");
      l->type = dlisp_cdata;
      return l;
    }
}

/*@}*/
/* end of static functions.*/

/**
   Read inside bracket of lisp string.
   
   Read from a file stream, and return a dlisp_lispentry structure
   parsing the lisp construct.

   @return dlisp_lispentry
 */
dlisp_lispentry *
dlisp_read_lisp_bracket (FILE*f/**Input file stream*/)
{
  dlisp_lispentry * l = NULL, *current = NULL;

  force_read_char (f, '(', " \t");
  skip_spaces(f);

  do 
    {
      if (current)
	  current = (current -> next = read_lisp_entity (f));
      else
	current = l = read_lisp_entity (f);
      skip_spaces(f);
    } while ( peek_char (f) != ')' );

  force_read_char(f, ')', NULL);
  
  return l;
}

/** 
    read a string of lisp text, with many brackets, and create a tree representation of it.

    @return dlisp_lispentry structure filled with parsed lisp data.
 */
dlisp_lispentry * 
dlisp_read_lisp_text (FILE*f /** Input file stream */)
{
  dlisp_lispentry * l = NULL, * current = NULL;
  int c;
  
  while ((c = peek_char (f)) != EOF)
    {
      if (c == ')')
	lisp_error (f, "Syntax error, ) exists on unexpected place");
      
      if (current)
	current = current -> next = read_lisp_entity(f);
      else
	current = l = read_lisp_entity(f);
      skip_spaces(f);
    }
  return l;
}

