/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/frame/record.hpp
 *
 * @brief [LEVEL: beta] The @ref diagnostics::Record class
 *
 * $Id: record.hpp,v 1.19 2005/06/23 09:54:20 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @test diagnostics/frame/record.t.cpp
 */

#ifndef DIAGNOSTICS__FRAME__RECORD_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__FRAME__RECORD_HPP__INCLUDE_GUARD

// used in the interface for system types and compile time switching
#include <diagnostics/frame/platform.hpp>

// used in the interface by value
#include <diagnostics/frame/level.hpp>

// used in the interface by value
#include <diagnostics/frame/type.hpp>

// used in the interface by reference
#include <string>

// used in the interface by reference
#include <iostream>

DIAGNOSTICS_NAMESPACE_BEGIN;

/**
 * @class Record diagnostics/frame/record.hpp
 *
 * @brief A Record is constructed for each invocation of @ref
 * Logging_Facility::log. If @ref DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED ==
 * 1 then, in addition to the arguments of the log-method, a Record
 * has further fields, which contain the hostname, the process id, the
 * thread id, and a time stamp.
 *
 * This class is intentionally kept flat. It is a pure data-carrier
 * with pretty printing. Also, sub-classing is intentionally a
 * non-option: It would require the registration of record-types which
 * would make the start-up of the overall framework a relatively
 * complex procedure.
 *
 * @nosubgrouping
 */
class Record
{
    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Types
     * @{
     */
private:
    typedef Record Self;
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Disallowed Creation
     * @{
     */
private:
    Record();
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Creation
     * @{
     */
public:
    /**
     * @brief assembly cstr  -- copies everything
     *
     * @throw never
     */
    Record(Level_t const level,
	   Type_t const type,
	   int const nr_what,
	   ::std::string const & str_what,
	   char const * const base_file_name,
	   char const * const file_name,
	   int const line
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
	   ,
	   Pid_t const pid,
	   Tid_t const tid,
	   Sec_t const sec,
	   Usec_t const usec,
	   char const * const hostname
#endif
	   );

    /**
     * @brief copy cstr -- copies everything
     *
     * @throw never 
     */
    Record(Self const &);

    /**
     * @throw never
     */
    Self & operator=(Self const &r);

    /**
     * @throw never
     */
    bool operator==(Self const &r) const;

    /**
     * @throw never
     */
    inline bool operator!=(Self const &r) const { return !operator==(r); }
    

    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /*
     * @name Accessors
     * @{
     */
public:
    inline Level_t               level()          const   { return m_level; }
    inline Type_t                type()           const   { return m_type; }
    inline int                   nr_what()        const   { return m_nr_what; }
    inline ::std::string const & str_what()       const   { return m_str_what; }
    inline ::std::string const & base_file_name() const   { return m_base_file_name; }
    inline ::std::string const & file_name()      const   { return m_file_name; }
    inline int                   line()           const   { return m_line; }
    
						
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    inline Pid_t                 pid()          const   { return m_pid; }
    inline Tid_t                 tid()          const   { return m_tid; }
    inline Sec_t                 sec()          const   { return m_sec; }
    inline Usec_t                usec()         const   { return m_usec; }
    inline ::std::string const & hostname()     const   { return m_hostname; }
#endif
    // *}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Encapsulated States
     * @{
     */
private:
    Level_t m_level;
    Type_t m_type;
    int m_nr_what;
    ::std::string m_str_what;
    ::std::string m_base_file_name;
    ::std::string m_file_name;
    int m_line;

#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    Pid_t m_pid;
    Tid_t m_tid;
    Sec_t m_sec;
    Usec_t m_usec;
    ::std::string m_hostname;
#endif
    // @}
};

DIAGNOSTICS_NAMESPACE_END;


::std::ostream & operator<<(::std::ostream & stream, ::diagnostics::Record const & record);

#endif

// vim:ts=4:sw=4
