#ifndef __HPP_TRICKLEBUFFER
#define __HPP_TRICKLEBUFFER

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//

//
//  The TrickleHandler:
//              read/write optimized data buffers for trickling
//
//  20040811::mad@madness.at
//

#include "Socket.hpp"
#include "String.hpp"
#include <stdexcept>
#include <fstream>
#include <fcntl.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <time.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>

#include "OptionContainer.hpp"
#include "HTTPHeader.hpp"

#ifdef __GCCVER3
using namespace std;
#endif

extern OptionContainer o;

class FileBuffer
{
private:
  int fd;
  int len;
  string fname;

public:
    FileBuffer (int _fd);       // use existing tempfile

   ~FileBuffer ();

  int doOpen (char *tempFileName);
  int doAppend (const char *_data, int _len);
  int doRead (char *_data, int _len, int _offset);
  int doRead (char *_data, int _len);

  int getFileLength () const
  {
    return len;
  }
};


class MemoryBuffer
{
private:
  int buffersize, tricklesize, len;
  char *buffer;

public:
    MemoryBuffer (int _buffersize, int _tricklesize);   // create new memory buffer

   ~MemoryBuffer ();

  char *prepareAppend ();       // returns address where we may write upto
  // $tricklesize bytes; otherwise return NULL
  char *commitAppend (int _len);        // update pos w/ actual bytes written
  char *doAppend (const char *_data, int _len);

  char *getData () const
  {
    return buffer;
  }
  int getBufferSize () const
  {
    return buffersize;
  }
  int getTrickleSize () const
  {
    return tricklesize;
  }
  int getLength () const
  {
    return len;
  }
  int setLength (int _len)
  {
    return len = _len;
  }
};



class TrickleBuffer
{
private:
  FileBuffer fbuf;
  MemoryBuffer mbuf;

public:
    TrickleBuffer (int _buffersize, int _tricklesize, int _fd);
   ~TrickleBuffer ();

  char *prepareAppend ();
  char *commitAppend (int _len);

  int doOpen (char *tempFileName);
  int doFlushToDisk (bool flushIt);
  int doTrickle (char *_data, int _len, int _offset);
  int doRead (char *_data, int _len, int _offset);

  int getLength () const
  {
    return fbuf.getFileLength () + mbuf.getLength ();
  }
  int getFileLength () const
  {
    return fbuf.getFileLength ();
  }
  int getTrickleSize () const
  {
    return mbuf.getTrickleSize ();
  }
};



class TrickleHandler
{
private:
  int first_delay, following_delay, trickle_delay;
  int trickle_length, rc_in, rc_out;
  int percent_chunk, percent_done, percent_scale;
  int trickle_pos, content_length, last_size;
  int proxyfd, peerfd, maxfd, tempbuffsize;
  bool dlmgr;
  bool dlmgr_header;
  bool scanIt;
  bool dg_data;
  char *tbuf_ptr;
  time_t time_last, time_current;
  TrickleBuffer tbuf;
  HTTPHeader *docheader;

  int doNegTrickle (Socket & _sock);
  int doNeg1Trickle (Socket & _sock);
  int doPosTrickle (Socket & _sock);
  int doDlManager (Socket & _sock);

  bool doSend (Socket & _sock, char *block, int size) throw ();
  int selectEINTR (int numfds, fd_set * readfds, fd_set * writefds, fd_set * exceptfds, struct timeval *timeout);

public:
    TrickleHandler (int _max_buf, int _size, int _fd, HTTPHeader * _docheader, bool _dlmgr);
   ~TrickleHandler ();

  int doOpen (char *tempFileName);

  int prepareTrickle (Socket & _proxysock, Socket & _peerconn, int _contentlength, int _tricklepos);
  // setup trickle obj for Neg1Trickle (e.g. after forked scanning)
  int prepareTrickle (Socket & _proxysock, Socket & _peerconn, int _tricklepos);

  int doReceive (Socket & _proxysock, Socket & _peerconn);
  int doReadAll (Socket & _peerconn, int _from_pos);
  int commitTrickle ();
  bool doTrickle (Socket & _peerconn);

  int getTricklePos () const
  {
    return trickle_pos;
  }
  int getFileLength () const
  {
    return tbuf.getFileLength ();
  }
  int getLength () const
  {
    return tbuf.getLength ();
  }
};

#endif
