//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

//  Written by Aecio F. Neto (afn@harvest.com.br)
//              Harvest Consultoria (www.harvest.com.br)
//
//  and somewhat hacked by m4d.
//

#include "ClamDScan.hpp"


extern OptionContainer o;

// Constructor
ClamDEngine::ClamDEngine ():stream_scan (false)
{
}

// Deconstructor
ClamDEngine::~ClamDEngine ()
{
}


int
ClamDEngine::scanFile (const char *_fname)
{
  int sockd = 0;
  int infected = 0;

#ifdef DGDEBUG
  std::cout << "Entering ClamDEngine scanFile" << std::endl;
  std::cout << "Before calling dconnect" << std::endl;
#endif

  if ((sockd = dconnect ()) < 0)
    {
      err_str = "Error connecting to clamav daemon";
#ifdef DGDEBUG
      std::cout << err_str << std::endl;
#endif
      infected = VirusEngine::AV_FAIL;
    }
  else
    {
#ifdef DGDEBUG
      std::cout << "Before calling dsfile/dsstream" << std::endl;
#endif
      if (stream_scan)
        {
          infected = dsstream (sockd, _fname);
        }
      else
        {
          infected = dsfile (sockd, _fname);
          close (sockd);
        }
    }

  return infected;
}


int
ClamDEngine::dsfile (int sockd, const char *filename)
{
  int infected = 0;
  char buff[4096], scancmd[512];
  FILE *fd;

#ifdef DGDEBUG
  std::cout << "Entering dsfile" << std::endl;
#endif

  snprintf (scancmd, 512, "CONTSCAN %s", filename);

#ifdef DGDEBUG
  std::cout << "Before sending clamd command: " << scancmd << std::endl;
#endif

  if (write (sockd, scancmd, strlen (scancmd)) <= 0)
    {
      // Error writing to clamd socket
      infected = VirusEngine::AV_FAIL;
    }

  // reading response from clamd
  if ((fd = fdopen (dup (sockd), "r")) == NULL)
    {
      // Error opening socket for reading
      err_str = "Error opening clamd socket for reading";
#ifdef DGDEBUG
      std::cout << err_str << std::endl;
#endif
      infected = VirusEngine::AV_FAIL;
      return infected;
    }

  while (fgets (buff, sizeof (buff), fd))
    {
      if (strstr (buff, "FOUND\n"))
        {
          infected = VirusEngine::AV_VIRUS;
        }
      if (strstr (buff, "ERROR\n"))
        {
          infected = VirusEngine::AV_FAIL;
        }
    }
#ifdef DGDEBUG
  std::cout << "Got reply from clamd: '" << buff << "'" << std::endl;
#endif

  if (infected == VirusEngine::AV_VIRUS)
    {
      buff[strlen (buff) - 7] = '\0';
      setVirusName (&buff[strlen (filename) + 2]);
#ifdef DGDEBUG
      std::cout << "Virus found: '" << vname << "'" << std::endl;
#endif
    }
  else if (infected == VirusEngine::AV_FAIL)
    {
      buff[strlen (buff) - 7] = '\0';
      setErrString (&buff[strlen (filename) + 2]);
#ifdef DGDEBUG
      std::cout << "Error found: '" << err_str << "'" << std::endl;
#endif
    }
  fclose (fd);

  return infected;
}

int
ClamDEngine::dsstream (int sockd, const char *_fname)
{
  int rfd = 0, wsockd = 0, loopw = 60, bread = 0, port = 0, infected = VirusEngine::AV_CLEAN;
  struct sockaddr_in server;
  struct sockaddr_in peer;
  socklen_t peer_size;
  char buff[4096], *pt;

#ifdef DGDEBUG
  std::cout << "Entering dsstream" << std::endl;
#endif

  if (write (sockd, "STREAM", 6) <= 0)
    {
      setErrString (strerror (errno));
      return VirusEngine::AV_FAIL;
    }

  memset (buff, 0, sizeof (buff));
  while (loopw)
    {
      read (sockd, buff, sizeof (buff));
      if ((pt = strstr (buff, "PORT")))
        {
          pt += 5;
          sscanf (pt, "%d", &port);
          break;
        }

      loopw--;
    }

  if (!loopw)
    {
      setErrString (strerror (errno));
      return VirusEngine::AV_FAIL;
    }


  /* connect to clamd */
  if ((wsockd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
    {
      setErrString (strerror (errno));
      return VirusEngine::AV_FAIL;
    }

  server.sin_family = AF_INET;
  server.sin_port = htons (port);

  peer_size = sizeof (peer);
  if (getpeername (sockd, (struct sockaddr *) &peer, &peer_size) < 0)
    {
      setErrString (strerror (errno));
      close (wsockd);
      return VirusEngine::AV_FAIL;
    }

  switch (peer.sin_family)
    {
    case AF_UNIX:
      server.sin_addr.s_addr = inet_addr ("127.0.0.1");
      break;
    case AF_INET:
      server.sin_addr.s_addr = peer.sin_addr.s_addr;
      break;
    default:
      close (wsockd);
      return VirusEngine::AV_FAIL;
    }

  if (connect (wsockd, (struct sockaddr *) &server, sizeof (struct sockaddr_in)) < 0)
    {
      setErrString (strerror (errno));
      close (wsockd);
      return VirusEngine::AV_FAIL;
    }


  if ((rfd = open (_fname, O_RDONLY)) < 0)
    {
      setErrString (strerror (errno));
      close (wsockd);
      return VirusEngine::AV_FAIL;
    }

  while ((bread = read (rfd, buff, sizeof (buff))) > 0)
    {
      if (write (wsockd, buff, bread) <= 0)
        {
          setErrString (strerror (errno));
          close (wsockd);
          close (rfd);
          return VirusEngine::AV_FAIL;
        }
    }

  close (rfd);
  close (wsockd);
  memset (buff, 0, sizeof (buff));
  while ((bread = read (sockd, buff, sizeof (buff))) > 0)
    {
      if (strstr (buff, "FOUND\n"))
        {
          buff[strlen (buff) - 7] = '\0';
          setVirusName (&buff[8]);
          infected = VirusEngine::AV_VIRUS;
        }
      if (strstr (buff, "ERROR\n"))
        {
          buff[strlen (buff) - 7] = '\0';
          setErrString (&buff[8]);
          return VirusEngine::AV_FAIL;
        }
      memset (buff, 0, sizeof (buff));
    }

  return infected;
}

int
ClamDEngine::dconnect ()
{
  int sockd = -1;
  struct sockaddr_un un_server;
  struct sockaddr_in in_server;
  struct hostent *he;

  const char *LocalSocket = o.clamdsocket.c_str ();

  if (LocalSocket)
    {
      if (LocalSocket[0] == '/')
        {
          // unix socket
#ifdef DGDEBUG
          std::cout << "Clamd: connect to unix socket " << LocalSocket << std::endl;
#endif
          un_server.sun_family = AF_UNIX;
          strncpy (un_server.sun_path, LocalSocket, sizeof (un_server.sun_path));

          if ((sockd = socket (AF_UNIX, SOCK_STREAM, 0)) < 0)
            {
              return -1;
            }
          if (connect (sockd, (struct sockaddr *) &un_server, sizeof (struct sockaddr_un)) < 0)
            {
              return -1;
            }
        }
      else
        {
          // tcp socket
          string socketstr = o.clamdsocket;
          string host;
          string port;

          string::size_type p = socketstr.find (':');
          if (p == string::npos)
            {
              host = socketstr;
              port = "3310";
            }
          else
            {
              host = socketstr.substr (0, p);
              port = socketstr.substr (p + 1, socketstr.length ());
            }

#ifdef DGDEBUG
          std::cout << "Clamd: tcp connect to " << host << " " << port << std::endl;
#endif

          if ((sockd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
            {
              return -1;
            }
          in_server.sin_family = AF_INET;
          in_server.sin_port = htons (atoi (port.c_str ()));

          if ((he = gethostbyname (host.c_str ())) == 0)
            {
              close (sockd);
              return -1;
            }
          in_server.sin_addr = *(struct in_addr *) he->h_addr_list[0];

          if (strcmp (he->h_addr_list[0], "127.0.0.1") == 0)
            {
              // localhost; no need to do stream scanning
              stream_scan = false;
            }
          else
            {
              stream_scan = true;
            }

          if (connect (sockd, (struct sockaddr *) &in_server, sizeof (struct sockaddr_in)) < 0)
            {
              close (sockd);
              return -1;
            }
        }
    }
  return sockd;
}
