
/*
 * Copyright (c) 2000 David Stes.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published 
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: item.m,v 1.6 2001/02/04 17:06:20 stes Exp $
 */

#include <curses.h>
#include <assert.h>
#include <stdio.h>
#include <menu.h>
#include <unistd.h>
#include <stdlib.h>

#include <Object.h>
#include <ordcltn.h>
#include <ocstring.h>
#include "item.h"
#include "var.h"
#include "frame.h"
#include "menufram.h"
#include "cmdmenu.h"
#include "cursel.h"
#include "process.h"

id mkitem(id name,id action,id description,id itemmsg)
{
  id item = [Item new];
  [item name:name];
  [item action:action];
  [item description:description];
  [item itemmsg:itemmsg];
  return item;
}

@implementation Item

- (unsigned)hash
{
  return [[self e_name] hash];
}

- (BOOL)isEqual:x
{
  return (self == x)?YES:([self compare:x] == 0);
}

- (int)compare:x
{
  id a,b;
  a=[self e_name];
  b=[x e_name];
  return [a compare:b];
}

- (BOOL)isnop
{
  /* without evaluation testing for nop */
  if (action == nil) return YES;
  assert([action isKindOf:(id)[OrdCltn class]]);
  if ([action size]==1 && [[action at:0] isKindOf:(id)[String class]] && !strcmp([[action at:0] str],"nop")) return YES;
  return NO;
}

- (BOOL)isselectable
{
  return item_opts(eti_item) & O_SELECTABLE;
}

- setsiginthandler
{
  assert(activeframe);
  [activeframe setsiginthandler];
  if (oninterrupt) u_oninterrupt = oninterrupt;
  if (interrupt) u_interrupt = expandbool(interrupt,NO);
  return self;
}

- mark
{
  id c;
  int v=item_value(eti_item);
  [self setsiginthandler];
  if (action) c=expandstr(action); /* eval backq, but do not evalcmd */
  [activeframe setsiginthandler]; /* reset default */
  set_item_value(eti_item,!v);
  return self;
}

- eval
{
  id c;
  [self setsiginthandler];
  c=(action)?expandstr(action):nil;
  [activeframe closeshortterm];
  if (c) evalcmd(c);
  [activeframe setsiginthandler]; /* reset default */
  return self;
}

- action:a
{
  action = a;
  return self;
}

- name:a
{
  name = a;
  return self;
}

- show:a
{
  show = a;
  return self;
}

- interrupt:a
{
  interrupt = a;
  return self;
}

- oninterrupt:a
{
  oninterrupt = a;
  return self;
}

- selected:a
{
  selected = a;
  return self;
}

- (BOOL)isselected
{
  return eti_item != NULL && item_value(eti_item) != 0;
}

- setupselected
{
  if (eti_item) {
    /* works apparently only after multi-val menu has been created */
    int ok = set_item_value(eti_item,expandbool(selected,NO));
    assert(ok == E_OK);
  }
  return self;
}

- setselected
{
  int v=item_value(eti_item);
  if (v) {
    [Var define:"SELECTED" as:[String str:"true"] env:YES];
  } else {
    [Var define:"SELECTED" as:[String str:"false"] env:YES];
  }
  return self;
}

- setlininfo
{
  if (lininfo) {
    [Var define:"LININFO" as:expandstr(lininfo) env:YES];
  } else {
    [Var undefine:"LININFO"];
  }
  return self;
}

- lininfo:a
{
  lininfo = a;
  return self;
}

- inactive:a
{
  inactive = a;
  return self;
}

- description:a
{
  description = a;
  return self;
}

- itemmsg:a
{
  itemmsg = a;
  return self;
}

- destroy
{
  if (eti_item) free_item(eti_item);
  eti_item = NULL;
  return self;
}

- (BOOL)hasdescription
{
  return description != nil;
}

- (int)width
{
  assert(eti_item);return width;
}

- (BOOL)showing
{
  if (show) {
    return expandbool(show,YES);
  } else {
    return YES;
  }
}

- create
{
  id d;
  int ok;

  if (eti_item) [self destroy];
  if (![self showing]) return self;

  /* Linux ncurses returns eti_item NULL if newline contained in name */
  /* original FMLI seems to print the newlines ... */

  e_name = expandstr(name);
  if (e_name) { char *p = strchr([e_name str],'\n'); if (p) *p = '\0'; }

  d = expandstr(description);
  if (d) { char *p = strchr([d str],'\n'); if (p) *p = '\0'; }

  if (d) {
    eti_item = new_item([e_name size]?[e_name str]:" ",[d str]);
  } else {
    eti_item = new_item([e_name size]?[e_name str]:" ","");
  }
   
  width = [e_name size]; /* description, numcols always 1, no width needed */

  if (expandbool(inactive,NO)) {
    set_item_opts(eti_item,item_opts(eti_item) & (~O_SELECTABLE));
  }

  assert(eti_item);
  set_item_userptr(eti_item,(char*)self);

  return self;
}

- itemmsg
{
  if (itemmsg) {
    id d = expandstr(itemmsg);
    return d;
  } else {
    return nil;
  }
}

- e_name
{
  if (!e_name) e_name = expandstr(name);
  return e_name;
}

- (ITEM*)eti_item
{
  return eti_item;
}

- writeOn:process delim:(char*)d
{
  id s;
  int ok;
  if (lininfo) {
    s = expandstr(lininfo);
  } else {
    s = e_name;
  }
  [process dstwrite:[s str]];
  return self;
}

@end
 
