---------------------------------------------------------------------
-- A protocol GUI for the CurryTest tool:
--
-- The GUI process messages sent by running tests with the test tool
-- and summarizes the results in a GUI.
--
-- If the currytest tool is executed in batch mode, the return code is
-- positive if some error occurred.
--
-- @author Michael Hanus
-- @version July 2013
---------------------------------------------------------------------

import Socket
import ReadShowTerm
import GUI
import Read
import Assertion
import System
import List
import FlatCurry.Types
import FlatCurry.Files
import IO
import IOExts -- use IORefs
import FileGoodies  (stripSuffix)
import Distribution
import Directory
import FilePath     ((</>))

---------------------------------------------------------------------
-- Check arguments and call main function:
main :: IO ()
main = do
  args <- getArgs
  case args of
   ["-h"]     -> printUsage
   ["--help"] -> printUsage
   ["-?"]     -> printUsage
   "--window":modnames -> startGUI modnames
   "-w":modnames -> startGUI modnames
   "-window":modnames -> startGUI modnames
   ["-f",portnums] -> forwardMessages (readNat portnums)
   _:_ -> do let testmods = map stripSuffix args
             rcs <- mapIO (testModule putStrFlush 0) testmods
             if all (==0) rcs
              then exitWith 0
              else do
                putStrLn "FAILURE IN SOME TEST OCCURRED!!!"
                putStrLn $ "FAILED TEST MODULES:" ++
                    concatMap (\ (rc,tmod) -> if rc==0 then "" else ' ':tmod)
                              (zip rcs testmods)
                exitWith 1
   _ -> do putStrLn $ "ERROR: Illegal arguments for currytest: " ++
                      concat (intersperse " " args) ++ "\n"
           printUsage
           exitWith 1
 where
   putStrFlush s = putStr s >> hFlush stdout

printUsage :: IO ()
printUsage = putStrLn $ unlines
  [ "A tool to run simple tests for Curry programs"
  , ""
  , "Usage:"
  , ""
  , "    curry test [--window|-w] <module_names>"
  ]

-- This operation creates a new socket to receive messages that are forwarded
-- to a continues connection to a socket with the argument port number:
forwardMessages :: Int -> IO ()
forwardMessages guiportnum = do
  (portnum,socket) <- listenOnFresh
  putStrLn $ "Forwarding messages from port "++show portnum++
             " to port "++show guiportnum++"..."
  guihandle <- connectToSocket "localhost" guiportnum
  hPutStrLn guihandle (show portnum) -- first message is my port number
  hFlush guihandle
  acceptAndForwardMessage guihandle socket
 where
   acceptAndForwardMessage guihandle socket = do
     (_,h) <- socketAccept socket
     str <- hGetLine h
     hClose h
     --putStrLn ("MESSAGE: "++str)
     if str=="TERMINATE" then sClose socket else do
       hPutStrLn guihandle str
       hFlush guihandle
       acceptAndForwardMessage guihandle socket

terminateForwardMessages :: Int -> IO ()
terminateForwardMessages portnum = do
  h <- connectToSocket "localhost" portnum
  hPutStrLn h "TERMINATE"
  hClose h

startGUI :: [String] -> IO ()
startGUI modnames = do
  (guiportnum,socket) <- listenOnFresh
  system (installDir++"/bin/currytest -f "++show guiportnum++" &")
  (_,inhandle) <- socketAccept socket
  portnums <- hGetLine inhandle
  let portnum = readNat portnums
  --putStrLn ("Using port number: "++show portnum)
  stateref <- newIORef (0,[])
  mapIO_ (addModule stateref) (map stripSuffix modnames)
  mods <- getModules stateref
  runHandlesControlledGUI "CurryTest Tool"
                          (protocolGUI portnum mods stateref) [inhandle]
  terminateForwardMessages portnum

-- A text edit widget with vertical scrollbar.
TextEditVScroll :: [ConfItem] -> Widget
TextEditVScroll confs =
   Row []
     [TextEdit ([WRef txtref, Fill]++confs),
      ScrollV txtref [FillY]]     where txtref free

---------------------------------------------------------------------
-- Functions to manipulate the GUI state (pair of module name list and
-- module index):

-- add a module name:
addModule :: IORef ((Int,[String])) -> String -> IO ()
addModule ref modname = do
  (_,mods) <- readIORef ref
  writeIORef ref (length mods+1,mods++[modname])

-- delete all modules:
deleteModules :: IORef ((Int,[String])) -> IO ()
deleteModules ref = writeIORef ref (0,[])

-- get list of modules as string representation:
getModules :: IORef ((Int,[String])) -> IO (String)
getModules ref = do
  (i,mods) <- readIORef ref
  return (concatMap (\ (c,n)->c++n++"\n")
                    (zip (replace "==>" i (repeat "   ")) mods))

-- intialize module index for testing:
initModuleIndex :: IORef ((Int,[String])) -> IO ()
initModuleIndex ref = do
  (_,mods) <- readIORef ref
  writeIORef ref (0,mods)

-- get next module to be tested (or Nothing):
getNextTestModule :: IORef ((Int,[String])) -> IO (Maybe String)
getNextTestModule ref = do
  (i,mods) <- readIORef ref
  if i >= length mods
   then return Nothing
   else writeIORef ref (i+1,mods) >> return (Just (mods!!i))

-- get name of current test module:
getCurrentTestModule :: IORef ((Int,[String])) -> IO String
getCurrentTestModule ref = do
  (i,mods) <- readIORef ref
  return (mods!!(i-1))


---------------------------------------------------------------------
-- The definition of the protocol GUI together with a handler
-- "ext_handler" that is responsible to handle the external messages
-- sent during running the test on the program:
protocolGUI :: Int -> String -> IORef ((Int,[String])) -> (Widget,[Handle -> GuiPort -> IO ([_])])
protocolGUI portnum initmods stateref =
 (Row []
   [Col [LeftAlign]
     [Row [] [Label [Text "Test cases:"],
                Entry [WRef rtestnum, Text "0", Width 5],
                Label [Text "Failures:"],
                Entry [WRef rfailnum, Text "0", Width 5,
                       Background "green"],
                Label [Text "Status:"],
                Entry [WRef rstatus, Text "ready", Width 10,
                       Background "green"]],
      Label [Text "Failed test cases:"],
      TextEditScroll [WRef rfail, Height 10, Width 60,
                      Background "green"],
      Label [Text "Test protocol:"],
      TextEditScroll [WRef rprot, Height 15, Width 60,
                        Background "white"]],
    Col [LeftAlign]
     [Row [LeftAlign] [Button starttest [Text "Run test"],
                       Button openfile  [Text "Add test module"],
                       Button delete    [Text "Clear test modules"],
                       Button exitGUI   [Text "Exit"]],
      Label [Text "Modules to be tested:"],
      TextEditVScroll [WRef rmods, Height 10, Width 60,
                         Text initmods, Background "white"],
      Label [Text "Compilation messages:"],
      TextEditScroll [WRef rcmsgs, Height 15, Width 60,
                        Background "white"]]
   ], [ext_handler])
 where
   rtestnum,rfailnum,rstatus,rfail,rprot,rcmsgs,rmods free

   openfile gp =
      getOpenFileWithTypes curryFileTypes >>= \filename ->
      if null filename then done else
      addModule stateref (stripSuffix filename) >>
      showModules gp

   delete gp = do
      deleteModules stateref
      showModules gp

   starttest gp = do
     setConfig rfailnum (Background "green") gp
     setConfig rfail (Background "green") gp
     setValue rtestnum "0" gp
     setValue rfailnum "0" gp
     setValue rfail "" gp
     setValue rprot "" gp
     setValue rcmsgs "" gp
     setConfig rstatus (Background "yellow") gp
     setValue rstatus "testing" gp
     initModuleIndex stateref
     startTestModule gp

   -- test a module, if present:
   startTestModule gp =
     showModules gp >>
     getNextTestModule stateref >>=  \nextmod ->
     maybe (setValue rstatus "ready" gp >>
            setConfig rstatus (Background "green") gp)
           (\m -> testModule (printCompMsg gp) portnum m >> done)
           nextmod

   -- print a compilation message in corresponding widget:
   printCompMsg gp msg = appendValue rcmsgs msg gp

   -- update list of modules in widget:
   showModules gp = do
     mods <- getModules stateref
     setValue rmods mods gp
     (row,_) <- readIORef stateref
     seeText rmods (row,1) gp

   ext_handler h gp = do
     msgstring <- hGetLine h
     processTestMsg (readUnqualifiedTerm ["Assertion","Prelude"] msgstring) gp
     return []

   processTestMsg (TestModule m) gp =
      appendValue rprot
                  (take 60 (repeat '=')++"\nTesting module: "++m++"\n") gp

   processTestMsg TestFinished gp = startTestModule gp

   processTestMsg TestCompileError gp = do
     setConfig rfailnum (Background "red") gp
     setConfig rfail (Background "red") gp
     updateValue incrText rfailnum gp
     mod <- getCurrentTestModule stateref
     appendValue rfail ("Compilation error in module: "++mod++"\n") gp
     startTestModule gp

   processTestMsg (TestCase s b) gp = do
      updateValue incrText rtestnum gp
      if b then done
           else do setConfig rfailnum (Background "red") gp
                   setConfig rfail (Background "red") gp
                   updateValue incrText rfailnum gp
                   mod <- getCurrentTestModule stateref
                   appendValue rfail ("Module: "++mod++"\n") gp
                   appendValue rfail (s++"\n") gp
      appendValue rprot s gp

-- Curry file types:
curryFileTypes :: [(String,String)]
curryFileTypes = [("Curry Files",".curry"),
                  ("Literate Curry files",".lcurry")]

-- increment number text string:
incrText :: String -> String
incrText s = show (readInt s + 1)


---------------------------------------------------------------------------
-- Main function to test a module:
-- Arg 1: function for printing compilation messages
-- Arg 2: port number of socket where the test messages are sent
--        (or 0 if no gui defined)
-- Arg 3: module name
-- The return code is positive if some tests failed and the GUI is not used
testModule :: (String -> IO _) -> Int -> String -> IO Int
testModule prtmsg portnum modname = do
  prtmsg ("Loading module \""++modname++"\"...\n")
  prog_or_error <- tryReadFlatCurry modname
  testModuleIfPossible prtmsg portnum modname prog_or_error

testModuleIfPossible :: (String -> IO _) -> Int -> String -> Either Prog (String) -> IO Int
testModuleIfPossible prtmsg portnum _ (Right errmsg) = do
  prtmsg ("ERROR: compilation not successful:\n\n"++errmsg++"\n")
  if portnum==0 then done else showTestCompileError portnum
  return 1
testModuleIfPossible prtmsg portnum modname (Left prog) =
  execTestFunctions prtmsg portnum modname (getTestFunctionNames prog)

execTestFunctions :: (String -> IO _) -> Int -> String -> [String] -> IO Int
execTestFunctions prtmsg portnum _ [] = do
  prtmsg "No test functions found.\n\n"
  if portnum==0 then done else showTestEnd portnum
  return 0
execTestFunctions prtmsg portnum modname fs@(_:_) = do
  prtmsg ("Exported top-level test functions:\n"
            ++ concatMap (++" ") fs ++ "\n\n")
  if portnum/=0 then showTestMod portnum modname else done
  let testgoal =
         "putStrLn (take 60 (repeat (chr 61))) >> " ++
         "putStrLn (\"Testing module \\\""++modname++"\\\"...\") >> " ++
         concat
           (intersperse " `seqStrActions` "
             (map (\f -> "checkAssertion \"" ++ f ++ "\" " ++
                         (if portnum/=0 then "(showTestCase "++show portnum++") "
                                        else "return ") ++ f)
                  fs)) ++
         " >>= writeAssertResult " ++
         (if portnum/=0 then " >> showTestEnd "++show portnum
                        else " >>= exitWith")
      execCall = case curryCompiler of
        "pakcs" -> "echo ':l "++modname++"\n:add Assertion\n:add System\n"++
                   testgoal++" ' | \"" ++
                   installDir++"/bin/pakcs\" -Dshowfcyload=no -quiet 2>&1"
        "kics2" -> "echo ':l "++modname++"\n:add Assertion\n:add System\n"++
                   ":save "++testgoal++"\n:q\n' | \""++
                   installDir++"/bin/kics2\" && "++ ("." </> modname) ++
                   " && rm "++modname++" 2>&1"
        _ -> error "CurryTest: unknown Curry compiler!"
  --putStrLn testgoal
  if portnum==0 then system execCall
                else system (execCall ++ " &") >> return 0


-- Extract all test functions from a module:
getTestFunctionNames :: Prog -> [String]
getTestFunctionNames (Prog _ _ _ funs _) =
   map funcname . filter isExportedFunc . filter hasAssertType $ funs
 where
   isExportedFunc (Func _ _ vis _ _) = vis==Public

   funcname (Func (_,fname) _ _ _ _) = fname


hasAssertType :: FuncDecl -> Bool
hasAssertType (Func _ _ _ texp _) =
 case texp of
   TCons tc _ -> tc==("Assertion","Assertion")
   _          -> False


-- Tries to read a FlatCurry program.
-- Returns either (Left prog) (if reading was successful)
-- or (Right msg) where msg is the string of error messages from the parser
tryReadFlatCurry :: String -> IO (Either Prog String)
tryReadFlatCurry mname = do
  pofile <- getPOFilename
  callFrontendWithParams FCY (setLogfile pofile defaultParams) mname
  exfcy <- doesFileExist (flatCurryFileName mname)
  -- check whether parsing was ok and return appropriate value:
  if exfcy
   then
     do prog <- readFlatCurryFile (flatCurryFileName mname)
        system ("rm "++pofile)
        return (Left prog)
   else
     do msgs <- readFile pofile
        system ("rm "++pofile)
        return (Right msgs)
 where
   -- compute name for auxiliary file for parser outputs:
   getPOFilename =
     do pid <- getPID
        return ("/tmp/pakcsoutput_"++show pid)
