#!/usr/local/bin/python
"""
Cubictemp v 0.4

Copyright (c) 2004, Nullcube Pty Ltd
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

*   Redistributions of source code must retain the above copyright notice, this
    list of conditions and the following disclaimer.
*   Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.
*   Neither the name of Nullcube nor the names of its contributors may be used to
    endorse or promote products derived from this software without specific prior
    written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

"""
import cgi, sre, __builtin__

class tempException(Exception): pass

def escape(s):
    """
        Replace special characters '&', '<', '>', ''', and '"' with the
        appropriate HTML escape sequences.
    """
    s = s.replace("&", "&amp;") # Must be done first!
    s = s.replace("<", "&lt;")
    s = s.replace(">", "&gt;")
    s = s.replace('"', "&quot;")
    s = s.replace("'", "&#146;")
    return s


class Temp:
    _cubictemp_unescaped = 1
    # An expression can either be an (if ... then ... else) construct, or a
    # simple expression.
    _strExpression = r"""
                        (?P<flavor>@|\$)!
                            (
                                # An if construct:
                                (\s*if\s+
                                    (?P<cond>.+?)
                                \s+then\s+
                                    (?P<true>.+?)
                                \s+else\s+
                                    (?P<false>.+?)
                                \s*)

                                # ... or a plain expression:
                                |(?P<expr>.+?)
                            )
                        !(?P=flavor)
                    """
    _reExpression = sre.compile(_strExpression, sre.X)

    # There are 3 kinds of blocks - for blocks, named blocks, and raw blocks:
    #           - A block needs to start and end with exactly the same
    #             whitespace indentation. 
    #           - Any nested block should have a greater level of indentation
    #             than the enclosing block. 
    _strBlock = r"""
                        # First the block opening marker. We take note of the
                        # whitespace, so we can match it against the closing
                        # marker. 
                        ^(?P<space>\s*)        
                        <!--\(\s*               
                            (
                                    for\s+(?P<varName>\w+)\s+in\s+(?P<iterable>.+)
                                |   block\s+(?P<blockName>.+)
                                |   (?P<raw>raw)
                            )
                        \s*\)(-->)?.*\n          
                            (?P<contents>(.*(\n))*?)      

                        # Now for the end marker. 
                        ^(?P=space)             # Check for the required whitespace
                        (<!--)?\(\s*           
                            end
                        \s*\)-->?.*\n?          # This could be the end of the file, so 
                                                # newline is not required. 
                    """
    _reBlock = sre.compile(_strBlock, sre.X|sre.M)

    def __init__(self, input, **nsDict):
        self.nsDict, self.input = nsDict, input
        self.blockCache = {}
        self.evalCache = {}

    def __repr__(self):
        return "".join(self.__render(self.input, self.nsDict))

    def __call__(self, **override):
        ns = self.nsDict.copy()
        ns.update(override)
        return "".join(self.__render(self.input, ns))

    def _findBlocks(self, text):
        """
            Keep a cache of block matches for given pieces of text. 
        """
        if not self.blockCache.has_key(text):
            self.blockCache[text] = list(sre.finditer(self._reBlock, text))
        return self.blockCache[text]

    def _ceval(self, expr, nsDict):
        """
            Similar to eval, but cache pre-compiled expressions.
        """
        if not self.evalCache.has_key(expr):
            self.evalCache[expr] = compile(expr, "<string>", "eval")
        return eval(self.evalCache[expr], {}, nsDict)

    def __render(self, input, nsDict):
        def glorb(match):
            try:
                if match.group("expr"):
                    ret = self._ceval(match.group("expr"), nsDict)
                else:
                    if eval(match.group("cond"), {}, nsDict):
                        ret = self._ceval(match.group("true"), nsDict)
                    else:
                        ret = self._ceval(match.group("false"), nsDict)
            # Any other errors we need to catch here?
            except (NameError, SyntaxError), value:
                raise tempException("%s" % (value))
            if match.group("flavor") == "@":
                if not getattr(ret, "_cubictemp_unescaped", 0):
                    return escape(str(ret))
            return str(ret)

        matchList = self._findBlocks(input)
        output = []
        end = 0
        for match in matchList:
            if match.start() > end:
                # We've skipped some data...
                output.append(self._reExpression.sub(glorb, input[end:match.start()]))
            groupDict = match.groupdict()
            if groupDict["iterable"]:
                try:
                    loopIter = iter(eval(groupDict["iterable"], {}, nsDict))
                except TypeError:
                    raise tempException("Cannot loop over %s." % groupDict["iterable"])
                except (NameError, SyntaxError), val:
                    raise tempException(val)
                for i in loopIter:
                    nsDict[groupDict["varName"]] = i
                    output.extend(self.__render(groupDict["contents"], nsDict))
            elif groupDict["blockName"]:
                nsDict[groupDict["blockName"]] = Temp(groupDict["contents"], **nsDict)
            elif groupDict["raw"]:
                output.append(groupDict["contents"])
            end = match.end()

        if end < len(input):
            # We have some data left over...
            output.append(self._reExpression.sub(glorb, input[end:]))
        return output


class File(Temp):
    def __init__(self, filename, **nsDict):
        self.filename = filename
        data = open(filename).read()
        Temp.__init__(self, data, **nsDict)

    def __repr__(self):
        try:
            return Temp.__repr__(self)
        except tempException, val:
            raise tempException, "%s: %s"%(self.filename, str(val))
