/* mkset -- create a level set for Crimson Fields
   Copyright (C) 2000, 2001 Jens Granseuer

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/* create a data set for WF, containing graphics and definitions
   for units and terrain

   currently, all data (except the graphics) is hardcoded into
   the program

   level designers can specify custom data files to adjust the
   graphics and/or units to their likes
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "SDL.h"

#include "editor.h"

#define NUM_TERRAIN_TYPES	58
#define NUM_SOUNDS		0

#define FID_SET			MakeID('M','S','E','T')

extern const struct UnitType UModels[];
extern const struct TerrainType TModels[];

const char *set_sounds[] = { NULL };

static char *PutDump( char *dest, const char *from, short nn );
static char *PutRun( char *dest, short nn, char cc );
static long ByteRun1_pack( const char *src, char *destbuf, unsigned long srcsize );

struct PicHeader {
  unsigned short colors;
  unsigned short width;
  unsigned short height;
  unsigned long packed_size;
};

struct DatHeader {
  unsigned long fid;
  unsigned short units;
  unsigned short terrains;
  unsigned char sounds;
};


int main( int argc, char *argv[] ) {
  SDL_Surface *bmp;
  int status, i, j;
  char *packbuf, *upackbuf;
  SDL_RWops *out;
  struct DatHeader dhead = { FID_SET, NUM_UNIT_TYPES, NUM_TERRAIN_TYPES, NUM_SOUNDS };

  if ( argc != 3 ) {
    fprintf(stderr, "Invalid number of arguments\n"
                    "Usage: %s <graphics.bmp> <outfile>\n",
                    argv[0] );
    exit(-1);
  }


  if ( SDL_Init(0) < 0 ) {
    fprintf(stderr, "Couldn't init SDL: %s\n", SDL_GetError());
    exit(-1);
  }
  atexit(SDL_Quit);

  out = SDL_RWFromFile( argv[2], "wb" );
  if ( !out ) {
    fprintf(stderr, "Couldn't open output file %s\n", argv[2] );
    exit(-1);
  }

  status = 0;

  /* write data file header */
  SDL_RWwrite( out, (char *)&dhead.fid, sizeof(unsigned long), 1 );
  SDL_WriteLE16( out, dhead.units );
  SDL_WriteLE16( out, dhead.terrains );

  /* write sound file names */
  SDL_RWwrite( out, &dhead.sounds, sizeof(unsigned char), 1 );
  for ( i = 0; i < NUM_SOUNDS; i++ ) {
    unsigned char len = strlen( set_sounds[i] );
    SDL_RWwrite( out, &len, sizeof(unsigned char), 1 );
    SDL_RWwrite( out, set_sounds[i], sizeof(char), len );
  }

  /* write unit data */
  for ( i = 0; i < NUM_UNIT_TYPES; i++ ) {
    SDL_WriteLE16( out, UModels[i].ut_terrain );
    SDL_WriteLE16( out, UModels[i].ut_image );
    SDL_RWwrite( out, &UModels[i].ut_moves, sizeof(unsigned char), 23 );
    SDL_RWwrite( out, UModels[i].ut_name, sizeof(char), 20 );
  }

  /* write terrain data */
  for ( i = 0; i < NUM_TERRAIN_TYPES; i++ ) {
    SDL_WriteLE16( out, TModels[i].tt_type );
    SDL_WriteLE16( out, TModels[i].tt_image );
    SDL_RWwrite( out, &TModels[i].tt_att_mod, sizeof(char), 3 );
    SDL_WriteLE32( out, TModels[i].tt_color );
  }

  /* load graphics */
  bmp = SDL_LoadBMP(argv[1]);
  if ( bmp ) {
    if ( bmp->format->BitsPerPixel == 8 ) {
      SDL_Palette *palette = bmp->format->palette;
      struct PicHeader pict = { palette->ncolors, bmp->w, bmp->h, 0 };

      packbuf = (char *)malloc( 2 * bmp->w * bmp->h );
      upackbuf = &packbuf[ bmp->w * bmp->h ];

      /* copy surface lines to buffer */
      for ( i = 0; i < bmp->h; i++ ) {
        for ( j = 0; j < bmp->w; j++ ) {
          upackbuf[j + i * bmp->w] = *((char *)bmp->pixels + j + i * bmp->pitch);
        }
      }

      pict.packed_size = ByteRun1_pack( upackbuf, packbuf, bmp->w * bmp->h );

      SDL_WriteLE16( out, pict.colors );
      SDL_WriteLE16( out, pict.width );
      SDL_WriteLE16( out, pict.height );
      SDL_WriteLE32( out, pict.packed_size );

      for ( i = 0; i < palette->ncolors; ++i )
        SDL_RWwrite( out, (char *)&palette->colors[i], sizeof(Uint8), 3 );

      SDL_RWwrite( out, packbuf, sizeof(char), pict.packed_size );
      free( packbuf );
    } else fprintf(stderr, "Couldn't find palette\n" );
    SDL_FreeSurface( bmp );
  } else {
    fprintf(stderr, "Couldn't open BMP %s\n", argv[2] );
    exit(-1);
  }

  SDL_RWclose( out );
  return 0;
}

/**********************************************************************
* cmpByteRun1 compression algorithm
*
* control bytes:
*  [0..127]   : followed by n+1 bytes of data
*  [-1..-127] : followed by byte to be repeated (-n)+1 times
*  -128       : NOOP
**********************************************************************/

#define DUMP	0
#define RUN	1

#define MINRUN 3
#define MAXRUN 128
#define MAXLEN 128	/* otherwise exceeds uchar range */

/* buffer size must be at least 2 * MAXLEN */
static char brbuf[2*MAXLEN];

static char *PutDump( char *dest, const char *from, short nn ) {
  short i;
  *dest++ = nn-1;
  for ( i = 0; i < nn; i++ ) *dest++ = *from++;
  return dest;
}

static char *PutRun( char *dest, short nn, char cc ) {
  *dest++ = -(nn-1);
  *dest++ = cc;
  return dest;
}

/* Compresses given data using cmpByteRun1 compression. returns size of packed data. */
static long ByteRun1_pack( const char *src, char *destbuf, unsigned long srcsize ) {
  char *dest = destbuf, c, lastc;
  short mode = DUMP;
  short nbuf = 1;			/* number of chars in buffer */
  short rstart = 0;			/* buffer index current run starts */

  brbuf[0] = lastc = *src++;

  for ( srcsize--; srcsize; srcsize-- ) {
    brbuf[nbuf++] = c = *src++;
    if ( mode == DUMP ) {
      /* If the buffer is full, write the length byte, then the data */
      if ( nbuf > MAXLEN ) {
        dest = PutDump( dest, brbuf, MAXLEN );
        brbuf[0] = c;
        nbuf = 1; rstart = 0;
      } else if ( c == lastc ) {
        if ( nbuf-rstart >= MINRUN ) {
          if ( rstart > 0 ) dest = PutDump( dest, brbuf, rstart );
          mode = RUN;
        } else if ( rstart == 0 ) mode = RUN;
      } else rstart = nbuf-1;		/* first of run */ 

    /* mode == RUN */
    } else if ( (c != lastc) || (nbuf-rstart > MAXRUN) ) {
      /* output run */
      dest = PutRun( dest, nbuf-rstart-1, lastc );
      brbuf[0] = c;
      nbuf = 1; rstart = 0;
      mode = DUMP;
    }

    lastc = c;
  }

  if ( mode == DUMP ) dest = PutDump( dest, brbuf, nbuf );
  else dest = PutRun( dest, nbuf-rstart, lastc );

  return dest-destbuf;
}

