// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// widget.cpp
//
// History:
//  30-11-2000 - created
////////////////////////////////////////////////////////////////////////

#include "widget.h"
#include "sound.h"

////////////////////////////////////////////////////////////////////////
// NAME       : Widget::Widget
// DESCRIPTION: Initialize the widget object. After initialization, it
//              still needs to be pointed to a surface to draw to,
//              usually by attaching it to a Window.
// PARAMETERS : id     - widget ID; by default, this ID is returned to
//                       the event handler upon widget activation
//              x      - left edge of widget
//              y      - top edge of widget
//              w      - widget width
//              h      - widget height
//              key    - keystroke to activate widget
//              flags  - widget flags (see widget.h for details)
//              title  - widget title, may be NULL
//              window - window to attach widget to
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Widget::Widget( short id, short x, short y, unsigned short w, unsigned short h,
        short key, unsigned short flags, const char *title, Window *window ) :
        Rect( x, y, w, h ) {
  font = NULL;

  this->id = id;
  this->flags = flags;
  this->key = key;
  this->title = title;
  surface = window;

  clicked = false;

  window->AddWidget( this );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Widget::Show
// DESCRIPTION: Copy the widget to the display surface.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Widget::Show( void ) const {
  surface->Show( *this );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Widget::Push
// DESCRIPTION: Change the widget state to 'clicked', then redraw and
//              reblit it to reflect that change graphically.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
//   25-04-2001 - play sound effect when called
////////////////////////////////////////////////////////////////////////

void Widget::Push( void ) {
  if ( !clicked ) {
    play_audio( SND_GUI_PRESSED, 0 );
    clicked = true;
    Draw();
    Show();
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Widget::Release
// DESCRIPTION: Change the widget state to 'released', then redraw and
//              reblit it to reflect that change graphically.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Widget::Release( void ) {
  clicked = false;
  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : Widget::HandleEvent
// DESCRIPTION: Distribute system events to the proper handling
//              functions.
// PARAMETERS : event - event received by the (window) event handler
// RETURNS    : GUI status
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status Widget::HandleEvent( const SDL_Event &event ) {

  switch ( event.type ) {
  case SDL_MOUSEBUTTONDOWN: return MouseDown( event.button ); break;
  case SDL_MOUSEBUTTONUP:   return MouseUp( event.button ); break;
  case SDL_KEYDOWN:         return KeyDown( event.key.keysym ); break;
  case SDL_KEYUP:           return KeyUp( event.key.keysym ); break;
  case SDL_MOUSEMOTION:     return MouseMove( event.motion ); break;
  default: return GUI_OK;
  }
} 

