// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// container.cpp
//
// History:
//  12-12-2000 - created
////////////////////////////////////////////////////////////////////////

#include "SDL_endian.h"

#include "container.h"

////////////////////////////////////////////////////////////////////////
// NAME       : UnitContainer::UnitContainer
// DESCRIPTION: Initialize a unit container.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

UnitContainer::UnitContainer( void ) :
  uc_slots( UC_MAX_SLOTS ), uc_slots_full(0),
  uc_max_weight( UC_MAX_WEIGHT ) {}

////////////////////////////////////////////////////////////////////////
// NAME       : UnitContainer::InsertUnit
// DESCRIPTION: Put a unit into the container. No checks are performed.
// PARAMETERS : unit - unit to insert
// RETURNS    : 1 if the owner of the building changed because of the
//              insertion (an enemy unit with the U_CONQUER flag came
//              in), -1 on error, 0 otherwise
//
// HISTORY
//   12-06-2001 - handle dummies specially
////////////////////////////////////////////////////////////////////////

short UnitContainer::InsertUnit( Unit *unit ) {
  short rc = 0;

  if ( unit->IsDummy() ) unit->SetFlags( U_SHELTERED );
  else {
    UCNode *n = new UCNode( unit );
    if ( n ) {
      unit->SetFlags( U_SHELTERED );
      uc_units.AddTail( n );
      uc_slots_full++;

      if ( Owner() != unit->Owner() ) rc = 1;

      // if a transport is coming in, we must unload it
      if ( unit->IsTransport() ) {
        Transport *t = static_cast<Transport *>(unit);
        short num = t->FullSlots();

        while ( num-- ) {
          Unit *u = t->GetUnit( 0 );
          t->RemoveUnit( u );
          InsertUnit( u );
        }
      }
    } else rc = -1;
  }
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : UnitContainer::RemoveUnit
// DESCRIPTION: Take a unit out of the container.
// PARAMETERS : unit - unit to remove from container
// RETURNS    : -
//
// HISTORY
//   12-06-2001 - handle dummies specially
////////////////////////////////////////////////////////////////////////

void UnitContainer::RemoveUnit( Unit *unit ) {
  if ( unit->IsDummy() ) unit->UnsetFlags( U_SHELTERED );
  else {

    UCNode *n = static_cast<UCNode *>( uc_units.Head() );

    while ( n ) {
      if ( n->uc_unit == unit ) {
        n->Remove();
        delete n;

        unit->UnsetFlags( U_SHELTERED );
        uc_slots_full--;
        break;
      }
      n = static_cast<UCNode *>( n->Next() );
    }
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : UnitContainer::GetUnit
// DESCRIPTION: Get the unit in the specified slot from the container.
// PARAMETERS : slot - slot number for the unit you want
// RETURNS    : pointer to the unit in slot, or NULL if none
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Unit *UnitContainer::GetUnit( short slot ) const {
  UCNode *n = static_cast<UCNode *>( uc_units.GetNode( slot ) );
  if ( n ) return n->uc_unit;
  return NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : UnitContainer::Allow
// DESCRIPTION: Check whether a unit is allowed to enter.
// PARAMETERS : unit - unit to check permission for
// RETURNS    : true if unit may enter, false if not
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool UnitContainer::Allow( const Unit *unit ) const {
  if ( unit->Weight() > uc_max_weight ) return false;

  if ( unit->IsTransport() )
     return( uc_slots > uc_slots_full + static_cast<const Transport *>(unit)->FullSlots() );
  return( uc_slots > uc_slots_full );
}


////////////////////////////////////////////////////////////////////////
// NAME       : Transport::Transport
// DESCRIPTION: Create a new transport instance.
// PARAMETERS : type   - unit type definition
//              player - transport controller
//              id     - unique transport unit identifier
//              x      - horizontal position on map
//              y      - vertical position on map
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Transport::Transport( const UnitType *type, Player *player, unsigned short id,
                      short x, short y ) :
     Unit( type, player, id, x, y ), t_crystals(0) {
  uc_slots = u_type->ut_trans_slots;
  uc_max_weight = u_type->ut_trans_weight;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Transport::Transport
// DESCRIPTION: Load a transport from a data file.
// PARAMETERS : file    - data file descriptor
//              types   - array of unit type definitions
//              players - array of players
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Transport::Transport( SDL_RWops *file, const UnitType *types, Player **players ) :
           Unit( file, types, players ) {
  t_crystals = SDL_ReadLE16( file );
  uc_slots = u_type->ut_trans_slots;
  uc_max_weight = u_type->ut_trans_weight;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Transport::Save
// DESCRIPTION: Save transport information to a file.
// PARAMETERS : file - save file descriptor
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int Transport::Save( SDL_RWops *file ) const {
  Unit::Save( file );
  SDL_WriteLE16( file, t_crystals );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Transport::SetPosition
// DESCRIPTION: When a transport is moved on the map, it must not only
//              keep track of its own position, but update the
//              information for all units inside as well.
// PARAMETERS : x - new horizontal coordinate
//              y - new vertical coordinate
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Transport::SetPosition( short x, short y ) {
  Unit::SetPosition( x, y );

  UCNode *n = static_cast<UCNode *>( uc_units.Head() );
  while ( n ) {
    n->uc_unit->SetPosition( x, y );
    n = static_cast<UCNode *>( n->Next() );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Transport::Hit
// DESCRIPTION: When a transport is hit chances are that the units
//              inside are damaged as well. If the transport is
//              destroyed, so are the carried units.
// PARAMETERS : damage - amount of damage taken
// RETURNS    : true if transport was destroyed, false otherwise
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool Transport::Hit( unsigned short damage ) {
  if ( Unit::Hit( damage ) ) {                  // destroyed
    UCNode *n = static_cast<UCNode *>( uc_units.Head() );
    while ( n ) {
      n->uc_unit->Hit( MAX_GROUP_SIZE );
      n = static_cast<UCNode *>( n->Next() );
    }
    return true;
  } else {            // randomly damage units inside
    UCNode *n = static_cast<UCNode *>( uc_units.Head() );
    while ( n ) {
      n->uc_unit->Hit( random( 0, damage ) );
      n = static_cast<UCNode *>( n->Next() );
    }
    return false;
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Transport::Allow
// DESCRIPTION: Check whether a unit is allowed to enter the transport.
// PARAMETERS : unit - unit to check permission for
// RETURNS    : true if unit may enter, false if not
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool Transport::Allow( const Unit *unit ) const {
  if ( unit->Owner() == u_player ) return UnitContainer::Allow( unit );
  return false;
}

