/* text.simple-filters.c - Plugin code for various filters
 * 
 * This program is part of Crank, a cryptanalysis tool
 * Copyright (C) 2000 Matthew Russell
 *
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License (LICENSE) for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "crank-interface.h"

/* Global data */

static int remove_numbers;
static int remove_punctuation;
static int remove_whitespace;
static int to_uppercase;
static int to_lowercase;
static int group_in_fives; /* Use cryptogram telegraph style:- fhsnm ojfwe sljfo ojjw llfaf....*/

static GtkWidget *checkbox_remove_numbers;
static GtkWidget *checkbox_remove_punctuation;
static GtkWidget *checkbox_remove_whitespace;
static GtkWidget *checkbox_to_uppercase;
static GtkWidget *checkbox_to_lowercase;
static GtkWidget *checkbox_group_in_fives;

/* Constants for the user interface controls */
#define REMOVE_NUMBERS     1
#define REMOVE_PUNCTUATION 2
#define REMOVE_WHITESPACE  3
#define TO_UPPERCASE       4
#define TO_LOWERCASE       5
#define GROUP_IN_FIVES     6

/* Function declares */
static GtkWidget *add_opt(char *txt, int type, GtkWidget *hbox);
static void cb_check_toggle(GtkWidget *widget, gpointer gdata);

/* Plugin Interface */
/* ---------------- */

const char name[] = "text.simple-filters";
const char version[] = VERSION;
const char author[] = "Matthew Russell";
const char description[] = "Various filters for changing case, stripping punctuation, numbers and whitespace, and grouping letters into fives.";
const int interface_version = 1;
const int plugin_type = PLUGIN_MODE;
const char menu_string[] = "/Text/Simple Filters";

int boot(void) {
    remove_numbers = FALSE;
    remove_punctuation = FALSE;
    remove_whitespace = FALSE;
    to_uppercase = FALSE;
    to_lowercase = FALSE;
    group_in_fives = FALSE;
    return PLUGIN_BOOT_OK;
}

void init_plugin(void) {;}

void deactivate_plugin(void) {;}

GtkWidget *make_widget(char *text) {
    GtkWidget *hbox;
    hbox = gtk_hbox_new(5, FALSE);
    checkbox_remove_numbers = add_opt("Remove numbers",     REMOVE_NUMBERS, hbox);
    checkbox_remove_punctuation = add_opt("Remove punctuation", REMOVE_PUNCTUATION,hbox);
    checkbox_remove_whitespace = add_opt("Remove whitespace",  REMOVE_WHITESPACE, hbox);
    checkbox_to_uppercase = add_opt("To uppercase",       TO_UPPERCASE, hbox);
    checkbox_to_lowercase = add_opt("To lowercase",       TO_LOWERCASE, hbox);
    checkbox_group_in_fives = add_opt("Group in fives",     GROUP_IN_FIVES, hbox);
    return hbox;
}

char *transform(char *text) {
    char *newtext, 
	 c, /* From old string */ 
 	 C; /* After case changes */
    int i, position = 0, length = strlen(text);
    newtext = malloc((sizeof(char) * (length + 1) * 6) / 5); /* The 6/5 is to accomodate the possible expansion of the string due to 'group in fives' */

    for (i = 0; i < length; i++) {
	c = text[i];
	if (remove_numbers && isdigit(c))
	    continue;
	if (remove_punctuation && ispunct(c))
	    continue;
	if ((remove_whitespace || group_in_fives) && isspace(c))
	    continue;

        if (to_uppercase)
	    C = toupper(c);
	else if (to_lowercase)
	    C = tolower(c);
	else
	    C = c;

	newtext[position++] = C;
	
	if (group_in_fives && !((1 + position) % 6))
	    newtext[position++] = ' ';
	
    }
    newtext[position] = '\0';
    return newtext;
}

void reset(void) {
    remove_numbers = FALSE;
    remove_punctuation = FALSE;
    remove_whitespace = FALSE;
    to_uppercase = FALSE;
    to_lowercase = FALSE;
    group_in_fives = FALSE;

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox_remove_numbers), remove_numbers); 
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox_remove_whitespace), remove_whitespace); 
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox_remove_punctuation), remove_punctuation); 
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox_to_uppercase), to_uppercase); 
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox_to_lowercase), to_lowercase); 
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox_group_in_fives), group_in_fives); 
}

/* Plugin implementation */
/* --------------------- */

/* Toggles the global preprocessing options */
static void check_toggle(int option_number) {
    switch (option_number) {
    case REMOVE_NUMBERS:
	TOGGLE(remove_numbers); break;
    case REMOVE_PUNCTUATION:
	TOGGLE(remove_punctuation); break;
    case REMOVE_WHITESPACE:
	TOGGLE(remove_whitespace); break;
    case TO_UPPERCASE:
	TOGGLE(to_uppercase); break;
    case TO_LOWERCASE:
	TOGGLE(to_lowercase); break;
    case GROUP_IN_FIVES:
	TOGGLE(group_in_fives); break;
    default:
	g_warning("Unknown filter type in %s - %d\n", name, option_number); break;
    }
    display_text();
}

static GtkWidget *add_opt(char *txt, int type, GtkWidget *hbox) {
    GtkWidget *opt;
    opt = gtk_check_button_new_with_label(txt);
    switch (type) {
    case REMOVE_NUMBERS:
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(opt), remove_numbers); break;
    case REMOVE_PUNCTUATION:
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(opt), remove_punctuation); break;
    case REMOVE_WHITESPACE:
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(opt), remove_whitespace); break;
    case TO_UPPERCASE:
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(opt), to_uppercase); break;
    case TO_LOWERCASE:
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(opt), to_lowercase); break;
    case GROUP_IN_FIVES:
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(opt), group_in_fives); break;
    default:
	g_warning("Unknown filter type in %s - %d\n", name, type); break;
    }
    gtk_box_pack_start(GTK_BOX(hbox), opt, TRUE, TRUE, 0);
    gtk_signal_connect(GTK_OBJECT(opt), "toggled", cb_check_toggle, GINT_TO_POINTER(type));
    gtk_widget_show(opt);
    return opt;
}

/* Callbacks */
/* --------- */

static void cb_check_toggle(GtkWidget *widget, gpointer gdata) {
    check_toggle(GPOINTER_TO_INT(gdata));
}
