//
// cppopt.h
// C++ Argument Parsing Header File
//
// The function cppopt::parse provided here attempts to parse a string
// by the following rules:
//
// Any leading or trailing spaces or tabs are stripped.
//
// If the first character in the argument list is not '-', then any
// words (separated by spaces unless enclosed by double-quotes) are
// placed in vector<string> cppopt::Options.preopt.  If it is '-',
// then it is parsed as if it were found below as ' -'.
//
// The beginning of an option is marked by ' -' unless it is the very
// first option, as noted above.  The next character is then checked.
// If the second character is '-', then the third character is
// checked.  If the third character is ' ', then that marks the end of
// basic option checking, and everything after the space up until the
// end of the line is stored in the string cppopt::Options.postopt.
//
// If the second character was '-' but the third character was not a
// space, then the option is processed as a long option, the text of
// which up until the next space is stored as the first element of
// map<string,string> cppopt::Options.longopt, and the text following
// that up until the next ' -' or end of line is placed into the
// second element.
//
// If the second character was not '-', then that character is placed
// as the first element of map<char,string> cppopt::Options.shortopt,
// and any following characters up until the next ' -' or end of line
// is placed into the second element, discarding any leading spaces.
//
// Subsequent options that match earlier ones will override the values
// in the earlier ones.
//
// Options are allowed to take one argument, delineated by whitespace.
// If multiple arguments have to be dealth with, use commas, and then
// parse them later.  Extra nonoption elements that are not preceded
// by an option are added to vector<string> cppopt::target
//
//
// Notable quirks:
//
// No distinction is made between "foo bar" and "foo" "bar" as
// adjacent options.  Better modularization might allow for this
// later.
//

#ifndef _CPPOPT_H
#define _CPPOPT_H

#include <map>
#include <string>
#include <vector>

namespace cppopt
{
    class Options
    {
    public:
	vector<string> preopt;
	map<char,string> shortopt;
	map<string,string> longopt;
	vector<string> target;
	string postopt;

	Options();
	~Options();

	void clear();		// Clears all values
	bool empty();		// Tests if any values are present

	// These functions test for the presence of particular option
	// types.  With no argument, they check for the presence of
	// any option at all.  With an argument, they check for the
	// presence of that specific element.
	//
	bool haspreopt();
	bool haspreopt(string opt);
	bool hasshortopt();
	bool hasshortopt(char opt);
	bool haslongopt();
	bool haslongopt(string opt);
	bool hastarget();
	bool hastarget(string opt);
	bool haspostopt();

	// Debugging functions to send to stdout the parsed contents
	void listall();
	void listpreopts();
	void listshort();
	void listlong();
	void listtargets();
    };

    string stripwhitespace(string unstripped);
    Options parse(string optstring);
    Options parse(int argc, char **argv);

}

#endif // _CPPOPT_HPP

// Local variables:
// mode: c++
// End:
